// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o1_jit/code_emitter.h,v 1.2 2001/08/13 09:59:51 xhshi Exp $
//



#ifndef _CODE_EMITTER_H_
#define _CODE_EMITTER_H_

#include "x86.h"
#include "Mem_Manager.h"

#define MAX_X86_INSTRUCTION_LEN     14

struct Profile_Rec ;

class Code_Emitter {
public:
    Code_Emitter(Mem_Manager& m,unsigned byteCodeSize);
    ~Code_Emitter();
	
	//
	// get the number of instruction bytes that have been emitted
	//
	unsigned get_size();

	//
	// copy the instructions into buffer; buffer must be large
	// enough to hold all instructions
	//
	void	copy(char *buffer);

	static unsigned estimate_mem_size(unsigned byteCodeSize);

////////////////////////////////////////////////////////////////////////////////
//
// inc(rement), dec(rement), not, neg(ate) instructions
//
////////////////////////////////////////////////////////////////////////////////
	void emit_inc(const M_Opnd *m) {
		_check_inst_space();
		_next = inc(_next,m);
	}
	void emit_inc(const R_Opnd *r) {
		_check_inst_space();
		_next = inc(_next,r);
	}
	void emit_dec(const M_Opnd *m) {
		_check_inst_space();
		_next = dec(_next,m);
	}
	void emit_dec(const R_Opnd *r) {
		_check_inst_space();
		_next = dec(_next,r);
	}
	void emit_not(const RM_Opnd *rm) {
		_check_inst_space();
		_next = _not(_next,rm);
	}
	void emit_neg(const RM_Opnd *rm) {
		_check_inst_space();
		_next = neg(_next,rm);
	}
	void emit_nop() {
		_check_inst_space();
		_next = nop(_next);
	}
////////////////////////////////////////////////////////////////////////////////
//
// alu instructions: add, or, adc, sbb, and, sub, xor, cmp
//
////////////////////////////////////////////////////////////////////////////////
	void emit_alu(X86_ALU_Opcode opc,const RM_Opnd *rm,const Imm_Opnd *imm) {
		_check_inst_space();
		_next = alu(_next,opc,rm,imm);
	}
	void emit_alu(X86_ALU_Opcode opc,const M_Opnd *m,const R_Opnd *r) {
		_check_inst_space();
		_next = alu(_next,opc,m,r);
	}
	void emit_alu(X86_ALU_Opcode opc,const R_Opnd *r,const RM_Opnd *rm) {
		_check_inst_space();
		_next = alu(_next,opc,r,rm);
	}
////////////////////////////////////////////////////////////////////////////////
//
// test instruction
//
////////////////////////////////////////////////////////////////////////////////
    void emit_test(const RM_Opnd *rm,const Imm_Opnd *imm) {
		_check_inst_space();
		_next = test(_next,rm,imm);
    }
    void emit_test(const RM_Opnd *rm,const R_Opnd *r) {
		_check_inst_space();
		_next = test(_next,rm,r);
    }
////////////////////////////////////////////////////////////////////////////////
//
// shift instructions: shl, shr, sar, shld, shrd
//
////////////////////////////////////////////////////////////////////////////////
	void emit_shift(X86_Shift_Opcode opc,const RM_Opnd *rm,
				const Imm_Opnd *imm) {
		_check_inst_space();
		_next = shift(_next,opc,rm,imm);
	}
	void emit_shift(X86_Shift_Opcode opc,const RM_Opnd *rm) {
		_check_inst_space();
		_next = shift(_next,opc,rm);
	}
	void emit_shift(X86_Shift_Opcode opc,const RM_Opnd *rm,
				const R_Opnd *r, const Imm_Opnd *imm) {
		_check_inst_space();
		_next = shift(_next,opc,rm,r,imm);
	}
	void emit_shift(X86_Shift_Opcode opc,const RM_Opnd *rm,
	                        const R_Opnd *r) {
		_check_inst_space();
		_next = shift(_next,opc,rm,r);
	}
////////////////////////////////////////////////////////////////////////////////
//
// multiply instructions: mul, imul
//
////////////////////////////////////////////////////////////////////////////////
	void emit_mul(const RM_Opnd *rm,int is_signed) {
		_check_inst_space();
		_next = mul(_next,rm,is_signed);
	}
	void emit_imul(const R_Opnd *r,const RM_Opnd *rm) {
		_check_inst_space();
		_next = imul(_next,r,rm);
	}
	void emit_imul(const R_Opnd *r,const Imm_Opnd *imm) {
		_check_inst_space();
		_next = imul(_next,r,imm);
	}
	void emit_imul(const R_Opnd *r,const RM_Opnd *rm,
			   const Imm_Opnd *imm) {
		_check_inst_space();
		_next = imul(_next,r,rm,imm);
	}
////////////////////////////////////////////////////////////////////////////////
//
// divide instructions: div, idiv
//
////////////////////////////////////////////////////////////////////////////////
	void emit_div(const RM_Opnd *rm,int is_signed) {
		_check_inst_space();
		_next = div(_next,rm,is_signed);
	}
////////////////////////////////////////////////////////////////////////////////
//
// data movement: mov
//
////////////////////////////////////////////////////////////////////////////////
	void emit_mov(const M_Opnd *m,const R_Opnd *r,X86_Opnd_Size sz=opnd_32) {
		_check_inst_space();
		_next = mov(_next,m,r,sz);
	}
	void emit_mov(const R_Opnd *r,const RM_Opnd *rm) {
		_check_inst_space();
		_next = mov(_next,r,rm);
	}
	void emit_mov(const R_Opnd *r,const Imm_Opnd *imm) {
		_check_inst_space();
		_next = mov_imm32(_next,r,imm->value);
	}
	void emit_mov(const M_Opnd *m,const Imm_Opnd *imm,X86_Opnd_Size sz=opnd_32) {
		_check_inst_space();
		_next = mov(_next,m,imm,sz);
	}
////////////////////////////////////////////////////////////////////////////////
//
// load effective address: lea
//
////////////////////////////////////////////////////////////////////////////////
	void emit_lea(const R_Opnd *r,const M_Opnd *m) {
		_check_inst_space();
		_next = lea(_next,r,m);
	}
////////////////////////////////////////////////////////////////////////////////
//
// conversions, i.e., widening instructions
//
////////////////////////////////////////////////////////////////////////////////
	void emit_widen(const R_Opnd *r,const RM_Opnd *rm,
				unsigned is_signed,unsigned is_half) {
		_check_inst_space();
		_next = widen(_next,r,rm,is_signed,is_half);
	}
	void emit_cdq() {
		_check_inst_space();
		_next = cdq(_next);
	}
////////////////////////////////////////////////////////////////////////////////
//
// floating-point instructions
//
////////////////////////////////////////////////////////////////////////////////
	//
	//		st(0) = st(0) fp_op m{32,64}real
	//
	void emit_fp_op_mem(X86_FP_Opcode opc,const M_Opnd *mem,int is_double) {
		_check_inst_space();
		_next = fp_op_mem(_next,opc,mem,is_double);
	}
	//
	//		st(0) = st(0) fp_op st(i)
	//
	void emit_fp_op(X86_FP_Opcode opc,unsigned i) {
		_check_inst_space();
		_next = fp_op(_next,opc,i);
	}
	//
	//		st(i) = st(i) fp_op st(0)	; optionally pop stack
	//
	void emit_fp_op(X86_FP_Opcode opc,unsigned i,unsigned pop_stk) {
		_check_inst_space();
		_next = fp_op(_next,opc,i,pop_stk);
	}
	//
	//		compare st(0),st(1) and pop stack twice
	//
	void emit_fcompp() {
		_check_inst_space();
		_next = fcompp(_next);
	}
	void emit_fnstsw() {
		_check_inst_space();
		_next = fnstsw(_next);
	}
	void emit_fnstcw(const M_Opnd *mem) {
		_check_inst_space();
		_next = fnstcw(_next,mem);
	}
	void emit_fldcw(const M_Opnd *mem) {
		_check_inst_space();
		_next = fldcw(_next,mem);
	}
	void emit_fchs() {
		_check_inst_space();
		_next = fchs(_next);
	}
	void emit_frem() {
		_check_inst_space();
		_next = frem(_next);
	}
	void emit_fxch(unsigned i) {
		_check_inst_space();
		_next = fxch(_next,i);
	}
	//
	// load single (double) floating point from memory
	// into fp register stack
	//
	void emit_fld(const M_Opnd *mem,int is_double) {
		_check_inst_space();
		_next = fld(_next,mem,is_double);
	}
	//
	// load extended double (80 bits) from memory
	// into fp register stack
	//
	void emit_fld80(const M_Opnd *mem) {
		_check_inst_space();
		_next = fld80(_next,mem);
	}
	//
	// load (long) int from memory into fp register stack
	//
	void emit_fild(const M_Opnd *mem,int is_long) {
		_check_inst_space();
		_next = fild(_next,mem,is_long);
	}
	//
	// push st(i) onto fp register stack
	//
	void emit_fld(unsigned i) {
		_check_inst_space();
		_next = fld(_next,i);
	}
	//
	// push the constants 0.0 and 1.0 onto the fp register stack
	//
	void emit_fldz() {
		_check_inst_space();
		_next = fldz(_next);
	}
	void emit_fld1() {
		_check_inst_space();
		_next = fld1(_next);
	}

	//
	// store single (double) floating point value from FP
	// register stack to memory, optionally popping the stack
	//
	void emit_fst(const M_Opnd *mem,int is_double,unsigned pop_stk) {
		_check_inst_space();
		_next = fst(_next,mem,is_double,pop_stk);
	}
	//
	// store (long) int from FP regsiter stack to memory and
	// pop the register stack
	//
	void emit_fist_pop(const M_Opnd *mem,int is_long) {
		_check_inst_space();
		_next = fist_pop(_next,mem,is_long);
	}

	//
	// st(0) = st(i)
	// Pop top of stack st(0)
	void emit_fstp(unsigned i) {
		_check_inst_space();
		_next = fstp(_next,i);
        }

////////////////////////////////////////////////////////////////////////////////
//
// stack push and pop instructions
//
////////////////////////////////////////////////////////////////////////////////
	void emit_push(const M_Opnd *rm) {
		_check_inst_space();
		_next = push(_next,rm);	
	}
	void emit_push(const Imm_Opnd *imm) {
		_check_inst_space();
		_next = push(_next,imm);
	}
	void emit_push(const R_Opnd *r) {
		_check_inst_space();
		_next = push(_next,r);
	}
	void emit_pop(const R_Opnd *r) {
		_check_inst_space();
		_next = pop(_next,r);
	}
	void emit_pop(const M_Opnd *mem) {
		_check_inst_space();
		_next = pop(_next,mem);
	}
////////////////////////////////////////////////////////////////////////////////
//
// control-flow instructions
//
////////////////////////////////////////////////////////////////////////////////
	//
	// jump with 32-bit relative
	//
	void emit_jump32(const Imm_Opnd *imm) {
		_check_inst_space();
		_next = jump32(_next,imm);
	}
	//
	// jump with 8-bit relative
	//
	void emit_jump8(const Imm_Opnd *imm) {
		_check_inst_space();
		_next = jump8(_next,imm);
	}
	//
	// register indirect jump
	//
	void emit_jump(const RM_Opnd *rm) {
		_check_inst_space();
		_next = jump(_next,rm);
	}
	//
	// jump to target address
	//
	void emit_jump(char *target) {
		_check_inst_space();
		_next = jump(_next,target);
	}
	//
	// jump with displacement
	//
	void emit_jump(int disp) {
		_check_inst_space();
		_next = jump(_next,disp);
	}
	//
	// conditional branch with 8-bit branch offset
	//
	void emit_branch8(X86_CC cc,
					   const Imm_Opnd *imm,unsigned is_signed) {
		_check_inst_space();
		_next = branch8(_next,cc,imm,is_signed);
	}
	//
	// conditional branch with 32-bit branch offset
	//
	void emit_branch32(X86_CC cc,
						const Imm_Opnd *imm,unsigned is_signed) {
		_check_inst_space();
		_next = branch32(_next,cc,imm,is_signed);
	}
	//
	// conditional branch
	//
	void emit_branch(X86_CC cc,char *target,unsigned is_signed) {
		_check_inst_space();
		_next = branch(_next,cc,target,is_signed);
	}
	//
	// conditional branch with displacement
	//
	void emit_branch(X86_CC cc,int disp,unsigned is_signed) {
		_check_inst_space();
		_next = branch(_next,cc,disp,is_signed);
	}

	//
	// call with displacement
	//
	void emit_call(const Imm_Opnd *imm) {
		_check_inst_space();
		_next = call(_next,imm);
	}
	//
	// indirect call through register or memory location
	//
	void emit_call(const RM_Opnd *rm) {
		_check_inst_space();
		_next = call(_next,rm);
	}
	//
	// call target address
	//
	void emit_call(char *target) {
		_check_inst_space();
		_next = call(_next,target);
	}
	//
	// return instruction
	//
	void emit_ret() {
		_check_inst_space();
		_next = ret(_next);
	}
	void emit_ret(Imm_Opnd *imm) {
		_check_inst_space();
		_next = ret(_next,imm);
	}
////////////////////////////////////////////////////////////////////////////////
//
// stack frame allocation instructions: enter & leave
//
////////////////////////////////////////////////////////////////////////////////
	//
	//	enter frame_size 
	//
	void emit_enter(const Imm_Opnd *imm) {
		_check_inst_space();
		_next = enter(_next,imm);
	}
	//
	// leave
	//
	void emit_leave() {
		_check_inst_space();
		_next = leave(_next);
	}
////////////////////////////////////////////////////////////////////////////////
//
// load SF, ZF, AF, PF, and CF flags from eax
//
////////////////////////////////////////////////////////////////////////////////
	void emit_sahf() {
		_check_inst_space();
		_next = sahf(_next);
	}
////////////////////////////////////////////////////////////////////////////////
//
// methods for reserving space and for allocating space for data objects
//
////////////////////////////////////////////////////////////////////////////////
	void reserve_inst_bytes(unsigned nbytes) {
		//
		// make sure there is enough space for nbytes instructions
		// must come from code space
		//
        if (_next + nbytes > _end)
            _alloc_arena(default_arena_size);

		_next += nbytes;
	}
	//
	// get the offset of the next instruction 
	//
	unsigned get_offset() {
		return _arena_begin_offset + (_next - _arena->bytes);
	}
	//
	// get the next pointer (used for patching)
	//
	char *get_next() {
		return _next;
	}

private:
	char *_copy(char *buffer,Arena *a);
	void _check_inst_space() {
	//
	// _check_inst_space() makes sure there is enough space
	// to emit an x86 instruction
	//
	// an x86 instruction is at most 14 bytes; if we don't have
	// this much space left, then allocate a new arena 
	//
		if (_next + MAX_X86_INSTRUCTION_LEN > _end)
            _alloc_arena(default_arena_size);
	}
	void	_alloc_arena(unsigned nbytes);

	//
	// offset of first instruction in _arena relative to beginning 
	//	of generated code
	//
	unsigned _arena_begin_offset;	
	Arena *_arena;				// code arena
	char *_next;				// address for next code emission
	char *_end;					// guard address for code
	Mem_Manager& _mem_manager;	// memory manager for allocating arenas
	//
	// For statistics, we need a "unsigned* offset_buf" array to record inner-instrumented counter's address 
	// and it's offset in the method.
	// The size of this array must be ensured for large enough from outside
	// the unsiged variable "offset_buf_offset" is used as the length-counter of the array
	//
public:
	unsigned* offset_buf ;
	unsigned  offset_buf_offset ;
	struct Profile_Rec* prof_rec ;		   // point to current prof_rec.
	unsigned  inner_bb_cnt_offset ;// offset of inner basic block counter.
};

#endif // _CODE_EMITTER_H_
