// Copyright (C)  2000 Intel Corporation.  All rights reserved.
// 
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o1_jit/o1_code_gen.cpp,v 1.4 2001/11/10 05:21:02 gwu2 Exp $
//


#include "defines.h"
#include <iostream.h>
#include <string.h>
#include "jit_intf.h"
#include "internal_jit_intf.h"
#include "jit.h"
#include "Mem_Manager.h"
#include "code_emitter.h"
#include "code_gen.h"
#include "cg_prepass.h"
#include "stack.h"
#include "operand.h"
#include "lazy_code_selector.h"
#include "gc_tags.h"
#include "jcfg.h"
#include "regalloc.h"
#include "register_allocator.h"
#include "profiling.h"
#include "level_1a_jit_intf.h"
#include "jit_common.h"

#ifdef DEBUG_L1A
#include "mtable.h"
#endif // DEBUG_L1A

#ifdef VTune_Support
#include "vtune.h"
#endif // VTune_Support

Jit_Method_Info *o1_method_get_info_block(Method_Handle mh) {
	Small_Method_Info *smi = (Small_Method_Info *)method_get_info_block(mh,o1_jit);
    assert(smi != NULL);
	if (smi->mi == NULL) { // generate method info lazily
		JIT_Result res = o1_jit->gen_method_info(
                                 jit_get_comp_handle(o1_jit), mh,
                                 o1_jit->jit_flags);
		assert(res == JIT_SUCCESS);
	}
    assert(smi->mi != NULL);
	return smi->mi;
}

static unsigned estimate_method_info_size(unsigned num_call_sites,
										  unsigned maxStack,
										  unsigned n_vars)
{
	// size of Jit_Method_Info
	unsigned size = sizeof(Jit_Method_Info);
	// plus size of Call_Site_Info array (first element within Jit_Method_Info)
	if (num_call_sites > 1)
		size += (num_call_sites - 1) * sizeof(Call_Site_Info);
	// plus size of type vectors
	size += n_vars;								// var_type_vector
	size += num_call_sites * (maxStack << 1);	// call site type_vectors; num_out_args <= maxStack
	// Esp_Record array is currently a linked list; the space is allocated elsewhere.
	// the bytecode array is not needed until the permanent method_info is generated.
	return size;
}

static void init_method_info(Jit_Method_Info *method_info,
                             Register_Allocator *regalloc,
                             const unsigned char *bc,
                             unsigned bc_length,
                             unsigned num_call_sites,
                             unsigned maxStack,
                             unsigned n_gc_tag_words,
                             unsigned n_arg_words,
                             unsigned n_vars,
                             unsigned callee_saved_reg_count,
                             const char *method_name,
							 Method_Handle methodHandle,
                             Profile_Rec *prof_rec)
{
	method_info->name = method_name;
	method_info->cnt = 0;
	method_info->num_spills = maxStack;
	method_info->num_gc_tag_words = n_gc_tag_words;
	method_info->num_in_args = n_arg_words;
    method_info->num_vars = n_vars;
    method_info->num_call_sites = num_call_sites;
    method_info->is_esp_based = (method_get_num_handlers(methodHandle) == 0);
    method_info->num_callee_saved_regs = callee_saved_reg_count;
    method_info->code = bc;
    method_info->code_len = bc_length;
    regalloc->init_rra(method_info->rra);
    regalloc->update_method_info(method_info);
#ifdef VAR_CLONING
    method_info->ref_var_bitmap = 0;
#endif //VAR_CLONING
	method_info->runtime_throws = NULL;

//    method_info->prof_rec = prof_rec;

    method_info->frame_size = method_info->num_in_args +
							  method_info->num_vars +
                              method_info->num_spills +
							  method_info->num_callee_saved_regs +
                              method_info->num_gc_tag_words +
							  1 /* return IP */;
	if (method_info->is_esp_based == 0) method_info->frame_size++;	/* ebp */

	// type_vectors go directly after the cs_info array since there is
	// not an Esp_Record array at this point
	char *_ptr = (char *)&method_info->cs_info[num_call_sites];
	// var_type_vector
	method_info->var_type_vector = _ptr;
	for (unsigned j = 0; j < n_vars; j++)
		*_ptr++ = NO_TYPE_CHAR;
	// type_vectors
	for (unsigned i = 0; i < num_call_sites; i++) {
		Call_Site_Info *csi = &method_info->cs_info[i];
		csi->stack_depth = csi->num_out_args = csi->ret_IP = csi->call_IP =
            csi->precall_IP = csi->num_records = 0;
		csi->esp_record = NULL;
		// initialize type_vector
		csi->type_vector_length = maxStack;
		csi->type_vector = _ptr;
#ifdef LOCAL_CALLEE
		csi->stack_ref_in_local_callee = 0;
#endif
		csi->is_caller_pop = 0;
		csi->bc = NULL;
        csi->outarg_bv = 0;
        csi->m_handle = NULL;
        csi->returns_ref = 0;
		for (unsigned j = 0; j < csi->type_vector_length; j++)
			*_ptr++ = NO_TYPE_CHAR;
	}
}

static void set_bytecode_visited_info(Jit_Method_Info *method_info,
									  //Mem_Manager &mem_manager,
									  Bit_Vector *visited)
{
	//method_info->is_visited = new(mem_manager) Bit_Vector(mem_manager, visited);
	method_info->is_visited = visited;
}

const char *O1a_envvar_METHODS = NULL;
JIT_Result	O1a_gen_code(
	Compile_Handle   compilationHandle,
    Class_Handle  classHandle,
    Method_Handle methodHandle,
    const BYTE *  byteCodeAddr,
    size_t        byteCodeSize,

    unsigned      maxLocals,
    unsigned      maxStack,
	JIT_Flags     flags,
    Dbg           *dbg_support,
	CODE_MI       code_mi) {

	const char *method_name = method_get_name(methodHandle);
	const char *class_name = class_get_name(classHandle);

#ifdef DEBUG_L1A
    static O3_Method_Table *mtable = NULL;
    static unsigned call_count=0;
    if (call_count++==0)
        mtable = new O3_Method_Table(O1a_envvar_METHODS, "METHODS", true);
	//
	// first decide whether this method should be jitted
	//
    if (!mtable->accept_this_method(methodHandle))
        return JIT_FAILURE;
    cout << "O1a compiles " << class_name << "." << method_name << endl;
#endif // DEBUG_L1A


    //
    // Call VM to determine if the method is fp_strict_mode
    //
    bool fp_strict_mode = false;
    //
    // For Java FP spec, we need to save/restore FPU control word for double 
    // operations so one extra word is reserved.
    //
    if (fp_strict_mode)
        maxStack += 2;

	unsigned size_estimate = CG_Prepass::estimate_mem_size(byteCodeSize,maxStack,maxLocals);
	size_estimate += Code_Emitter::estimate_mem_size(byteCodeSize);
	size_estimate += Operand::estimate_mem_size(byteCodeSize,maxStack);


	Mem_Manager mem_manager(size_estimate);
	Code_Emitter code_emitter(mem_manager,byteCodeSize);

    // CG_Prepass needs access to at least one Register_Allocator method,
    // so we need to create the Register_Allocator object first.
    Register_Allocator *regalloc;
#if (REG_ALLOC_METHOD == REG_ALLOC_NONE)
    regalloc = new(mem_manager) Register_Allocator_None(mem_manager);
#endif // REG_ALLOC_METHOD
#if (REG_ALLOC_METHOD == REG_ALLOC_SIMPLE)
    regalloc = new(mem_manager) Register_Allocator_Simple(mem_manager);
#endif // REG_ALLOC_METHOD
#if (REG_ALLOC_METHOD == REG_ALLOC_CHOW)
    regalloc = new(mem_manager) Register_Allocator_Priority(mem_manager);
#endif // REG_ALLOC_METHOD

    CG_Prepass prepass(maxLocals,byteCodeAddr,byteCodeSize,classHandle,methodHandle,
        compilationHandle,mem_manager,maxStack,regalloc);
    if (prepass.prepass_failed) {
        return JIT_FAILURE;
    }

	//
	// we are using extra words for gc tagging of locals bitvector
	//
    unsigned n_gc_tag_words = prepass.n_aloaded > 0 ? GC_Tags_n_words(maxLocals) : 0;
    
    unsigned callee_saved_reg_count;
    callee_saved_reg_count =
        regalloc->register_allocate(&prepass,methodHandle,classHandle,
        maxLocals,byteCodeAddr,byteCodeSize);
    
    unsigned n_arg_words = n_words_of_method_arg_type(methodHandle);
    if (!method_is_static(methodHandle))
        // add one more word for the this pointer
        n_arg_words++;

	// if a method has N args but uses only K args ( N > K), then the case that
	// maxLocals < n_arg_words may happen.
	//
    if(maxLocals < n_arg_words)
        maxLocals = n_arg_words; 

	unsigned n_vars = maxLocals - n_arg_words;

    //
    // create profiling record
    //
	Profile_Rec *prof_rec = NULL;
	if (instrumenting) {
		//::
		// If we want "statistics", instrument every BB!
		// So we need a "bigger" slot for prof_rec!
		//::
		int bb_num = statistics ? prepass.num_blocks : prepass.num_entries_back_edge ;
		if (code_mi != cm_gen_method){ 
			prof_rec = create_profile_rec(methodHandle, compilationHandle,
										  bb_num,
										  byteCodeSize);
		}else {
			//
			// profiling record has been generated already.  We allocate a temporay
			// profiling record just for the purpose of producing method info
			//
			unsigned sz = estimate_prof_rec_size(bb_num);
			prof_rec = (Profile_Rec*)mem_manager.alloc(sz);
		}
	}

	// method info block
	unsigned mi_size_estimate = estimate_method_info_size(prepass.num_call_sites,maxStack,n_vars);
	Mem_Manager info_manager(mi_size_estimate);
	Jit_Method_Info *method_info = (Jit_Method_Info *)info_manager.alloc(mi_size_estimate);
	init_method_info(method_info, regalloc, byteCodeAddr, byteCodeSize,
        prepass.num_call_sites, maxStack, n_gc_tag_words, n_arg_words,
        n_vars, callee_saved_reg_count, method_name, methodHandle, prof_rec);

	//set_bytecode_visited_info(method_info, info_manager, prepass.is_visited);
	set_bytecode_visited_info(method_info, prepass.is_visited);

#ifdef VAR_CLONING
	char *local_var_info = prepass.local_var_info;
	if(prepass.var_cloning_failed) {
		method_info->ref_var_bitmap = INVALID_RV_MASK;
	} else if(maxLocals > MAX_REF_VARS) {
		method_info->ref_var_bitmap = INVALID_RV_MASK;
	} else {
		bool lv_conflict = false;
		Ref_Var_Bitmap_Type mask = 1;
		Ref_Var_Bitmap_Type rbt = 0;
		if (method_is_static(methodHandle)) {
			rbt = 0;
		} else {
			rbt = 1;
		}
		for (unsigned j=0; j < maxLocals; j++, mask <<= 1) {
			unsigned lvi_int = local_var_info[j];
			Local_Var_Info &lvi = *((Local_Var_Info*)(&lvi_int));
			if(lvi.ref_var) {
				rbt |= mask;
				if(lvi.non_ref_var) {
					lv_conflict = true;
					break;
				}
			}
		}
		if(lv_conflict) {
			method_info->ref_var_bitmap = INVALID_RV_MASK;
		} else {
			method_info->ref_var_bitmap = rbt;
		}
	}
#endif //VAR_CLONING
	// choose esp- or ebp-based frame
	ESP_Frame	esp_frame(n_arg_words,n_vars,
        n_gc_tag_words,maxStack,callee_saved_reg_count);
	EBP_Frame	ebp_frame(n_arg_words,n_vars,
        n_gc_tag_words,maxStack,callee_saved_reg_count);

	Frame *frame;
	if (method_info->is_esp_based)
		frame = &esp_frame;
	else
		frame = &ebp_frame;
	// pre-allocate commonly used operands so as to avoid allocating them duing the
	// lazy code generation phase
	Pre_Alloc_Operand_Pool op_pool(*frame,maxStack,mem_manager);

	Stack	stack(mem_manager,code_emitter,op_pool,*frame,maxStack,prepass,fp_strict_mode);
	JIT_Result res;

#ifdef VTune_Support
	iJIT_Method_Load* pInfo;
	pInfo = (iJIT_Method_Load*)malloc(sizeof(iJIT_Method_Load)) ;
    iJIT_Method_Load& mInfo = (*pInfo);
#endif // VTune_Support

	res = select_code(mem_manager, code_emitter, prepass,
		  *frame, stack, op_pool, compilationHandle, classHandle,
		  methodHandle, byteCodeAddr, byteCodeSize, flags, 
		  method_info, regalloc, prof_rec, dbg_support, code_mi
#ifdef VTune_Support
        , &mInfo
#endif // VTune_Support
		);


#ifdef VTune_Support
    if (code_mi != cm_gen_method && res==JIT_SUCCESS) {
        if (VTuneModeFlags & iJIT_BE_NOTIFY_ON_LOAD) {
            int notify_VTune =
                iJIT_NotifyEvent(iJVM_EVENT_TYPE_METHOD_LOAD_FINISHED, (void *)&mInfo);
        }
    }
#endif // VTune_Support

	return res;
}

