// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o3_jit/flow_graph.h,v 1.6 2002/01/10 03:12:09 xhshi Exp $
//

#ifndef _FLOW_GRAPH_H_
#define _FLOW_GRAPH_H_

#define COMPACT_NODE

#include <assert.h>
#include "Mem_Manager.h"
#include "cg_prepass.h"
#include "ir.h"
#include "dlink.h"
#include "x86_emitter.h"
#include "overridden.h"
#include "internal_jit_intf.h"

// calculate a signed integer offset based on an array of unsigned chars.
#define OFFSET2(array,idx) ((((char*)array)[idx] << 8) | (array)[(idx)+1])
#define OFFSET4(array,idx) (((array)[idx] << 24) | ((array)[(idx)+1] << 16) | \
                            ((array)[(idx)+2] << 8) | (array)[(idx)+3])
#define NO_BC_IDX (0x12345678)

// Save memory by setting Cfg_Int to "short".  Don't make it
// any smaller, or things will break.  For example, if a "try" region
// contains many basic blocks, there can be many edges that come into
// an Eh_Node.  Also, don't make it "unsigned".
typedef short Cfg_Int;

class Cfg_Node;
class Cfg_Node_List;
class Eh_Node;
class Eh_In_Header;
class Expressions;
class Stack;
class Inst;
class Live_LCSE;
class Bit_Vector;
class Bit_Vector_Group;
class GCTrack_Operand;
class Back_Edge;
class Work_Set;
class WL_Node;
struct O3_Profile_Rec;
struct Profile_Rec;

class Closure {};

typedef	void (*Apply_Func)(Cfg_Node *node, Closure *c);

#ifdef O3_VTune_Support
class O3_VTune_Closure: public Closure{
public:
    O3_VTune_Closure(Expressions& e, Flow_Graph *fg)
        : exprs(e), fg(fg){}
    Expressions& exprs;
    Flow_Graph *fg;
} ;
#endif

#ifdef STAT_INDIRECT_CALL
class Instrument_Closure: public Closure{
public:
    Instrument_Closure(Expressions& e, Flow_Graph *fg)
        : exprs(e), fg(fg){}
    Expressions& exprs;
    Flow_Graph *fg;
} ;
#endif

struct bvp_homeloc_mapping
{
    bvp_homeloc_mapping(Mem_Manager &mem, unsigned nb) : bvps(NULL), opnds(NULL), mem(mem),
        n_bits(nb), b_size(0), b_capacity(0), o_size(0), o_capacity(0), frame_offsets(NULL) {}
    void *operator new(size_t sz, Mem_Manager& m) { return m.alloc(sz); }
    unsigned *bvps;
    GCTrack_Operand **opnds;
    int b_size, b_capacity, o_size, o_capacity;
    int *frame_offsets;
    bool *use_var_offset;
    Mem_Manager &mem;
    unsigned n_bits;
};

class Cfg_Node_List : public Dlink
{
public:
    void *operator new(size_t sz, Mem_Manager& m) { return m.alloc(sz); }
    Cfg_Node_List() {Cfg_Node_List(NULL);}
    Cfg_Node_List(Cfg_Node *n): _node(n) {}
    Cfg_Node *node() { return _node; }
    Cfg_Node_List *next() {return (Cfg_Node_List *)_next;}
    Cfg_Node_List *prev() {return (Cfg_Node_List *)_prev;}
private:
    Cfg_Node *_node;
};

struct Eh_Entry
{
  Class_Handle class_handle;  // a pointer to the class of the exception object
  unsigned cp_index;  // constant pool entry of class_handle
  Cfg_Node *handler;
};

class Eh_Node : public Dlink
{
    friend class Cfg_Node;
public:
    void *operator new(size_t sz, Mem_Manager& m) { return m.alloc(sz); }
    Eh_Node(int lbl);
    Eh_Node() { Eh_Node(-1); }

    Cfg_Node *in_edges(Cfg_Int i) { return _in_edges[i]; }
    struct Eh_Entry *out_edges(Cfg_Int i) { return &_out_edges[i]; }
    Cfg_Int in_edge_size() { return _in_edge_size; }
    Cfg_Int out_edge_size() { return _out_edge_size; }
    
    void add_edge(Mem_Manager &mm, Cfg_Node *dest, void *type, unsigned cp_idx);
    void delete_edge(Cfg_Node *dest);
    bool is_identical(Eh_Node *node);
    Cfg_Node *bounds_exception() { return _bounds_exception; }
    void set_bounds_exception(Cfg_Node *n) {
        assert(_bounds_exception == NULL); _bounds_exception = n; }
    Eh_Node *next() {return (Eh_Node *)_next;}
    Eh_Node *prev() {return (Eh_Node *)_prev;}
    
    int label;
#ifdef PRINTABLE_O3
    int unique_label;
#endif // PRINTABLE_O3
    unsigned short latest_traversal;
    Bit_Vector_Group *live;
    Cfg_Node *idom;

private:
    Cfg_Node **_in_edges;
    struct Eh_Entry *_out_edges;
    Cfg_Int _in_edge_capacity;  // initially 4
    Cfg_Int _in_edge_size;
    Cfg_Int _out_edge_capacity;  // initially 1
    Cfg_Int _out_edge_size;
    Cfg_Node *_bounds_exception;
};

class Flow_Graph
{
public:
  Flow_Graph(const unsigned char *bc, unsigned code_length,
             unsigned maxLocals, CG_Prepass *prepass,
             Mem_Manager &m,
             Cfg_Int inline_depth,
             unsigned maxStack,
             Eh_Node *caller_eh, // Eh_Node of the caller, for inlining
             Cfg_Node *caller_subr,
             Flow_Graph *caller_fg,
             unsigned caller_bc_idx,
             Compile_Handle cmpl_handle, Method_Handle methodHandle, Class_Handle classHandle);
  ~Flow_Graph() {}
  void *operator new(size_t sz, Mem_Manager& m) { return m.alloc(sz); }

  void Build_IR(Compile_Handle compilation_handle,
                Expressions &expressions,
                Stack&        stack,
                bool gc_requires_write_barriers);
  void Build_IR_node(Cfg_Node *root,
                     Compile_Handle compilation_handle,
                     Expressions &expressions,
                     Stack&        stack,
                     char *stack_sig,
                     int stack_depth,
                     Mem_Manager &build_mm,
					 Cfg_Node       *prev_bb,
                     int is_extended_bb,
                     bool gc_requires_write_barriers);
  void Build_IR_eh(Eh_Node *root,
                   Compile_Handle compilation_handle,
                   Expressions &expressions,
                   Stack&        stack,  // mimic stack
                   Mem_Manager &build_mm,
                   bool         gc_requires_write_barriers);
  const unsigned char *bytecodes() { return _bytecodes; }
  unsigned bc_length() { return _bc_length; }
  Method_Handle m_handle() { return _m_handle; }
  Class_Handle c_handle() { return _c_handle; }
  Compile_Handle cmpl_handle() { return _cmpl_handle; }
  CG_Prepass *prepass() { return _prepass; }
  Cfg_Node *epilog() { return _epilog; }
  void make_traversal_number_safe(Flow_Graph *fg) {
      if (fg->traversal_number > traversal_number)
          traversal_number = fg->traversal_number;
  }

  void apply(Apply_Func fun, Closure *c, bool traverse_sub_graph = false);
//#ifdef INLINE_NATIVE
  Cfg_Node* new_cfg_node(Cfg_Node* node) ;
//#endif
  Cfg_Node *split_cfg_node(Cfg_Node *node);
  Cfg_Node *split_cfg_node_pred(Cfg_Node *node);
  Cfg_Node *splice_cfg_nodes(Cfg_Node *pred, Cfg_Node *succ);
  Cfg_Node *prolog() { return nodes; }
  Eh_Node *handlers() { return &_handlers; }
  Eh_Node *create_eh_node();
  void merge_eh_info(Flow_Graph *merged_fg);
  void dead_code_eliminate(Expressions &exprs, bool compute_live_refs);
  // set idom for each node
  void build_dom_tree();
  // loop transformations
  unsigned loop_entry_normalization();
  unsigned loop_exit_normalization();
  unsigned loop_linearization();
  Cfg_Node *peel_loop(Cfg_Node *header, unsigned peel_factor, Mem_Manager &mm, Back_Edge &be, int block_limit=50);
  Cfg_Node *unroll_loop(Cfg_Node *header, unsigned unroll_factor, Mem_Manager &mm, Back_Edge &be, int block_limit=50);
  void find_loop_depths_and_headers(); 
  void duplicate_loop_headers();
  void set_callee_saved_regs(unsigned regs) { _callee_saved_registers_used = regs; }
  unsigned callee_saved_registers_used() { return _callee_saved_registers_used; }
  void set_home_locations(int n);
  int num_home_locations() { return _num_home_locations; }
  void emit_code(O3_Emitter& xe, X86_Opnd_Pool& xp, Frame& frame, Expressions &exprs);
  char *code_block;
  Cfg_Node *create_bounds_exception_block(Cfg_Node *orig_node);
  Cfg_Node *bounds_exception() { return _bounds_exception; }
  void set_bounds_exception(Cfg_Node *n) {
      assert(_bounds_exception == NULL); _bounds_exception = n; }
  void register_exceptions();
  unsigned short traversal_num() {return traversal_number;}
  void set_traversal_num(unsigned short n) {
      /*assert(n >= traversal_number);*/ traversal_number = n;
  }
  void set_need_linearization() {_need_linearization = true;}
  bool need_linearization() {return _need_linearization;}
  int reassign_label();
  void linearize();
  void prune_unreachable();
  void remove_empty_blocks();
  void create_dataflow_ordering(Mem_Manager &mm, Cfg_Node **&nodearray,
                                int &num_nodes);
  bvp_homeloc_mapping *bvpmap;
  Mem_Manager &mem_manager;
  Operand *this_pointer_of_method;
  void home_location_assignment();
  void set_prof_rec(Profile_Rec *rec) {_prof_rec = rec;}
  Profile_Rec *prof_rec() {return _prof_rec;}

  Flow_Graph *calling_fg;
#ifdef PRINTABLE_O3
  void print_cfg(const char *suffix);
  void dumpjit_dfs(ostream &cout);
  void dumpjit_dfs_node(Cfg_Node *root, ostream &cout);
  void dumpjit_dot_node(Cfg_Node *root, ostream &cout);
  void dumpjit_dot_dfs_bounds_nodes(ostream &cout);
  void dumpjit_dot_hierarchical(ostream &cout);
  unsigned calling_bc_idx;
  char *prefix_str;  // inlining prefix to print

  // The following are just for printing the inlining hierarchy
  // in the dot files.
  Cfg_Node **nodes_in_fg;
  int nodes_in_fg_size, nodes_in_fg_capacity;
  Flow_Graph **fgs_in_fg;
  int fgs_in_fg_size, fgs_in_fg_capacity;

#endif // PRINTABLE_O3
  Cfg_Node_List linear_node_ordering; // created by linearize()
  int num_fg_nodes;
  O3_Profile_Rec *o3_prof_rec;
#ifdef PLDI_OVERRIDDEN
  Overridden_Rec  *overridden_rec;
#endif
  Inst *first_push_inst, *first_pop_inst, *return_inst;
  bool remove_all_bounds_checks;
  bool has_fp;
  bool has_virtual_inline;
  unsigned num_jsr;
  //
  // For O3 statistics
  //
  void O3_statistics_dump(ostream& cout) ;
  unsigned* inner_counter ;	   //for inner bb instrument
  unsigned  inner_counter_num ;//for inner bb instrument

private:
  Cfg_Int inlining_depth;  // for debugging output
  const unsigned char *_bytecodes;
  unsigned _bc_length;
  Method_Handle _m_handle;
  Class_Handle _c_handle;
  Compile_Handle _cmpl_handle;
  CG_Prepass *_prepass;
  Cfg_Node *nodes;
  Cfg_Node *_epilog;
  Eh_Node _handlers;
  Profile_Rec *_prof_rec; // profiling info collected by O1
  Cfg_Node *create_flow_graph(unsigned bc_start, Return_Address_Tracking *stk);
  Cfg_Node *create_epilog(Expressions &exprs, Cfg_Node *pred, Cfg_Node *enclosing_subr);
  void add_exception_info(Eh_Node *caller_eh);
  void create_flow_graph_handlers(Return_Address_Tracking *stk);
  void vm_get_nth_eh_info(Eh_Node *caller_eh, unsigned nth,
                          unsigned &begin, unsigned &end,
                          Cfg_Node *&handler, Class_Handle &type,
                          unsigned &cp_index,
                          Return_Address_Tracking *stk);
  void eliminate(Cfg_Node *node, Expressions &exprs, GCTrack_Operand *where_is_ret_addr,
      Bit_Vector_Group *&tmp, Bit_Vector_Group *&region_tmp, 
      Bit_Vector *scratch, Operand *sync_this_ptr, Bit_Vector *node_bv);
  void mark_subroutines(Cfg_Node *caller_subr);
  void mark_subroutines_node(Cfg_Node *node, Cfg_Node *subr);
  void mark_subroutines_eh(Eh_Node *eh);
  unsigned short traversal_number;
  unsigned max_stack;
  int _next_cfg_label;
  int _next_eh_label;
  bool liveness_changed;
  bool contains_merge;
  bool _need_linearization;
  Return_Address_Tracking *retstack;
  unsigned _callee_saved_registers_used;
  int _num_home_locations;
  Cfg_Node_List _bounds_nodes;
  Cfg_Node *_bounds_exception;
  bool did_dead_elim;
};

class Cfg_Extra_Info {
public:
    virtual bool is_tableswitch() {return false;}
    virtual unsigned data_space_needed() {return 0;}
};

// This is to hold the "match" information 
class lookupswitch_info : public Cfg_Extra_Info
{
public:
    void *operator new(size_t sz, Mem_Manager& m) { return m.alloc(sz); }
    int size;  // number of entries, not including default
    int *matches;  // branch targets are encoded in the Cfg_Node
	unsigned* offsets;	//::remember the targets for future optimization. sxh-- 2001.10.12
						//:: 0: default , 1~size: cases
    Cfg_Node *node; // node containing the switch
};

class tableswitch_info : public Cfg_Extra_Info
{
public:
    virtual bool is_tableswitch() {return true;}
    void *operator new(size_t sz, Mem_Manager& m) { return m.alloc(sz); }
    int low, high;
    unsigned data_space_needed() {return (high - low + 1)*sizeof(int);}
};

#ifdef COMPACT_NODE
class Subr_Info
{
public:
    void *operator new(size_t sz, Mem_Manager& m) { return m.alloc(sz); }
    Subr_Info(): enclosing_subr(NULL), jsr_succ(NULL), jsr_pred(NULL),
        where_is_ret_addr_bb(NULL), where_is_ret_addr_subr(NULL) {}


    Cfg_Node *enclosing_subr;  // which BB is start of immediately enclosing jsr subroutine
    Cfg_Node *jsr_succ;  // Successor node of a jsr instruction, needed for GC analysis
    Cfg_Node *jsr_pred;  // back-pointer for jsr_succ
    GCTrack_Operand *where_is_ret_addr_subr;
    GCTrack_Operand *where_is_ret_addr_bb;
};
#endif // COMPACT_NODE

class Cfg_Node
{
    friend class Eh_Node;
public:
    // "linear_pred" is the node's predecessor in the linearization.
    Cfg_Node(Mem_Manager &mm, int lbl, Flow_Graph *fg, Cfg_Node_List *linear_pred);
    void *operator new(size_t sz, Mem_Manager& m) { return m.alloc(sz); }
    Cfg_Extra_Info *extra_info;
    
    // Convention: fall-through or default edge is always stored first.
    Cfg_Node *out_edges(Cfg_Int i) { return _out_edges[i]; }
    Cfg_Int out_edge_size() { return _out_edge_size; }
    Cfg_Node *in_edges(Cfg_Int i) { return _in_edges[i]; }
    Cfg_Int in_edge_size() { return _in_edge_size; }
    Eh_Node *eh_out_edge() { return _eh_out_edge; }
    Eh_In_Header *eh_in_edge() { return _eh_in_edge; }
    Inst *IR_instruction_list() {return (Inst *) &_IR_instruction_list; }
    unsigned first_bc_idx() { return _first_bc_idx; }
    unsigned bc_length() { return _bc_length; }
    void set_bytecodes(unsigned first, unsigned length) { _first_bc_idx = first; _bc_length = length; }
    
    void replace_edge(Mem_Manager &mm, Cfg_Node *old_dest, Cfg_Node *new_dest);
    // Add an edge from "this" to "dest".
    void add_edge(Mem_Manager &mm, Cfg_Node *dest);
    // Delete the edge from "this" to "dest".
    void delete_edge(Cfg_Node *dest);
    // Swap two edges
    void swap_edges();
    // Split the edge from "this" to "dest".
    Cfg_Node *split_edge(Mem_Manager &mm, Cfg_Node *dest);
    // Add an edge from "this" to "dest".
    void add_eh_edge(Mem_Manager &mm, Eh_Node *dest);
    // Delete the edge from "this" to "dest".
    void delete_eh_edge(Eh_Node *dest);
    Cfg_Node *get_fallthrough();
    Cfg_Node *get_branch_target();
    void apply(unsigned short traversal_number, Apply_Func fun, Closure *c, 
               Flow_Graph *fg, bool traverse_sub_graph);

    Cfg_Node *clone(Mem_Manager &m);

    int dominates(Cfg_Node *node);
    void set_traversing(bool value) { _traversing = value; }
    bool traversing() { return (bool)_traversing; }
    Cfg_Node *idom; // immediate dominator
	Cfg_Node *loop_header;

    Bit_Vector_Group *live;
    Bit_Vector_Group *region_live;
#ifdef COMPACT_NODE
    Cfg_Node *get_enclosing_subr() { return (_subr_info ? _subr_info->enclosing_subr : NULL); }
    Cfg_Node *get_jsr_succ() { return (_subr_info ? _subr_info->jsr_succ : NULL); }
    Cfg_Node *get_jsr_pred() { return (_subr_info ? _subr_info->jsr_pred : NULL); }
    GCTrack_Operand *get_where_is_ret_addr_subr() { return (_subr_info ? _subr_info->where_is_ret_addr_subr : NULL); }
    GCTrack_Operand *get_where_is_ret_addr_bb() { return (_subr_info ? _subr_info->where_is_ret_addr_bb : NULL); }
    void set_enclosing_subr(Cfg_Node *subr) {
        if (_subr_info == NULL && subr == NULL) return;
        _create_subr_info(flowgraph->mem_manager);
        _subr_info->enclosing_subr = subr;
    }
    void set_jsr_succ(Cfg_Node *succ) {
        if (_subr_info == NULL && succ == NULL) return;
        _create_subr_info(flowgraph->mem_manager);
        _subr_info->jsr_succ = succ;
    }
    void set_jsr_pred(Cfg_Node *pred) {
        if (_subr_info == NULL && pred == NULL) return;
        _create_subr_info(flowgraph->mem_manager);
        _subr_info->jsr_pred = pred;
    }
    void set_where_is_ret_addr_subr(GCTrack_Operand *opnd) {
        if (_subr_info == NULL && opnd == NULL) return;
        _create_subr_info(flowgraph->mem_manager);
        _subr_info->where_is_ret_addr_subr = opnd;
    }
    void set_where_is_ret_addr_bb(GCTrack_Operand *opnd) {
        if (_subr_info == NULL && opnd == NULL) return;
        _create_subr_info(flowgraph->mem_manager);
        _subr_info->where_is_ret_addr_bb = opnd;
    }
#else // COMPACT_NODE
    Cfg_Node *get_enclosing_subr() { return _enclosing_subr; }
    Cfg_Node *get_jsr_succ() { return _jsr_succ; }
    Cfg_Node *get_jsr_pred() { return _jsr_pred; }
    GCTrack_Operand *get_where_is_ret_addr_subr() { _where_is_ret_addr_subr; }
    GCTrack_Operand *get_where_is_ret_addr_bb() { return _where_is_ret_addr_bb; }
    void set_enclosing_subr(Cfg_Node *subr) {
        _enclosing_subr = subr;
    }
    void set_jsr_succ(Cfg_Node *succ) {
        _jsr_succ = succ;
    }
    void set_jsr_pred(Cfg_Node *pred) {
        _jsr_pred = pred;
    }
    void set_where_is_ret_addr_subr(GCTrack_Operand *opnd) {
        _where_is_ret_addr_subr = opnd;
    }
    void set_where_is_ret_addr_bb(GCTrack_Operand *opnd) {
        _where_is_ret_addr_bb = opnd;
    }
#endif // COMPACT_NODE
#ifdef PRINTABLE_O3
    int unique_label;
#endif // PRINTABLE_O3
    Flow_Graph *flowgraph;
    unsigned short latest_traversal;
    short label;
    int loop_depth() { return _loop_depth; }
    void inc_loop_depth() { _loop_depth ++; }
    void dec_loop_depth() { _loop_depth --; }
    void set_loop_depth(int d) { _loop_depth = (unsigned char)d; }
    void set_code_offset(unsigned off) {_code_offset = off;}
    unsigned code_offset() {return _code_offset;}
    void set_code_length(unsigned len) { _code_length = len; }
    unsigned code_length() { return _code_length; }
    Cfg_Node_List *linearization_node() { return _linearization_node; }
    void set_linearization_node(Cfg_Node_List *list) { _linearization_node = list; }
    void set_mark(char c)   {_mark = c;}
    char mark() {return _mark;}
    Live_LCSE *live_lcse() {return _live_lcse;}
    void set_live_lcse(Live_LCSE *l) {assert(l != NULL && _live_lcse == NULL); _live_lcse = l;}
    void set_cold_code() {_cold_code = true;}
    bool is_cold() {return (bool)_cold_code;}
    bool is_empty_block() {return IR_instruction_list()->next() == IR_instruction_list();}
    void set_cold_non_inlined() { _cold_non_inlined = true; }
    bool is_cold_non_inlined() { return (bool)_cold_non_inlined; }
    bool already_set_subr() { return (bool)_already_set_subr; }
    void set_already_set_subr() { _already_set_subr = 1; }
    bool did_fp_spilling() { return (bool) _did_fp_spilling; }
    void set_did_fp_spilling() { _did_fp_spilling = 1; }
    bool is_athrow_block() {return _contain_athrow;}
    void set_athrow_block() {_contain_athrow = 1;}
    
private:
    Cfg_Node **_out_edges;
    Cfg_Node **_in_edges;
    Cfg_Int _out_edge_size;
    Cfg_Int _out_edge_capacity;
    Cfg_Int _in_edge_size;
    Cfg_Int _in_edge_capacity;  // initially 4
    Eh_Node *_eh_out_edge;  // can only be one out edge
    Eh_In_Header *_eh_in_edge;
    Cfg_Node_List *_linearization_node;
    Dlink _IR_instruction_list;
    Live_LCSE *_live_lcse;
    Subr_Info *_subr_info;
    unsigned _first_bc_idx, _bc_length;  // possibly invalid after transformations
    unsigned _traversing:1;  // whether this node or its successors are being traversed
    unsigned _cold_code:1;   // indicate if the block is executed infrequently
    unsigned _cold_non_inlined:1; // the "cold" portion of an inlined virtual method
    unsigned _already_set_subr:1;
    unsigned _did_fp_spilling:1;
    unsigned _contain_athrow:1; // set if the block contains athrow
	unsigned char _loop_depth;
    char _mark; // mark node for whatever purpose
    unsigned _code_offset;  // code offset of the first native inst of the block
    unsigned _code_length;  // number of bytes of machine code emitted for the block
#ifdef COMPACT_NODE
    void _create_subr_info(Mem_Manager &mm) { if (_subr_info == NULL) _subr_info = new(mm) Subr_Info; }
#else // COMPACT_NODE
    Cfg_Node *_enclosing_subr;  // which BB is start of immediately enclosing jsr subroutine
    Cfg_Node *_jsr_succ;  // Successor node of a jsr instruction, needed for GC analysis
    Cfg_Node *_jsr_pred;  // back-pointer for jsr_succ
    GCTrack_Operand *_where_is_ret_addr_subr;
    GCTrack_Operand *_where_is_ret_addr_bb;
#endif // COMPACT_NODE
};

class Eh_In_Header
{
    friend class Eh_Node;
public:
    Eh_In_Header(): _eh_in_edges(NULL), _eh_in_edge_capacity(0), _eh_in_edge_size(0) {}
    Eh_Node *eh_in_edges(Cfg_Int i) { return _eh_in_edges[i]; }
    Cfg_Int eh_in_edge_size() { return _eh_in_edge_size; }
private:
    Eh_Node **_eh_in_edges;
    Cfg_Int _eh_in_edge_capacity;  // initially 2
    Cfg_Int _eh_in_edge_size;
};

// Data structure for keeping track of return addresses on the Java
// operand stack and in the local variables.  We record the fact that
// the operand does or does not contain a ReturnAddress, and if it does,
// what is the bytecode of the entry point of the subroutine.  We record
// a -1 if it does not contain a ReturnAddress.  We count on the verifier
// to shield us from the nasty cases.
// Note: the only way to put a ReturnAddress into a local is through an
// astore.
class Return_Address_Tracking
{
public:
    Return_Address_Tracking(unsigned num_locals, unsigned max_stack, Mem_Manager &mem):
    _num_locals(num_locals), _max_stack(max_stack), _m(mem) {
        _stack = (unsigned *)_m.alloc(_max_stack * sizeof(unsigned));
        _vars = (unsigned *)_m.alloc(_num_locals * sizeof(unsigned));
        reset();
    }
    void *operator new(size_t sz, Mem_Manager& m) { return m.alloc(sz); }
    void reset() { _stack_top = 0; for (unsigned i=0; i<_num_locals; i++) _vars[i] = (unsigned)-1; }
    void push(unsigned idx) { assert(_stack_top<_max_stack); _stack[_stack_top++] = idx; }
    void push() { push((unsigned)-1); }
    unsigned pop() { assert(_stack_top>0); return _stack[--_stack_top]; }
    void store(unsigned varno) { assert(varno<_num_locals); _vars[varno] = pop(); }
    Return_Address_Tracking *clone() {
        Return_Address_Tracking *result = new(_m) Return_Address_Tracking(_num_locals, _max_stack, _m);
        unsigned i;
        result->_stack_top = _stack_top;
        for (i=0; i<_max_stack; i++) result->_stack[i] = _stack[i];
        for (i=0; i<_num_locals; i++) result->_vars[i] = _vars[i];
        return result;
    }
    //unsigned top() { assert(_stack_top>0&&_stack_top<=_num_locals); return _stack[_stack_top-1]; }
    unsigned size() { return _stack_top; }
    unsigned getvar(unsigned var) { assert(var<_num_locals); return _vars[var]; }
private:
    unsigned _num_locals, _max_stack;
    unsigned _stack_top;
    Mem_Manager &_m;
    unsigned *_stack, *_vars;
};

// Makes sure there's enough space in the array for one more element.
// TYPE: the type of an array element
// ARRAY: the array
// CAPACITY: current capacity of the array (should have type Cfg_Int)
// SIZE: current size of the array (should have type Cfg_Int)
// INITIAL_CAPACITY: number of elements to allocate on first allocation
// MM: the memory manager, of type Mem_Manager
#define RESIZE_ARRAY(TYPE, ARRAY, CAPACITY, SIZE, INITIAL_CAPACITY, MM) \
do {                                                                    \
  if ((SIZE) >= (CAPACITY))                                             \
    {                                                                   \
      if ((CAPACITY) == 0)                                              \
        {                                                               \
          CAPACITY = (INITIAL_CAPACITY);                                \
          SIZE = 0;                                                     \
          ARRAY = (TYPE *)(MM).alloc((CAPACITY) * sizeof(TYPE));        \
        }                                                               \
      else                                                              \
        {                                                               \
          int new_capacity = 2 * (CAPACITY);                            \
          TYPE *new_array =                                             \
            (TYPE *)(MM).alloc(new_capacity * sizeof(TYPE));            \
          memcpy(new_array,(ARRAY),(CAPACITY)*sizeof(TYPE));            \
          ARRAY = new_array;                                            \
          CAPACITY = new_capacity;                                      \
        }                                                               \
    }                                                                   \
} while (0);

/////////////////////////////////////////////////////////////////
// closure classes
/////////////////////////////////////////////////////////////////
class Num_Nodes : public Closure {
public:
    Num_Nodes() : count(0) {}
    int count;
};

class Back_Edge : public Closure {
public:	
    Back_Edge(Mem_Manager &mm) : 
      mm(mm), back_edge_heads(NULL), back_edge_tails(NULL), 
      back_edge_count(0), back_edge_capacity(0) {}
    Cfg_Node *head_of(Cfg_Node *tail) {
        int i;
        for (i = 0; i < back_edge_count; i++)
            if (back_edge_tails[i] == tail) return back_edge_heads[i];
        return NULL;
    }
    
    Mem_Manager &mm;
    Cfg_Node **back_edge_heads, **back_edge_tails;
    int back_edge_count;
    int back_edge_capacity;
};

/////////////////////////////////////////////////////////////////
// Implement traversing flow graph non-recursively
/////////////////////////////////////////////////////////////////
class WL_Node : public Dlink 
{
public:
    void     *node;
    WL_Node() : node(NULL) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
};

class WL_DFS_Node : public WL_Node 
{
public:
    unsigned parent_id;
    bool     is_eh;

    WL_DFS_Node() {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
    void set(unsigned pid, void *nd, bool eh)
    {
        parent_id = pid;
        node = nd;
        is_eh = eh;
    }
};

class Work_Set 
{
public:
    Dlink free_list;
    Dlink work_list; 

    bool  is_empty() {return work_list.get_next() == &work_list;}
    void  free(Dlink *n) 
    {
        n->unlink();
        n->insert_before(&free_list); 
    }
    Dlink *get_free_node()
    {
        Dlink *n = free_list.get_next();
        if (n != &free_list) // there is a free node
        {
            n->unlink();
            return n;
        }
        return NULL;
    }
};


extern void find_back_edges(Cfg_Node *node, Closure *c);
extern void loop_transformation(Flow_Graph *fg, bool &did_peeling);

extern bool is_artificially_split_edge(Cfg_Node *&pred, Cfg_Node *&succ);
extern bool is_edge_into_hot_inlined(Cfg_Node *&pred, Cfg_Node *&succ);
extern bool is_edge_into_cold_inlined(Cfg_Node *&pred, Cfg_Node *&succ);
extern bool is_edge_outof_hot_inlined(Cfg_Node *&pred, Cfg_Node *&succ);
extern bool is_edge_outof_cold_inlined(Cfg_Node *&pred, Cfg_Node *&succ);

#endif // _FLOW_GRAPH_H_
