// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/base/bytereader.cpp,v 1.4 2001/08/03 03:13:39 xli18 Exp $
//



#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>

#ifndef ORP_POSIX
#include <io.h>
#else
#include <unistd.h>
#include <iostream.h>
#endif

#include <fcntl.h> 
#include <fstream.h>
#include <string.h>
#include <assert.h>


#include "bytereader.h"


////////////////////////////////////////////////////////////////////////////
// begin ByteReader



//
// Offset and length are 64 bits long, but our code only handles 32 bits long
// class files.
//
ByteReader::ByteReader(int _file_handle, int64 _offset, int64 _len)
{
    offset       = (int)_offset;
    len          = (int)_len;
    curr = start = new uint8[len];
    byte_array_owner = true;
    assert(offset == 0);
#ifdef ORP_NT
    int bytes_read = _read(_file_handle, start, len);
#else
    int bytes_read = read(_file_handle, start, len);
#endif
    error = (bytes_read < 0) || (bytes_read < len);
    if(error) {
        // This will cause the first read to fail.
        end = start;
    } else {
        end = start + len;
    }
    //printf("Initialized ByteReader of length 0x%x\n", len);
} //ByteReader::ByteReader




ByteReader::ByteReader(uint8 *bytecode, int _offset, int _len)
{
    offset       = _offset;
    len          = _len;
    curr = start = bytecode + offset;
    byte_array_owner = false;
    end = start + len;
} //ByteReader::ByteReader




ByteReader::~ByteReader()
{
    if(byte_array_owner) {
        delete []start;
    }
} //ByteReader::~ByteReader



int ByteReader::get_offset()
{
    return offset + (curr - start);
} //ByteReader::get_offset



void ByteReader::dump(int num_bytes)
{
    printf("--- begin dump\n");
    int num_columns = 8;
    bool first_time = true;
    bool first_col = true;
    int curr_offset = get_offset();
    uint8 *curr_byte = curr;
    for(int i = 0; i < num_bytes; i++, curr_offset++, curr_byte++) {
        if(first_time || !(curr_offset & (num_columns - 1))) {
            if(!first_time) {
                printf("\n");
            }
            printf("%04x: ", curr_offset);
            if(first_time) {
                first_time = false;
                int filler_cols = curr_offset & (num_columns - 1);
                for(int col = 0; col < filler_cols; col++) {
                    printf("        ", col);
                }
            }
        }
        uint8 b = *curr_byte;
        printf("%02x", b);
        if(b >= 32 && b < 127) {
            printf(" '%c'  ", b);
        } else {
            printf("  .   ");
        }
    }
    printf("\n");
    printf("--- end dump\n");
} //ByteReader::dump



int ByteReader::find_byte(uint8 byte)
{
    for(uint8 *curr_byte = curr; curr_byte < end; curr_byte++) {
        if(*curr_byte == byte) {
            return offset + (curr_byte - start);
        }
    }
    return -1;
} //ByteReader::go_to_offset



bool ByteReader::go_to_offset(int _offset)
{
    int num_bytes_to_skip = _offset - get_offset();
    return skip(num_bytes_to_skip);
} //ByteReader::go_to_offset



bool ByteReader::skip(int num_bytes)
{
    if((curr + num_bytes) > end) {
#ifdef _DEBUG
			cerr << "ByteReader::skip: eof!" << endl;
#endif
        return false;
    }

    curr += num_bytes;
	return true;
} //ByteReader::skip



bool ByteReader::parse_u4_be(uint32 *val)
{
    if((curr + 4) > end) {
#ifdef _DEBUG
			cerr << "ByteReader::parse_u4_be: eof!" << endl;
#endif
        return false;
    }
	uint32 result = 0;
	for (int i = 0; i < 4; i++) {
		uint32 x = (uint32)*curr++;
		result = (result << 8) + x;
	}
	*val = result;
	return true;
} //ByteReader::parse_u4_be



bool ByteReader::parse_u4_le(uint32 *val)
{
    if((curr + 4) > end) {
#ifdef _DEBUG
			cerr << "ByteReader::parse_u4_le: eof!" << endl;
#endif
        return false;
    }
	uint32 result = 0;
    curr += 4;
    uint8 *curr_byte = curr;
	for (int i=0; i < 4; i++) {
		uint32 x = (uint32)*(--curr_byte);
		result = (result << 8) + x;
	}
	*val = result;
	return true;
} //ByteReader::parse_u4_le



bool ByteReader::parse_u2_be(uint16 *val)
{
    if((curr + 2) > end) {
#ifdef _DEBUG
			cerr << "ByteReader::parse_u2_be: eof!" << endl;
#endif
        return false;
    }
	uint16 result = 0;
	for (int i=0; i < 2; i++) {
		uint16 x = (uint16)*curr++;
		result = (result << 8) + x;
	}
	*val = result;
	return true;
} //ByteReader::parse_u2_be



bool ByteReader::parse_u2_le(uint16 *val)
{
    if((curr + 2) > end) {
#ifdef _DEBUG
			cerr << "ByteReader::parse_u2_le: eof!" << endl;
#endif
        return false;
    }
	uint16 result = 0;
    curr += 2;
    uint8 *curr_byte = curr;
	for (int i = 0; i < 2; i++) {
		uint16 x = (uint16)*(--curr_byte);
		result = (result << 8) + x;
	}
	*val = result;
	return true;
} //ByteReader::parse_u2_le



bool ByteReader::parse_u1(uint8 *val)
{
    if((curr + 1) > end) {
#ifdef _DEBUG
			cerr << "ByteReader::parse_u1: eof!" << endl;
#endif
        return false;
    }
	*val = *curr++;
	return true;
} //ByteReader::parse_u1



bool ByteReader::read_bytes(uint8 *buffer, int len)
{
    if((curr + len) > end) {
#ifdef _DEBUG
			cerr << "ByteReader::read_bytes: eof!" << endl;
#endif
        return false;
    }
    for(int i = 0; i < len; i++) {
	    *buffer++ = *curr++;
    }
	return true;
} //ByteReader::read_bytes




// end ByteReader
////////////////////////////////////////////////////////////////////////////







