// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/base/jit_runtime_support.cpp,v 1.15 2002/01/10 22:49:39 ssubram5 Exp $
//



#include "platform.h"
#include <iostream.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <float.h>

#ifdef OBJECT_SPLITTING
#include <stddef.h>
#endif // OBJECT_SPLITTING

#include "object_layout.h"
#include "orp_types.h"
#include "Class.h"
#include "environment.h"
#include "method_lookup.h"
#include "stack_manipulation.h"
#include "exceptions.h"
#include "orp_synch.h"
 
#include "gc_for_orp.h"
 
#include "root_set_enum.h"
#include "ini.h"
#include "orp_utils.h"
#include "orp_threads.h"
#include "orp_stats.h"

#include "jit_runtime_support_common.h"
#include "jit_runtime_support.h"
#include "orp_synch.h"

#ifdef ORP_POSIX
#include "platform2.h"
#endif

#ifdef CLI_TESTING
#ifdef CLI_OCL
#include "../include/cli_misc.h"
#endif
#endif


int32 orp_rt_imul_common(int32 v1, int32 v2)
{
    return v1 * v2;
} //orp_rt_imul_common



int32 orp_rt_idiv_common(int32 v1, int32 v2)
{
    assert(v2);
    return v1 / v2;
} //orp_rt_idiv_common



int32 orp_rt_irem_common(int32 v1, int32 v2)
{
    assert(v2);
    return v1 % v2;
} //orp_rt_irem_common




/////////////////////////////////////////////////////////////
// begin utf8 support


//
// See ORP Spec, Section 4.4.7
//
unsigned get_unicode_length_of_utf8(const char *utf8)
{
    unsigned len = 0;
    uint8 ch;
    while(ch = *utf8++) {
        len++;
        if(ch & 0x80) {
            assert(ch & 0x40);
            if(ch & 0x20) {
                uint8 x = ch;
                uint8 y = *utf8++;
                uint8 z = *utf8++;
            } else {
                uint8 x = ch;
                uint8 y = *utf8++;
            }
        } 
    }
    return len;
} //get_unicode_length_of_utf8



unsigned get_utf8_length_of_unicode(const uint16 *unicode, unsigned unicode_length)
{
    unsigned len = 0;
    for(unsigned i = 0; i < unicode_length; i++) {
        uint16 ch = unicode[i];
        if(ch == 0) {
            len += 2;
        } else if(ch < 0x80) {
            len += 1;
        } else if(ch < 0x800) {
            len += 2;
        } else {
            len += 3;
        }
    }
    return len;
} //get_utf8_length_of_unicode



void pack_utf8(char *utf8_string, const uint16 *unicode, unsigned unicode_length)
{
    unsigned utf8_len = get_utf8_length_of_unicode(unicode, unicode_length);
    char *s = utf8_string;
    for(unsigned i = 0; i < unicode_length; i++) {
        unsigned ch = unicode[i];
        if(ch == 0) {
            *s++ = (char)0xc0;
            *s++ = (char)0x80;
        } else if(ch < 0x80) {
            *s++ = ch;
        } else if(ch < 0x800) {
            unsigned b5_0 = ch & 0x3f;
            unsigned b10_6 = (ch >> 6) & 0x1f;
            *s++ = 0xc0 | b10_6;
            *s++ = 0x80 | b5_0;
        } else {
            unsigned b5_0 = ch & 0x3f;
            unsigned b11_6 = (ch >> 6) & 0x3f;
            unsigned b15_12 = (ch >> 12) & 0xf;
            *s++ = 0xe0 | b15_12;
            *s++ = 0x80 | b11_6;
            *s++ = 0x80 | b5_0;
        }
    }
    *s = 0;
} //pack_utf8



void unpack_utf8(uint16 *unicode, const char *utf8_string)
{
    const uint8 *utf8 = (const uint8 *)utf8_string;
    unsigned len = 0;
    uint16 ch;
    while(ch = (uint16)*utf8++) {
        len++;
        if(ch & 0x80) {
            assert(ch & 0x40);
            if(ch & 0x20) {
                uint16 x = ch;
                uint16 y = (uint16)*utf8++;
                uint16 z = (uint16)*utf8++;
                uint16 u =
                *unicode++ = ((0x0f & x) << 12) + ((0x3f & y) << 6) + ((0x3f & z));
            } else {
                uint16 x = ch;
                uint16 y = (uint16)*utf8++;
                uint16 u =
                *unicode++ = ((0x1f & x) << 6) + (0x3f & y);
            }
        } else {
            uint16 u =
            *unicode++ = ch;
        }
    }
} //unpack_utf8



// end utf8 support
/////////////////////////////////////////////////////////////



/////////////////////////////////////////////////////////////
// begin strings

#ifdef OBJECT_SPLITTING
extern unsigned int get_proper_array_size(Partial_Reveal_Class * clss, unsigned int size);
#endif // OBJECT_SPLITTING



static JavaArrayOfChar *
create_array_of_chars_from_ORP_string(String *str, bool or_null)
{
    Class *clss = ORP_Global_State::loader_env->ArrayOfChar_Class;
    assert(clss->vtable);

    const char *bytes = str->bytes;
    unsigned len = get_unicode_length_of_utf8(bytes);

#if 1
    unsigned sz = orp_array_size(clss, len);
#else
    unsigned sz = sizeof(JavaArray) + (len) * sizeof(J_Char); // 2
    assert (sizeof(J_Char) == 2); // Should always be true??

    sz = sz + OBJECT_HEADER_SIZE;
    // align the object upwards
    sz = (((sz + (GC_OBJECT_ALIGNMENT - 1)) 
          / GC_OBJECT_ALIGNMENT) * GC_OBJECT_ALIGNMENT);
#endif

    JavaArrayOfChar *_array;
    if(or_null) {

#ifdef OBJECT_SPLITTING
		_array = (JavaArrayOfChar *)gc_malloc_or_null(
                        get_proper_array_size((Partial_Reveal_Class * )clss, sz), 
                        (Partial_Reveal_VTable *)clss->vtable);
#else
#ifdef GC_REWORK
        _array = (JavaArrayOfChar *)gc_malloc_or_null(sz, (VTable *)clss->vtable);
#else
        _array = (JavaArrayOfChar *)gc_malloc_or_null(sz, (Partial_Reveal_VTable *)clss->vtable);
#endif
#endif // OBJECT_SPLITTING
        if(!_array) {
            return 0;
        }
    } else {

#ifdef OBJECT_SPLITTING
        _array = (JavaArrayOfChar *)gc_malloc(
                        get_proper_array_size((Partial_Reveal_Class * )clss, sz), 
                        (Partial_Reveal_VTable *)clss->vtable);
#else 

#ifdef GC_REWORK   
        _array = (JavaArrayOfChar *)gc_malloc(sz, (VTable *)clss->vtable);
#else
        _array = (JavaArrayOfChar *)gc_malloc(sz, (Partial_Reveal_VTable *)clss->vtable);
#endif
#endif // OBJECT_SPLITTING

    }
    _array->length = len;
    unpack_utf8(&(_array->body[0]), bytes);
    gc_heap_wrote_object((Java_java_lang_Object *)_array);
    assert(_array->vt == clss->vtable);
    return _array;
} //create_array_of_chars_from_ORP_string



static JavaArrayOfChar *
create_array_of_chars_from_ORP_string(String *str)
{
    return create_array_of_chars_from_ORP_string(str, false);
} //create_array_of_chars_from_ORP_string



Java_java_lang_String *create_java_lang_String_from_C_string(const char *str)
{
    volatile JavaArrayOfChar *char_array =
        create_array_of_chars_from_C_string(str);

    Class *jls_clss = ORP_Global_State::loader_env->JavaLangString_Class;

    GC_Frame f;
    orp_push_gc_frame(&f, (void *)&char_array, sizeof(char_array));

    Java_java_lang_String *jls =
        (Java_java_lang_String *)class_alloc_new_object(jls_clss);

    orp_pop_gc_frame(&f);

    set_java_lang_string_fields(jls, (JavaArrayOfChar *)char_array, 0, char_array->length);
    return jls;
} //create_java_lang_String_from_C_string



JavaArrayOfString *orp_create_array_of_strings(char **p_str, int length)
{
    Class *clss = ORP_Global_State::loader_env->JavaLangString_Class;
    volatile JavaArrayOfString *arr =
        (JavaArrayOfString *)orp_anewarray_resolved(clss, length);

    GC_Frame f;
    orp_push_gc_frame(&f, (void *)&arr, sizeof(arr));

    for(int i = 0; i < length; i++) {
        String *s = ORP_Global_State::loader_env->string_pool.lookup(p_str[i]);
        Java_java_lang_String *v = orp_instantiate_cp_string_resolved(s);
//                arr->body[i] = (Java_java_lang_String *)v;
        
        gc_heap_slot_write_ref ((Java_java_lang_Object *)arr, 
            (Java_java_lang_Object **)&(arr->body[i]), 
            (Java_java_lang_Object *)v);
    }

    orp_pop_gc_frame(&f);

    return (JavaArrayOfString *)arr;
} //orp_create_array_of_strings


#ifdef CLI_TESTING
#ifdef CLI_OCL


Java_java_lang_String *orp_instantiate_cp_cli_string_resolved(String *str)
{
	Java_java_lang_String *java_str = orp_instantiate_cp_string_resolved(str);
	assert(java_str);

	// Just replace the vtable pointer before returning.	
	extern Global_CLI_Env *global_cli_env;
	assert(global_cli_env);
	volatile VTable *system_string_vtable = global_cli_env->System_String->vtable;
	assert(system_string_vtable);

	// Do LOCKED CMPXCHG and stick a new vtable into the object to 
	// change its type from java.lang.String to System.String.
	// Is this the easiest(not best) way to produce a CLI string object??
	// I think so, given that the field layout, initializtion and usage 
	// are the same between Java and CLI for String.
	
	volatile VTable *orig_vt = str->intern->vt;
    volatile VTable *what_it_is_now = 
        (volatile VTable  *)InterlockedCompareExchangePointer (
			(PVOID *)(&(str->intern->vt)),(PVOID)system_string_vtable, (PVOID)orig_vt);

	if ((what_it_is_now != orig_vt) && (what_it_is_now != system_string_vtable)) { 
		// BAD!!
		assert(0);
		return NULL;
	} 
	return str->intern;
} // orp_instantiate_cp_cli_string_resolved

#endif // CLI_OCL
#endif // CLI_TESTING


//
// We enter here with gc disabled and create a Java_java_lang_string.
// 
Java_java_lang_String *orp_instantiate_cp_string_resolved(String *str)
{
    assert(  !orp_is_gc_enabled(p_TLS_orpthread) );

    if(str->intern) {
        return str->intern;
    }

    Class *jlc = ORP_Global_State::loader_env->JavaLangString_Class;

    volatile Java_java_lang_String *lang_string =
        (Java_java_lang_String *)class_alloc_new_object(jlc);

    GC_Frame f;
    orp_push_gc_frame(&f, (void *)&lang_string, sizeof(lang_string));

    JavaArrayOfChar *char_array = create_array_of_chars_from_ORP_string(str, false);

    set_java_lang_string_fields((Java_java_lang_Object *)lang_string, char_array, 0, char_array->length);

    assert(  !orp_is_gc_enabled(p_TLS_orpthread) );

    // We now will atomically update the str data structure.
    // Note that other thread might be trying to make the same update.
    // The GC won't be able to enumerate here since we are disabled. This
    // means that there are no race conditions with the GC.

    volatile Java_java_lang_String *result = 
        (volatile Java_java_lang_String *)InterlockedCompareExchangePointer ((PVOID *)(&(str->intern)), 
            (PVOID)lang_string, (PVOID)NULL);
    
    if (result == NULL) {
        // Note the successful write of the object. 
        gc_heap_write_global_slot ((Java_java_lang_Object **)(&(str->intern)), (Java_java_lang_Object *)lang_string);
    }
    // Some other thread may have beaten us to the slot or the 
    // sapphire may update slot so refresh.
    lang_string = str->intern;

    orp_pop_gc_frame(&f);
    assert ((lang_string == str->intern));
    return (Java_java_lang_String *)lang_string;
} //orp_instantiate_cp_string_resolved


Java_java_lang_String *
orp_instantiate_cp_string_slow(Class *c, unsigned cp_index)
{
#ifdef ORP_STATS
    orp_stats_total.num_instantiate_cp_string_slow++;
#endif
    assert(cp_index < c->cp_size);
	Const_Pool *cp = c->const_pool;

	String *str = cp[cp_index].string;
    assert(cp_is_cnst(cp, cp_index));

    return orp_instantiate_cp_string_resolved(str);
} //orp_instantiate_cp_string_slow



//
// This function checks if a string has already an interned representation.
// If so, it returns it.
//
// Note that the ORP spec (p. 291) implies that every execution of ldc
// should allocate a new copy of the constant string.  This can't be what
// they intended, because it would make it hard to implement Java semantics
// efficiently.  See Sec 3.10.5 of the Java Language Spec.
//
// Otherwise, it attempts to allocate a string. If allocation fails,
// it returns NULL.
//
Java_java_lang_String *orp_instantiate_cp_string_fast(Class *c, unsigned cp_index)
{
#ifdef ORP_STATS
    orp_stats_total.num_instantiate_cp_string_fast++;
#endif
    assert(  !orp_is_gc_enabled(p_TLS_orpthread) );

    assert(cp_index < c->cp_size);
	Const_Pool *cp = c->const_pool;

	String *str = cp[cp_index].string;
    assert(cp_is_cnst(cp, cp_index));

    if(str->intern) {
#ifdef ORP_STATS
        orp_stats_total.num_instantiate_cp_string_fast_returned_interned++;
#endif
        return str->intern;
    }

    Class *jls_class = ORP_Global_State::loader_env->JavaLangString_Class;
    Java_java_lang_String *lang_string =
        (Java_java_lang_String *)class_alloc_new_object_or_null(jls_class);

    if(!lang_string)
        return 0;

    JavaArrayOfChar *char_array = create_array_of_chars_from_ORP_string(str, true);

    if(!char_array)
        return 0;

    set_java_lang_string_fields((Java_java_lang_Object *)lang_string, char_array, 0, char_array->length);

    assert(  !orp_is_gc_enabled(p_TLS_orpthread) );

    // We now will atomically update the str data structure.
    // Note that other thread might be trying to make the same update.
    // The GC won't be able to enumerate here since we are disabled. This
    // means that there are no race conditions with the GC.

    Java_java_lang_String *result = 
        (Java_java_lang_String *)InterlockedCompareExchangePointer ((PVOID *)(&(str->intern)), 
            (PVOID)lang_string, (PVOID)NULL);

    if (result != NULL) {
        // Some other thread beat us to the compare exchange so use that value
        // and forget about the one we just created.
        lang_string = result;
    }
    assert ((lang_string == str->intern));
#ifdef ORP_STATS
    orp_stats_total.num_instantiate_cp_string_fast_success_long_path++;
#endif
    return lang_string;
} //orp_instantiate_cp_string_fast



JavaArrayOfChar *create_array_of_chars(int length)
{
    assert(length >= 0);
    Class *clss = ORP_Global_State::loader_env->ArrayOfChar_Class;
    assert(clss->vtable);
 
#if 1
    unsigned sz = orp_array_size(clss, length);
#else
    unsigned sz = sizeof(JavaArray) + (length) * sizeof(J_Char);
    
    sz = sz + OBJECT_HEADER_SIZE;
    // align the object upwards
    sz = (((sz + (GC_OBJECT_ALIGNMENT - 1)) 
          / GC_OBJECT_ALIGNMENT) * GC_OBJECT_ALIGNMENT);
#endif

#ifdef OBJECT_SPLITTING
	// Leave the cold section unused....
    JavaArrayOfChar *_array = (JavaArrayOfChar *)gc_malloc(
                                    get_proper_array_size((Partial_Reveal_Class * )clss, sz), 
                                    (Partial_Reveal_VTable *)clss->vtable);
#else 
#ifdef GC_REWORK
    JavaArrayOfChar *_array = (JavaArrayOfChar *)gc_malloc(sz, (VTable *)clss->vtable);
#else
    JavaArrayOfChar *_array = (JavaArrayOfChar *)gc_malloc(sz, (Partial_Reveal_VTable *)clss->vtable);
#endif
#endif // OBJECT_SPLITTING

    // _array->length = length;
    gc_heap_slot_write_int32 ((Java_java_lang_Object *)_array,
        &(_array->length),
        length);

    assert(_array->vt == clss->vtable);
    return _array;
} //create_array_of_chars



JavaArrayOfChar *create_array_of_chars_from_C_string(const char *str)
{
    String *s = ORP_Global_State::loader_env->string_pool.lookup(str);
    return create_array_of_chars_from_ORP_string(s, false);
} //create_array_of_chars_from_string




// end strings
/////////////////////////////////////////////////////////////



/////////////////////////////////////////////////////////////
// begin Java object allocation


Java_java_lang_Object *class_alloc_new_object_or_null(Class *c)
{
#ifdef ORP_STATS
    orp_stats_total.num_class_alloc_new_object_or_null++;
#endif
    assert(!orp_is_gc_enabled(p_TLS_orpthread) );
    assert(strcmp(c->name->bytes, "java/lang/Class"));
#ifdef GC_REWORK
    
    return (Java_java_lang_Object *)gc_malloc_or_null(c->instance_data_size, (VTable *)c->vtable);
#else
    return (Java_java_lang_Object *)gc_malloc_or_null(c->instance_data_size, (Partial_Reveal_VTable *)c->vtable);
#endif
} //class_alloc_new_object_or_null




Java_java_lang_Object *class_alloc_new_object_with_finalizer_or_null(Class *c)
{
    // Checks
    unsigned int class_constraints = c->class_properties;
    unsigned int real_size = get_instance_data_size (c);
    
    assert(c->has_finalizer);
    assert(get_prop_finalizable (class_constraints));
    assert(real_size != c->instance_data_size);
    // end checks

    // ORP_STATS counts need to be redone inside the gc.
    // Finalization is an class property.
    // Perhaps an assert to make sure that c is a class that finalizes.
    return class_alloc_new_object_or_null(c);
} //class_alloc_new_object_with_finalizer_or_null



// end Java object allocation
/////////////////////////////////////////////////////////////



/////////////////////////////////////////////////////////////
// begin Java array allocation

JavaArray *orp_new_array_from_class(Class *clss, int length)
{
    assert(length >= 0);
    assert(  !orp_is_gc_enabled(p_TLS_orpthread) );

    assert(clss->name->bytes[0] == '[');
    assert(clss->name->len > 1);

    char typ = clss->name->bytes[1];
    switch(typ) {
    case JAVA_TYPE_CLASS:
    case JAVA_TYPE_ARRAY:
        return (JavaArray *)orp_anewarray_resolved(clss->array_element_class, length);
    case JAVA_TYPE_INT:
        return orp_newarray(JAVA_NEWARRAY_TYPE_INT, length);
    case JAVA_TYPE_LONG:
        return orp_newarray(JAVA_NEWARRAY_TYPE_LONG, length);
    case JAVA_TYPE_BOOLEAN:
        return orp_newarray(JAVA_NEWARRAY_TYPE_BOOLEAN, length);
    case JAVA_TYPE_CHAR:
        return orp_newarray(JAVA_NEWARRAY_TYPE_CHAR, length);
    case JAVA_TYPE_FLOAT:
        return orp_newarray(JAVA_NEWARRAY_TYPE_FLOAT, length);
    case JAVA_TYPE_DOUBLE:
        return orp_newarray(JAVA_NEWARRAY_TYPE_DOUBLE, length);
    case JAVA_TYPE_BYTE:
        return orp_newarray(JAVA_NEWARRAY_TYPE_BYTE, length);
    case JAVA_TYPE_SHORT:
        return orp_newarray(JAVA_NEWARRAY_TYPE_SHORT, length);
    default:
        cout << "Type: " << (char)typ << endl;
        assert(0);
        orp_exit(1);
        break;
    }
    assert(0);
    return 0;
} //orp_new_array_from_class



//
// Create a Java array
// There are two arguments:
// - array length
// - class id of the element type
// Return value (eax):
//  - Reference to the new object
//
// GC may happen in this functions, so the callers must protect all the
// references before invoking orp_anewarray_resolved.
//
JavaArrayOfObject *orp_anewarray_resolved(Class *cc, int length)
{
    assert(length >= 0);
    assert(  !orp_is_gc_enabled(p_TLS_orpthread) );

    assert(cc);
    const unsigned max_name_length = 1000;
    char array_name[max_name_length + 2];   // extra chars: '[' and '\0'
    assert(cc->name->len < max_name_length);

    if(cc->name->bytes[0] == '[') {
        sprintf(array_name, "[%s", cc->name->bytes);
    } else {
        sprintf(array_name, "[L%s;", cc->name->bytes);
    }
    Global_Env *env = ORP_Global_State::loader_env;
    String *arr_str = env->string_pool.lookup(array_name);
    Loader_Exception exc;
    Class *arr_clss = class_load_verify_prepare(env, arr_str, &exc);
    assert(arr_clss);

#if 1
    unsigned sz = orp_array_size(arr_clss, length);
#else
    unsigned sz = sizeof(JavaArray) + (length) * sizeof(J_Reference);
 
    sz = sz + OBJECT_HEADER_SIZE;
    // align the object upwards
// Something like this might be faster.    
//    sz = (void *)( (unsigned)b + 7);
//    sz = (void *)( (unsigned)b & ~0x07);
    sz = (((sz + (GC_OBJECT_ALIGNMENT - 1)) 
          / GC_OBJECT_ALIGNMENT) * GC_OBJECT_ALIGNMENT);
#endif
#ifdef OBJECT_SPLITTING
    JavaArrayOfObject *object_array = 
                    (JavaArrayOfObject *)gc_malloc(
                                        get_proper_array_size((Partial_Reveal_Class * )arr_clss, sz), 
                                        (Partial_Reveal_VTable *)arr_clss->vtable);
#else
#ifdef GC_REWORK
	JavaArrayOfObject *object_array = (JavaArrayOfObject *)gc_malloc(sz, (VTable *)arr_clss->vtable);
#else
	JavaArrayOfObject *object_array = (JavaArrayOfObject *)gc_malloc(sz, (Partial_Reveal_VTable *)arr_clss->vtable);
#endif
#endif // OBJECT_SPLITTING

//    object_array->length = length;
    gc_heap_slot_write_int32 ((Java_java_lang_Object *)object_array,
        &(object_array->length),
        length);

    assert(object_array->vt == arr_clss->vtable);

    return object_array;
} //orp_anewarray_resolved



JavaArrayOfObject *orp_anewarray_resolved_array_type(Class *arr_clss, int length)
{
#ifdef ORP_STATS
    orp_stats_total.num_anewarray++;
#endif
    assert(  !orp_is_gc_enabled(p_TLS_orpthread) );

    // This assertion fails because ORP marks
    // "[[I" as an array of primitives,
    //assert(!arr_clss->is_array_of_primitives);

#if 1
    unsigned sz = orp_array_size(arr_clss, length);
#else
    unsigned sz = sizeof(JavaArray) + (length) * sizeof (J_Reference);
 
    sz = sz + OBJECT_HEADER_SIZE;
    // align the object upwards
    sz = (((sz + (GC_OBJECT_ALIGNMENT - 1)) 
          / GC_OBJECT_ALIGNMENT) * GC_OBJECT_ALIGNMENT);
#endif

#ifdef OBJECT_SPLITTING
    JavaArrayOfObject *object_array = 
                        (JavaArrayOfObject *)gc_malloc(
                                    get_proper_array_size((Partial_Reveal_Class * )arr_clss, sz), 
                                    (Partial_Reveal_VTable *)arr_clss->vtable);
#else
#ifdef GC_REWORK
    
    JavaArrayOfObject *object_array = 
        (JavaArrayOfObject *)gc_malloc(sz, (VTable *)arr_clss->vtable);
#else
    JavaArrayOfObject *object_array = 
        (JavaArrayOfObject *)gc_malloc(sz, (Partial_Reveal_VTable *)arr_clss->vtable);
#endif
#endif // OBJECT_SPLITTING

//    object_array->length = length;
    gc_heap_slot_write_int32 ((Java_java_lang_Object *)object_array,
        &(object_array->length),
        length);
    assert(object_array->vt == arr_clss->vtable);

    return object_array;
} //orp_anewarray_resolved_array_type

static unsigned orp_primitive_array_element_size(ORP_Data_Type typ)
{
    switch(typ) {
    case ORP_DATA_TYPE_INT8:
    case ORP_DATA_TYPE_UINT8:
    case ORP_DATA_TYPE_BOOLEAN:
        return sizeof (int8);
    case ORP_DATA_TYPE_INT16:
    case ORP_DATA_TYPE_UINT16:
    case ORP_DATA_TYPE_CHAR:
        return sizeof(int16);
    case ORP_DATA_TYPE_UINT32:
    case ORP_DATA_TYPE_INT32:
    case ORP_DATA_TYPE_F4:
        return sizeof(int32);
    case ORP_DATA_TYPE_UINT64:
    case ORP_DATA_TYPE_INT64:
    case ORP_DATA_TYPE_F8:
        return sizeof(int64);
    case ORP_DATA_TYPE_CLASS:
    case ORP_DATA_TYPE_ARRAY:
        return sizeof(void *);
    default:
        assert(0);
        return 0;
    }
} //orp_primitive_array_element_size



static unsigned orp_primitive_array_size(ORP_Data_Type typ, int length)
{
    unsigned sz;
#if 0
    switch(typ) {
    case ORP_DATA_TYPE_INT8:
    case ORP_DATA_TYPE_UINT8:
    case ORP_DATA_TYPE_BOOLEAN:
        sz = sizeof(ORP_Vector_int8) + (length-1) * sizeof (int8);
        break;
    case ORP_DATA_TYPE_INT16:
    case ORP_DATA_TYPE_UINT16:
    case ORP_DATA_TYPE_CHAR:
        sz = sizeof(ORP_Vector_int16) + (length-1) * sizeof (int16);
        break;
    case ORP_DATA_TYPE_UINT32:
    case ORP_DATA_TYPE_INT32:
    case ORP_DATA_TYPE_F4:
        sz = sizeof(ORP_Vector_int32) + (length-1) * sizeof (int32);
        break;
    case ORP_DATA_TYPE_UINT64:
    case ORP_DATA_TYPE_INT64:
    case ORP_DATA_TYPE_F8:
        sz = sizeof(ORP_Vector_int64) + (length-1) * sizeof (int64);
        break;
    case ORP_DATA_TYPE_CLASS:
    case ORP_DATA_TYPE_ARRAY:
        sz = sizeof(ORP_Vector_Ref) + (length-1) * sizeof (void *);
        break;
    default:
        assert(0);
        break;
    }
    sz = ( ((sz + (GC_OBJECT_ALIGNMENT - 1)) & (~(GC_OBJECT_ALIGNMENT - 1))) );
    sz += OBJECT_HEADER_SIZE;
#else
    // This does not work because of alignement done for 8 byte longs.
    sz = sizeof(ORP_Vector) + (length) * orp_primitive_array_element_size(typ);
#ifndef EIGHT_BYTE_ALIGN_ARRAY
    if (orp_primitive_array_element_size(typ) == 8) {
        sz = sizeof(ORP_Vector_int64) + (length-1) * sizeof (int64);
    }
#endif
    sz += OBJECT_HEADER_SIZE;
    sz = ( ((sz + (GC_OBJECT_ALIGNMENT - 1)) & (~(GC_OBJECT_ALIGNMENT - 1))) );
    assert((sz % GC_OBJECT_ALIGNMENT) == 0);
#endif
    return sz;
} //orp_primitive_array_size



static Class *atype_to_class(unsigned atype)
{
    Global_Env *env = ORP_Global_State::loader_env;
    Class *clss = 0;
    switch(atype) {
    case JAVA_NEWARRAY_TYPE_CHAR:
        clss = env->ArrayOfChar_Class;
        break;
    case JAVA_NEWARRAY_TYPE_FLOAT:
        clss = env->ArrayOfFloat_Class;
        break;
    case JAVA_NEWARRAY_TYPE_DOUBLE:
        clss = env->ArrayOfDouble_Class;
        break;
    case JAVA_NEWARRAY_TYPE_BOOLEAN:
        clss = env->ArrayOfBoolean_Class;
        break;
    case JAVA_NEWARRAY_TYPE_BYTE:
        clss = env->ArrayOfByte_Class;
        break;
    case JAVA_NEWARRAY_TYPE_SHORT:
        clss = env->ArrayOfShort_Class;
        break;
    case JAVA_NEWARRAY_TYPE_INT:
        clss = env->ArrayOfInt_Class;
        break;
    case JAVA_NEWARRAY_TYPE_LONG:
        clss = env->ArrayOfLong_Class;
        break;
    default:
        assert(0);
        break;
    }
    return clss;
} //atype_to_class



unsigned orp_array_size(Class *vector_class, int length)
{
    unsigned sz;
    if(vector_class->is_array_of_primitives) {
        ORP_Data_Type primitive_type = (ORP_Data_Type)vector_class->name->bytes[1];
        assert(vector_class->vtable);
        sz = orp_primitive_array_size(primitive_type, length);
        assert((sz % GC_OBJECT_ALIGNMENT) == 0);
    } else {
        sz = sizeof(ORP_Vector_Ref) + (length-1) * sizeof (J_Reference);
        sz += OBJECT_HEADER_SIZE;
        // I think this is the fastest way to round up but the compiler probable does just as good a job.
        sz = ( ((sz + (GC_OBJECT_ALIGNMENT - 1)) & (~(GC_OBJECT_ALIGNMENT - 1))) );
#ifdef _DEBUG
        unsigned sz1 = sz + ((sz % GC_OBJECT_ALIGNMENT) ? (GC_OBJECT_ALIGNMENT - (sz % GC_OBJECT_ALIGNMENT)) : 0 ); 
        assert (sz == sz1);
#endif
        assert((sz % GC_OBJECT_ALIGNMENT) == 0);
    }
    return sz;
} //orp_array_size



static unsigned primitive_array_size(unsigned atype, int length)
{
    unsigned sz;
#if 1
    Class *clss = atype_to_class(atype);
    sz = orp_array_size(clss, length);
#else
    switch(atype) {
    case JAVA_NEWARRAY_TYPE_CHAR:
        sz = sizeof(JavaArrayOfChar) + (length-1) * sizeof(J_Char);
        break;
    case JAVA_NEWARRAY_TYPE_FLOAT:
        sz = sizeof(JavaArrayOfFloat) + (length-1) * sizeof(J_Float);
        break;
    case JAVA_NEWARRAY_TYPE_DOUBLE:
        sz = sizeof(JavaArrayOfDouble) + (length - 1) * sizeof(J_Double); // 8
        break;
    case JAVA_NEWARRAY_TYPE_BOOLEAN:
        sz = sizeof(JavaArrayOfBoolean) + (length-1)* sizeof(J_Boolean); // 1
        assert (sizeof(J_Boolean) == 1);
        break;
    case JAVA_NEWARRAY_TYPE_BYTE:
        sz = sizeof(JavaArrayOfByte) + (length-1)*sizeof (J_Byte);
        assert (sizeof (J_Byte) == 1);
        break;
    case JAVA_NEWARRAY_TYPE_SHORT:
        sz = sizeof(JavaArrayOfShort) + (length-1) * sizeof (J_Short);
        assert (sizeof(J_Short) == 2);
        break;
    case JAVA_NEWARRAY_TYPE_INT:
        sz = sizeof(JavaArrayOfInt) + (length-1) * sizeof (J_Int);
        assert (sizeof(J_Int) == 4);
        break;
    case JAVA_NEWARRAY_TYPE_LONG:
        sz = sizeof(JavaArrayOfLong) + (length-1) * sizeof (J_Long);
        assert (sizeof(J_Long) == 8);
        break;
    default:
        assert(0);
    }

    sz = sz + OBJECT_HEADER_SIZE;
    // align the object upwards
    sz = (((sz + (GC_OBJECT_ALIGNMENT - 1)) 
            / GC_OBJECT_ALIGNMENT) * GC_OBJECT_ALIGNMENT);
#endif
    return sz;
} //primitive_array_size



// Create a Java array
// There are two arguments:
//  - atype
//  - length
// Return value (eax):
//  - Reference to the new object
JavaArray *orp_newarray(unsigned atype, int length)
{
#ifdef ORP_STATS
    orp_stats_total.num_newarray++;
#endif
    assert(  !orp_is_gc_enabled(p_TLS_orpthread) );

    if (length < 0) {
        throw_java_exception("java/lang/NegativeArraySizeException");
    }

    Global_Env *env = ORP_Global_State::loader_env;

    switch(atype) {
    case JAVA_NEWARRAY_TYPE_CHAR:
#ifdef ORP_STATS
        orp_stats_total.num_newarray_char++;
#endif
        return (JavaArray *)create_array_of_chars(length);
    case JAVA_NEWARRAY_TYPE_FLOAT:
        {
#ifdef ORP_STATS
            orp_stats_total.num_newarray_float++;
#endif
            Class *clss = env->ArrayOfFloat_Class;
            assert(clss->vtable);
            unsigned sz = primitive_array_size(atype, length);
#ifdef OBJECT_SPLITTING
            JavaArrayOfFloat * _array = 
                        (JavaArrayOfFloat *) gc_malloc(
                                        get_proper_array_size((Partial_Reveal_Class * )clss, sz), 
                                        (Partial_Reveal_VTable *)clss->vtable);
#else
#ifdef GC_REWORK
            
            JavaArrayOfFloat * _array = 
                        (JavaArrayOfFloat *)gc_malloc(sz, (VTable *)clss->vtable);
#else
            JavaArrayOfFloat * _array = 
                        (JavaArrayOfFloat *)gc_malloc(sz, (Partial_Reveal_VTable *)clss->vtable);
#endif
#endif // OBJECT_SPLITTING
            gc_heap_slot_write_int32 ((Java_java_lang_Object *)_array,
                &(_array->length),
                length);
            assert(_array->vt == clss->vtable);
            return (JavaArray *)_array;
        }
    case JAVA_NEWARRAY_TYPE_DOUBLE:
        {
#ifdef ORP_STATS
            orp_stats_total.num_newarray_double++;
#endif
            Class *clss = env->ArrayOfDouble_Class;
            assert(clss->vtable);
            unsigned sz = primitive_array_size(atype, length);
#ifdef OBJECT_SPLITTING
            JavaArrayOfDouble * _array = (JavaArrayOfDouble *)
				gc_malloc(get_proper_array_size((Partial_Reveal_Class * )clss, sz), 
                          (Partial_Reveal_VTable *)clss->vtable);
#else
#ifdef GC_REWORK
            JavaArrayOfDouble * _array = (JavaArrayOfDouble *)
                gc_malloc(sz, (VTable *)clss->vtable);
#else
            JavaArrayOfDouble * _array = (JavaArrayOfDouble *)
                gc_malloc(sz, (Partial_Reveal_VTable *)clss->vtable);
#endif
#endif // OBJECT_SPLITTING 
            gc_heap_slot_write_int32 ((Java_java_lang_Object *)_array,
                &(_array->length),
                length);

            assert(_array->vt == clss->vtable);
            return (JavaArray *)_array;
        }
    case JAVA_NEWARRAY_TYPE_BOOLEAN:
        {
#ifdef ORP_STATS
            orp_stats_total.num_newarray_boolean++;
#endif
            Class *clss = env->ArrayOfBoolean_Class;
            assert(clss->vtable);
            unsigned sz = primitive_array_size(atype, length);
#ifdef OBJECT_SPLITTING
     		JavaArrayOfBoolean * _array = (JavaArrayOfBoolean *)
				gc_malloc(get_proper_array_size((Partial_Reveal_Class * )clss, sz), 
                          (Partial_Reveal_VTable *)clss->vtable);
#else
#ifdef GC_REWORK
            JavaArrayOfBoolean * _array = (JavaArrayOfBoolean *)
                gc_malloc(sz, (VTable *)clss->vtable);
#else
			JavaArrayOfBoolean * _array = (JavaArrayOfBoolean *)
                gc_malloc(sz, (Partial_Reveal_VTable *)clss->vtable);
#endif
#endif // OBJECT_SPLITTING
            gc_heap_slot_write_int32 ((Java_java_lang_Object *)_array,
                &(_array->length),
                length);

            assert(_array->vt == clss->vtable);
            return (JavaArray *)_array;
        }
    case JAVA_NEWARRAY_TYPE_BYTE:
        {
#ifdef ORP_STATS
            orp_stats_total.num_newarray_byte++;
#endif
            Class *clss = env->ArrayOfByte_Class;
            assert(clss->vtable);
            unsigned sz = primitive_array_size(atype, length);
#ifdef OBJECT_SPLITTING
			JavaArrayOfByte * _array = (JavaArrayOfByte *)
				gc_malloc(get_proper_array_size((Partial_Reveal_Class * )clss, sz), 
                          (Partial_Reveal_VTable *)clss->vtable);
#else
#ifdef GC_REWORK
            
            JavaArrayOfByte * _array = (JavaArrayOfByte *)
                gc_malloc(sz, (VTable *)clss->vtable);
#else
            JavaArrayOfByte * _array = (JavaArrayOfByte *)
                gc_malloc(sz, (Partial_Reveal_VTable *)clss->vtable);
#endif
#endif // OBJECT_SPLITTING
            gc_heap_slot_write_int32 ((Java_java_lang_Object *)_array,
                &(_array->length),
                length);

            assert(_array->vt == clss->vtable);
            return (JavaArray *)_array;
        }
    case JAVA_NEWARRAY_TYPE_SHORT:
        {
#ifdef ORP_STATS
            orp_stats_total.num_newarray_short++;
#endif
            Class *clss = env->ArrayOfShort_Class;
            assert(clss->vtable);
            unsigned sz = primitive_array_size(atype, length);
#ifdef OBJECT_SPLITTING
			JavaArrayOfShort * _array = (JavaArrayOfShort *)
				gc_malloc(get_proper_array_size((Partial_Reveal_Class * )clss, sz), 
                          (Partial_Reveal_VTable *)clss->vtable);
#else
#ifdef GC_REWORK
            
            JavaArrayOfShort * _array = (JavaArrayOfShort *)
                gc_malloc(sz, (VTable *)clss->vtable);
#else
            JavaArrayOfShort * _array = (JavaArrayOfShort *)
                gc_malloc(sz, (Partial_Reveal_VTable *)clss->vtable);
#endif
#endif // OBJECT_SPLITTING
            gc_heap_slot_write_int32 ((Java_java_lang_Object *)_array,
                &(_array->length),
                length);

            assert(_array->vt == clss->vtable);
            return (JavaArray *)_array;
        }
    case JAVA_NEWARRAY_TYPE_INT:
        {
#ifdef ORP_STATS
            orp_stats_total.num_newarray_int++;
#endif
            Class *clss = env->ArrayOfInt_Class;
            assert(clss->vtable);
            unsigned sz = primitive_array_size(atype, length);
#ifdef OBJECT_SPLITTING            
			JavaArrayOfInt * _array = (JavaArrayOfInt *)
				gc_malloc(get_proper_array_size((Partial_Reveal_Class * )clss, sz), 
                          (Partial_Reveal_VTable *)clss->vtable);
#else
#ifdef GC_REWORK
            
            JavaArrayOfInt * _array = (JavaArrayOfInt *)
                gc_malloc(sz, (VTable *)clss->vtable);
#else
            JavaArrayOfInt * _array = (JavaArrayOfInt *)
                gc_malloc(sz, (Partial_Reveal_VTable *)clss->vtable);
#endif

#endif // OBJECT_SPLITTING           
            gc_heap_slot_write_int32 ((Java_java_lang_Object *)_array,
                &(_array->length),
                length);

            assert(_array->vt == clss->vtable);
            return (JavaArray *)_array;
        }
    case JAVA_NEWARRAY_TYPE_LONG:
        {
#ifdef ORP_STATS
            orp_stats_total.num_newarray_long++;
#endif
            Class *clss = env->ArrayOfLong_Class;
            assert(clss->vtable);
            unsigned sz = primitive_array_size(atype, length);
#ifdef OBJECT_SPLITTING
            JavaArrayOfLong * _array = (JavaArrayOfLong *)
					gc_malloc(get_proper_array_size((Partial_Reveal_Class * )clss, sz), 
                              (Partial_Reveal_VTable *)clss->vtable);
#else
#ifdef GC_REWORK
            JavaArrayOfLong * _array = (JavaArrayOfLong *)
                    gc_malloc(sz, (VTable *)clss->vtable);
#else
            JavaArrayOfLong * _array = (JavaArrayOfLong *)
                    gc_malloc(sz, (Partial_Reveal_VTable *)clss->vtable);
#endif
#endif // OBJECT_SPLITTING
            gc_heap_slot_write_int32 ((Java_java_lang_Object *)_array,
                &(_array->length),
                length);

            assert(_array->vt == clss->vtable);
            return (JavaArray *)_array;
        }
    default:
        assert(0);
    }

    assert(0);
    return 0;
} //orp_newarray



// This is more elegant than orp_newarray (no big, ugly switch
// statement).  We should convert all our JIT's to call
// ORP_RT_NEW_VECTOR instead of ORP_RT_ANEWARRAY_RESOLVED and
// ORP_RT_NEWARRAY.
// This function should also replace orp_new_array_from_class.
ORP_Vector *orp_new_vector(Class *vector_class, int length)
{
    Global_Env *env = ORP_Global_State::loader_env;
    if(vector_class->is_array_of_primitives) {
        ORP_Data_Type primitive_type = (ORP_Data_Type)vector_class->name->bytes[1];
        assert(vector_class->vtable);
        unsigned sz = orp_primitive_array_size(primitive_type, length);
#ifdef GC_REWORK
        sz = ( ((sz + (GC_OBJECT_ALIGNMENT - 1)) & (~(GC_OBJECT_ALIGNMENT - 1))) );
        ORP_Vector *vector = (ORP_Vector *)
            gc_malloc(sz, (VTable *)vector_class->vtable);
#else
        ORP_Vector *vector = (ORP_Vector *)
            gc_malloc(sz, (Partial_Reveal_VTable *)vector_class->vtable);
#endif
        gc_heap_slot_write_int32 ((Java_java_lang_Object *)vector,
            &(vector->length),
            length);

        assert(vector->vt == vector_class->vtable);
        return (ORP_Vector *)vector;
    } else {
        return (ORP_Vector *)orp_anewarray_resolved_array_type(vector_class, length);
    }
    return 0;
} //orp_new_vector



static JavaArray *
orp_multianewarray_recursive(Class    *c,
                             int      *dims_array,
                             unsigned  dims,
                             Boolean   firstDimension)
{
    int length = *dims_array;
    assert(length >= 0);
    assert(c->name->bytes[0] == '[');
    assert(c->name->len > 1);

    if(dims == 1) {
        return (JavaArray *)orp_new_array_from_class(c, length);
    } else {
        const char *elem_type_name = c->name->bytes + 1;
        String *elem_type_str =
            ORP_Global_State::loader_env->string_pool.lookup(elem_type_name);
        Loader_Exception ld_exc;
        Class *elem_type_clss = class_load_verify_prepare(ORP_Global_State::loader_env,
                                                          elem_type_str,
                                                          &ld_exc);
        assert(elem_type_clss);
        volatile JavaArrayOfObject *object_array =
            orp_anewarray_resolved(elem_type_clss, length);

        GC_Frame f;
        orp_push_gc_frame(&f, (void *)&object_array, sizeof(object_array));

        for(int i = 0; i < length; i++) {
            JavaArray *elem = orp_multianewarray_recursive(elem_type_clss,
                                                           dims_array + 1,
                                                           dims - 1,
                                                           FALSE);
//            object_array->body[i] = (Java_java_lang_Object *)elem;

            gc_heap_slot_write_ref ((Java_java_lang_Object *)object_array, 
                (Java_java_lang_Object **)&(object_array->body[i]), 
                (Java_java_lang_Object *)elem);
        }

        orp_pop_gc_frame(&f);

        //assert(object_array->p_vtable->clss == c);
        return (JavaArray *)object_array;
    }
} //orp_multianewarray_recursive



// Create a multidimensional Java array
// There is a variable # of arguments:
//  - class id
//  - number of dimensions
//  - count1
//  ...
// Return value (eax):
//  - Reference to the new object
//
// This is __cdecl function and the caller must pop the arguments.
//
JavaArray *orp_multianewarray_resolved(Class *cc, unsigned dims, ...)
{
#ifdef ORP_STATS
    orp_stats_total.num_multianewarray++;
#endif
    assert(  !orp_is_gc_enabled(p_TLS_orpthread) );

    const unsigned max_dims = 100;
    int dims_array[max_dims];
    assert(dims <= max_dims);

    va_list args;
    va_start(args, dims);
    for(unsigned i = 0; i < dims; i++) {
        int d = va_arg(args, int);
        if (d < 0) {
            throw_java_exception("java/lang/NegativeArraySizeException");
        }
        dims_array[(dims - 1) - i] = d;
    }
    va_end(args);

    JavaArray *arr = orp_multianewarray_recursive(cc, dims_array, dims, TRUE);
    return arr;
} //orp_multianewarray_resolved





// end Java array allocation
/////////////////////////////////////////////////////////////



/////////////////////////////////////////////////////////////
// begin instanceof


Boolean orp_instanceof_class(Class *s, Class *t)
{
#ifdef ORP_STATS
    orp_stats_total.num_instanceof_class_reversed++;
#endif
    if(s == t)
        return TRUE;

    Global_Env *env = ORP_Global_State::loader_env;
    assert(env->JavaLangObject_Class);

    if(s->is_array) {
#ifdef ORP_STATS
        orp_stats_total.num_instanceof_s_is_array++;
#endif
        assert(*s->name->bytes == '[');
        if(t == env->JavaLangObject_Class) {
            return TRUE;
        }
        if(t == env->java_io_Serializable_Class) {
            return TRUE;
        }
        if(t == env->java_lang_Cloneable_Class) {
            return TRUE;
        }
        if(!t->is_array) {
            return FALSE;
        }

#ifdef ORP_STATS
        orp_stats_total.num_instanceof_class_reversed_recursive++;
#endif
        return orp_instanceof_class(s->array_element_class, t->array_element_class);
    } else {
        assert(*s->name->bytes != '[');
        if(!class_is_interface(t)) {
            for(Class *c = s; c; c = c->super_class) {
                if(c == t)
                    return TRUE;
            }
        } else {
#ifdef ORP_STATS
            orp_stats_total.num_instanceof_t_is_intf++;
#endif
            for(Class *c = s; c; c = c->super_class) {
                unsigned n_intf = c->n_superinterfaces;
		        for(unsigned i = 0; i < n_intf; i++) {
                    Class *intf = c->superinterfaces[i].clss;
                    assert(intf);
                    assert(class_is_interface(intf));
                    assert(class_is_interface(t));
                    if(orp_instanceof_class(intf, t)) {
#ifdef ORP_STATS
                        orp_stats_total.num_instanceof_t_is_intf_success++;
#endif
                        return TRUE;
                    }
                }
            }
        }
    }

    return FALSE;
} //orp_instanceof_class



#if 0 //defined(USE_IA64_JIT)

int orp_instanceof_class(Class *s, Class *t)
{
    return orp_instanceof_class_reversed(s, t);
}

#endif


int __stdcall orp_instanceof(Java_java_lang_Object *obj, Class *c)
{
    assert(  !orp_is_gc_enabled(p_TLS_orpthread) );

#ifdef ORP_STATS
    orp_stats_total.num_instanceof++;
    c->num_instanceof_slow++;
#endif
    if(!obj)
        return 0;
#ifdef ORP_STATS
    orp_stats_total.num_instanceof_non_null++;
#endif
    assert(obj->vt);
    Class *cc = obj->vt->clss;
    assert(cc);
    if(c == cc) {
        // Take care of the common case first.
#ifdef ORP_STATS
        orp_stats_total.num_instanceof_equal++;
#endif
        return 1;
    } else {
        return orp_instanceof_class(cc, c);
    }
} //orp_instanceof



// end instanceof
/////////////////////////////////////////////////////////////



/////////////////////////////////////////////////////////////
// begin get_interface_vtable


void *orp_get_interface_vtable(Java_java_lang_Object *obj, Class *iid)
{
    assert(obj);
    Class *clss = obj->vt->clss;
    unsigned num_intfc = clss->n_intfc_table_entries;
#ifdef ORP_STATS
    orp_stats_total.num_invokeinterface_calls++;
    switch(num_intfc) {
    case 1:
        orp_stats_total.num_invokeinterface_calls_size_1++;
        break;
    case 2:
        orp_stats_total.num_invokeinterface_calls_size_2++;
        break;
    default:
        orp_stats_total.num_invokeinterface_calls_size_many++;
        break;
    }
    if(num_intfc > orp_stats_total.invokeinterface_calls_size_max) {
        orp_stats_total.invokeinterface_calls_size_max = num_intfc;
    }
#endif
    for(unsigned i = 0; i < num_intfc; i++) {
        Class *intfc = clss->intfc_table_descriptors[i];
        if(intfc == iid) {
#ifdef ORP_STATS
            switch(i) {
            case 0:
                orp_stats_total.num_invokeinterface_calls_searched_1++;
                break;
            case 1:
                orp_stats_total.num_invokeinterface_calls_searched_2++;
                break;
            default:
                orp_stats_total.num_invokeinterface_calls_searched_many++;
                break;
            }
            if(i > orp_stats_total.invokeinterface_calls_searched_max) {
                orp_stats_total.invokeinterface_calls_searched_max = i;
            }
#endif
            unsigned char **table = clss->vtable->intfc_table->entry[i].table;
            return (void *)table;
        }
    }
    // assert(0);
    throw_java_exception("java/lang/IncompatibleClassChangeError");
    // Should throw IncompatibleClassChangeError here
    return NULL;
} //orp_get_interface_vtable


// end get_interface_vtable
/////////////////////////////////////////////////////////////
