// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/base/method.cpp,v 1.9 2001/11/21 00:54:06 michal Exp $
//


#include "platform.h"
#include <iostream.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

#include "orp_types.h"
#include "Class.h"
#include "environment.h"
#include "method_lookup.h"
#include "stack_manipulation.h"
#include "exceptions.h"
#include "compile.h"
#include "nogc.h"
#include "root_set_enum.h"
#include "object_layout.h"
#include "orp_utils.h"
#include "jit_intf.h"
#include "jit_intf_cpp.h"
#include "orp_synch.h"


/////////////////////////////////////////////////////////////////
// begin Method
/////////////////////////////////////////////////////////////////



Arg_List_Iterator initialize_arg_list_iterator(const char *descr)
{
    while(*descr != '(') {
        descr++;
    }
    assert(*descr == '(');
    return descr + 1;
} //initialize_iterator



Java_Type curr_arg(Arg_List_Iterator it)
{
    return (Java_Type)*(char *)it;
} //curr_arg



Class_Handle get_curr_arg_class(Arg_List_Iterator it,
                                Method_Handle m,
                                Loader_Exception *exc)
{
    Global_Env *env = ORP_Global_State::loader_env;
    Java_Type t = curr_arg(it);
    assert(t == JAVA_TYPE_CLASS || t == JAVA_TYPE_ARRAY);
    Arg_List_Iterator next = advance_arg_iterator(it);
    int len = ((char *)next) - ((char *)it);
    char *name = new char[len + 1];
    memcpy(name, it, len);
    name[len] = 0;
    String *n;
    if(name[0] == 'L') {
        // strip out 'L' and ';'
        name[len - 1] = 0;
        n = env->string_pool.lookup(name + 1);
    } else {
        n = env->string_pool.lookup(name);
    }
    delete []name;
    Class *clss = ((Method *)m)->get_class();
    Class *c = load_class_by_loader(env, n, clss->class_loader, exc);
    return c;
} //get_curr_arg_class



Arg_List_Iterator advance_arg_iterator(Arg_List_Iterator it)
{
    char *iter = (char *)it;
    while(*iter == '[')
        iter++;

    if(*iter == ')') {
        return iter;
    }

    if(*iter == 'L') {
        while(*iter++ != ';')
            ;
        return iter;
    }

    iter++;

    return iter;
} //advance_arg_iterator

#ifdef CLI_TESTING
char * Method::get_parameter_java_type(Arg_List_Iterator it)
{
    char *iter = (char *)it;
    uint32 count = 0;

    iter = (char *)it;

    if ( (*iter == '[') || (*iter == 'L') )
    {
        iter++; // skip [ or L
        while(*iter != ';') {
            iter++;
            count++;
        }  
        iter = (char *)it;
        iter++; // skip the '[' or 'L'
        char * rv = (char *)malloc(count +1);  // + 1 for null terminated string
        memset(rv, 0, count + 1);
        strncpy(rv, iter, count);
        return rv;
    }
    else
    {
        assert( *iter == 'B' || *iter == 'C' || *iter == 'D' ||
        *iter == 'F' || *iter == 'I' || *iter == 'J' ||
        *iter == 'S' || *iter == 'Z' || *iter == 'V');
        
        char * rv = (char *)malloc(2);
        memset(rv, 0, 2);
        strncpy(rv, iter, 1);
        return rv;
    }

}
#endif // CLI_TESTING

// Return the return type of this method.
Java_Type Method::get_return_java_type()
{
    const char *descr = get_descriptor();
    while(*descr != ')')
        descr++;
    return (Java_Type)*(descr + 1);
} //Method::get_return_java_type



Class_Handle method_get_return_type_class(Method_Handle m,
                                          Loader_Exception *exc)
{
    Method *method = (Method *)m;
    Global_Env *env = ORP_Global_State::loader_env;
    Java_Type t = method->get_return_java_type();
    assert(t == JAVA_TYPE_CLASS || t == JAVA_TYPE_ARRAY);
    const char *descr = method->get_descriptor();
    while(*descr != ')')
        descr++;
    descr++;
    String *n;
    if(descr[0] == 'L') {
        descr++;
        int len = strlen(descr);
        char *name = new char[len];
        memcpy(name, descr, len);
        assert(name[len - 1] == ';');
        name[len - 1] = 0;
        n = env->string_pool.lookup(name);
        delete []name;
    } else {
        n = env->string_pool.lookup(descr);
    }

    Class *clss = method->get_class();
    unsigned gc_enabled = orp_disable_gc();
    Class *c = load_class_by_loader(env, n, clss->class_loader, exc);
    if(gc_enabled)
	    orp_enable_gc();
    return c;
} //method_get_return_type_class



// Previously this method is not implemented
// Return the return class of this method (for non-primitive return types)
Class* Method::get_return_class_type()
{
/*
    const char *descr = get_descriptor();
    while(*descr != ')')
        descr++;
	char type = *(++descr); //pass through ')'
	if(type != 'L' && type != '[')return NULL;
	const char *p = descr;
	while(*p != ';')
		p++;
	int len = p - descr + 1;
	char *name = new char[len+1];
	
	if(type == 'L'){ //remove lead 'L' and tail ';'
		descr++;
		len -= 2; 
	}
	strncpy(name, descr, len);
	name[len] = '\0';
	String *s = ORP_Global_State::loader_env->string_pool.lookup(name);
	delete[] name;

    Loader_Exception ld_exc;
	
	orp_disable_gc(); 
    Class *clss =
        class_load_verify_prepare(ORP_Global_State::loader_env, s, &ld_exc);
	orp_enable_gc();
*/
	Loader_Exception ld_exc;
	Class *clss = (Class*)method_get_return_type_class(this, &ld_exc);
	
	return clss;
} //Method::get_return_class_type



Handler::Handler()
{
    _start_pc    = 0;
    _end_pc      = 0;
    _handler_pc  = 0;
    _catch_type  = NULL;
#ifdef CLI_TESTING
    _is_cli      = false;
    _cli_handler = 0;
#endif
} //Handler::Handler



unsigned Method::num_bc_exception_handlers()
{
    return _n_handlers;
} //Method::num_bc_exception_handlers



Handler *
Method::get_bc_exception_handler_info(unsigned eh_number)
{
    assert(eh_number < _n_handlers);
    Handler *h = _handlers + eh_number;
    return h;
} //Method::get_bc_exception_handler_info



void Method::set_num_target_exception_handlers(JIT *jit, unsigned n)
{
    JIT_Specific_Info *jit_info = get_JIT_specific_info(jit);
    jit_info->_num_target_exception_handlers       = n;
    jit_info->_target_exception_handlers = new Target_Exception_Handler_Ptr[n];
    memset(jit_info->_target_exception_handlers, 0, n * sizeof(Target_Exception_Handler_Ptr));
} //Method::set_num_target_exception_handlers



unsigned Method::get_num_target_exception_handlers(JIT *jit) {
    return get_JIT_specific_info(jit)->_num_target_exception_handlers; 
} //Method::get_num_target_exception_handlers



void Method::set_target_exception_handler_info(JIT *jit,
                                               unsigned eh_number,
                                               void *start_ip,
                                               void *end_ip,
                                               void *handler_ip,
                                               Class *catch_clss,
                                               bool exc_obj_is_dead
                                               )
{
    JIT_Specific_Info *jit_info = get_JIT_specific_info(jit);
    assert(eh_number < jit_info->_num_target_exception_handlers);
    jit_info->_target_exception_handlers[eh_number] =
        new Target_Exception_Handler(start_ip,
                                     end_ip,
                                     handler_ip,
                                     catch_clss,
                                     exc_obj_is_dead);
} //Method::set_target_exception_handler_info



Target_Exception_Handler_Ptr
Method::get_target_exception_handler_info(JIT *jit,
                                          unsigned eh_num)
{
    JIT_Specific_Info *jit_info = get_JIT_specific_info(jit);
    return jit_info->_target_exception_handlers[eh_num];
} //Method::get_target_exception_handler_info



Arg_List_Iterator Method::get_argument_list()
{
    return initialize_arg_list_iterator(get_descriptor());
} //Method::get_argument_list


unsigned Method::get_num_arg_bytes()
{
    unsigned nb;

    if(is_static())
        nb = 0;
    else
        nb = 4;

    Arg_List_Iterator iter = get_argument_list();
    Java_Type typ;
    while((typ = curr_arg(iter)) != JAVA_TYPE_END) {
        switch(typ) {
        case JAVA_TYPE_LONG:
        case JAVA_TYPE_DOUBLE:
            nb += 8;
            break;
#ifdef CLI_TESTING
        case JAVA_TYPE_STAR:
			nb += 4;
			// Added so that we can skip over whatever follows "*"
			iter = advance_arg_iterator(iter);
			break;
#endif
		default:
            nb += 4;
            break;
        }
        iter = advance_arg_iterator(iter);
    }

    return nb;
} //Method::get_num_arg_bytes



unsigned Method::get_num_ref_args()
{
    unsigned nargs;

    if(is_static())
        nargs = 0;
    else
        nargs = 1;

    Arg_List_Iterator iter = get_argument_list();
    Java_Type typ;
    while((typ = curr_arg(iter)) != JAVA_TYPE_END) {
        switch(typ) {
        case JAVA_TYPE_CLASS:
        case JAVA_TYPE_ARRAY:
            nargs++;
            break;
        default:
            break;
        }
        iter = advance_arg_iterator(iter);
    }

    return nargs;
} //Method::get_num_ref_args



unsigned Method::get_num_args()
{
    unsigned nargs;

    if(is_static()) {
        nargs = 0;
    } else {
        nargs = 1;
    }

    Arg_List_Iterator iter = get_argument_list();
    Java_Type typ;
    while((typ = curr_arg(iter)) != JAVA_TYPE_END) {
        nargs++;
        iter = advance_arg_iterator(iter);
    }

    return nargs;
} //Method::get_num_args



void *Method::allocate_code_block(size_t size, JIT *jit)
{
    _code_block_size = size;
    if(!size) {
        _code_block = NULL;
    } else {
        // Try to guarantee that _code_block is always 16-byte aligned
        // (i.e., aligned to a cache line).  We assume that
        // gc_malloc_fixed_code_for_jit() always returns 8-byte-aligned data.
        _code_block = gc_malloc_fixed_code_for_jit(size + 8);
        assert((((unsigned)_code_block) & 7) == 0);
        if (((unsigned)_code_block) & 8)
            _code_block = (void *)((char *)_code_block + 8);
    }

    JIT_Specific_Info *jit_info = get_JIT_specific_info(jit);
    jit_info->_code_block       = _code_block;
    jit_info->_code_block_size  = _code_block_size;
    return _code_block;
} //Method::allocate_code_block



// Read/Write data block.
void *Method::allocate_rw_data_block(size_t size, JIT *jit)
{
    void *rw_data_block;
    if(!size) {
        rw_data_block = NULL;
    } else {
        rw_data_block = gc_malloc_fixed_data_C(size);
    }
    JIT_Specific_Info *jit_info   = get_JIT_specific_info(jit);
    jit_info->_rw_data_block      = rw_data_block;
    jit_info->_rw_data_block_size = size;
    return rw_data_block;
} //Method::allocate_rw_data_block



// Allocate memory for extra information needed by JIT.
void *Method::allocate_jit_info_block(size_t size, JIT *jit)
{
    _jit_info_block_size = size;
    Byte *jit_info_block;
    if(!size) {
        jit_info_block = NULL;
    } else {
        jit_info_block = (Byte *)gc_malloc_fixed_data_C(size + sizeof(JIT **));
    }

    // Store a pointer to the JIT before the JIT info block.
    *(JIT **)jit_info_block = jit;
    _jit_info_block = jit_info_block + sizeof(JIT **);

    JIT_Specific_Info *jit_info = get_JIT_specific_info(jit);
    jit_info->_jit_info_block = _jit_info_block;
    jit_info->_jit_info_block_size = _jit_info_block_size;
    return _jit_info_block;
} //Method::allocate_jit_info_block



void Method::set_jit(JIT *jit)
{
    _jit = jit;
    get_JIT_specific_info(jit);
} //Method::set_jit



JIT_Specific_Info *Method::get_JIT_specific_info_no_create(JIT *jit)
{
    JIT_Specific_Info *jit_info;
    for(jit_info = get_first_JIT_specific_info(); jit_info; jit_info = jit_info->_next) {
        if(jit_info->get_jit() == jit) {
            return jit_info;
        }
    }
    return 0;
} //Method::get_JIT_specific_info_no_create



JIT_Specific_Info *Method::get_JIT_specific_info(JIT *jit)
{
    JIT_Specific_Info *jit_info = get_JIT_specific_info_no_create(jit);
    if(jit_info) {
        return jit_info;
    }

    jit_info = (JIT_Specific_Info *)gc_malloc_fixed_data_C(sizeof(JIT_Specific_Info));
    memset(jit_info, 0, sizeof(JIT_Specific_Info));
    jit_info->set_jit(jit);
    jit_info->set_method(this);
    jit_info->_next = _jits;
    _jits = jit_info;
    return jit_info;
} //Method::get_JIT_specific_info



void *Method::allocate_JIT_data_block(size_t size, JIT *jit)
{
    JIT_Data_Block *block =
        (JIT_Data_Block *)gc_malloc_fixed_data_C(size + sizeof(JIT_Data_Block) - 1);
    JIT_Specific_Info *jit_info;
    for(jit_info = _jits; jit_info; jit_info = jit_info->_next) {
        if(jit_info->get_jit() == jit) {
            block->next = jit_info->_data_blocks;
            jit_info->_data_blocks = block;
            break;
        }
    }
    if(!jit_info) {
        jit_info = (JIT_Specific_Info *)gc_malloc_fixed_data_C(sizeof(JIT_Specific_Info));
        memset(jit_info, 0, sizeof(JIT_Specific_Info));
        jit_info->set_jit(jit);
        jit_info->_next = _jits;
        _jits = jit_info;
        block->next = jit_info->_data_blocks;
        jit_info->_data_blocks = block;
    }
    return &(block->bytes[0]);
} //Method::allocate_JIT_data_block



void *Method::get_first_JIT_data_block(JIT *jit)
{
    JIT_Specific_Info *jit_info;
    for(jit_info = _jits; jit_info; jit_info = jit_info->_next) {
        if(jit_info->get_jit() == jit) {
            return &(jit_info->_data_blocks->bytes[0]);
        }
    }
    assert(0);
    return 0;
} //Method::get_first_JIT_data_block



void *Method::get_first_JIT_data_block(void *info_block)
{
    JIT *jit = *(((JIT **)info_block) - 1);
    return get_first_JIT_data_block(jit);
} //Method::get_first_JIT_data_block



void *Method::get_next_JIT_data_block(void *b)
{
    JIT_Data_Block *block = (JIT_Data_Block *)(((char *)b) - sizeof(JIT_Data_Block *));
    JIT_Data_Block *next = block->next;
    if(next) {
        return &(next->bytes[0]);
    } else {
        return 0;
    }
} //Method::get_next_JIT_data_block



void Method::add_vtable_patch(void *patch)
{
    p_vtable_patch_lock->_lock();                         // vvv

    if(!_vtable_patch) {
        VTable_Patches *vp =
            (VTable_Patches *)gc_malloc_fixed_data_C(sizeof(VTable_Patches));
        memset(vp, 0, sizeof(VTable_Patches));
        _vtable_patch = vp;
    }


    VTable_Patches *curr_vp = _vtable_patch;
    for(int i = 0; i < MAX_VTABLE_PATCH_ENTRIES; i++) {
        if(!curr_vp->patch_table[i]) {
            curr_vp->patch_table[i] = patch;
            p_vtable_patch_lock->_unlock();               // ^^
            return;
        }
    }
    VTable_Patches *new_vp =
        (VTable_Patches *)gc_malloc_fixed_data_C(sizeof(VTable_Patches));
    memset(new_vp, 0, sizeof(VTable_Patches));
    new_vp->next = curr_vp;
    _vtable_patch = new_vp;
    new_vp->patch_table[0] = patch;


    p_vtable_patch_lock->_unlock();                     // ^^^
} //Method::add_vtable_patch



void Method::apply_vtable_patches()
{
    p_vtable_patch_lock->_lock();

    if(!_vtable_patch) {
        // Constructors are never entered into a vtable.
        p_vtable_patch_lock->_unlock();
        return;
    }

    void *code_addr = get_code_addr();
#ifdef CLI_TESTING
    Class *clss = get_class();
    if(clss->is_value_type) {
        //printf("%s is value type\n", get_class()->name->bytes);
        set_unboxer_addr(create_unboxer(this));
        code_addr = get_unboxer_addr();
    }
#endif

    VTable_Patches *vp = (VTable_Patches *)_vtable_patch;
    for(; vp; vp = vp->next) {
        for(int i = 0; i < MAX_VTABLE_PATCH_ENTRIES; i++) {
            if(vp->patch_table[i]) {
                *((void **)vp->patch_table[i]) = code_addr;
            }
        }
    }

    p_vtable_patch_lock->_unlock();
} //Method::apply_vtable_patches



unsigned Method::num_exceptions_method_can_throw() 
{
	return _n_exceptions;	
} //Method::num_exceptions_method_can_throw



String* Method::get_exception_name (int n) 
{
	if (!_exceptions || (n >= _n_exceptions)) return NULL;
	return _exceptions[n];
}



//////////////////////////////////////////////////////
// begin inlining support


struct Inline_Record {
    JIT *jit;
    Method *caller;
    Inline_Record *next;
};



void Method::set_inline_assumption(JIT *jit, Method *caller) 
{
#if 0
    printf("Jit 0x%x inlined %s.%s in %s.%s\n",
           (int)jit,
           caller->get_class()->name->bytes,
           caller->get_name()->bytes,
           get_class()->name->bytes,
           get_name()->bytes
           );
#endif
    Inline_Record *ir = (Inline_Record *)malloc(sizeof(Inline_Record));
    ir->jit    = jit;
    ir->caller = caller;
    ir->next   = inline_records;
    inline_records = ir;
} //Method::set_inline_assumption



void Method::method_was_overridden() 
{
#if 0
    printf("Method %s.%s was overrriden\n",
           get_class()->name->bytes,
           get_name()->bytes
           );
#endif

    _flags.is_overridden = 1;

    while(inline_records) {
        Inline_Record *curr = inline_records;
        curr->jit->method_was_overridden(curr->caller, this);
        inline_records = curr->next;
        free(curr);
    }
} //Method::method_was_overridden



// end inlining support
//////////////////////////////////////////////////////




//////////////////////////////////////////////////////
// begin nop analysis


Method_Handle resolve_special_method_env(Global_Env *env,
                                         Class_Handle c,
                                         unsigned index,
                                         Loader_Exception *exc);


enum Nop_Stack_State {
    NS_StackEmpty,
    NS_ThisPushed,
    NS_ThisAndZeroPushed
};

void Method::_set_nop()
{
    bool verbose = false;

    Global_Env *env = ORP_Global_State::loader_env;
    if(get_signature() != env->Default_Constructor_Signature) {
        return;
    }

    if(is_native()) {
        return;
    }
    //return;
    unsigned len = _byte_code_length;
    if(!len) {
        return;
    }
    Byte *bc = _byte_codes;
    Nop_Stack_State stack_state = NS_StackEmpty;
    if(verbose) {
        printf("=========== nop[%d]: %s.%s%s\n", len, get_class()->name->bytes, get_name()->bytes, get_descriptor());
    }
    for(unsigned idx = 0; idx < len; idx++) {
        Byte b = bc[idx];
        if(verbose) {
            printf("\tbc[%d]=%d, state=%d\n", idx, b, stack_state);
        }
        if(b == 0xb1) {   // return
            if(verbose) {
                printf("+++++++ nop: %s.%s%s\n", get_class()->name->bytes, get_name()->bytes, get_descriptor());
            }
            _flags.is_nop = TRUE;
            return;
        }
        switch(stack_state) {
        case NS_StackEmpty:
            switch(b) {
    		case 0x2a:  // aload_0
                stack_state = NS_ThisPushed;
                break;
            default:
                return;
            }
            break;
        case NS_ThisPushed:
            switch(b) {
    		case 0x01:  // aconst_null
    		case 0x03:  // iconst_0
                stack_state = NS_ThisAndZeroPushed;
                break;
    		case 0xb7:  // invokespecial
                {
                    unsigned index = (bc[idx + 1] << 8) + bc[idx + 2];
                    if(verbose) {
                        printf("\tinvokespecial, index=%d\n", index);
                    }
                    Method_Handle mh = resolve_special_method_env(ORP_Global_State::loader_env,
                                                                  get_class(),
                                                                  index,
                                                                  0);
                    Method *callee = (Method *)mh;
                    if(!callee) {
                        if(verbose) {
                            printf("\tinvokespecial, callee==null\n");
                        }
                        return;
                    }
                    if(callee == this) {
                        return;
                    }
                    if(verbose) {
                        printf("invokespecial: %s.%s%s\n", callee->get_class()->name->bytes, callee->get_name()->bytes, callee->get_descriptor());
                    }
                    if(!callee->is_nop()) {
                        return;
                    }
                    const char *descr = callee->get_descriptor();
                    if(descr[1] != ')') {
                        return;
                    }
                    if(verbose) {
                        printf("invokespecial nop: %s.%s%s\n", callee->get_class()->name->bytes, callee->get_name()->bytes, callee->get_descriptor());
                    }
                }
                stack_state = NS_StackEmpty;
                idx += 2;
                break;
            default:
                return;
            }
            break;
        case NS_ThisAndZeroPushed:
            switch(b) {
    		case 0xb5:  // putfield
                stack_state = NS_StackEmpty;
                if(verbose) {
                    printf("\tputfield\n");
                }
                idx += 2;
                break;
            default:
                return;
            }
            break;
        default:
            assert(0);
            return;
        }
    }
    assert(0);
} //Method::_set_nop


// end nop analysis
//////////////////////////////////////////////////////



#ifdef INTERPRETER

void Method::create_interpreter_Object_Handle_Struct(void *p_first, void *p_remaining)
{
    int32 *first_arg = (int32 *)p_first;
    int32 *remaining_args = (int32 *)p_remaining;

    unsigned n_ref_args = get_num_ref_args();
    if(is_static())
        n_ref_args++;

    Object_Handle_Struct *p_ohs = (Object_Handle_Struct *)
        malloc( sizeof(Object_Handle_Struct) * n_ref_args);
    // BUGBUG need to free the malloc'ed space above
    memset(p_ohs, 0, sizeof(Object_Handle_Struct) * n_ref_args);

    for (unsigned xx = 0; xx < n_ref_args; xx++) {
        p_ohs[xx].next = &(p_ohs[xx + 1]);
        p_ohs[xx + 1].prev = &(p_ohs[xx]);        
    }

    p_ohs[0].java_reference = (Java_java_lang_Object *)first_arg[0];
    first_arg[0] = (long)&(p_ohs[0]);

    unsigned ohs_cursor = 1;
    unsigned arg_cursor = 0;

    Arg_List_Iterator iter = get_argument_list();
    Java_Type typ;
    while((typ = curr_arg(iter)) != JAVA_TYPE_END) {
        switch(typ) {
        case JAVA_TYPE_CLASS:
        case JAVA_TYPE_ARRAY:
            {
                p_ohs[ohs_cursor].java_reference = (Java_java_lang_Object *)remaining_args[arg_cursor];
                remaining_args[arg_cursor] = (long)&(p_ohs[ohs_cursor]);
                ohs_cursor++;
                break;
            }
        default:
            break;
        }
        iter = advance_arg_iterator(iter);
        arg_cursor++;
    }

    assert(ohs_cursor == n_ref_args);

    extern ORP_thread *p_TLS_orpthread;

    interp_frame *p_cursor = p_TLS_orpthread->p_lif;
    while (p_cursor) {
        assert(p_cursor->p_jni_object_handles == 0);
        p_cursor = p_cursor->p_prev;
    }

    assert(p_TLS_orpthread->p_lif->p_jni_object_handles == 0);
    p_TLS_orpthread->p_lif->p_jni_object_handles = (void *)p_ohs;
}

#endif  // INTERPRETER
    
/////////////////////////////////////////////////////////////////
// end Method
/////////////////////////////////////////////////////////////////

