// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/base/orp.cpp,v 1.18 2002/01/10 08:45:31 xli18 Exp $
//


#include "platform.h"
#include <iostream.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

#include "orp_types.h"
#include "Class.h"
#include "environment.h"
#include "method_lookup.h"
#include "stack_manipulation.h"
#include "exceptions.h"
#include "compile.h"
#include "gc_for_orp.h"
#include "root_set_enum.h"
#include "object_layout.h"
#include "orp_utils.h"
#include "jit_intf.h"
#include "orp_synch.h"
#include "orp_threads.h"
#include "orp_stats.h"
#include "orp_vtune.h"
#include "ini.h"

#ifdef OBJECT_LOCK_V2
#include "object_generic_olv2.h"
#include "thread_manager_olv2.h"
#include "mon_enter_exit_olv2.h"
#else
#include "object_generic.h"
#include "thread_manager.h"
#include "mon_enter_exit.h"
#endif

#include "jni.h"
#include "jni_direct.h"
#include "native_utils.h"



#ifdef _DEBUG
#include "jvmdi_clean.h"
#endif

#ifdef ORP_POSIX
#include "platform2.h"
#endif

Class *class_java_lang_class = 0;

Global_Env *ORP_Global_State::loader_env = 0;



// It is unclear if this can be changed to return a 
// (Java_java_lang_Object *) 
void *gc_malloc_fixed_data_java(unsigned size, VTable *p_vtable)
{
#ifdef GC_REWORK
    return (void *)gc_malloc(size, (VTable *)p_vtable);
#else
    return (void *)gc_malloc(size, (Partial_Reveal_VTable *)p_vtable);
#endif
} //gc_malloc_fixed_data_java


/////////////////////////////////////////////////////////////////
// begin VM data structures alloc
/////////////////////////////////////////////////////////////////


JavaArray *orp_pinned_java_array(Java_Type elem_type, int length)
{
    return 0;
} //orp_pinned_java_array


/////////////////////////////////////////////////////////////////
// end VM data structures alloc
/////////////////////////////////////////////////////////////////



/////////////////////////////////////////////////////////////////
// begin Class
/////////////////////////////////////////////////////////////////


Java_Type class_get_field_type_no_resolve(Class *clss, unsigned cp_index)
{
	Const_Pool *cp = clss->const_pool;
    assert(cp_index <= clss->cp_size);
    assert(cp_is_fieldref(cp, cp_index));
    unsigned class_index = cp[cp_index].cp_tag.class_index;
    unsigned name_type_index = cp[cp_index].cp_tag.name_and_type_index;
    Signature *sig = cp[name_type_index].signature;
	String *const descriptor = sig->descriptor;
    return JAVA_TYPE_BOOLEAN;
} //class_get_field_type_no_resolve




Field *class_resolve_static_field(Class *clss, unsigned cp_index)
{
    Compilation_Handle ch;
    ch.env = ORP_Global_State::loader_env;
    Field_Handle fh = resolve_static_field((Compile_Handle)&ch,
                                           (Class_Handle)clss,
                                            cp_index,
                                            0);
    return (Field *)fh;
} //class_resolve_static_field



Method *class_resolve_static_method(Class *clss, unsigned cp_index)
{
    Compilation_Handle ch;
    ch.env = ORP_Global_State::loader_env;
    Field_Handle fh = resolve_static_method((Compile_Handle)&ch,
                                            (Class_Handle)clss,
                                            cp_index,
                                            0);
    return (Method *)fh;
} //class_resolve_static_field

Method *class_resolve_virtual_method(Class *clss, unsigned cp_index)
{
    Compilation_Handle ch;
    ch.env = ORP_Global_State::loader_env;
    Field_Handle fh = resolve_virtual_method((Compile_Handle)&ch,
                                               (Class_Handle)clss,
                                               cp_index,
                                               0);
    return (Method *)fh;
}

Method *class_resolve_special_method(Class *clss, unsigned cp_index)
{
    Compilation_Handle ch;
    ch.env = ORP_Global_State::loader_env;
    Field_Handle fh = resolve_special_method((Compile_Handle)&ch,
                                               (Class_Handle)clss,
                                               cp_index,
                                               0);
    return (Method *)fh;
}


Method *class_resolve_nonstatic_method(Class *clss, unsigned cp_index)
{
    Compilation_Handle ch;
    ch.env = ORP_Global_State::loader_env;
    Field_Handle fh = resolve_nonstatic_method((Compile_Handle)&ch,
                                               (Class_Handle)clss,
                                               cp_index,
                                               0);
    return (Method *)fh;
} //class_resolve_static_field



Method *class_resolve_interface_method(Class *clss, unsigned cp_index)
{
    Compilation_Handle ch;
    ch.env = ORP_Global_State::loader_env;
    Field_Handle fh = resolve_interface_method((Compile_Handle)&ch,
                                               (Class_Handle)clss,
                                               cp_index,
                                               0);
    return (Method *)fh;
} //class_resolve_static_field



Field *class_resolve_nonstatic_field(Class *clss, unsigned cp_index)
{
    Compilation_Handle ch;
    ch.env = ORP_Global_State::loader_env;
    Field_Handle fh = resolve_nonstatic_field((Compile_Handle)&ch,
                                              (Class_Handle)clss,
                                              cp_index,
                                              0);
    return (Field *)fh;
} //class_resolve_static_field



Class *class_resolve_class(Class *clss, unsigned cp_index)
{
	Const_Pool *cp = clss->const_pool;
    assert(cp_index <= clss->cp_size);
    if(cp_is_resolved(cp, cp_index)) {
        Class *c = cp[cp_index].clss;
        return c;
    } else {
        String *s = cp[cp_index].string;
        Loader_Exception ld_exc;
        Class *c = class_load_verify_prepare(ORP_Global_State::loader_env, s, &ld_exc);
        if(!c) {
            orp_cout << "Couldn't load a class" << endl;
            assert(0);
            orp_exit(1);
        }
        assert(c);    // for now

        cp[cp_index].clss = c;
        cp_set_resolved(cp, cp_index);

        return c;
    }
} //class_resolve_class




// A run-time error occurs if called for an index which does not represent
// a constant of type String, Integer, Float, Long or Double.
Java_Type class_get_cp_const_type(Class *clss, unsigned cp_index)
{
	Const_Pool *cp = clss->const_pool;
    assert(cp_is_cnst(cp, cp_index));
    switch(cp_tag(cp, cp_index)) {
    case CONSTANT_String:
        return JAVA_TYPE_STRING;
    case CONSTANT_Integer:
            return JAVA_TYPE_INT;
    case CONSTANT_Float:
            return JAVA_TYPE_FLOAT;
    case CONSTANT_Long:
            return JAVA_TYPE_LONG;
    case CONSTANT_Double:
            return JAVA_TYPE_DOUBLE;
    
    default:
        assert(0);
    }
    return JAVA_TYPE_INVALID;
} //class_get_cp_const_type



// Returns an address of an int, float, etc.
// For a string it returns a pointer to the utf8 representation of the string.
const void *class_get_addr_of_constant(Class *clss, unsigned cp_index)
{
	Const_Pool *cp = clss->const_pool;
    assert(cp_tag(cp, cp_index) != CONSTANT_String);

    return (void *)(cp + cp_index);
} //class_get_addr_of_constant



// Initializes a class.

void class_initialize1(Class *clss, bool throw_exception)
{
    // the following code implements the 11-step class initialization program
    // described in page 226, section 12.4.2 of Java Language Spec, 1996
    // ISBN 0-201-63451-1

    // ---  step 1   ----------------------------------------------------------

    orp_monitor_enter( (Java_java_lang_Object *)clss );
#ifdef _DEBUG
    if (clss->p_initializing_thread) {
        volatile ORP_thread * p_scan = p_active_threads_list;
        while (p_scan) {
            if (p_scan == (ORP_thread *)clss->p_initializing_thread) break;
            p_scan = p_scan->p_active;
        }
        assert(p_scan);     // make sure p_initializer_thread is legit
    }
#endif  

    // ---  step 2   ----------------------------------------------------------

    while (  ( (ORP_thread *)(clss->p_initializing_thread) != p_TLS_orpthread)  &&
             (clss->state == ST_Initializing)                      ) {

         java_lang_Object_wait ( (Java_java_lang_Object *)clss, 0);
    }

    // ---  step 3   ----------------------------------------------------------

    if ( (ORP_thread *)(clss->p_initializing_thread) == p_TLS_orpthread) {
        orp_monitor_exit( (Java_java_lang_Object *)clss );
        return;
    }

    // ---  step 4   ----------------------------------------------------------

    if (clss->state == ST_Initialized) {
        orp_monitor_exit( (Java_java_lang_Object *)clss );
        return;
    }

    // ---  step 5   ----------------------------------------------------------

    if (clss->state == ST_Error) {

        orp_monitor_exit( (Java_java_lang_Object *)clss );
        throw_java_exception("java/lang/NoClassDefFoundError");
        assert(0); // previous line should never return
    }

    // ---  step 6   ----------------------------------------------------------

    assert(clss->state == ST_Prepared);
    clss->state = ST_Initializing;
    assert(clss->p_initializing_thread == 0);
    clss->p_initializing_thread = (void *)p_TLS_orpthread;
    orp_monitor_exit( (Java_java_lang_Object *)clss );

    // ---  step 7 ------------------------------------------------------------

    if(clss->super_class) {
        class_prepare(clss->super_class);
        class_initialize(clss->super_class);

        if(clss->super_class->state == ST_Error) {

#if defined(USE_IA64_JIT)
            printf("exceptions are not implemented on IA64\n");
            assert(0);
            orp_exit(1);
#else
            orp_monitor_enter( (Java_java_lang_Object *)clss );

            clss->state = ST_Error;
            clss->p_initializer_error_object =
            clss->super_class->p_initializer_error_object;
            clss->p_initializing_thread = 0;
            java_lang_Object_notifyAll( (Java_java_lang_Object *)clss );

            orp_monitor_exit( (Java_java_lang_Object *)clss );

            Frame_Context context;
            // Force an unwind from a native in the first iteration
            context.ljf   = (uint32)get_orp_last_java_frame();
            context.eip   = 0;
            context.p_eip = &(context.eip);
            Boolean result = unwind_to_next_java_frame(&context, FALSE);
            if (result) {

                orp_athrow(FALSE,
                           *context.p_edi,
                           *context.p_esi,
                           *context.p_ebx,
                           *context.p_ebp,
                           context.esp - 8,
                           *context.p_eip,
                           (Java_java_lang_Object *)clss->p_initializer_error_object);
            }
            else {
                print_uncaught_exception_message(stderr,
                    (Java_java_lang_Throwable*) clss->p_initializer_error_object);
                orp_exit(1);
            }

            assert(0);
#endif
        }

    }

    // ---  step pre-8   ------------------------------------------------------
    // This is an extra step needed to initialize static final fields

    unsigned n_fields = clss->n_fields;
    for(unsigned i = 0; i < n_fields; i++) {
        Field *f = &clss->fields[i];
        if(f->is_static() && f->is_final() && f->get_const_value_index()) {
            //printf("Field %s.%s has a value\n", f->get_class()->name->bytes, f->get_name()->bytes);
            void *addr = f->get_address();
            Const_Java_Value cv = f->get_const_value();
            switch(f->get_java_type()) {
            case JAVA_TYPE_BYTE:
            case JAVA_TYPE_BOOLEAN:
                *(int8 *)addr = (int8)cv.i;
                break;
            case JAVA_TYPE_CHAR:
            case JAVA_TYPE_SHORT:
                *(int16 *)addr = (int16)cv.i;
                break;
            case JAVA_TYPE_INT:
                *(int32 *)addr = cv.i;
                break;
            case JAVA_TYPE_LONG:
                *(int64 *)addr = cv.j;
                break;
            case JAVA_TYPE_FLOAT:
                *(float *)addr = cv.f;
                break;
            case JAVA_TYPE_DOUBLE:
                *(double *)addr = cv.d;
                break;
            default:
                assert(!strcmp("Ljava/lang/String;", f->get_descriptor()));
                String *str = cv.string;
                Java_java_lang_String *jls = create_java_lang_String_from_C_string(str->bytes);
                *(Java_java_lang_String **)addr = jls;
                break;
            }
        }
    }


    // ---  step 8   ----------------------------------------------------------


    Method *meth = clss->static_initializer;

    if(!meth) {
        orp_monitor_enter( (Java_java_lang_Object *)clss );
        clss->state = ST_Initialized;
        clss->p_initializing_thread = 0;
        java_lang_Object_notifyAll ( (Java_java_lang_Object *)clss );
        orp_monitor_exit( (Java_java_lang_Object *)clss );
        return;
    }

    void *a = meth->get_indirect_address();
    void *aa = *(void **)a;

    Java_java_lang_Object *p_error_object = 0;

    orp_execute_java_method_array(meth, 0, 0);
    p_error_object = (Java_java_lang_Object *)get_current_thread_exception();
    clear_current_thread_exception();

    // ---  step 9   ----------------------------------------------------------

    if(!p_error_object) {
        orp_monitor_enter( (Java_java_lang_Object *)clss );
        clss->state = ST_Initialized;
        clss->p_initializing_thread = 0;
        assert(clss->p_initializer_error_object == 0);
        java_lang_Object_notifyAll ( (Java_java_lang_Object *)clss );
        orp_monitor_exit( (Java_java_lang_Object *)clss );
        return;
    }

    // ---  step 10  ----------------------------------------------------------

    if(p_error_object) {

        Class *p_error_class = p_error_object->vt->clss;
        Class *jle = ORP_Global_State::loader_env->java_lang_Error_Class;

        while(p_error_class && p_error_class != jle) {
            p_error_class = p_error_class->super_class;
        }

        if((!p_error_class) || (p_error_class != jle) ) {
            Class *eiie =
                ORP_Global_State::loader_env->java_lang_ExceptionInInitializerError_Class;

            //BUGBUG if class_alloc_new_object() below hits out of memory
            //BUGBUG then throw OutOfMemoryError
            clss->p_initializer_error_object = class_alloc_new_object(eiie);
            // Run <init> here!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        } else {
            clss->p_initializer_error_object = p_error_object;
        }

    // ---  step 11  ----------------------------------------------------------

        orp_monitor_enter( (Java_java_lang_Object *)clss );
        clss->state = ST_Error;
        clss->p_initializing_thread = 0;
        java_lang_Object_notifyAll ( (Java_java_lang_Object *)clss );
        orp_monitor_exit( (Java_java_lang_Object *)clss );

#ifdef USE_IA64_JIT
        printf("exceptions are not implemented on IA64\n");
        assert(0);
        orp_exit(1);
#else
        Frame_Context context;
        // Force an unwind from a native in the first iteration
        context.ljf   = (uint32)get_orp_last_java_frame();
        context.eip   = 0;
        context.p_eip = &(context.eip);
        Boolean result = unwind_to_next_java_frame(&context, FALSE);

        if (result) {
            orp_athrow(FALSE,
                       *context.p_edi,
                       *context.p_esi,
                       *context.p_ebx,
                       *context.p_ebp,
                       context.esp - 4,
                       *context.p_eip,
                       (Java_java_lang_Object *)clss->p_initializer_error_object);
        }
        else {
            print_uncaught_exception_message(stderr,
                (Java_java_lang_Throwable*) clss->p_initializer_error_object);
            orp_exit(1);
        }

            assert(0);
#endif
    }
    // end of 11 step class initialization program

} //class_initialize1


void class_initialize_from_jni(Class *clss, bool throw_exception)
{
    // jni functions assume gc is enabled, but "corp" orp internals assume 
    // gc is disabled, thus disable gc when calling from jni

    bool prev_state = orp_disable_gc();
    assert(prev_state);
    class_initialize1(clss, throw_exception);
    orp_enable_gc();
} // class_initialize_from_jni


void class_initialize(Class *clss)
{
    if(clss->state == ST_Initialized) {
        return;
    }
    class_initialize1(clss, true);
} //class_initialize



/////////////////////////////////////////////////////////////////
// end Class
/////////////////////////////////////////////////////////////////




/////////////////////////////////////////////////////////////////
// begin Field
/////////////////////////////////////////////////////////////////


#if 1

// For static fields
void *Field::get_address()
{
    assert(is_static());
#ifdef _DEBUG
    Class *c = get_class();
    assert(c);
    switch(c->state) {
    case ST_Prepared:
    case ST_Initializing:
    case ST_Initialized:
        break;
    default:
        assert(0);
        break;
    }
#endif

    return (char *)(get_class()->static_data_block) + get_offset();
} //Field::get_address

#endif


/////////////////////////////////////////////////////////////////
// end Field
/////////////////////////////////////////////////////////////////






////////////////////////////////////////////////////////////
// begin array offset calculation


int get_array_offset(Java_Type element_type)
{
    switch(element_type) {
    case JAVA_TYPE_BYTE:
        {
            JavaArrayOfByte a;
            return (int)(((uint64)&(a.body[0])) - (uint64)&a);
        }
    case JAVA_TYPE_CHAR:
        {
            JavaArrayOfChar a;
            return (int)(((uint64)&(a.body[0])) - (uint64)&a);
        }
    case JAVA_TYPE_DOUBLE:
        {
            JavaArrayOfDouble a;
            return (int)(((uint64)&(a.body[0])) - (uint64)&a);
        }
    case JAVA_TYPE_FLOAT:
        {
            JavaArrayOfFloat a;
            return (int)(((uint64)&(a.body[0])) - (uint64)&a);
        }
    case JAVA_TYPE_INT:
        {
            JavaArrayOfInt a;
            return (int)(((uint64)&(a.body[0])) - (uint64)&a);
        }
    case JAVA_TYPE_LONG:
        {
            JavaArrayOfLong a;
            return (int)(((uint64)&(a.body[0])) - (uint64)&a);
        }
    case JAVA_TYPE_SHORT:
        {
            JavaArrayOfShort a;
            return (int)(((uint64)&(a.body[0])) - (uint64)&a);
        }
    case JAVA_TYPE_BOOLEAN:
        {
            JavaArrayOfBoolean a;
            return (int)(((uint64)&(a.body[0])) - (uint64)&a);
        }
    case JAVA_TYPE_CLASS:
    case JAVA_TYPE_ARRAY:
        {
            JavaArrayOfObject a;
            return (int)(((uint64)&(a.body[0])) - (uint64)&a);
        }
    default:
        assert(0);
        return 0;
    }
} //get_array_offset



// end array offset calculation
////////////////////////////////////////////////////////////

void orp_exit(int exit_code)
{

#ifdef CONCURRENCY_ANALYSIS
    ORP_thread *p_thr = p_active_threads_list;
    if(!p_thr) printf("No more threads left\n");
	uint64 end_time = readTimeStampCounter();
    for(; p_thr ; p_thr = p_thr->p_active ) {
#ifdef ORP_POSIX
    	fprintf(f_concur, "THREAD_END( tid = %d, time = %llu )\n", p_thr->thread_index, end_time);
#else
    	fprintf(f_concur, "THREAD_END( tid = %d, time = %I64u )\n", p_thr->thread_index, end_time);
#endif //#ifdef ORP_POSIX
    }
#endif //#ifdef CONCURRENCY_ANALYSIS

#ifdef GC_SAPPHIRE
    //
    // If one thread is exiting and another thread is doing a GC then
    // we will be deleting data structures out from under the GC thread.
    // These are not only the gc data structures but the JIT structures as well
    // which is why this code is here instead of inside the gc wrapup code.
    // Prevent this by grabbing a lock needed by GC_SAPPHIRE.
    p_sapphire_lock->_lock();
#endif

#ifdef GC_HOTFIELD_STATS
    if (collect_hotfield_stats) {
        // Write out to disk the field access statistics that we have collected.
        extern void process_field_access_stats();
        process_field_access_stats();
    }
#endif

#ifndef NO_JVMDI_SUPPORT
#ifdef _DEBUG

    if (jvmdi_hook_function)
    {
        assert(0);

        JVMDI_Event *p_event_record = 0;
        Allocate(sizeof(JVMDI_Event), (jbyte**)&p_event_record );
        memset(p_event_record, 0, sizeof(JVMDI_Event) );

        p_event_record->kind = JVMDI_EVENT_VM_DEATH;

        extern const struct JNIEnv_ *jni_native_intf;

        jvmdi_hook_function( (struct JNIEnv_ *)jni_native_intf, p_event_record);
    }

#endif
#endif

    methods.unload_all();
#ifdef ORP_VTUNE_SUPPORT
    vtune_notify_native_unload_all();
    vtune_notify_special_unload_all();
#endif
    orp_delete_all_jits();

    orp_thread_shutdown();
#ifdef ORP_STATS
    orp_stats_total.print();
#endif
 
    gc_wrapup();
 
#ifdef DUMP_IA64_STUBS
    extern FILE *ia64_stubs_out_file;
    if(ia64_stubs_out_file) {
        fclose(ia64_stubs_out_file);
    }
#endif

#ifdef GC_SAPPHIRE
      // We are exiting so no need to unlock.
//    p_sapphire_lock->_unlock();
#endif

    exit(exit_code);
} //orp_exit

// Two routines for accessing system properties when not in jni context
void __stdcall PropPut(char *pProp, char *pVal)
{
	assert(pProp);
#if 0 
	// Is this needed?
	assert(pVal);
#endif

	Global_Env *env = ORP_Global_State::loader_env;    
	String *name  = env->string_pool.lookup("java/lang/System");
	Class *clsys = env->class_table.lookup(name);
	if(!clsys)return;

	int p_len = strlen(pProp);
	jchar* jprop = CToUnicode(NULL, pProp, p_len);
	assert(jprop);
	jstring p = NewString(NULL, jprop, p_len);
	free((void*)jprop);

	int v_len = strlen(pVal);
	jchar* jval = CToUnicode(NULL, pVal, v_len);
	assert(jval);
	jstring v = NewString(NULL, jval, v_len);
	free((void*)jval);

	J_Value args[2];
	name  = env->string_pool.lookup("setProperty");	
	String *descr = env->string_pool.lookup("(Ljava/lang/String;Ljava/lang/String;)Ljava/lang/String;");
	Signature *sig = env->sig_table.lookup(name, descr);
	Method *method = class_lookup_method_recursive(clsys, sig);
	args[0].r = ((Object_Handle)p)->java_reference;
	args[1].r = ((Object_Handle)v)->java_reference;
	set_current_thread_exception(0);
	J_Value rtrn;
	unsigned gc_enabled = orp_disable_gc();  
	orp_execute_java_method_array(method, &rtrn, args);
	if(gc_enabled)
		orp_enable_gc();  

	delete jprop;
	delete jval;
}

void __stdcall PropGet(char *pProp, ExpandableMemBlock &mb)
{
	assert(pProp);

	mb.SetCurrentPos(0);

	Global_Env *env = ORP_Global_State::loader_env;    
	String *name  = env->string_pool.lookup("java/lang/System");
	Class *clsys = env->class_table.lookup(name);
	if(!clsys)return;

	int p_len = strlen(pProp);
	jchar* jprop = CToUnicode(NULL, pProp, p_len);
	assert(jprop);
	jstring p = NewString(NULL, jprop, p_len);
	free((void*)jprop);

	J_Value args[1];
	name  = env->string_pool.lookup("getProperty");	
	String *descr = env->string_pool.lookup("(Ljava/lang/String;)Ljava/lang/String;");
	Signature *sig = env->sig_table.lookup(name, descr);
	Method *method = class_lookup_method_recursive(clsys, sig);
	args[0].r = ((Object_Handle)p)->java_reference;
	set_current_thread_exception(0);
	J_Value rtrn;
	unsigned gc_enabled = orp_disable_gc();  
	orp_execute_java_method_array(method, &rtrn, args);
	if(gc_enabled)
		orp_enable_gc();  
	
	Object_Handle v = orp_create_local_object_handle();
	v->java_reference = (Java_java_lang_Object*)rtrn.r;
	const char* val = GetStringUTFChars(NULL, v, NULL);
	mb.AppendBlock((char*)val);
	mb.toString();
	ReleaseStringUTFChars(NULL, v, val);

	delete jprop;
}

//:M
ORP_thread *get_p_TLS_orpthread()
{
    return p_TLS_orpthread;
}
