// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/class_loader/Prepare.cpp,v 1.23 2002/01/13 12:59:16 gwu2 Exp $
//


#include "platform.h"
#include <assert.h>
#include "String_Pool.h"
#include "Class.h"
#include "nogc.h"
#include "Package.h"

#include "orp_utils.h"
#include "gc_for_orp.h"
#include "environment.h"
 
#ifdef _DEBUG
#include "jni.h"
#include "jvmdi_clean.h"
#endif

#ifdef ORP_POSIX
#include "platform2.h"
#endif

#ifdef PREPARE_TRACE
static ostream& trace = cerr;
#endif

//#define NEW_JAVA_LANG_CLASS_LAYOUT

//#define REMOVE_FINALIZE_FROM_VTABLES 

static void class_initialize_if_no_side_effects(Class *clss);



#ifdef CLI_TESTING
unsigned orp_sizeof_type(String *descriptor);
#endif



static unsigned sizeof_type(String *descriptor)
{
    switch (descriptor->bytes[0]) {
#ifdef CLI_TESTING
    case ORP_DATA_TYPE_UINT8:
    case ORP_DATA_TYPE_UINT16:
    case ORP_DATA_TYPE_UINT32:
    case ORP_DATA_TYPE_INTPTR:
    case ORP_DATA_TYPE_UINTPTR:
    case ORP_DATA_TYPE_UINT64:
    case ORP_DATA_TYPE_VALUE:
        return orp_sizeof_type(descriptor);
#endif
    case 'B':
    case 'C':
    case 'Z':
    case 'S':
    case 'F':
    case 'I':
        return 4;
    case 'L': 
    case '[': 
#ifdef POINTER64
        return 8;
#else
        return 4;
#endif
    case 'J':
    case 'D':
        return 8;
    default:
        assert(0);
    }

    return 0;
}
// Given a clss that is an primitive array return the size
// of an element in an instance of the class.
// Beware the the class is not fully formed at this time.

unsigned sizeof_primitive_array_element(Class *p_class) 
{
	const String *elt_type = p_class->name;
	char elt = elt_type->bytes[1];
    unsigned int sz;
    switch (elt) {
	case 'C' : 
		sz = 2;
    	break;			
	case 'B':
		sz = 1;
		break;
	case 'D':
		sz = 8;
        break;
	case 'F':
		sz = 4;
        break;
	case 'I':
		sz = 4;
        break;
	case 'J':
		sz = 8;
        break;
	case 'S':
		sz = 2;
        break;
	case 'Z' : // boolean
		sz = 1;
        break;
#ifdef CLI_TESTING
	case ORP_DATA_TYPE_UINT8:
		sz = 1;
		break;
	case ORP_DATA_TYPE_UINT16:
		sz = 2;
		break;
	case ORP_DATA_TYPE_UINT32:
		sz = 4;
        break;
	case ORP_DATA_TYPE_INTPTR:
	case ORP_DATA_TYPE_UINTPTR:
		sz = sizeof(POINTER_SIZE_INT);
        break;
	case ORP_DATA_TYPE_UINT64:
		sz = 8;
        break;
#endif
	case '[':
		sz = 4; // This works for the H32, V64, R32 version.
        assert (OBJECT_REF_SIZE == 4);
        break;
	default:
		printf("Unimplemented %c\n", elt);
		assert(0);
		return 0;
	}
    return sz;
}

//
// Is this array class a one dimensional array (vector) with a primitive component type.
//
bool
is_vector_of_primitives(Class *p_class)
{
    //
	// I parse the following character of the class name
	// to see if it is an array of arrays.
	if (*(p_class->name->bytes + 1) == '[') // An array of array
		return false;
	if (*(p_class->name->bytes + 1) == 'L') // An array of objects
		return false;
	if (p_class->is_array_of_primitives == 0) // base type is not primitive
		return false;
    if (p_class->is_array) {
    	return true;   
    }
    assert (0); // Should never be called unless p_class is an array.
    return true;
}


#ifndef NO_JVMDI_SUPPORT
#ifdef _DEBUG

void jvmdi_event_class_prepare(Class *clss)
{
    if (report_jvmdi_event_class_prepare == true)
    {
        Object_Handle jt = orp_create_global_object_handle();
        jt->java_reference = (struct Java_java_lang_Object *)
                                p_TLS_orpthread->p_java_lang_thread;

        Object_Handle jc = orp_create_global_object_handle();
        jc->java_reference = (struct Java_java_lang_Object *)jc;

        JVMDI_Event *p_event_record = 0;
        Allocate(sizeof(JVMDI_Event), (jbyte**)&p_event_record );
        memset(p_event_record, 0, sizeof(JVMDI_Event) );

        p_event_record->kind = JVMDI_EVENT_CLASS_PREPARE;
        p_event_record->u.class_event.thread = jt;
        p_event_record->u.class_event.clazz = jc;
        extern JVMDI_EventHook jvmdi_hook_function;
        extern const struct JNIEnv_ *jni_native_intf;

        jvmdi_hook_function( (struct JNIEnv_ *)jni_native_intf, p_event_record);
    }
}
#endif
#endif


// Function to zero out array
void inline zero_integer_array(unsigned int *arr, int elems)
{
    assert(arr);
    assert(elems > 0);

    for (int i = 0; i < elems; i++)
        arr[i] = (unsigned int)0;
    return;
}

void assign_offsets_to_class_fields(Class *clss)
{
	for (int i=0; i<clss->n_fields; i++) {
		Field& field = clss->fields[i];
        int field_size = sizeof_type(field.get_signature()->descriptor);
		if (field.is_static()) {
			// static (i.e., class) data field
			// ??? is this needed for interface static constants ???
          #ifdef POINTER64
            if (field_size==8) {
                if ((clss->static_data_size%8)!=0) {
                    clss->static_data_size += 4;
                    assert((clss->static_data_size%8)==0);
                }
            }
          #endif
			field._offset = clss->static_data_size;
			clss->static_data_size += field_size;
            clss->n_static_fields += 1;
		} else {
            // instance data field
            int offset = clss->unpadded_instance_data_size;
            int sz = field_size;
            int inc = sz;
            // Check if alignment padding is needed.
            if(offset % sz) {
                int delta = offset % sz;
                offset += sz - delta;
                inc += sz - delta;
            }
            field._offset = offset;
			clss->unpadded_instance_data_size += inc;
			char c_type = *(field.get_signature()->descriptor->bytes);
            if ((c_type == '[') || (c_type == 'L')) {
                clss->n_instance_refs += 1;
            }
		}

	}
}

// See why we should add this in class_prepare() STEP 20
void assign_values_to_class_static_final_fields(Class *clss)
{
	for (int i=0; i<clss->n_fields; i++) {
		Field& field = clss->fields[i];
        if (field.is_final() && field.is_static()) {
			Java_Type type = field.get_java_type();
			Const_Java_Value cvalue = field.get_const_value();
			void *field_addr = field.get_address();
			switch(type){
			case '[':
			case 'L':
				if(cvalue.object == NULL){ //cvalue.string == NULL
					/* We needn't deal with this case, because the object field must
					 * be set in static initializer
					 */
					break;
				}
				if(strcmp(field.get_descriptor(), "Ljava/lang/String;") == 0){
				    unsigned gc_was_enabled = orp_disable_gc();
					Java_java_lang_String *str = orp_instantiate_cp_string_resolved(cvalue.string);
					*((Java_java_lang_String **)field_addr) = str;
					if(gc_was_enabled)
						orp_enable_gc();
				}else{
					assert(0);
				}
				break;
			default:
				int field_size = sizeof_type(field.get_signature()->descriptor);
				memmove(field_addr, (void*)&cvalue, field_size);
			}
		}
	}
}

void build_gc_class_ref_map(Class *clss)
{	
	// Add 1 to the size of the information since it includes a zero delimiter.
    // Please note where this get deleted when we unload a class!!!!????
    // It should be done by a call to the gc side of the interface.
    unsigned *local_gc_information = new unsigned[clss->n_instance_refs + 1]; 
    unsigned int current_index = 0;

    // Copy the superclasses gc_information into this refs_offset_map incrementing current_index as needed.
    if (clss->super_class) { // We might be in initialization.
        assert (clss->n_instance_refs >= clss->super_class->n_instance_refs);

        // Ask the GC to fill the local_gc_information with the super class offsets.
#ifdef GC_REWORK
        gc_get_class_ref_map ((VTable *)clss->super_class->vtable, 
                              clss->super_class->n_instance_refs + 1,
                              local_gc_information);
#else
        gc_get_class_ref_map ((Partial_Reveal_Class *)clss->super_class, 
                              clss->super_class->n_instance_refs + 1,
                              local_gc_information);
#endif
        current_index = clss->super_class->n_instance_refs;
    } else {
        assert (current_index == 0);

//#ifdef CLASS_LOADER_BOOTSTRAP
#ifndef OLD_VERSION_CLASSPATH
		extern bool bootstrapped;
		// gloss over bootstrap inconsistency
		if (bootstrapped == true) { 
			assert (clss->n_instance_refs == 0);
		}
#else
		assert (clss->n_instance_refs == 0);
#endif
    }
	assert (current_index <= clss->n_instance_refs);
    
	for (int i = 0; i<clss->n_fields; i++) {
		Field& field = clss->fields[i];
		if (field.is_static()) {
			// static (i.e., class) data field
            // Since refs_offset only deals with instance fields 
            // this can be skipped. We don't change current offset for statics.
		} else {
			// instance data field
			//
			char c_type = *(field.get_signature()->descriptor->bytes);
			if ((c_type == '[') || (c_type == 'L')) {
                assert (field._offset != 0); // Only the vtable can have this offset.
				local_gc_information[current_index] = field._offset;
                current_index = current_index + 1;
            } 
        }
	}
    assert (current_index == (clss->n_instance_refs));
    local_gc_information[current_index] = 0; 
	// delimit with 0 since and offsetwill never be zero, that is where the vtable is we are OK.
#ifdef GC_REWORK
    gc_class_ref_map ((Partial_Reveal_Class *)clss, local_gc_information);
#else
    gc_class_ref_map ((Partial_Reveal_Class *)clss, local_gc_information);
#endif

    delete local_gc_information;
	// gc_information is not created, and populated and zero deliminted.
    // Pass this to the GC since it responsible for the format of the
    // information saved in clss->gc_information.
}

void assign_offsets_to_class_methods(Class *clss)
{
    int i;
	Method **super_vtable_descriptors = NULL;
	unsigned n_super_virtual_method_entries = 0;
	if (clss->super_class != NULL) {
		super_vtable_descriptors = clss->super_class->vtable_descriptors;
		n_super_virtual_method_entries=clss->super_class->n_virtual_method_entries;
	}
	// offset of next entry in vtable
#ifdef POINTER64
	unsigned next_vtable_offset = clss->n_virtual_method_entries << 3;  
#else
	unsigned next_vtable_offset = clss->n_virtual_method_entries << 2;  
#endif
	if (!class_is_interface(clss)) {
		// classes have an additional overhead for the 
		// class pointer and interface table
		next_vtable_offset += VTABLE_OVERHEAD;
	}
	for (i=0; i<clss->n_methods; i++) {
		Method& method = clss->methods[i];
		if (method.is_static()) {
			// static (i.e., class) method
			if (method.is_clinit()) {
				method._offset = 0;
				method._index = 0;
			} else {
				// better not be an interface!
				// To do : make sure this is not an interface
                assert(!class_is_interface(clss));
				method._offset = clss->static_method_size;
#ifdef POINTER64
				clss->static_method_size += 8;
#else
				clss->static_method_size += 4;
#endif
			}
		} else {
			//
			// virtual method
			// first look up in super class' virtual method table; 
			// if not found, then assign a new offset
			//
			// skip over initializers
			if (method.is_init())
				continue;

#ifdef REMOVE_FINALIZE_FROM_VTABLES
			//
			// skip over finalize() method, but remember it
			if (method.is_finalize()) {
				clss->finalize_method = &method;
				continue;
			}
#endif
			unsigned off = 0;
			unsigned index = 0;
			if (super_vtable_descriptors != NULL) {
				int same_package_as_super = 1;
#ifdef CLI_TESTING
                if(!clss->cli_type_def)
#endif
				if(strcmp((char *)clss->package->get_name()->bytes, (char *)clss->super_class->package->get_name()->bytes) )
					same_package_as_super = 0;

				Signature *s = method.get_signature();
				for (unsigned j=0; j<n_super_virtual_method_entries; j++) {
					Method *m = super_vtable_descriptors[j];
					if (s == m->get_signature()) {
						if(!same_package_as_super && !(m->is_public()|m->is_protected()|m->is_private())) break;
						off = m->get_offset();
						index = m->get_index();
						// mark superclass' method as being overwridden
                        m->method_was_overridden();
                        break;
					}
				}
			}
			if(off == 0 || class_is_interface(clss)) {
				// didn't find matching signature in super class
				// add new entry to this class' vtable
				off = next_vtable_offset;
				index = clss->n_virtual_method_entries;
#ifdef POINTER64
				next_vtable_offset += 8;  
#else
				next_vtable_offset += 4;  
#endif
				clss->n_virtual_method_entries++;
			}
			method._offset = off;
			method._index = index;
		}
	}
	    //
    // Figure out which methods don't do anything
    //
	for (i=0; i<clss->n_methods; i++) {
		Method& method = clss->methods[i];
        method._set_nop();
    }

}


void build_class_interface_table_descriptors(Class *clss)
{	
	// At this point we know the number of virtual methods in the vtable we now need to find 
	// the number of virtual methods that are in the interface part of the vtable, before we 
	// can allocate _vtable and _vtable_descriptors

	// Compute number of entries in intfc_table;note that _n_intfc_table_entries has been initialized 
	// earlier also, this is an upperbound because we will later eliminate duplicate entries in this table
    unsigned i;
	for (i=0; i<clss->n_superinterfaces; i++) {
		Class *intfc = clss->superinterfaces[i].clss;
		clss->n_intfc_table_entries += intfc->n_intfc_table_entries;
	}
	// allocate the intfc_table_descriptors
	if (clss->n_intfc_table_entries != 0) {
		clss->intfc_table_descriptors = (Class**)gc_malloc_fixed_data_C(
                sizeof(Class*)*clss->n_intfc_table_entries);
	} else {
		clss->intfc_table_descriptors = NULL;
	}
    unsigned k;
	for (k=0; k < clss->n_intfc_table_entries; k++) {
		clss->intfc_table_descriptors[k] = NULL;
	}
	// fill in intfc_table_descriptors with the descriptors from the superclass and the superinterfaces
	unsigned intfc_table_entry = 0;
	if (clss->super_class != NULL) {
		for (unsigned i = 0; i < clss->super_class->n_intfc_table_entries; i++) {
			clss->intfc_table_descriptors[intfc_table_entry] 
				= clss->super_class->intfc_table_descriptors[i];
			intfc_table_entry++;
		}
	}
	for (k = 0; k < clss->n_superinterfaces; k++) {
		Class *intfc = clss->superinterfaces[k].clss;
		for (unsigned i = 0; i < intfc->n_intfc_table_entries; i++) {
			clss->intfc_table_descriptors[intfc_table_entry]
				= intfc->intfc_table_descriptors[i];
			intfc_table_entry++;
		}
	}
	// if this class is an interface, add it to the interface table
	if (class_is_interface(clss)) {
		clss->intfc_table_descriptors[intfc_table_entry] = clss;
		intfc_table_entry++;
	}
	// sort the interfaces in intfc_table_descriptors, eliminating duplicate entries
	unsigned last_min_id = 0;
	for (i = 0; i < clss->n_intfc_table_entries; i++) {
		//
		// select the next interface C with smallest id and insert 
		// into i'th position; delete entry of C if C is the same
		// as i-1'th entry
		//
		Class *intfc = clss->intfc_table_descriptors[i];
		unsigned min_index = i;			// index of intfc with min id
		unsigned min_id = intfc->id;	// id of intfc with min id
		for (unsigned k = i+1; k < clss->n_intfc_table_entries; k++) {
			unsigned id = clss->intfc_table_descriptors[k]->id;
			if (id < min_id) {
				// new min
				min_index = k;
				min_id = id;
				continue;
			}
		}
		// if the id of the min is the same as the i-1'th entry's id, then we have a duplicate
		if (min_id == last_min_id) {
			// duplicate found -- insert the last entry in place of the duplicate's entry
			clss->intfc_table_descriptors[min_index] 
				= clss->intfc_table_descriptors[clss->n_intfc_table_entries-1];
			clss->n_intfc_table_entries--;
			continue;
		}
		last_min_id = min_id;
		if (min_index == i)
			continue;
		// swap i'th entry with min entry
		Class *min_intfc = clss->intfc_table_descriptors[min_index];
		clss->intfc_table_descriptors[min_index] = clss->intfc_table_descriptors[i];
		clss->intfc_table_descriptors[i] = min_intfc;
	}
}


void intialize_static_fields_for_interface(Class *clss)
{
    // Initialize static fields
    unsigned i;
	for (i=0; i<clss->n_fields; i++) {
		Field& field = clss->fields[i];
		if (field.is_static() && field.get_const_value_index()) {
            char *field_addr = ((char *)clss->static_data_block) + field._offset;
            Const_Java_Value field_const_value = field.get_const_value();
            switch(field.get_java_type()) {
            case JAVA_TYPE_INT:
                *((int32 *)field_addr) = field_const_value.i;
                break;
            case JAVA_TYPE_SHORT:
            case JAVA_TYPE_CHAR:
                *((int16 *)field_addr) = (int16)field_const_value.i;
                break;
            case JAVA_TYPE_BYTE:
            case JAVA_TYPE_BOOLEAN:
                *((int8 *)field_addr) = (int8)field_const_value.i;
                break;
            case JAVA_TYPE_LONG:
                *(((int32 *)field_addr))     = field_const_value.l.lo_bytes;
                *(((int32 *)field_addr) + 1) = field_const_value.l.hi_bytes;
                break;
            case JAVA_TYPE_DOUBLE:
                *(((int32 *)field_addr))     = field_const_value.l.lo_bytes;
                *(((int32 *)field_addr) + 1) = field_const_value.l.hi_bytes;
                break;
            case JAVA_TYPE_FLOAT:
                *((float *)field_addr) = field_const_value.f;
                break;
            case JAVA_TYPE_CLASS:
            {
                unsigned gc_enabled = orp_disable_gc();
                // It must be a String
                assert(strcmp(field.get_descriptor(), "Ljava/lang/String;") == 0);
                Java_java_lang_String *str = orp_instantiate_cp_string_resolved(field_const_value.string);
                *((Java_java_lang_String **)field_addr) = str;
                if(gc_enabled)
					orp_enable_gc();
                break;
            }
            default:
                // This should never happen.
                assert(0);
                break;
            }
        }
    }
	clss->n_virtual_method_entries = 0;	// interfaces don't have vtables
	for (i=0; i<clss->n_methods; i++) {
		Method *method = &clss->methods[i];
		if (method->is_clinit()) {
            assert(clss->static_initializer == method);
		}
	}
	clss->state = ST_Prepared;
#ifdef PREPARE_TRACE
	trace << "finished preparing interface " << clss->this_name->bytes << endl;
#endif
#if PREPARE_TRACE >= 2
	dump_prepare_info(trace);
#endif
    class_initialize_if_no_side_effects(clss);
    p_load_class_lock->_unlock();                      

#ifndef NO_JVMDI_SUPPORT
#ifdef _DEBUG
    jvmdi_event_class_prepare(clss);
#endif
#endif

	return;
}

void populate_vtable_descriptors_table_and_override_methods(Class *clss)	
{
	// Populate _vtable_descriptors first with _n_virtual_method_entries from super class
	if (clss->super_class != NULL) {
		for (unsigned i = 0; i < clss->super_class->n_virtual_method_entries; i++) {
			clss->vtable_descriptors[i] = clss->super_class->vtable_descriptors[i];
		}
	}
	// NOW OVERRIDE with this class' methods
    unsigned i;
	for (i=0; i<clss->n_methods; i++) {
		Method *method = &clss->methods[i];
		if (method->is_clinit()) {
            assert(clss->static_initializer == method);
		} 
        if (method->is_static() || 
			   method->is_init()
#ifdef REMOVE_FINALIZE_FROM_VTABLES
			   || method->is_finalize()
#endif
               )
			continue;
		clss->vtable_descriptors[method->get_index()] = method;
	}
	// finally, the interface methods
	unsigned index = clss->n_virtual_method_entries;
	for (i = 0; i < clss->n_intfc_table_entries; i++) {
		Class *intfc = clss->intfc_table_descriptors[i];
		for (unsigned k = 0; k < intfc->n_methods; k++) {

			// Find method with matching signature and replace
			Signature *sig = intfc->methods[k].get_signature();

    		if (intfc->methods[k].is_clinit()) {
                continue;
            }

			Method *method = NULL;
			for (unsigned j = 0; j < clss->n_virtual_method_entries; j++) {
				if (clss->vtable_descriptors[j]->get_signature() == sig) {
					method = clss->vtable_descriptors[j];
					break;
				}
#ifdef CLI_TESTING
                Signature *sig_clss_meth = clss->vtable_descriptors[j]->get_signature();

                if (strcmp(sig_clss_meth->descriptor->bytes, sig->descriptor->bytes)) 
                    continue;

#define SIGBUFF_SIZE 128
                char sigbuff[SIGBUFF_SIZE];
                memset(sigbuff, 0, SIGBUFF_SIZE);
                assert( (strlen(intfc->methods[k].get_class()->name->bytes) + 
                       strlen(sig->name->bytes) ) < (SIGBUFF_SIZE - 1) );

                strcpy(sigbuff, intfc->methods[k].get_class()->name->bytes);
                strcat(sigbuff, ".");
                strcat(sigbuff, sig->name->bytes);
                int32 xx;
                for (xx = 0; xx < SIGBUFF_SIZE; xx++) {
                    if (sigbuff[xx] == '/')
                        sigbuff[xx] = '.';
                }

                char *cursor = strstr(sig_clss_meth->name->bytes, sigbuff);
                if (!cursor)
                    continue;

                // therefor name and descriptor match

				method = clss->vtable_descriptors[j];
				break;
#endif // CLI_TESTING

			}
			if (method == NULL && class_is_abstract(clss)) {
            // wgs: I think we should comment out this assert, because there're many 
			// cases ORP/Classpath will run apps built on previous JDK version, and 
			// without implementations of newly added methods for specific interfaces,
			// we allow them to continue to run
			/* 
				printf("**************\n");
				printf("No implementation in class %s for method %s of interface %s. \n\n", 
					clss->name->bytes,
					sig->name->bytes,
					intfc->name->bytes);
				printf("Check whether you used another set of class library.\n");
				printf("**************\n");
			*/
			//assert(0);
			assert(0); //wgs: After discussion with Michal, we think we should remove comments to assert
			// To do : problem!!!
			}
			clss->vtable_descriptors[index] = method;
			index++;
		}
	}
}

void point_class_vtable_entries_to_stubs(Class *clss)
{
	for (unsigned i = 0; i < clss->n_virtual_method_entries; i++) {
        assert(clss->vtable_descriptors[i]);
		Method& method = *(clss->vtable_descriptors[i]);
        assert(!method.is_static());
        if(!method.is_static()) {
            unsigned meth_idx = method.get_index();
#ifdef POINTER64
            assert((method.get_offset() - 16) >> 3 == method.get_index());  
#else
            assert((method.get_offset() - VTABLE_OVERHEAD) >> 2 == method.get_index());  //::
#endif
		    clss->vtable->methods[meth_idx] =
                (unsigned char *)method.get_code_addr();
            method.add_vtable_patch(&(clss->vtable->methods[meth_idx]));
            assert(method.get_code_addr());
        }
    }
}

Intfc_Table *create_populate_class_interface_table(Class *clss)		
{
	Intfc_Table *intfc_table;
	if (clss->n_intfc_table_entries != 0) {
		unsigned vtable_offset = clss->n_virtual_method_entries;
        // shouldn't it be called vtable_index?
		intfc_table = create_intfc_table(clss->n_intfc_table_entries);
        unsigned i;
		for (i = 0; i < clss->n_intfc_table_entries; i++) {
			Class *intfc = clss->intfc_table_descriptors[i];
			intfc_table->entry[i].intfc_id = intfc->id;
			intfc_table->entry[i].table = &clss->vtable->methods[vtable_offset];
			vtable_offset += intfc->n_methods;
            if(intfc->static_initializer) {
                // Don't count static initializers of interfaces.
                vtable_offset--;
            }
		}
        // Set the vtable entries to point to the code address.
    	unsigned meth_idx = clss->n_virtual_method_entries;
	    for (i = 0; i < clss->n_intfc_table_entries; i++) {
		    Class *intfc = clss->intfc_table_descriptors[i];
		    for (unsigned k = 0; k < intfc->n_methods; k++) {
        		if (intfc->methods[k].is_clinit()) {
                    continue;
                }
    	Method *method = clss->vtable_descriptors[meth_idx];
        //wgs: also for running apps that depend on old versions of JDK, just remove assert
        //assert(method);
        //wgs: After discussion with Michal, we think we should remove comments to assert
        assert(method);
		clss->vtable->methods[meth_idx] =
                    (unsigned char *)method->get_code_addr();
                method->add_vtable_patch(&(clss->vtable->methods[meth_idx]));
	    		meth_idx++;
            }
        }
	} else {
		intfc_table = NULL;
	}
	return intfc_table;
} 

void initialize_interface_class_data(Class *clss)
{
	// this is an interface
	clss->instance_data_size = 0;	// no instance data
    clss->unpadded_instance_data_size = 0;
    clss->allocated_size = 0;
    clss->n_instance_refs      = 0; // RDS
    clss->n_static_fields      = 0; // RDS
	clss->n_virtual_method_entries = 0;	// thus no virtual method entries
	clss->n_intfc_table_entries = 1;	// need table entry for this interface

	return;
}

void initialize_java_lang_object_class(Class *clss)
{
	// java.lang.Object -- Java ROOT.
    clss->instance_data_size = 0; // set below use the unpadded_instace_data_size.
    clss->allocated_size = 0;     // set below.
#ifndef OLD_OBJ_LAYOUT
    clss->unpadded_instance_data_size = sizeof(Java_java_lang_Object);
#else
    clss->unpadded_instance_data_size = OBJECT_VTABLE_POINTER_SIZE;	// only ptr to vtable
#endif
    clss->n_instance_refs    = 0; // RDS
	clss->n_static_fields    = 0; // RDS
	clss->n_virtual_method_entries = clss->n_intfc_table_entries = 0;
}


void  initialize_regular_class_data(Class *clss)
{
	clss->instance_data_size = 0; // set below.
    clss->allocated_size = 0;     // set below.
	// Roll over instance size, instance refs and static fields # of super class.
    clss->unpadded_instance_data_size = clss->super_class->unpadded_instance_data_size;
//#ifdef NEW_JAVA_LANG_CLASS_LAYOUT
#ifndef OLD_VERSION_CLASSPATH
    if(!strcmp("java/lang/Class", clss->name->bytes)) {
        clss->unpadded_instance_data_size = 
            ( (sizeof(Class) + (GC_OBJECT_ALIGNMENT - 1)) / GC_OBJECT_ALIGNMENT) * GC_OBJECT_ALIGNMENT;
    }
#endif
    clss->n_instance_refs = clss->super_class->n_instance_refs; // RDS
    clss->n_static_fields = clss->super_class->n_static_fields;
	// Roll over all virtual methods and interface methods of super class.
	clss->n_virtual_method_entries = clss->super_class->n_virtual_method_entries;
	clss->n_intfc_table_entries = clss->super_class->n_intfc_table_entries;
}


//
// prepares a class:
//	(1) assign offsets
//		- offset of instance data fields
//		- virtual methods in vtable
//		- static data fields in static data block
//		- static methods in static method block
//
//	(2) create class vtable
//	(3) create static field block
//	(4) create static method block
//
//
//

Loader_Result class_prepare(Class *clss)
{
    p_load_class_lock->_lock();                           

	//
	//
	// STEP 1 ::: SIMPLY RETURN IF already prepared, initialized or currently initializing.
	//
	//
    switch(clss->state) {
    case ST_Prepared: case ST_Initializing: case ST_Initialized: {
    
            p_load_class_lock->_unlock();  
		    return LD_OK;
        }
    default:
        break;
    }

	//
	//
	// STEP 2 ::: PREPARE SUPER-INTERFACES
	//
	//
	Loader_Result result;
	unsigned i;
	for (i=0; i<clss->n_superinterfaces; i++) {
		result = class_prepare(clss->superinterfaces[i].clss);
        if (result != LD_OK) {
            p_load_class_lock->_unlock();              // ^^^
			return result;
        }
        if ( !class_is_interface(clss->superinterfaces[i].clss) ) {
            p_load_class_lock->_unlock();              // ^^^
            return LD_ParseError;
        }
	}
	//
	//
	// STEP 3 ::: PREPARE SUPERCLASS if needed; simply initialize if interface.
	//
	//
	if (class_is_interface(clss)) {

		initialize_interface_class_data(clss);

	} else if (clss->super_class != NULL) {

		// Regular class with non-zero super-class.
		result = class_prepare(clss->super_class);
        if (result != LD_OK) {
            p_load_class_lock->_unlock();          
			return result;
        }
        // CLASS_VTABLE_REWORK - these will eventually be moved into the vtable but we don't have one yet.
        // Beforp we start adding properties make sure they are clear.
        assert (clss->class_properties == 0);
		if(clss->super_class->has_finalizer) {
            clss->has_finalizer = 1;
        }
        initialize_regular_class_data(clss);

	} else {

		initialize_java_lang_object_class(clss);
	}
	clss->static_data_size = 0;
	clss->static_method_size = 0;
	//
	//
	// STEP 4 :::: ASSIGN OFFSETS to the class and instance data FIELDS
	//
	//
	assign_offsets_to_class_fields(clss);
	//
	//
	// STEP 5 :::: Build GC REFERENCE OFFSET MAP
	//
	//
	build_gc_class_ref_map(clss);
	//
	//
	// STEP 6 :::: ASSIGN OFFSETS to the class and virtual METHODS
	//
	//
	assign_offsets_to_class_methods(clss);
	//
	//
	// STEP 7 :::: Calculate # of INTERFACES METHODS and build interface table DESCRIPTORS for C
	//
	//
	build_class_interface_table_descriptors(clss);
	//
	//
	// STEP 8 :::: Create the static field and method blocks
	//
	//
	clss->static_data_block = new char[clss->static_data_size]; 
	memset(clss->static_data_block, 0, clss->static_data_size);
	clss->static_method_block = (unsigned char**) new char[clss->static_method_size];    
	memset(clss->static_method_block, 0, clss->static_method_size);
	//
	//
	// STEP 9 :::: For INTERFACES intialize static fields and return.
	//
	//
	if (class_is_interface(clss)) {
		
		intialize_static_fields_for_interface(clss);
		// DONE for interfaces
		return LD_OK;
	}
	//
	//
	// STEP 10 :::: COMPUTE number of interface method entries.
	//
	//
	for (i = 0; i < clss->n_intfc_table_entries; i++) {
		Class *intfc = clss->intfc_table_descriptors[i];
		clss->n_intfc_method_entries += intfc->n_methods;
    	if (intfc->static_initializer) {
            // Don't count static initializers of interfaces.
            clss->n_intfc_method_entries--;
        }
	}
	//
	//
	// STEP 11 :::: ALLOCATE the Vtable descriptors array 
	//
	//
	unsigned n_vtable_entries = clss->n_virtual_method_entries + clss->n_intfc_method_entries;
	if (n_vtable_entries != 0) {
		clss->vtable_descriptors = new Method*[n_vtable_entries];
	} else {
		clss->vtable_descriptors = NULL;
	}
	//
	//
	// STEP 12 :::: POPULATE with interface descriptors and virtual method descriptors.
	//				Also, OVERRIDE superclass' methods with those of this one's
	//
	populate_vtable_descriptors_table_and_override_methods(clss);	
	//
	//
	// STEP 13 :::: CREATE VTABLE and set the Vtable entries to point to the 
	//				code address (a stub or jitted code) 
	//
	//
	clss->vtable = create_vtable(clss, n_vtable_entries);
	for (i = 0; i < n_vtable_entries; i++) {
		// need to populate with pointers to stubs or compiled code
		clss->vtable->methods[i] = NULL;	// for now
	}
	clss->vtable->clss = clss;
	// Set the vtable entries to point to the code address (a stub or jitted code)
	point_class_vtable_entries_to_stubs(clss);

	//
	//
	// STEP 14 :::: CREATE and POPULATE the CLASS INTERFACE TABLE
	//
	//
	clss->vtable->intfc_table = create_populate_class_interface_table(clss);		
	
	//
	//
	// STEP 15 :::: HANDLE JAVA CLASSCLASS separately	
	//
	//

	// Make sure on one hasn't prematurely set these fields since all calculations
    // up to this point should be based on clss->unpadded_instance_data_size.
    assert (clss->instance_data_size == 0);
    assert (clss->allocated_size == 0);
//#ifdef NEW_JAVA_LANG_CLASS_LAYOUT
#ifndef OLD_VERSION_CLASSPATH
#else
	// Handle 
    if(!strcmp("java/lang/Class", clss->name->bytes)) {
        // Objects of this class are at the same time ORP data structures and are larger than implied 
		// by the class file definition of java.lang.Class.
        // No fields should be known by this routine except the vtable. By unpadded
        // we are refering to gc padding not padding done due to alignment constraints.
        assert(clss->unpadded_instance_data_size == (OBJECT_VTABLE_POINTER_SIZE + sizeof_java_lang_class()));
        clss->unpadded_instance_data_size = 
            ( (sizeof(Class) + (GC_OBJECT_ALIGNMENT - 1)) / GC_OBJECT_ALIGNMENT) * GC_OBJECT_ALIGNMENT;
    }
#endif
    // Add any needed padding including the OBJECT_HEADER which is used to hold
    // things like gc forwarding pointers, mark bits, hashes and locks..
    clss->allocated_size = 
        (((clss->unpadded_instance_data_size + (GC_OBJECT_ALIGNMENT - 1)) 
          / GC_OBJECT_ALIGNMENT) * GC_OBJECT_ALIGNMENT) + OBJECT_HEADER_SIZE;
#ifdef GC_REWORK
    // Move the size to the vtable.
    clss->vtable->allocated_size = clss->allocated_size;
#endif
    clss->instance_data_size = clss->allocated_size;

    // CLASS_VTABLE_REWORK - these will eventually be eliminated.
    // We now need to se the class_properties and possible set the high bit in the clss->instance_data_size.
    if(!strcmp("java/lang/Class", clss->name->bytes)) {
        set_prop_pinned (clss); // java/lang/Class is always pinned.
    }    

	//
	//
	// STEP 16 :::: HANDLE PINNING and Class PROPERTIES if needed.
	//
	//

    if (clss->super_class) {
        if (get_prop_pinned (clss->super_class->class_properties)) {
            // If the super class is pinned then this class is pinned.
            set_prop_pinned (clss);
        }
    }
    // Set up the class_properties field.
    if (clss->is_array == 1) {
        // Arrays that are not primitives use 4 bytes for each element on IA32
        // and 8 bytes on IA64
        // Put this information into the gc_information field of the class.
        // Since this field usually holds a pointer cast 4 to a pointer.

#ifdef POINTER64
        clss->array_element_size = 8;
#else
        clss->array_element_size = 4; 
#endif
        set_prop_array (clss);
        if (is_vector_of_primitives (clss)) {
            clss->array_element_size = sizeof_primitive_array_element (clss);
            set_prop_non_ref_array (clss);
        }
#ifdef GC_REWORK
        clss->vtable->array_element_size = clss->array_element_size;
#endif // GC_REWORK
    }
    
    if (clss->has_finalizer) {
        set_prop_finalizable(clss);
    }
#ifdef GC_REWORK
    // Now that we have a vtable we can get the gc_infomation out of the class structure
    // and put it into the vtable structure.

    clss->vtable->gc_information = clss->gc_information;
#endif
	//
	//
	// STEP 17 :::: HANDLE ALIGNMENT and Class FINALIZER if needed.
	//
	//

    if (clss->alignment) {
        if (clss->alignment != GC_OBJECT_ALIGNMENT) { 
            // The GC will align on 4 byte boundaries by default on IA32....
#ifdef POINTER64
            assert(0);
#endif
            // Make sure it is a legal mask.
            assert ((clss->alignment & CL_PROP_ALIGNMENT_MASK) <= CL_PROP_ALIGNMENT_MASK);
            set_prop_alignment_mask (clss, clss->alignment);
            // make sure constraintbit was set.
            assert (get_instance_data_size(clss) != clss->instance_data_size);
        }
    }

	//
	//
	// STEP 18 :::: SET Class ALLOCATED SIZE to INSTANCE SIZE
	//
	//

    // Finally set the allocated size field.
    clss->allocated_size = get_instance_data_size(clss);
 

	//
	//
	// STEP 19 :::: SET class to ST_Prepared state and return.
	//
	//

	clss->state = ST_Prepared;
#ifdef PREPARE_TRACE
		trace << "finished preparing class " << clss->this_name->bytes << endl;
#if PREPARE_TRACE >= 2
		dump_prepare_info(trace);
#endif
#endif
    class_initialize_if_no_side_effects(clss);
    p_load_class_lock->_unlock();                         // ^^^

	//
	// STEP 20 :::: ASSIGN VALUE to static final fields
	//
	// Generally speaking final value is inlined, so we needn't worry about the 
	//     initialization of to those static final fields. But when we use reflection 
	//     mechanisms-Field.getXXX()- to access them, we got null values. Consider this,
	//     We must initialize those static final fields.
	// Also related to this is Binary Compatibility chapter of the JLS.  
	//       Section 13.4.8 
	// 
	assign_values_to_class_static_final_fields(clss);

#ifndef NO_JVMDI_SUPPORT
#ifdef _DEBUG
    jvmdi_event_class_prepare(clss);
#endif
#endif

#ifdef GC_REWORK
    assert (clss->class_properties == clss->vtable->class_properties);
    assert (clss->allocated_size == clss->vtable->allocated_size);
    assert (clss->gc_information == clss->vtable->gc_information);
    assert (clss->array_element_size == clss->vtable->array_element_size);
#endif

	return LD_OK;
} //class_prepare



static void class_initialize_if_no_side_effects(Class *clss)
{
    if(clss->state == ST_Initialized) {
        return;
    }

    Class *c;
    for(c = clss; c; c = c->super_class) {
        if(c->state == ST_Initialized) {
            // 1. c and all its superclasses have been initialized.
            // 2. all the superclases of clss which are a subclass of c
            //    have no static constructors.
            break;
        }
        if(c->static_initializer) {
            // c is not initialized and has a static constructor.
            // Side effects are possible.
            return;
        }
    }

    // If we get here, initializing clss has no side effects
    for(c = clss; c; c = c->super_class) {
        if(c->state == ST_Initialized) {
            break;
        }
        c->state = ST_Initialized;
    }
} //class_initialize_if_no_side_effects



