// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/car.cpp,v 1.1.1.1 2001/07/23 07:25:38 xli18 Exp $
//

#include "platform.h"
#include "card_table.h"
#include "train.h"
#include "train_generation.h"
#include "gc_asserts.h"
#include "gc_consts.h"
#include "gc_plan.h"
#include "gc_hooks.h"
#include "car.h"
#include "object_list.h"
#include "finalize.h"
 
#include "orp_for_gc.h"
 
Car::Car(unsigned long    car_id,
         Gc_Fast_Hooks    *p_gc_hooks,
         Gc_Plan          *p_gc_plan,
		 Train            *p_container,
         Train_Generation *p_train_generation,
         Card_Table       *p_card_table,
		 Block_Store      *p_block_store)
		  : Block_List(car_id,
                       p_gc_hooks,
                       p_gc_plan,
		               p_train_generation,
		               NULL,
                       p_card_table,
		               p_block_store)
{
	//
	// Discover the sizes of blocks in cars.
	//
	_car_block_size_bytes = 
		_p_gc_plan->car_block_size_bytes();

    //
    // Create a structure to hold the list of objects that are
    // candidates for finalization (when they become unreachable.)
    //
    _p_finalizable_object_list = new Object_List();

	//
	// Queues to deal with weak reference objects.
	// These need to be enumerated for the GC. Each car has a set of these
	// queues as does each step (or step_nursery) generation.
	// They are created lazily to reduce normal case overhead and so that the
	// creation can happen after the car has been fully initialized. You can't
	// really do it now because the car doesn't have any blocks associated with it yet.
	//
    _p_weak_object_list = NULL;

    _p_soft_object_list = NULL;

    _p_phantom_object_list = NULL;

	_maximum_blocks_in_car = _p_gc_plan->car_size_blocks();

	// Though Gc_Space owns the following, it
	// does not get initialized by the Gc_Space
	// constructor since Spaces and Nurseries
	// currently don't need their own remembered
	// sets.
	_p_container_write_barrier = new Remembered_Set();

	_p_train            = p_container;
	_p_train_generation = p_train_generation;

    _p_base = 0;
    _p_ceiling = 0;

    //
	// Since cars get scanned only once and then discarded,
	// it is sufficient to set the current block ID, which
	// is used during cheney scans to advance the scan pointer,
	// once here. Cars don't have a seperate "to" and "from" space like
    // steps do since cars are added to over the course of several 
    // collections.
	//
    assert(sizeof(_current_block_id) == 4);
    _current_block_id   = 0xFFFFffff;

#ifdef GC_TRAIN_V5
    // Each car has a remembered set holding all slots with pointers that
    // might hold interesting pointers.
    // These are scanned at each collection to find pointers into the focus car,
    // into the YOS and into the focus train.
    _p_car_remembered_set = new Remembered_Set();
#endif // GC_TRAIN_V5
#ifdef PER_CAR_BARRIERS // NOT USED
    _p_old_to_young = new Remembered_Set();
#endif

#if (GC_DEBUG>0)
	car_creation_hook(this);
#endif

    assert (_maximum_blocks_in_car > 0); // Just to be sure we can move something
                                         // into the car if we need to.
#ifndef GC_TRAIN_V5
#ifdef GC_GEN_V3
    assert (car_id == 0); // Only one car/train in GC_GEN_V3 collector.
#endif // GC_GEN_V3
#endif // GC_TRAIN_V5
}

Car::~Car() 
{
    
#ifdef CLASS_DEMOGRAPHICS
    walk_car(walk_report_dead_object);
#endif

#if (GC_DEBUG>0)
	car_deletion_hook(this);
#endif


    delete _p_finalizable_object_list;

	//
	// Since this car is going away, we need to update any old-to-young
	// references in younger generations (i.e. flush them).
	//
    // This is where pointers from destroyed cars are flushed form
    // the young object space remembered set.
	_p_train_generation->flush_rs_entries_of_car(this);

    delete _p_container_write_barrier;
#ifdef GC_TRAIN_V5
    delete _p_car_remembered_set;
#endif // GC_TRAIN_V5
#ifdef PER_CAR_BARRIERS
    delete _p_old_to_young;
#endif

    //
    // Free all the constituent blocks.
    //

    // We need to free the blocks that this car uses by returning them
    // to the block store. This is done by the block list destructor.
#ifdef GC_TRAIN_TRACE
//    if (stats_gc) {
        orp_cout << "Train Trace - Freeing blocks used by car in train " << get_train_id() << endl;
//    }
#endif
}

unsigned long 
Car::get_train_id() 
{
	return _p_train->get_train_id();
}

//
// The write barrier of all pointers from younger MOS areas into this car.
// This includes cars in the same train. This is a conservative approximation.

Remembered_Set *
Car::get_car_remembered_set () 
{
    return _p_container_write_barrier;   
};

//
// We are being notified that there is a new incoming pointer
// into our car. We now need to determine if this warrants
// updating our write barrier.
//
// THIS CHECK CAN BE DONE USING THE GENERATIONAL ENCODING
// Rick - MAKE IT SO.
//
void
Car::check_update_car_write_barrier(Java_java_lang_Object **pp_obj_ref)
{
    Train *p_referring_train = 
        p_global_bs->p_get_reference_train(pp_obj_ref);

#if (GC_DEBUG>0)
	Car *p_target_car   = 
		p_global_bs->p_get_object_car(*pp_obj_ref);
    assert (p_target_car == this);
#endif
//            TO RICK - I think we have an object that has been moved to a younger train with
//                a reference into the focus car and for some reason the object being refered to
//                by the object that was moved should be moved but isn't....
    if (p_referring_train->get_train_id() > _p_train->get_train_id()) {
        //
        // The referring object is in a younger, more recently created, train. Therefore we need
        // to update the write barrier remembered set for this car in the older train.
        // 
        gc_trace (*pp_obj_ref, 
            "Car::check_update_car_write_barrier, pointer to this car object in younger train"); 

        update_container_write_barrier(pp_obj_ref);
#ifndef GC_TRAIN_V5
#ifdef GC_GEN_V3
        assert (0); // There should never be a pointer to a younger
                    // train in GC_GEN_V3. If the step promotes into
                    // the youngest train (train 1) then why shouldn't there
                    // be one.
#endif // GC_GEN_V3
#endif // GC_TRAIN_V5

    } else if (_p_train == p_referring_train) {
		//
		// The pointer originates and ends in our train.
		//
		Car *p_referring_car =
			p_global_bs->p_get_reference_car(pp_obj_ref);

        if (p_referring_car->get_id() > identity) { 
           gc_trace (*pp_obj_ref,
               "Car::check_update_car_write_barrier, pointer to this car object in same train but younger car");
 
			//
			// Pointer from a higher car in my train.
			//
			update_container_write_barrier(pp_obj_ref);
#ifndef GC_TRAIN_V5
#ifdef GC_GEN_V3
        assert (0); // There should never be more than one car in a
                    // train in GC_GEN_V3
#endif // GC_GEN_V3
#endif // GC_TRAIN_V5
		}
	}
	//
	// If the referring train is older, (p_referring_train < _p_train)
	// we don't need to record the reference. If both trains are the
	// same, we are guaranteed that the target object is in a younger
	// car, since it was scavenged into the end of the train.
}

//
// Do any required cleanup at the end of the collection.
//
void 
Car::cleanup() 
{
	cout << "Error: Car::cleanup stub" << endl;
	orp_exit(1); // stub
}

#if (GC_DEBUG>3)
void
Car::dump_container_write_barrier()
{
	_p_container_write_barrier->rewind();

	Java_java_lang_Object **pp_obj_ref;

	cout << "Dumping write barrier for car " << this << endl;
	cout << "Number of entries = " << _p_container_write_barrier->size();
	cout << endl;
	while ((pp_obj_ref = _p_container_write_barrier->next()) != NULL) {
		cout << "ref = " << pp_obj_ref << " ( " << *pp_obj_ref << " ) ";
		if (_p_train_generation->is_address_in_my_generation(pp_obj_ref)) {
			cout << " train = ";
			cout << p_global_bs->p_get_address_train(pp_obj_ref);
			cout << " car = " ;
			cout << p_global_bs->p_get_address_car(pp_obj_ref);
			cout << endl;
		} else {
			cout << " out of generation pointer" << endl;
		}
	}
	cout << endl << endl;
}
#endif // _DEBUG

#ifdef GC_SAPPHIRE
//
// sapphire_absorb_nurseries
// Comments - sapphire needs to move the nurseries into the focus car "from" space before a
//				sapphire collection can take place. This is done by linking the blocks of each
//              nursery onto the car block list.
//				Each thread is pushed out of allocation by pushing it to a gc safe point, halting
//              it and the absorbing its nursery.
// 
extern void orp_sapphire_release_nurseries_all_threads();
void Car::sapphire_absorb_nurseries()
{
//	orp_cout << " Absorbing nurseries into From space." << endl;
	Car *focus_car = p_global_bs->get_mature_generation()->p_nominate_focus_car();
	assert (this == focus_car);
	// Tell the YOS to evict the spent nurseries into the "from" c space focus car.
	// This will provide the needed nurseries when the threads need to release theirs.
	p_global_bs->get_young_generation()->sapphire_evict_spent_nurseries_into_focus_car(focus_car);
	// Now release all the nurseries owned by the threads to the spent nurseries list.
	orp_sapphire_release_nurseries_all_threads();
	// Now evict the just released spent nurseries into this, the "from" c space focus car.
	p_global_bs->get_young_generation()->sapphire_evict_spent_nurseries_into_focus_car(focus_car);
}
#endif  // GC_SAPPHIRE

void Car::enumerate_reference_queues ()
{
    if (_p_weak_object_list) {
        gc_add_root_set_entry((Java_java_lang_Object **)&_p_weak_object_list);
    }
    
    if (_p_soft_object_list) {
        gc_add_root_set_entry((Java_java_lang_Object **)&_p_soft_object_list);
    }
    
    if (_p_phantom_object_list) {
        gc_add_root_set_entry((Java_java_lang_Object **)&_p_phantom_object_list);
    }
#ifdef GC_TRACE_WEAK_REF
    orp_cout << "A car just enumerated the weak/soft/phantom reference queues" << endl;
#endif // GC_TRACE_WEAK_REF
} 

//
// Helper routine for reclaim_car.
// Evict all objects referenced from inside this generation.
// This routine is used by the reclaim_car method.
//
void
Car::_evict_inside_generation_refsxxx()
{
    assert (0);
}

// Filter out NULL and references that are not to the focus car
// and slots in the focus car holding references to the focus car.
bool
Car::_is_slot_interesting(Java_java_lang_Object **pp_obj_ref)
{
    assert (this == ((Mrl_Gc_V1 *)p_gc)->p_get_focus_car());

	if (*pp_obj_ref == NULL) {
		return false;
	}
    // Is the target object in the focus car?
    Gc_Space *p_container = 
			_p_block_store->p_get_object_container(*pp_obj_ref);
    if (!(p_container == this)) {
  
        gc_trace (*pp_obj_ref, 
            "Car::_is_slot_interesting, object is not in focus car, NOT interesting."); 
        return false;
    }
    // Is the slot in the focus car?
    if (is_address_in_this_car (pp_obj_ref)) {
        gc_trace (*pp_obj_ref, 
            "Car::_is_slot_interesting, slot is in focus car object, NOT interesting."); 
        return false;
    }
    
    gc_trace (*pp_obj_ref,
        "Car::_is_slot_interesting, pointer to focus car from outside focus car, found interesting."); 
    return true;
}
//
// Helper routine for reclaim_car.
// Starting a reclaim of this car: first take care of
// the incoming pointers into the MOS generation. This routine is used
// by the reclaim_car method. Since these slots are not in the MOS they
// do not have to be remembered.
//
void
Car::_evict_out_of_mos_refs(Remembered_Set *p_incoming_refs)
{
    p_incoming_refs->rewind();
    Java_java_lang_Object **pp_obj_ref;
    while (pp_obj_ref = p_incoming_refs->next()) {
        if (!(_is_slot_interesting (pp_obj_ref))) {
            // No it is not. It might have been interesting at one time but
            // remsets are conservative in nature and the slot could have
            // be changed.
            continue;
        }
        if (p_global_bs->is_address_in_heap(pp_obj_ref)) {
            if (_p_train_generation->is_address_in_my_generation(pp_obj_ref)) {
                continue;
            }
        }
        // The slot is not in MOS, it is either in YOS or outside of the heap. In either
        // case no remembered set needs to be maintained for it...

        if (is_object_forwarded(*pp_obj_ref)) {
            //
            // The object has already been moved - just update the ref.
			//
			update_reference_forwarded(pp_obj_ref);
		} else {
            //
            // The minor collection only handed us references into
			// the focus car. The Train generation will heuristically
			// place the object in the right train.
            //
            Java_java_lang_Object *temp = 
                _p_train_generation->p_move_train_object(pp_obj_ref);
            // The slot is outside MOS so the object had to have been moved.
            assert (temp);
            if (temp != NULL) {
                // It could be NULL if it is a pointer from a car in the focus train.
                *pp_obj_ref = temp;
            }
            //
            // Update the residency stats.
            //
		    _resident_object_count--;
		    _resident_occupied_bytes -= get_real_object_size_bytes(*pp_obj_ref);
        }
    }
    // At this point p_incoming_refs holds no roots that have
    // pointers into this car. We still have to do the cheney scan
    // and that could evict more objects out of the focus car.
}
//
// Helper routine for reclaim_car.
// Continuing a reclaim of this car: we have already taken care of
// the incoming pointers into the MOS generation. Take care of 
// slots in MOS that are not in the focus car. This will require
// some remset activity.
//
void
Car::_evict_out_of_train_refs(Remembered_Set *p_incoming_refs)
{
    p_incoming_refs->rewind();
    Java_java_lang_Object **pp_obj_ref;
    while (pp_obj_ref = p_incoming_refs->next()) {
        if (!(_is_slot_interesting (pp_obj_ref))) {
            // No it is not.
            continue;
        }
        if (is_object_forwarded(*pp_obj_ref)) {
            //
            // The object has already been moved - just update the ref.
			// Remembered set processing done below
			update_reference_forwarded(pp_obj_ref);
		} else {
            //
            // The minor collection only handed us references into
			// the focus car. The Train generation will heuristically
			// place the object in the right train.
            //
            Java_java_lang_Object *temp = 
                _p_train_generation->p_move_train_object(pp_obj_ref);
            if (temp != NULL) {
                // It could be NULL if it is a pointer from a car in the focus train.
                *pp_obj_ref = temp;
                continue; // No need to update remembered set logic.
            }
            //
            // Update the residency stats.
            //
		    _resident_object_count--;
		    _resident_occupied_bytes -= get_real_object_size_bytes(*pp_obj_ref);
        }
        _p_train_generation->
		    add_entry_to_generation_write_barriers(pp_obj_ref, *pp_obj_ref);
    }
    // At this point p_incoming_refs holds no roots that have
    // pointers into this car. We still have to do the cheney scan
    // and that could evict more objects out of the focus car.
}

//
// Helper routine for reclaim_car.
// Starting a reclaim of this car: Take care of slots in the focus train
// that are not in the focus car.
//
void
Car::_evict_out_of_car_refs(Remembered_Set *p_incoming_refs)
{
    p_incoming_refs->rewind();

    Java_java_lang_Object **pp_obj_ref;
    while (pp_obj_ref = p_incoming_refs->next()) {
        if (!(_is_slot_interesting (pp_obj_ref))) {
            // No it is not.
            continue;
        }
        if (is_object_forwarded(*pp_obj_ref)) {
            //
            // The object has already been moved - just update the ref.
			//
			update_reference_forwarded(pp_obj_ref);
		} else {
            //
            // The minor collection only handed us references into
			// the focus car. The Train generation will heuristically
			// place the object in the right train.
            //
			*pp_obj_ref = _p_train_generation->p_move_car_object(pp_obj_ref);
            assert (*pp_obj_ref);
            //
            // Update the residency stats.
            //
		    _resident_object_count--;
		    _resident_occupied_bytes -= get_real_object_size_bytes(*pp_obj_ref);
        }
        _p_train_generation->
		    add_entry_to_generation_write_barriers(pp_obj_ref, *pp_obj_ref);
    }
}

#if (GC_DEBUG>0)
void Car::inspect(unsigned int level)
{
	cout << "    car " << this << " id: " << identity << endl;
	cout << "    contains " << _resident_object_count << " objects" << endl;
	cout << "    train is " << _p_train ;
	cout << " ( " << _p_train->get_id() << " ) " << endl;

	for (unsigned int idx = 0; idx < _number_of_blocks; idx++) {
		cout << "    Inspecting block idx# " << idx << endl;
		_inspect_block(_p_block[idx], _p_block_end[idx], idx);
	}
}
#endif // _DEBUG

//
// My containing train wants to know:
// Are there any pointers into this car from outside this train.
//
bool
Car::inter_train_refs_exist()
{
	_p_container_write_barrier->rewind();

	Java_java_lang_Object **pp_obj_ref;

    while ((pp_obj_ref = _p_container_write_barrier->next()) != NULL) {

		if (_is_reference_into_this_car(pp_obj_ref)) {
			//
			// We just verified that this entry is not obsolete, ie. some
			// mutation subsequent to its insertion into this remembered
			// set did not cause it to now refer to an object in a
			// different container.
			//
            if (_p_train_generation->is_address_in_my_generation(pp_obj_ref)) {
//			if (is_address_in_my_generation(pp_obj_ref)) {
				//
				// OK, the pointer originates in mature space.
				//
				Train *p_remote_train = 
					p_global_bs->p_get_reference_train(pp_obj_ref);

				if (_p_train != p_remote_train) {
					//
					// OK, this is an incoming pointer from another train.
					//
					goto true_return;
				}
			} else {
				//
				// This is a pointer from outside mature space.
				// Therefore inter-train references DO indeed exist.
				//
				goto true_return;
			}
		}
	}
	//
	// Since we scanned the entire remembered set without stumbling
	// across an incoming pointer, we haven't discovered any inter-
	// train references into this car.
	//
	return false;
	
true_return:
#if (GC_DEBUG>2)
    //
    // Need to ensure the table state is consistent since we aborted scan.
    //
    _p_container_write_barrier->reset_scan();
#endif
    return true;
}

// Return true if object is in this car.
bool 
Car::is_address_in_this_car(void *p_addr)
{
    if (p_global_bs->is_address_in_heap (p_addr)) {
        return _is_in_my_car(p_addr);
    }
    return false;
}


//
// Is the specified address contained in my car.
//
bool 
Car::_is_in_my_car(void *p_addr) 
{
    bool result = false;
    Gc_Space *container = p_global_bs->p_get_address_container(p_addr);
    if (this == (Car *)container) {
        result = true;
    }
    return result;
#if 0
    assert (p_global_bs->is_address_in_heap (p_addr));
	for (unsigned long idx = 0; idx < _number_of_blocks; idx++) {
		if (_p_block_store->is_address_in_super_block(_p_block[idx], p_addr)) {
			assert (result);
            return true;
		}
	}
    assert (!result);
	return false;
#endif
}
 
//
// Is the specified address not contained in my car.
//
bool 
Car::_is_not_in_my_car(void *p_addr) 
{
	for (unsigned long idx = 0; idx < _number_of_blocks; idx++) {
		if (_p_block_store->is_address_in_super_block(_p_block[idx], p_addr)) {
			return false;
		}
	}
	return true;
}


//
// The argument is the location of a pointer possible to an object in this car.
// If it is a pointer into this car then return true otherwise return false.
//
// One needs to remember that remembered sets are only a conservative
// approximation of the pointers into a car. In particular a location can
// be placed into a remembered set because it holds a pointer into the
// focus car. However, the value in the location can be subsequentially
// changed, possible to null or to point to an object not even in this
// generation.
//
bool
Car::_is_reference_into_this_car(Java_java_lang_Object **pp_obj_ref)
{
	if (!(*pp_obj_ref)) {
	  return false;
	}

	if (is_object_not_in_my_generation (*pp_obj_ref)) {
		return false;
	}
	//
	// So it isn't null and it is in a train, check IDs
	//
    Train *p_train_object_is_in = 
        p_global_bs->p_get_object_train(*pp_obj_ref);
            
    if (p_train_object_is_in->get_train_id() != _p_train->get_train_id()) {
		return false;
	}

	if (p_global_bs->p_get_object_car(*pp_obj_ref) != this) {
		return false;
	}

	return true;
}

//
// After reclaiming the focus car, we process finalizable objects.
//
void
Car::_locate_and_process_finalizable_objects(bool doing_mos_collection)
{
//    orp_cout << " *** NOT processing finalizable object in car.cpp. " << endl;
//    return;

    Java_java_lang_Object *p_obj;
    Java_java_lang_Object *p_target_object;

    _p_finalizable_object_list->rewind();

    while ((p_obj = _p_finalizable_object_list->next()) != NULL) {
        if (!is_object_forwarded(p_obj)) {
            //
            // The object is toast: candidate for running finalizer
            //

            //
            // For the present, we move the object to the end of the
            // oldest train. The reasoning is that an attempt should
            // be made to collect it as soon as possible, assuming
            // that the ORP is eagerly running finalizers.
            //
 
          
            Java_java_lang_Object **pp_obj_ref = &p_obj;
            Java_java_lang_Object *p_source_object = p_obj;

            p_target_object = _p_train->p_scavenge_object(pp_obj_ref);
            orp_finalize_object(p_target_object);
 
 
        } else {
            //
            // The object is still alive: notify the car it now
            // resides in that the object is finalizable.
            //
            Object_Gc_Header *p_gc_hdr = get_object_gc_header(p_obj);
            //
            // Locate the new location of the object from the
            // forwarding address in the header.
            //
#ifdef POINTER64
            p_target_object =
                (Java_java_lang_Object *)((POINTER_SIZE_INT)*p_gc_hdr & 0xffffffffFFFFFFFE);
#else
            p_target_object =
                (Java_java_lang_Object *)((POINTER_SIZE_INT)*p_gc_hdr & 0xFFFFFFFE);
#endif

            //
            // Locate the target car.
            //
            Gc_Space *p_container =
                _p_block_store->p_get_object_container(p_target_object);
            //
            // Pass the responsibility onto the new car.
            //
            assert(p_container->is_car());

            ((Car *)p_container)->record_finalizable_object(p_target_object);
        }
    }

    //
    // OK, we have gotten all the candidates for finalization.
    // Now revive all finalizer-reachable objects.
    //
    p_container->execute_pending_cheney_scans(true);
    //
}

#if (GC_DEBUG>3)
//
// Return my name and ID in a printable string for debugging.
//
void 
Car::my_name() 
{
    cout << "Car with ID " << identity << " contained in ";
    _p_train->my_name();
}
#endif // _DEBUG

// We have a slot that is not in MOS referencing an object in the
// focus car. We need to move the object in the focus car and return the
// new location.

Java_java_lang_Object *
Car::p_evict_out_of_mos_ref (Java_java_lang_Object **pp_obj_ref)
{
    // Someone has already done the work.
    if (is_object_forwarded(*pp_obj_ref)) {
        //
        // The object has already been moved - just update the ref.
        //
        update_reference_forwarded(pp_obj_ref);
        return *pp_obj_ref;
    }
    Java_java_lang_Object *temp = _p_train_generation->p_move_train_object(pp_obj_ref);
    assert (temp != NULL);
    *pp_obj_ref = temp;
    return temp;
}

// If the slot is not in the same train as this car 
// evict it to the appropriate place.
Java_java_lang_Object *
Car::p_evict_if_out_of_train_object(Java_java_lang_Object **pp_obj_ref)
{
    // We should only be given "interesting slots that require the object
    // to be moved.
//    assert(_is_slot_interesting(pp_obj_ref));

    if (!(_is_slot_interesting (pp_obj_ref))) {
        // No it is not.
        return NULL;
    }
    if (is_object_forwarded(*pp_obj_ref)) {
        //
        // The object has already been moved - just update the ref.
        //
        update_reference_forwarded(pp_obj_ref);
        return *pp_obj_ref;
    } else {
        //
        // The minor collection only handed us references into
        // the focus car. The Train generation will heuristically
        // place the object in the right train.
        //
        Java_java_lang_Object *temp = _p_train_generation->p_move_train_object(pp_obj_ref);
        if (temp != NULL) { 
            // It could be NULL if it is a pointer from a car in the focus train.
            *pp_obj_ref = temp;
        
            //
            // Update the residency stats.
            // Toss this someday...
            //
            _resident_object_count--;
            _resident_occupied_bytes -= get_real_object_size_bytes(*pp_obj_ref);
        }
        return temp;
    }
    return NULL;
}

// We have a single object that needs to be evicted from the focus car.
// Why am I here instead of knowing what flavor of object I have ahead of time.

Java_java_lang_Object *
Car::p_evict_object(Java_java_lang_Object **pp_obj, bool doing_mos_collection) 
{
    assert (0);
    return NULL;
}
//
// Our container train was asked to reclaim the oldest train.
// It turned out that the entire train wasn't garbage. Therefore
// the train has singled me out as the oldest car, and asked 
// me to reclaim myself.
//
void
Car::reclaim_car(Remembered_Set *p_incoming_refs,
                 Remembered_Set *p_weak_refs)
{
#if (GC_DEBUG>0)
	car_reclaim_hook(this);
#endif
#ifdef GC_TRAIN_TRACE
    orp_cout << "Train trace - reclaiming a car in train " << get_train_id() << endl;
#endif

#if (GC_DEBUG>0)
    // Make sure all the p_incoming_refs are valid.
    p_incoming_refs->rewind();
    Java_java_lang_Object **pp_obj_ref_quick_check;
    while (pp_obj_ref_quick_check = p_incoming_refs->next()) {
        if (*pp_obj_ref_quick_check != NULL) {
            verify_is_object (*pp_obj_ref_quick_check, p_incoming_refs);
        }
    }
#endif

    //
    // First take care of the incoming pointers
    // (into the MOS generation, either from the ORP
    // live references, or from the young-to-old remembered set.)
    //
    // No remembered set activity is needed here.
    _evict_out_of_mos_refs(p_incoming_refs);
	//
	// Run all the pending cheney scans.
	//
	_p_train_generation->execute_pending_cheney_scans(true);

	//
	// Check if the train is now garbage by checking
	// for pointers that straddle trains, into the
	// train we belong to.
	//
	if (_p_train->no_incoming_refs()) {
		//
		// All done - there are no pointers from in mos with referents into the entire train.
		// There may be pointers from outside MOS.
        goto finalize;
		return;
	}

    //
    // Next take care of the intra-generation 
    // incoming pointers that were stored 
    // in my personal car's remembered set by the
    // hybrid barriers.
    //

    _evict_out_of_train_refs (_p_container_write_barrier);

	//
	// Do any remaining cheney scans.
	//
	_p_train_generation->execute_pending_cheney_scans(true);

    
    _evict_out_of_car_refs (_p_container_write_barrier);

	//
	// Do any remaining cheney scans.
	//
	_p_train_generation->execute_pending_cheney_scans(true);

    //
    // The car should now be garbage.
    //

    //
    // Now process finalizable objects
    //

finalize:
	// Update any soft pointers.
	_update_soft_references(false, true);

    //
    // Update any weak pointers to objects that have moved, zero ones that died.
    //
    _update_weak_references(true);

	_update_phantom_references(true);

    _locate_and_process_finalizable_objects(true);

}

//
// The young generation is notifying us that a finalizable,
// but still reachable, object has just been tenured into 
// our car.
//
void 
Car::record_finalizable_object(Java_java_lang_Object *p_obj) 
{
    _p_finalizable_object_list->add_entry(p_obj);
}

//
// If requested by the app, all finalizers are run before ORP exit.
//
void
Car::run_all_finalizers()
{
    Java_java_lang_Object *p_obj;

    _p_finalizable_object_list->rewind();

    while ((p_obj = _p_finalizable_object_list->next()) != NULL) {
        orp_finalize_object(p_obj);
    }
}

//
// Our container train is telling us to import this object into our
// car. We return a pointer to the moved object if there is enough space
// in our car.
//
Java_java_lang_Object *
Car::p_reserve_space(Java_java_lang_Object **pp_obj_ref)
{
	if (_will_fit_in_car(pp_obj_ref)) {
		Java_java_lang_Object *p_new_obj_location =
			Block_List::p_reserve_space(pp_obj_ref);
		//
		// Don't need write barrier because we don't maintain out-of
		// -generation pointers when we scavenge from outside the
		// generation, and we don't maintain old-to-young pointers
		// when we scavenge from the focus car. 
		//

		return p_new_obj_location;
	} 
	//
	// We return a null, indicating to our train that there is no
	// space left in this car, prompting it to add another car
	// and try again.
	//
    return NULL;
}

//
// Our container train is telling us to reserver this many bytes in our
// car. We return a pointer to the new object if there is enough space
// in our car.
//
Java_java_lang_Object *
Car::p_reserve_bytes(unsigned size)
{
	if (_will_bytes_fit_in_car(size)) {
		Java_java_lang_Object *p_new_obj_location =
			Block_List::p_reserve_bytes(size);
		//
		// Don't need write barrier because we don't maintain out-of
		// -generation pointers when we scavenge from outside the
		// generation, and we don't maintain old-to-young pointers
		// when we scavenge from the focus car. 
		//

		return p_new_obj_location;
	} 
	//
	// We return a null, indicating to our train that there is no
	// space left in this car, prompting it to add another car
	// and try again.
	//
    return NULL;
}

//
// Our container train is telling us to import this object into our
// car. We return a pointer to the moved object if there is enough space
// in our car.
//
Java_java_lang_Object *
Car::p_scavenge_object(Java_java_lang_Object **pp_obj_ref)
{
	if (_will_fit_in_car(pp_obj_ref)) {
		Java_java_lang_Object *p_new_obj_location =
			Block_List::p_scavenge_object(pp_obj_ref);
		//
		// Don't need write barrier because we don't maintain out-of
		// -generation pointers when we scavenge from outside the
		// generation, and we don't maintain old-to-young pointers
		// when we scavenge from the focus car. 
		//

		return p_new_obj_location;
	} 
	//
	// We return a null, indicating to our train that there is no
	// space left in this car, prompting it to add another car
	// and try again.
	//
    return NULL;
}

//
// Merge the entries in the incoming remembered set with entry.
// This is only valid for GC Spaces with their own remembered sets.
// (Such as cars, and unlike nurseries and steps.)
//
// RLH-TRAIN This adds a remember set entry from one car into this car.
void 
Car::update_container_write_barrier(Java_java_lang_Object **pp_obj_ref) 
{
#if (GC_DEBUG>0)
	//
	// Sanity check - verify all pointers are incoming.
	//
	Gc_Space *p_container = 
		_p_block_store->p_get_object_container(*pp_obj_ref);
    assert(p_container == this);
    // Also make sure this is a pointer from a car.
    // RLH-TRAIN .. new check below.
    _p_train_generation->is_address_in_my_generation(pp_obj_ref);
#endif
	_p_container_write_barrier->add_entry(pp_obj_ref);
}

// POSSIBLE RACE CONDITION HERE - USE INTERLOCKED_COMPARE_EXCHANGE
void Car::enqueue_soft_ref (java_lang_ref_Reference *soft_reference)
{
//    soft_reference->next = _p_soft_object_list;
    gc_heap_slot_write_ref ((Java_java_lang_Object *)soft_reference,
        (Java_java_lang_Object **)&(soft_reference->next),
        (Java_java_lang_Object *)_p_soft_object_list);
	_p_soft_object_list = soft_reference;
}
void Car::enqueue_weak_ref (java_lang_ref_Reference *weak_reference)
{
//    weak_reference->next = _p_weak_object_list;
    gc_heap_slot_write_ref ((Java_java_lang_Object *)weak_reference,
        (Java_java_lang_Object **)&(weak_reference->next),
        (Java_java_lang_Object *)_p_weak_object_list);
	_p_weak_object_list = weak_reference; 
}
void Car::enqueue_phantom_ref (java_lang_ref_Reference *phantom_reference)
{
//    phantom_reference->next = _p_phantom_object_list;

    gc_heap_slot_write_ref ((Java_java_lang_Object *)phantom_reference,
        (Java_java_lang_Object **)&(phantom_reference->next),
        (Java_java_lang_Object *)_p_phantom_object_list);
	_p_phantom_object_list = phantom_reference; 
}


// Delink a ref from the _p_mumble_object_list.
void Car::remove_weak_ref_Reference (java_lang_ref_Reference *p_ref_Reference, 
									 java_lang_ref_Reference *previous)
{
    
	if (previous == NULL) { // The first one on the list.
	    _p_weak_object_list = p_ref_Reference->next;
	} else {
//		previous->next = p_ref_Reference->next;		
    gc_heap_slot_write_ref ((Java_java_lang_Object *)previous,
        (Java_java_lang_Object **)&(previous->next),
        (Java_java_lang_Object *)p_ref_Reference->next);
    }
	return;
}
// Delink a ref from the linked list.
void Car::remove_soft_ref_Reference (java_lang_ref_Reference *p_ref_Reference, 
									 java_lang_ref_Reference *previous)
{
	if (previous == NULL) { // The first one on the list.
		_p_soft_object_list = p_ref_Reference->next;
	} else {
//        previous->next = p_ref_Reference->next;		
        gc_heap_slot_write_ref ((Java_java_lang_Object *)previous,
            (Java_java_lang_Object **)&(previous->next),
            (Java_java_lang_Object *)p_ref_Reference->next);
	}
	return;
}
// Delink a ref from the linked list.
void Car::remove_phantom_ref_Reference (java_lang_ref_Reference *p_ref_Reference, 
		  							    java_lang_ref_Reference *previous)
{
	if (previous == NULL) { // The first one on the list.
		_p_phantom_object_list = p_ref_Reference->next;
	} else {
//        previous->next = p_ref_Reference->next;		
        gc_heap_slot_write_ref ((Java_java_lang_Object *)previous,
            (Java_java_lang_Object **)&(previous->next),
            (Java_java_lang_Object *)p_ref_Reference->next);
	}
	return;
}

//
// This is executed for all cars at each collection.
//
// We have a list of java.lang.ref.Reference objects. They each have 
// a hidden field pointing to the referent. This field is only accessable 
// via the "get" method. If the referent was in a region that was being
// collected then either it was moved indicating that it is still alive
// or it was not moved indicating that the only references to it are 
// weak references. 
//
// If it was moved (and is alive) then the java.lang.ref.reference is 
//    dequeued from the queue associated with the old location of the object
//    and queued with the new location.
// If it was not moved (is unreachable) then the java.lang.ref.reference 
//    object will be places on the java.lang.ref.ReferenceQueue if on was associated 
//    with it when it was created.
//
//

//
// If a queue has been specified we need to move the reference into the queue.
// Similar code is in step_generation.cpp, If there is a bug here there 
// probable is one there.
//
// The logic for lock free enqueueing is based on the fact that remove uses a lock
// and also sets the which field to 1 if it is removing off head1 and to 2 if
// it is removing off of head 2. If which is set to 0 then nothing is being 
// removed at this time. This code just looks at which and if it is 
// 0 enqueues on head1,
// 1 enqueues on head2,
// 2 enqueues on head1.
//
// The mutator code for queues grabs the lock, sets which to either 1 or 2 based on
// if one has any objects on it, does its work and then sets which back to 0.
// Notice that this code will never be in conflict with the mutator code since it holds the
// gc_lock and does not touch the list being dealt with by the mutator.
//
// How does one write a test for this???
//
// No SMP issues since we don't touch any fields that are being used
// by the Java code.
//
// Race condition possible if Java code is enqueueing and then we also enqueue then
// we can have the same reference enqueued twice. 
// 
void process_weak_reference_in_car (Java_java_lang_ref_Reference *p_ref_Reference)
{
    // If there is a queue, enqueue the reference object.
    if (p_ref_Reference->queue) {
        // If the object has not already been enqueued, do it, if it has
        // been enqueued don't enqueue it again.
        if (p_ref_Reference->enqueued == false) {
            java_lang_ref_ReferenceQueue *the_queue = p_ref_Reference->queue;
            if (the_queue->which == 1){
                p_ref_Reference->next = the_queue->head2;
                the_queue->head2 = p_ref_Reference;
            } else {                
                assert ((the_queue->which == 2) || (the_queue->which == 0));
                p_ref_Reference->next = the_queue->head1;
                the_queue->head1 = p_ref_Reference;
            }
            orp_notify_reference_queue((Java_java_lang_Object *)the_queue);
            p_ref_Reference->enqueued = true;
        }
    }
    p_ref_Reference->referent = NULL; // Clear the referent.
#ifdef GC_TRACE_WEAK_REF
	orp_cout << "Weak Reference has been processed." << endl;
#endif // GC_TRACE_WEAK_REF
    return;
}

void process_soft_reference_in_car (Java_java_lang_ref_Reference *p_ref_Reference)
{
    // If there is a queue, enqueue the reference object.
    if (p_ref_Reference->queue) {
        // If the object has not already been enqueued, do it, if it has
        // been enqueued don't enqueue it again.
        if (p_ref_Reference->enqueued == false) {
            java_lang_ref_ReferenceQueue *the_queue = p_ref_Reference->queue;
            if (the_queue->which == 1){
                p_ref_Reference->next = the_queue->head2;
                the_queue->head2 = p_ref_Reference;
            } else {                
                assert ((the_queue->which == 2) || (the_queue->which == 0));
                p_ref_Reference->next = the_queue->head1;
                the_queue->head1 = p_ref_Reference;
            }
            orp_notify_reference_queue((Java_java_lang_Object *)the_queue);
            p_ref_Reference->enqueued = true;
        }
    } 
#ifdef GC_TRACE_WEAK_REF
	orp_cout << "Soft Reference has been processed." << endl;
#endif // GC_TRACE_WEAK_REF
    return;
}

void process_phantom_reference_in_car (Java_java_lang_ref_Reference *p_ref_Reference)
{
	assert (p_ref_Reference->queue != NULL);
	// All phantom references should have queues or an exception should have been
	// thrown at creation time.
    // If there is a queue, enqueue the reference object.

    // If the object has not already been enqueued do it, if it has
    // been enqueued don't enqueue it again.
    if (p_ref_Reference->enqueued == false) {
        java_lang_ref_ReferenceQueue *the_queue = p_ref_Reference->queue;
        if (the_queue->which == 1){
            p_ref_Reference->next = the_queue->head2;
            the_queue->head2 = p_ref_Reference;
        } else {                
            assert ((the_queue->which == 2) || (the_queue->which == 0));
            p_ref_Reference->next = the_queue->head1;
            the_queue->head1 = p_ref_Reference;
        }
        orp_notify_reference_queue((Java_java_lang_Object *)the_queue);
        p_ref_Reference->enqueued = true;
    } 
#ifdef GC_TRACE_WEAK_REF
    // The referent is not cleared in the case of a phantom reference.
	orp_cout << "Phantom Reference has been processed." << endl;
#endif //GC_TRACE_WEAK_REF
    return;
}

void
Car::_update_phantom_references(bool doing_mos_collection)
{
	// loop through all the objects on the list. All the referents have been moved
	// to new cars if they are strongly reachable (still alive). If they aren't 
	// alive then the referents are enqueued. My reading of the spec is that we
	// don't clear the referent even though we can not retrieve the referent. I assume this
	// is done so that the objects that the referent's referents are kept
	// alive so that they can be used.

	if (_p_phantom_object_list) {
		java_lang_ref_Reference *p_ref_Reference = (java_lang_ref_Reference *)_p_phantom_object_list;
		java_lang_ref_Reference *p_next = NULL;

	    while (p_ref_Reference != NULL) {
		    Java_java_lang_Object *p_obj;
			p_obj = p_ref_Reference->referent;
		    if (p_obj) {
				if (is_object_forwarded(p_obj)) { // It been moved out of this car?
					p_next = p_ref_Reference->next;
					p_ref_Reference->next = NULL;
					// gc_register_weak_ref will always add to the front of the list
					// of some other car.
					gc_register_soft_ref ((Partial_Reveal_JavaObject *)p_ref_Reference, (Partial_Reveal_JavaObject *)(get_forwarded_object(p_obj)));
				} else { // Object has died. Save it.
	    			// Assert that p_obj is in this car
					assert (this == p_global_bs->p_get_object_car(p_obj)); 
					p_next = p_ref_Reference->next;
					p_ref_Reference->next = NULL;
					// Move the referent to another car, treating the object as if it 
					// was reachable from a root.
					
		            Java_java_lang_Object *temp = 
					    _p_train_generation->p_move_train_object(&p_ref_Reference->referent);
					if (temp == NULL) {
						// ref_Reference is inside same train.
						temp = _p_train_generation->p_move_car_object(&p_ref_Reference->referent);
					};
                    //
			        // The object is only reachable via this this pointer.
					// enqueue the object if that has been requested.
					//
					process_phantom_reference_in_car (p_ref_Reference);

#ifdef GC_TRACE_WEAK_REF
						orp_cout << "Moving phantom object." << endl;
#endif // GC_TRACE_WEAK_REF

               	}
	        } else {
				// How can we have a ref.Reference with out a referent?
				// The application can clear the referent leaving us with a null referent.
				// In this case the ref.Reference object can be removed from the list.
				p_next = p_ref_Reference->next;
				p_ref_Reference->next = NULL; 
			}
			p_ref_Reference = p_next;
		}
	}
    // Cheney scan the spaces that might hold promoted objects.
	
    p_container->execute_pending_cheney_scans(true);
} 

void
Car::_update_soft_references(bool low_on_space, bool doing_mos_collection)
{
	// loop through all the objects on the list. All the referents have been moved
	// to new cars if they are strongly reachable (still alive). If they aren't 
	// alive and update_soft_reference we are low on space then the referents are 
	// cleared and enqueued.
	// If we are not low on space then the object is copied to new space
	// and kept alive.

	if (_p_soft_object_list) {
		java_lang_ref_Reference *p_ref_Reference = (java_lang_ref_Reference *)_p_soft_object_list;
		java_lang_ref_Reference *p_next = NULL;

	    while (p_ref_Reference != NULL) {
		    Java_java_lang_Object *p_obj;
			p_obj = p_ref_Reference->referent;
		    if (p_obj) {
				if (is_object_forwarded(p_obj)) { // It been moved out of this car?
					p_next = p_ref_Reference->next;
					p_ref_Reference->next = NULL;
					// gc_register_weak_ref will always add to the front of the list
					// of some other car.
					gc_register_soft_ref ((Partial_Reveal_JavaObject *)p_ref_Reference, (Partial_Reveal_JavaObject *)(get_forwarded_object(p_obj)));
				} else { // Object has died. Kill it if we are low_on_space
					// Assert that p_obj is in this car
					assert (this == p_global_bs->p_get_object_car(p_obj)); 
					if (low_on_space) {
						p_next = p_ref_Reference->next;
						p_ref_Reference->next = NULL;
						//
				        // The object is only reachable via this this pointer.
						// enqueue the object if that has been requested.
						//
						process_soft_reference_in_car (p_ref_Reference);
					} else {
						p_next = p_ref_Reference->next;
						p_ref_Reference->next = NULL;
						// Move the referent to another car, treating the object as if it 
						// was reachable from a root.
						
			            Java_java_lang_Object *temp = 
						    _p_train_generation->p_move_train_object(&p_ref_Reference->referent);
						
						if (temp == NULL) {
							// ref_Reference is inside same train.
							temp = _p_train_generation->p_move_car_object(&p_ref_Reference->referent);
						};
#ifdef GC_TRACE_WEAK_REF
						orp_cout << "Moving soft object." << endl;
#endif // GC_TRACE_WEAK_REF
					}
               	}
	        } else {
				// How can we have a ref.Reference with out a referent?
				// The application can clear the referent leaving us with a null referent.
				// In this case the ref.Reference object can be removed from the list.
				p_next = p_ref_Reference->next;
				p_ref_Reference->next = NULL; 
			}
			p_ref_Reference = p_next;
		}
	}
    // Cheney scan the spaces that might hold promoted objects.
	
    p_container->execute_pending_cheney_scans(true);
} 

void
Car::_update_weak_references(bool doing_mos_collection)
{
	// loop through all the objects on the list. All the referents have been moved
	// to new cars if they are still alive. If they aren't alive they will be cleared.
	// This means that this p_weak_object_list will be cleared and all
	// relevant ref.References will registered or queued
	if (_p_weak_object_list) {
		java_lang_ref_Reference *p_ref_Reference = (java_lang_ref_Reference *)_p_weak_object_list;
		java_lang_ref_Reference *p_next = NULL;

	    while (p_ref_Reference != NULL) {
		    Java_java_lang_Object *p_obj;
			p_obj = p_ref_Reference->referent;
		    if (p_obj) {
				if (is_object_forwarded(p_obj)) { // It been moved out of this car?
					p_next = p_ref_Reference->next;
					p_ref_Reference->next = NULL;
					// gc_register_weak_ref will always add to the front of the list
					// of some other car.
					gc_register_weak_ref ((Partial_Reveal_JavaObject *)p_ref_Reference, (Partial_Reveal_JavaObject *)(get_forwarded_object(p_obj)));
				} else { // Object has died.
					
	    			// Assert that p_obj is in this car
					assert (this == p_global_bs->p_get_object_car(p_obj)); 
					p_next = p_ref_Reference->next;
					p_ref_Reference->next = NULL;
					//
                    // The object is only reachable via this this pointer.
					// enqueue the object if that has been requested.
					//
					process_weak_reference_in_car (p_ref_Reference); 
               	}
	        } else {
				// How can we have a ref.Reference with out a referent?
				// The application can clear the referent leaving us with a null referent.
				// In this case the ref.Reference object can be removed from the list.
				p_next = p_ref_Reference->next;
				p_ref_Reference->next = NULL; 
			}
			p_ref_Reference = p_next;
		}
	}
    // Cheney scan the spaces that might hold promoted reference objects.
	
    p_container->execute_pending_cheney_scans(true);
} 

#if (GC_DEBUG>0)
//
// Debug routine to verify the integrity of the contents of this car's write barrier
// Verify that none of the incoming pointers are from obsolete space.
//
void
Car::verify_car_write_barrier()
{
    Java_java_lang_Object **pp_obj_ref;
    _p_container_write_barrier->rewind();
    while ((pp_obj_ref = _p_container_write_barrier->next()) != NULL) {
        Gc_Space *p_container = _p_block_store->p_get_address_container(pp_obj_ref);
        assert(p_container->is_car());
    }
}
#endif // _DEBUG

bool
Car::walk_car(bool (*func)(Object_Gc_Header *,
			               Remembered_Set *))
{
    for (unsigned idx = 0; idx < _number_of_blocks; idx++) {
        _walk_block(_p_block[idx],
                    _p_block_end[idx],
                    func);
    }
    return true;
}

bool
Car::_will_fit_in_car(Java_java_lang_Object **pp_obj_ref)
{
	if (_p_free == NULL) {
		_add_block();
	}

	if ((_will_not_fit_in_block(pp_obj_ref)) &&
		_number_of_blocks >= _maximum_blocks_in_car) {
		return false;
	} else if (_will_not_fit_in_block(pp_obj_ref)) {
		_add_block();
		return true;
	} else {
		return true;
	}
}

bool
Car::_will_bytes_fit_in_car(unsigned size)
{
	if (_p_free == NULL) {
		_add_block();
	}

	if ((_will_not_fit_in_block(size)) &&
		_number_of_blocks >= _maximum_blocks_in_car) {
		return false;
	} else if (_will_not_fit_in_block(size)) {
		_add_block();
		return true;
	} else {
		return true;
	}
}

// end file gc\car.cpp
