// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/include/nursery.h,v 1.1.1.1 2001/07/23 07:25:39 xli18 Exp $
//

#ifndef _nursery_H_
#define _nursery_H_

//
// A Nursery is a Gc_Space. It is a container, or a repository of
// Java objects. 
//
//                          Gc_Space (abstract)
//                              |
//             ------------------------------------
//             |                                  |
//             |                                  |
//        Block_List (abstract)                   |
//      ---------------------                     |
//      |                    |                 Nursery
//     Car                  Step
//
//
//
// Unlike Cars and Steps, Nurseries comprise contiguous blocks.
// Therefore a Nursery inherits from Gc_Space instead of Block_List.
//
// To avoid making Java object allocation a point of serialization in
// multi-threaded programs, each Java thread is given its own nursery.
// When this is exhausted, it gets additional nurseries from a pool.
// When the pool is exhausted, a minor collection is triggered.
//
//



#include "gc_space.h"
#include "gc_consts.h"
#include "car.h"
#include "gc_hooks.h"

class Block_Store;
class Card_Table;
class Generation;
class Gc_Fast_Hooks;
class Gc_Plan;
class Step;

#ifdef GC_ZERO_THREAD

#define NURSERY_CLEARED 1
#define NURSERY_NOT_CLEARED 0
#define NURSERY_CLEARING_IN_PROGRESS -1

#endif // GC_ZERO_THREAD


class Nursery : public Gc_Space {
public:
	Nursery(unsigned long nursery_number,
            Gc_Fast_Hooks *p_gc_hooks,
            Gc_Plan       *p_gc_plan,
            Generation    *p_container,
			Gc_Space      *p_superior,
            Card_Table    *p_card_table,
		    Block_Store   *p_bs,
			int            nursery_block_size_bytes);

    virtual ~Nursery();

    //
    // Return the base that this nursery starts at.
    //
    void *get_nursery_base ();

    //
    // Return the free pointer for this nursery.
    //
    void *get_nursery_free ();

#ifdef trashme
    //
    // Return the step that is this nurseries superior
    //
    Step *_get_superior ();
#endif

#if (GC_DEBUG>3)
	//
	// Beforp discarding old nursery blocks (during debugging),
	// inform the block store that these are now obsolete.
	// (i.e. associated with the obsolete_space container.)
	//
	void debug_flag_old_blocks_obsolete();
#endif // _DEBUG

#if (GC_DEBUG>4)
	//
	// Our generation is telling us to write protect the blocks
	// in this nursery and allocate fresh ones. This is for
	// debugging ORP problems.
	//
	void debug_write_protect();
#endif // _DEBUG

#if (GC_DEBUG>4)
	//
	// Our generation is telling us to leave the old blocks alone,
	// instead to allocate fresh ones. This is for debugging GC
	// problems.
	//
	void debug_replace_intact();
#endif // _DEBUG

#if (GC_DEBUG>4)
	//
	// Our generation is telling us to leave the old blocks alone,
	// instead to allocate fresh ones. This is for debugging GC
	// problems. This version writes a distinct pattern into the spent
	// nurseries.
	//
	void debug_replace_cleared();
#endif // _DEBUG
 
    Java_java_lang_Object *gc_malloc(unsigned object_size_bytes, Partial_Reveal_VTable *p_vtable);
 
#if (GC_DEBUG>3)
	void dump();  // dump textual representation to stdout
#endif // _DEBUG


    //
    // The containing generation traced a reachable object
    // down to my space. I'll just pass it to my superior
    // step for scavenging.
    //
    inline Java_java_lang_Object *
    Nursery::p_evict_object(Java_java_lang_Object **pp_obj_ref, bool doing_mos_collection) 
    {
    #if (GC_DEBUG>0)
	    evict_hook(pp_obj_ref);
    #endif

    #if (GC_DEBUG>2)
	    assert(_p_superior->is_step() == TRUE);
    #endif 
        gc_trace (*pp_obj_ref, "Nursery::p_evict_object ln 385"); 
	    return _p_superior->p_scavenge_object(pp_obj_ref);
    }


#if (GC_DEBUG>2)
	//
	//
	//
	Java_java_lang_Object 
        *p_scavenge_object(Java_java_lang_Object **pp_obj) {
		orp_cout << "Error: nursery asked to scavenge object!" << endl;
		orp_exit(1);
		return NULL;
	}
#endif

#if (GC_DEBUG>2)
	//
	// Do any pending cheney scans: override generic version:
	//
	void execute_cheney_scan(Generation *p_gen, bool doing_mos_collection) {
		orp_cout << "error: illegal request to nursery to execute cheney scan" << endl;
		orp_exit(1); // illegal for nurseries
	}
#endif // _DEBUG

	//
	// Do any required cleanup at the end of the collection.
	//
	virtual void cleanup();

	void reset(void) { 
        _p_free = _p_base; 
    }

#ifdef GC_SAPPHIRE
	bool sapphire_evict_blocks(Car *focus_car);
#endif // GC_SAPPHIRE
		
	//
    // GC_FALSE_COLLECTION  
    //
    // We are about to do a deferred collection and we need to evict the nursery block
    // from this nursery into whatever is superior.
    //
    // Returns true if successful 
    //         false otherwise.
    bool evict_blocks ();
    
    // GC_FALSE_COLLECTION
    //
    // Take the empty block that starts at new_block and overwrite the
    // current block. Reset all relevant data structures. 
    void replace_nursery_block (void *new_block, int size);

	//
	// routines for introspection
	//

	virtual bool is_nursery() {
		return true;
	}

	Nursery *p_next_nursery; // used to chain spent nurseries
    
#if (GC_DEBUG>0)
	void verify_empty_nursery ();
#else
    inline void verify_empty_nursery() 
    {
    }
#endif // (GC_DEBUG>0)
#if (GC_DEBUG>3)
	//
	// Debug routine to inspect details.
	//
	virtual void inspect(unsigned int level);
#endif // _DEBUG

#if (GC_DEBUG>0)
	//
	// Debugging: used to ensure multiple threads are not
	// accessing the same nursery.
	//
	void set_owner_thread(DWORD ThreadId)
	{
		_current_owner_thread = ThreadId;
	}
#endif

	//
	// Merge the entries in the incoming remembered set with entry.
    // This is only valid for GC Spaces with their own remembered sets.
    // (Such as cars, and unlike nurseries and steps.)
	//
	virtual void update_remembered_set(Java_java_lang_Object **pp_obj) {
		orp_cout << "Error: attempt to add entry to a nursery rs (I don't have one!)" << endl;
        orp_exit(1);
	}
#if (GC_DEBUG>3)
	virtual bool Nursery::verify_space();
#endif // _DEBUG

	inline virtual int my_block_size_bytes() {
		return _nursery_block_size_bytes;
	}

#ifdef GC_ZERO_THREAD

    // Need to make _is_nursery_clear a Public member

    // Value of NURSERY_CLEARED, means that nursery is clear. Zeroing thread is done with it.
    
    // Value of NURSERY_CLEARING_IN_PROGRESS, means means nursery is being cleared by Zeroing thread ...
    // mutator thread sleeps as it gets done.
    
    // Value of NURSERY_NOT_CLEARED means nursery has not yet been gotten 
    // to by Zeroing thread, so the mutator should clear it by itself.

    int _is_nursery_clear;

#endif // GC_ZERO_THREAD

private:

    // Used to initialize the nursery by constructor and replace block routines.
    void _init_nursery (void * new_block, int block_size_bytes);

	int _nursery_block_size_bytes;

	//
	// Bump the scan pointer forward past an object
	//
	virtual bool 
	_advance_scan_pointer(unsigned int real_object_size_bytes) {
		orp_exit(1); // illegal
		return true;
	}
#if (GC_DEBUG>0)
	//
	// Debugging: to ensure multiple threads are not
	// accessing the same nursery.
	//
	DWORD _current_owner_thread;
    bool on_free_list;
    bool on_spent_list;
    bool on_thread;
#endif
	
	//
	// This represents the threshold over which an object is deemed
	// 'large.' The GC Plan provides us with this information when
	// this nursery is created. THis value is cached here, instead
	// of checking with the GC Plan each time, since we would be
	// on the critical path (allocation) when we needed this information.
	//
	unsigned long _los_threshold_bytes;

	//
	// Walk the nursery and apply function argument to all 
	// recognized objects.
	//
	bool Nursery::_walk_nursery(bool(*func)(Object_Gc_Header *,
		                                    Remembered_Set *));

    // Set to false until the nursery has been zeroed out.
    // The clearing of the nursery is done in gc_malloc and
    // the nursery cleanup code sets this to false at the
    // end of each GC.

#ifndef GC_ZERO_THREAD
    
    bool _is_nursery_clear;

#endif // GC_ZERO_THREAD

};

#endif // _nursery_H_

