// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/include/nursery_step_gen.h,v 1.1.1.1 2001/07/23 07:25:39 xli18 Exp $
//


#ifndef _nursery_step_gen_H_
#define _nursery_step_gen_H_

//
// Generations which have nurseries and steps (usually the youngest gen.).
// A Step_Plus_Nursery_Generation is a Step_Generation with nurseries
// added.
//
//                Generation
//                    |
//                    |
//       ---------------------------
//       |                         |
//       |                         |
//    Step_Generation              |
//       |                         |
//       |                  Train_Generation
//       |
//  Step_Plus_Nursery_Generation
//



#include "step_generation.h"
#include "nursery.h"
#include "gc_space.h"

class Card_Table;
class Gc_Fast_Hooks;
class Gc_Plan;

class Step_Plus_Nursery_Generation : public Step_Generation {
public:
    Step_Plus_Nursery_Generation(unsigned char generation_number,
                                 Gc_Fast_Hooks *p_gc_hooks,
                                 Gc_Plan       *p_gc_plan,
		                         Gc_Interface  *p_container,
								 Generation    *p_superior,
                                 Card_Table    *p_card_table,
		                         Block_Store   *p_bs);

	virtual ~Step_Plus_Nursery_Generation();

    //
    // We have more threads than nurseries and need to
    // add some more.
    //
    void add_nurseries();

	//
	// The GC interface gives each generation an opportunity to
	// clean up after a collection of that generation.
	//
	virtual void cleanup();

#if (GC_DEBUG>1)
	void debug_list_all_nurseries();
 
	void debug_list_spent_nurseries();
 
	void debug_list_free_nurseries();

	void dump_nursery_stats();
#endif

#if (GC_DEBUG>3)
	void dump();  // dump textual representation to components to stdout
#endif // _DEBUG
    // 
    // verify_nurseries is called during debuggin to make sure the
    // spent nursery list and the free nursery list are in sync.
    //
    bool verify_nurseries ();

	//
	// A thread exhaused its nursery.
	//
	virtual Nursery *p_cycle_nursery(Nursery *p_nursery,
		                             bool returnNullOnFail);
#ifdef GC_SAPPHIRE 
    Nursery *p_sapphire_release_nursery(Nursery *p_nursery,
        bool returnNullOnFail);
	bool sapphire_move_spent_nurseries_into_focus_car();
	
//
// sapphire_move_spent_nurseries_into_focus_car
//
// Move the blocks in the spent nurseries into the focus car and refill
// the nurseries with empty blocks.
//
	bool sapphire_evict_spent_nurseries_into_focus_car(Car *focus_car);

#endif // GC_SAPPHIRE

    // GC_FALSE_COLLECTION
    // 
    // Move the blocks in the spent nursery into the step and refill the
    // nurseries with empty blocks. 
    // Return true if you were successful and a full collection is not needed.
    // otherwise return false. 
    //
    bool move_spent_nurseries_into_step();

	//
	// One of our nurseries encountered an allocation request for a
	// large object and passed it up to us.
	//
 
    virtual Java_java_lang_Object *
	Step_Plus_Nursery_Generation::gc_pinned_malloc(unsigned size, 
	                                               Partial_Reveal_VTable *p_vt,
												   bool return_null_on_fail,
                                                   bool double_align                                             
                                                   ) {
		//
		// We pass it up to our own container, who manages the LOS object.
		//
		return _p_container->gc_pinned_malloc(size, p_vt, return_null_on_fail,
                                              double_align      
                                              );
	}
 
    //
    // Routines for reflection.
    //
    virtual bool is_train_generation() {
        return false;
    }

    virtual bool is_step_generation() {
        return true;
    }

    virtual bool is_step_plus_nursery_generation() {
        return true;
    }

#if (GC_DEBUG>3)
	//
	// Routine to display the contents of the generation.
	//
	virtual void inspect(unsigned int level);
#endif

	//
	// Take a spent nursery and store it away till 
	// the next reclaim.
	//
	void store_spent_nursery(Nursery *p_nursery);

    //
    // Return the number of nurseries that have been filled but not
    // yet reclaimed. This is an estimate since there could be race conditions
    // while counting the number of nurseries.
    // Assumption - load of _p_spent_nurseries is done as volatile...

    int get_number_of_spent_nurseries ();

    unsigned long get_number_of_nurseries () {
        return _number_of_nurseries;
    }

#if (GC_DEBUG>3)
	virtual bool verify_generation();
#endif // _DEBUG

#ifdef GC_ZERO_THREAD

    HANDLE Zero_Thread_Handle;

    // Boolean indicating if Zero thread has suspended itself.
    bool zero_thread_is_sleeping;
    // The Zero thread will die when this boolean is set...(it is set to true by the step destructor)
    bool zero_thread_die_signal;

    Nursery * get_me_next_nursery_to_zero(Nursery *previous_nursery_zeroed);

#endif // GC_ZERO_THREAD

private:


	//
	// The number of nurseries handed out since the last collection
	//
	int _allocated_nurseries;

	//
	// The number of spent nurseries
    // not used due to race conditions, now we just count the number of nurseries
    // on the list. There are still race conditions but they do not compound. In
    // anycase when we calculate the number of nurseries we need to understand that
    // we are only getting an estimate.
	//
//	int _spent_nurseries;

#if (GC_DEBUG>4)
	//
	// This debug routine leaves the old nurseries intact (for the ORP to
	// access if it wants), instead to allocate new nursery blocks.
	//
	void _debug_replace_nurseries_intact();
#endif

#if (GC_DEBUG>4)
	//
	// This debug routine leaves the old nurseries intact (for the ORP to
	// access if it wants), instead to allocate new nursery blocks.
	// This version writes a distinct pattern into the old nurseries.
	//
	void _debug_replace_nurseries_cleared();
#endif

#if (GC_DEBUG>4)
	//
	// At debug time, our containing generation is telling us to
	// write protect all the blocks in the constituent nurseries
	// and allocate new blocks for debugging ORP problems.
	//
	void _debug_write_protect_and_replace_nurseries();
#endif

    unsigned long _current_nursery;
	//
	// At nursery creation time this routine takes all the
	// nurseries and links them up into a free list.
	//
	void _link_free_nurseries(unsigned int start_index);

    unsigned long _number_of_nurseries;

    Nursery *_p_nurseries[MAXIMUM_NURSERIES];

#ifdef GC_SAPPHIRE
    CRITICAL_SECTION _NurseryCriticalSection;
#endif // GC_SAPPHIRE
    //
    // Critical Section to mediate concurrent access
    // to the list of spent nurseries.
    //
//    CRITICAL_SECTION _SpentNurseryCriticalSection;
	//
	// Hook to hold unused/recycled nurseries available
	// for threads.
	//
	volatile Nursery *_p_free_nurseries;
	//
	// Hook to hold all the nurseries that are exhausted and
	// await reclamation.
	//
	volatile Nursery *_p_spent_nurseries;
	//
	// Unlink the spent nurseries. This is done after collecting
	// young space, but before an incremental collection of mature space.
	//
	void _unlink_spent_nurseries(Nursery *spent_nurseries); 	//

    void link_onto_free_list (Nursery *first_nursery, Nursery *last_nursery);
    // This list of nurseries is about to be placed somewhere so make
    // sure they are orphans. This means they are not on the free list,
    // the spent list, or associated with any thread. It asserts or return true.
    bool nurseries_are_orphans (Nursery *nurseries);

};


#endif // _nursery_step_gen_H_
