// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/step_generation.cpp,v 1.1.1.1 2001/07/23 07:25:39 xli18 Exp $
//

#include "platform.h"
#include "step_generation.h"
#include "card_table.h"
#include "car.h"
#include "train.h"
#include "remembered_set.h"
#include "gc_hooks.h"
#include "Block_Store.h"
#include "finalize.h"
 
#include "orp_for_gc.h"
 

Step_Generation::Step_Generation(unsigned long gen_num,
                                 Gc_Fast_Hooks *p_gc_hooks,
                                 Gc_Plan       *p_gc_plan,
		                         Gc_Interface  *p_container,
					             Generation    *p_superior,
                                 Card_Table    *p_card_table,
		                         Block_Store   *p_block_store)
    : Generation(gen_num, 
                 p_gc_hooks,
                 p_gc_plan,
                 p_container, 
                 p_superior, 
                 p_card_table,
                 p_block_store)
{
    int step_index = _number_of_steps = 
        _p_gc_plan->number_of_steps();

    while (--step_index >= 0)
	    _add_step(step_index);

#ifndef PER_CAR_BARRIERS
       //
       // This is a repository for incoming pointers from older generations.
       // Entries are detected when jit-ted code does putfields, and updated
       // lazily into this.
       //
       _p_old_to_young_remembered_set = new Remembered_Set();
#endif

       //
       // Create an object set for objects in this generation that
       // will need to be finalized when they become unreachable.
       //
    _p_finalizable_object_list = new Object_List();

	
	//
	// This queue holds all the reference object whose referent (as returned by get)
	// are in this step (or step_nursery) generation. Since all steps in this generation 
	// are collected at the same time we only need one set of queues. 
	// In MOS the queues are assocated with each car.
	// These objects are created lazily since creating them now before initialization 
	// is complete creates new issues.
	//
    _p_weak_object_list = NULL;

    _p_soft_object_list = NULL;

    _p_phantom_object_list = NULL;

#if (GC_DEBUG>0)
    step_generation_creation_hook(this);
#endif

}

//
// This support routine is used during generation initialization to
// add another step to the generation. Currently we don't dynamically
// vary the number of steps at run time.
//
void
Step_Generation::_add_step(unsigned long step_index)
{
	//
	// First determine the higher step that this to-be-constructed
	// step will be promoting objects into. (The "higher" step.)
	// If this is the highest step, the "higher" step will be NULL,
	// and objects will be tenured into a higher generation, instead
	// of being promoted into a higher step.
	//
	Step *p_higher_step = 
		step_index == _number_of_steps - 1
		? NULL : _p_step[step_index + 1];

	//
	// Now create the step.
	//
    _p_step[step_index] = new Step(step_index,
                                   _p_gc_hooks,
                                   _p_gc_plan,
		                           this,
								   p_higher_step,
                                   _p_card_table,
		                           _p_block_store);
}

//
// Every generation needs to define a policy for handling objects
// found during cheney scans.
// 
// This is the one for Step Generations and derivatives.
// Step Generations (and NS Generations) simply send eviction notices
// to all reachable objects.
//
// When a container (Gc_Space or Block_List) does a cheney scan,
// it queries the generation of the FROM space container of the object
// that was moved, for the policy for doing the cheney scan. That is
// how we get to this routine.
//
// In terms of Tri-color algorithms this routine scans a grey object,
// and turns objects it finds to grey using the evict notice.
//

//#ifdef GC_OFFSET_LOGIC

void 
Step_Generation::cheney_scan_execute_policy(Java_java_lang_Object *p_obj, bool doing_mos_collection)
{
    assert (!doing_mos_collection); // No cheney scanning of YOS can happen doing MOS collection (for now)
    // This object had better not be in a nursery, it could be in LOS however.
    assert (!(p_global_bs->is_object_in_nursery(p_obj))); 
    gc_trace (p_obj, "step_generation ln 150 scanning this object."); 
    // If this is an array call cheney_scan_array_execute_policy and return.
 
    Partial_Reveal_Class *p_class = (*((Partial_Reveal_VTable **)p_obj))->clss;
//    Partial_Reveal_Class *p_class = get_object_class(p_obj);
    bool array_p = is_array(p_class);

    if (array_p) {
        cheney_scan_array_execute_policy (p_obj, doing_mos_collection);
        return;
    }

    // DEBUG
    //
	// Get all the descendents of this object.
 
    gc_trace (p_obj, "cheney_scan(ning)_execute_policy ln 171 the object in the step_generation.cpp"); 

    //
	// Cycle through all the descendents.
	//
    unsigned int *offset_scanner = init_object_scanner (p_obj);
    assert(offset_scanner);

    Java_java_lang_Object **pp_child_ref;

	while ((pp_child_ref = p_get_ref(offset_scanner, p_obj)) != NULL) {
        // Move the scanner to the next reference.

        offset_scanner = p_next_ref (offset_scanner);
        if (*pp_child_ref == NULL) {
            continue;
        }

#if (GC_DEBUG>0)
		root_reference_hook(pp_child_ref);
#endif
		//_cheney_scan_reference_hook(pp_child_ref);

#if (GC_DEBUG>2)
		assert(*pp_child_ref != 0);
#endif
        assert (p_global_bs->is_object_in_heap(*pp_child_ref));
        //
		// OK, it is in our GC heap. Now check the generation
        // of the source of this reference.
		// I.e., what is the generation of the object we are scanning?
		//
        assert (is_object_in_my_generation(p_obj)); // For now we are only scanning YOS objects.
		if (is_object_in_my_generation(p_obj)) {
            //
			// We are scanning an object that is in a step or nursery in 
			// our generation.
			//
            // The object could be in LOS. 
            // Turn the object grey.
			_update_a_slot_in_step_or_nursery(pp_child_ref, doing_mos_collection);
		} else {
#ifdef GC_COPY_V2
            assert (0); // Only one generation so it had better be in it.
#endif
            assert (0); // The object has to be in the step generation since
            // we are only scanning objects in the step generation.....
            //
			// We are scanning an object that is in a car in mature space.
			//
            //
            // Why aren't we using the cheney scan policy in MOS, assert and
            // find out...??? For some reason execute_cheney_scan gets
            // passes a generation into it and uses that to figure out the
            // policy of the scan instead of using the policy of the 
            // area the object is located in. WHY I AM NOT SURE. I SUSPECT
            // I'll RIP THAT OUT SOMEDAY AND REINSERT THIS ASSERT. RLH
            // assert (0);
            //
            _update_a_slot_in_car(pp_child_ref, doing_mos_collection);
		}
    } 
    // DEBUG
/// 	delete p_descendents;
    // END DEBUG
}

//
// Use the old ways on arrays for now.
//
void 
Step_Generation::cheney_scan_array_execute_policy(Java_java_lang_Object *p_obj, bool doing_mos_collection)
{
    // This object had better not be in a nursery, it could be in LOS however.
    assert (!(p_global_bs->is_object_in_nursery(p_obj))); 
    gc_trace (p_obj, "Cheney scanning an array, in LOS or in a step."); 
    Partial_Reveal_Class *p_class = (*((Partial_Reveal_VTable **)p_obj))->clss;
    // delete Partial_Reveal_Class *p_class = get_object_class(p_obj);

    // If array is an array of primitives, then there are no references, so return.
    if (is_array_of_primitives(p_class))
        return;

    // Initialize the array scanner which will scan the array from the
    // top to the bottom.

    unsigned int offset = init_array_scanner (p_obj);

	//
	// Cycle through all the descendents.
	//
	Java_java_lang_Object **pp_child_ref;
	while ((pp_child_ref = p_get_array_ref(p_obj, offset)) != NULL) {
        offset = next_array_ref (offset);
        // Skip null entries.
        if (*pp_child_ref == NULL) {
            continue;
        }

#if (GC_DEBUG>0)
		root_reference_hook(pp_child_ref);
#endif
		//_cheney_scan_reference_hook(pp_child_ref);

#if (GC_DEBUG>0)
		assert(*pp_child_ref != 0);
#endif
        assert(*pp_child_ref != 0);
        assert(p_global_bs);

        //
        // First make sure the object pointed to is in our GC heap
        //
        assert (p_global_bs->is_object_in_heap(*pp_child_ref));
		//
		// OK, it is in our GC heap. Now check the generation
        // of the source of this reference.
		// I.e., what is the generation of the object we are scanning?
		//
		if (is_object_in_my_generation(p_obj)) {
		    //
			// We are scanning an object that is in YOS.
			//
            // The object could be in LOS. 
            // Turn the object grey.
			_update_a_slot_in_step_or_nursery(pp_child_ref, doing_mos_collection);
        } else {
#ifdef GC_COPY_V2
            assert (0); // Only one generation so it had better be in it.
#endif
			//
			// We are scanning an object that is in a car in mature space.
			//
            //
            // Why aren't we using the cheney scan policy in MOS, assert and
            // find out...??? For some reason execute_cheney_scan gets
            // passed a generation into it and uses that to figure out the
            // policy of the scan instead of using the policy of the 
            // area the object is located in. WHY I AM NOT SURE. I SUSPECT
            // I'll RIP THAT OUT SOMEDAY AND REINSERT THIS ASSERT. RLH
            // assert (0);
            //
			_update_a_slot_in_car(pp_child_ref, doing_mos_collection);
		}
    } // while end
    return;
}


// #else

//#endif // GC_OFFSET_LOGIC

bool 
Step_Generation::_is_object_in_mature_generation(Java_java_lang_Object *p_obj) 
{
	return _p_mature_generation->is_object_in_my_generation(p_obj);
}

inline bool 
Step_Generation::_is_reference_in_mature_generation(Java_java_lang_Object **pp_obj_ref) 
{
	return _p_mature_generation->is_reference_in_my_generation(pp_obj_ref);
}

// There are several remembered sets that might need updating.
// If the referent is in MOS and the object is in YOS we need
// to update the old to young remembered set.
// RLH-TRAIN
// If the referent is in YOS and the object is in the focus car
// it should have been evicted and an assert will hit.
// If the referent is in YOS and the object is in the focus train (but not the focus car)
// then the focus_train_is_alive bool will need to be set to true.
//
//
// If the referent is in YOS and the object is in MOS then we need
// to update the young to old remembered set.
//
// Finally if both the referent and the object are in MOS we need
// to let the MOS have a chance to update the car to car remembered sets.
//

void
Step_Generation::update_remembered_sets_as_needed (Java_java_lang_Object **pp_obj_ref, 
                                                   bool doing_mos_collection ) 
{

#if (GC_DEBUG>0)
    assert (p_global_bs->is_reference_in_heap (pp_obj_ref));
#endif

    // Now deal with possible remset issues. The only things we remember have 
    // references in MOS.
    if (_is_reference_in_mature_generation(pp_obj_ref)) { // MOS->?
        #ifdef GC_COPY_V2
            assert (0); // There should be no mature generation objects 
                        // in the straight copy version.
        #endif
        // If object is still in YOS update remset..
        if (is_object_in_my_generation(*pp_obj_ref)) { //MOS -> YOS 
            gc_trace (*pp_obj_ref, "A slot in MOS points to this YOS object."); 
            // Object is in YOS and reference is in MOS so we
            // need to update the _p_old_to_young_remembered_set
            // so it is around during the next collection.
            update_old_to_young_write_barrier (pp_obj_ref, *pp_obj_ref);     
        } else {// _is_object_in_my_generation(*pp_obj_ref) // MOS->MOS
 
            gc_trace (*pp_obj_ref, "A slot in MOS points to this MOS object."); 
            // The object got tenured into MOS 
            // and the reference is in MOS so let MOS deal with it.

	        _notify_mature_generation_possible_car_write_barrier_updates(pp_obj_ref);
        } // end if (_is_object_in_my_generation(*pp_obj_ref))
       return;
    } // end if (_is_reference_in_mature_generation(pp_obj_ref))

    // YOS->??  
    // Reference is in YOS, is object in MOS?

    // Short circuit this if we are not doing a MOS collection this time 
    //          around. 

        // RLH-TRAIN - This is needed even though we collect the focus car at 
        // the same time that we collect the nursery. If the pointer is into the
        // focus train but not the focus car we need to note that in order to keep
        // the train alive.
        if (!is_object_in_my_generation(*pp_obj_ref)) { // YOS->MOS
#ifdef GC_TRAIN_V5
//            return; // YOS->MOS but no collection of MOS happening
#endif // GC_TRAIN_V5
            // RLH-TRAIN this is not needed for train logic.
            // The object is in MOS and the reference is in YOS
//            assert (0); // stop and debug.... rlh 1/1/00
            _update_into_mature_generation_ref (pp_obj_ref, doing_mos_collection);
            return;
        }


    // We have returned if we have a pointer or an object in MOS so all
    // that is left is a pointer and and object in YOS.
    // YOS->YOS
    assert (is_object_in_my_generation(*pp_obj_ref));
    assert (is_reference_in_my_generation (pp_obj_ref));
    // WE HAVE AN OBJECT THAT WAS MOVED THAT COULD HOLD POINTERS INTO MOS WHERE DO WE
    // DEAL WITH THAT. This is dealt with by the cheney scan of the MOS spaces.        
}


//
// When supplying cheney scan policy, the object being scanned was discovered
// to be in our generation (i.e. a step or nursery).
//
// Actually I think this is an object discovered by a cheney scan. Not the object
// being scanned. ?????????????????????
//
// pp_child_ref - a pointer to a slot in an object that is being turned from 
// grey to black by a cheney scan.
// 
// If the object pointed to by the slot is white - turn it grey. 
// If it is grey leave it grey.
//
// If the object has been moved update the slot.
//
// CHANGE source_reference to slot someday...
void
Step_Generation::_update_a_slot_in_step_or_nursery(Java_java_lang_Object **pp_child_ref, 
                                                   bool doing_mos_collection)
{

#if (GC_DEBUG>2)
	assert(p_global_bs->is_object_in_heap(*pp_child_ref));
#endif // _DEBUG

	Java_java_lang_Object *p_target_object = *pp_child_ref;
	//
	// I'm scanning a step or a nursery in my generation.
	// Check the generation of the target of this reference.
	// i.e. the generation of the child object.
	//
	if (is_object_in_my_generation(p_target_object)) {
		//
		// Both source and target are in YOS generation.
		//
        // If the object is not white (needs to be forwarded or marked (LOS))
        // do it now and update the slot if need be.
		*pp_child_ref = _p_evict_object_or_update_forwarding(pp_child_ref, doing_mos_collection);
        assert (*pp_child_ref); 
        update_remembered_sets_as_needed(pp_child_ref, doing_mos_collection);
        return;

	} else {
		//
		// The source is in my generation and the target is in
		// the mature generation. No eviction notices are sent to
		// objects in mature space, since we are doing a minor
		// collection (of young space). Instead, we see if recording
		// this young-to-old pointer is necessary for the subsequent
		// incremental collection of mature space.
		//
#ifdef GC_FIXED_V1
        assert (0);
#endif // GC_FIXED_V1
#ifdef GC_COPY_V2
        assert (0); // No mature generations in GC_COPY_V2.
#endif // GC_COPY_V2

        // We are doing a cheney scan of the YOS and we have a YOS slot with
        // a pointer into the MOS and we also need to remember it if
        // we are about to collect the MOS.
        // RLH-TRAIN if the pointer is into the focus car the object will
        // be moved. If it is into the focus train but not the focus car
        // we need to note that so the entire train is not recovered.
 
        _update_into_mature_generation_ref(pp_child_ref, doing_mos_collection);
	}
}

//
// While supplying cheney scan policy, it was discovered that the object
// being scanned is in a car in the mature generation.
//

void
Step_Generation::_update_a_slot_in_car(Java_java_lang_Object **pp_child_ref, bool doing_mos_collection)
{
	Java_java_lang_Object *p_target_object = *pp_child_ref;

	if ((_is_object_in_mature_generation(p_target_object))) {
		//
		// The source and target are in mature space.
		// The mature generation should handle this.
		//
		_notify_mature_generation_possible_car_write_barrier_updates(pp_child_ref);
   	} else { 
        // object is in YOS, Reference is in MOS.
		//
		// The source is in mature space and the target is in mine, a step generation.
		// Evict the object if not already forwarded.
		//
		*pp_child_ref = _p_evict_object_or_update_forwarding(pp_child_ref, doing_mos_collection);
        assert (*pp_child_ref);

		//
		// See if it is still in my the YOS generation after eviction.
		//
		p_target_object = *pp_child_ref;

        if (is_object_in_my_generation(p_target_object)) {
			//
			// Yes, the target is still in my space. 
			// Since the source is in mature space, ensure that 
			// this is recorded in our generations write barrier.
			//
			add_entry_to_generation_write_barriers(pp_child_ref, *pp_child_ref);
		} else {
			//
			// The target just got tenured or was tenured earlier.
			// Now the source and target are both in mature space.
			// The mature generation should take care of this.
			//
//            orp_cout << "Promoted object" << endl;
            // It is possible that this was tenured due to a pointer from the ORP
            // enumeration. We need to let MOS know about this in case this collection
            // is one that requires an MOS collection. Otherwise this reference will
            // be dropped on the floor and the object will be moved without the
            // reference being updated.
			_notify_mature_generation_possible_car_write_barrier_updates(pp_child_ref); 
        }
    }

}

//
// Do any cleanup after a collection.
//
void
Step_Generation::cleanup()
{
	// 
	// Give each step an opportunity to clean up.
	//
	for (unsigned long step_idx = 0; step_idx < _number_of_steps; step_idx++) {
		_p_step[step_idx]->cleanup();
	}

	return;
}

//
// Evict an object if it hasn't already been. - ie the object is white turn it grey
//      by evicting it. If it is in LOS mark it (making it grey) and then scan it (making
//      it black).
// Also update the reference.
//
//
//

// RLH-TRAIN THIS CODE SHOULD WORK FOR OBJECTS IN MOS AS WELL AS YOS. If 

Java_java_lang_Object *
Step_Generation::_p_evict_object_or_update_forwarding(Java_java_lang_Object **pp_obj_ref, bool doing_mos_collection)
{

    assert (*pp_obj_ref); 
	gc_trace ((void *)*pp_obj_ref, "_p_evict_object_or_update_forwarding referent in slot "); 

	Java_java_lang_Object *p_target_object = *pp_obj_ref;
    Java_java_lang_Object *p_old_object = *pp_obj_ref; // Used for debugging.
    assert (p_target_object);

    assert (*pp_obj_ref);

    //
	// Only evict if object hasn't already been.
	//
    // If it is an LOS object see if it has been marked.
    //
	if (is_object_forwarded(p_target_object) || is_object_marked (p_target_object)) {        
        assert (*pp_obj_ref);
		//
		// It has already been evicted and set forwarded or if LOS marked.
		//
		if (p_global_bs->is_object_in_large_object_space(p_target_object)) {
			//
			// Large objects don't move. Therefore no forwarding necessary.
			//
            
            assert (*pp_obj_ref);
		} else {
			//
			// Update the reference to reflect the forwarded location.
			//
			update_reference_forwarded(pp_obj_ref);
            
            assert (*pp_obj_ref);
			p_target_object = *pp_obj_ref;   
            
            assert (*pp_obj_ref);
            assert (p_target_object);
		}
	} else {
		//
		// Object isn't forwarded - evict it.
		//
			Gc_Space *p_container =
				_p_block_store->p_get_object_container(*pp_obj_ref);
            // This could be in LOS.
			p_target_object = p_container->p_evict_object(pp_obj_ref, doing_mos_collection);
            //
            // Since the object is not on LOS and has been evicted p_target_object points
            // to the new object and the new object should not be forwarded.... 
            // 
            if (!(p_global_bs->is_object_in_large_object_space(p_target_object))) {
                assert (!(is_object_forwarded(p_target_object)));
            } else {
                // The object is in LOS and it should not have been moved/
                assert (p_target_object == p_old_object);
                // and it should have been marked
                assert (is_object_marked (p_target_object));
            }

            assert (*pp_obj_ref);
            // RLH-TRAIN I do not update the pointer since it is up to the
            // caller to update the pointer. 
//			update_reference_forwarded(pp_obj_ref);            
            assert (*pp_obj_ref);
            // RLH-TRAIN end of additional code.
            assert (p_target_object);
	}
    
    assert (p_target_object == *pp_obj_ref);
    assert (p_target_object);
	return p_target_object;
}

//
// We have finished a minor collection. Beforp we return we need to
// scan the spent nurseries, from spaces and LOS for unreachable 
// objects that have just become candidates for finalization.
//
void 
Step_Generation::_locate_and_process_finalizable_objects(bool doing_mos_collection)
{
//    orp_cout << "Not finalizing object in step generation. " << endl;
//    return;

    Java_java_lang_Object *p_obj;
    Java_java_lang_Object *p_source_object;
    Java_java_lang_Object *p_target_object;
    Java_java_lang_Object **pp_obj_ref;
    Object_List       *p_object_list = _p_finalizable_object_list;

    //
    // Create a new Object List for storing new state of objects
    //
    _p_finalizable_object_list = new Object_List();

    //
    // Iterate through the finalizable objects in our generation
    //
    p_object_list->rewind();
    while ((p_obj = p_object_list->next()) != NULL) {

#if (GC_DEBUG>3)
        assert(is_java_object(p_obj));
#endif

        Gc_Space *p_container = 
            _p_block_store->p_get_object_container(p_obj);
        if (p_container->is_large_object_space()) {
           if (is_object_unmarked(p_obj)) {
                //
                // Found an unreachable pinned object - temporarily revive it
                // and notify the ORP.
                //
                pp_obj_ref = &p_obj;
                p_source_object = p_obj;
                p_target_object = p_container->p_evict_object(pp_obj_ref, doing_mos_collection);
                orp_finalize_object(p_target_object);
           } else {
                //
                // This object doesn't move:just return it to the list
                //
                _p_finalizable_object_list->add_entry(p_obj);
           } // if (is_object_unmarked(p_obj))
        } else { // not in LOS
            if (!is_object_forwarded(p_obj)) {
                //
                // Found an unreachable object - temporarily revive it
                // and notify the ORP.
                //  
            
                pp_obj_ref = &p_obj;
                p_source_object = p_obj;
                p_target_object = p_container->p_evict_object(pp_obj_ref, doing_mos_collection);
                orp_finalize_object(p_target_object);
 
#if 0 // DEBUG DELETEME DELETE ME JUNK TRASH
                cout << "OOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOO" << endl;
                cout << "Finalizing young unreachable object " << p_source_object << " after moving to ";
                cout << p_target_object;
                cout << " of type " << p_obj->vt->clss->name->bytes << endl;
#endif
            } else {
                //
                // This object is still alive
                //
                //
                // Locate the new resting place:
                //
                Object_Gc_Header *p_gc_hdr = get_object_gc_header(p_obj);
                p_target_object     = 
#ifdef POINTER64
                    (Java_java_lang_Object *)((POINTER_SIZE_INT)*p_gc_hdr & 0xffffffffFFFFFFFE);
#else
                    (Java_java_lang_Object *)((POINTER_SIZE_INT)*p_gc_hdr & 0xFFFFFFFE);
#endif
 
                //
                // Check if it has been tenured
                //
                Generation *p_generation = 
                    p_global_bs->p_get_object_generation(p_target_object);

                if (p_generation==this) {
                    //
                    // Nope: we are still in young space
                    //
                    _p_finalizable_object_list->add_entry(p_target_object);

                } else {
                    //
                    // Yes: tenured: container has to be a car
                    //
                    p_container =
                        _p_block_store->p_get_object_container(p_target_object);

                    assert(p_container->is_car());
                    ((Car *)p_container)->record_finalizable_object(p_target_object);

                } // if (p_generation==this) 
            } // if (!is_object_forwarded(p_obj))
        } // if (p_container->is_large_objec_space())
    } // while ((p_obj = p_object_list->next()) != NULL
    //
    // OK, we have gotten all the candidates for finalization.
    // Now revive all finalizer-reachable objects.
    //
//    orp_cout << "Executing pending cheney scans due to finalizable objects." << endl;
    execute_pending_cheney_scans(false);
    //
    delete p_object_list;
}


#if (GC_DEBUG>3)
//
// Routine to display the contents of the generation.
//
void Step_Generation::inspect(unsigned int level)
{
	cout << "Step Generation: I am " << this << endl;
	for (unsigned long idx = 0; idx < _number_of_steps; idx++) {
		cout << "Inspecting Step #: " << idx << endl;
		_p_step[idx]->inspect(level);
	}
}
#endif // _DEBUG

//			
// We detected that an object got tenured into the mature space.
// Therefore we are notifying the mature generation that the
// inter-car write barriers may need updating.
//
void 
Step_Generation::_notify_mature_generation_possible_car_write_barrier_updates(Java_java_lang_Object **pp_obj_ref) 
{
    assert (_p_mature_generation->is_address_in_my_generation (pp_obj_ref));
	_p_mature_generation->add_entry_to_generation_write_barriers(pp_obj_ref, *pp_obj_ref);
}

//
// We are being called by the GC Interface Object
// to perform a minor collection. A collection of this step (+/-nursery) generation.
// p_orp_live_references ignored slots in the focus car if there is a focus car...
// If if discovers an object in the focus car when doing a cheney scan (or sweeping LOS)
// it evicts the object from the focus car.
//
Remembered_Set *
Step_Generation::
    p_reclaim_generation(Remembered_Set *p_orp_live_references,
                         Remembered_Set *p_weak_references,
						 bool *p_train_is_live)
{
// RLH-TRAIN this routine will collect the focus car as well as YOS.

#if (GC_DEBUG>0)
    minor_collection_start_hook(this);
#endif

#if (GC_DEBUG>3)
    _p_finalizable_object_list->debug_check_list_integrity();
    //_p_finalizable_object_list->debug_dump_list();
#endif

#if (GC_DEBUG>2)
    assert(_p_pending_scans->is_empty());
#endif // _DEBUG
	//
	//
	// This boolean will signal if any pointers into the oldest
	// train were found. This optimization will permit the 
	// incremental collection of mature space to quickly reclaim
	// a train full of garbage.
	//
	_train_is_live  = false;


#if (GC_DEBUG>2)
	assert(_p_young_to_old_rs->is_empty());
#if (GC_DEBUG>3)
#ifndef PER_CAR_BARRIERS
    _p_old_to_young_remembered_set->verify_no_obsolete_references();
#endif
#endif
#endif

#ifndef GC_FIXED_V1 // ndef
	//
	// Give each step an opportunity to set up cheney spaces
	// as a prelude to a cheney copy. This involves renaming the
	// TO spaces to FROM, and creating fresh TO spaces.
	//
	_set_up_cheney_spaces();
#endif

#ifdef GC_FIXED_V1

    //
	// Reclaim all objects reachable from the ORP live references.
	//
	_reclaim_generation(p_orp_live_references);

    // In the fixed Version 1 we can just do the scan and return, since
    // there is only one generation.
	execute_pending_cheney_scans(false);

#if (GC_DEBUG>0)
    minor_collection_end_hook(this);
#endif
	return _p_young_to_old_rs; // This should be empty since this is GC_FIXED_V1

#endif // GC_FIXED_V1

    _slow_tenure_reclaim(p_orp_live_references);

#ifdef GC_TRAIN_TRACE
    orp_cout << "Executing cheney scan due to YOS collection. " << endl;
#endif // GC_TRAIN_TRACE
    
    //
    // Give all steps an opportunity to complete any
    // pending cheney scans. (This will invoke the method of Block_List).
    //
	execute_pending_cheney_scans(false);

	// During the cheney scans, if any pointer into mature space was found
	// to be into the focus train, the routines would have set _train_is_alive
	// to true. 
	//
	if (_train_is_live) {
		*p_train_is_live = true;
	}

#if (GC_DEBUG>2)
	assert(_p_pending_scans->is_empty());
#if (GC_DEBUG>3)
    p_orp_live_references->verify_no_obsolete_references();
#ifndef PER_CAR_BARRIERS
//    orp_cout << "After YOS collection, verifying _p_old_to_young_remembered_set." << endl;
    _p_old_to_young_remembered_set->verify_no_obsolete_references();
#endif
#endif
#endif


#if (GC_DEBUG>3)
    //_p_finalizable_object_list->debug_dump_list();
    _p_finalizable_object_list->debug_check_list_integrity();
#endif


    //
    // Update all weak references to objects that have moved.
    //
	_update_soft_references(false, false);
    _update_weak_references(false);
	_update_phantom_references(false);

    //
    // We have finished a minor collection. Beforp we return we need to
    // scan the spent nurseries, from spaces and LOS for unreachable 
    // objects that have just become candidates for finalization.
    //
    _locate_and_process_finalizable_objects(false);

#if 0 // untested
    //
    // Give everybody a chance to clean up.
    //
    _cleanup();
#endif 

#if (GC_DEBUG>0)
    minor_collection_end_hook(this);
#endif

	//
	// The following remembered set should contain only
	// those references into the focus car.
	//
	return _p_young_to_old_rs;
}

void
Step_Generation::_slow_tenure_reclaim(Remembered_Set *p_orp_live_references)
{
	//
	// Reclaim all objects reachable from the ORP live references.
	//
	_reclaim_generation(p_orp_live_references);

    // RLH_TRAIN - this will need to reclaim the focus car also since
    // we don't want to waste time building these YOS to MOS write barriers.
    //

    return;
}

//
// This support routine is used by p_reclaim generation to cycle
// through a remembered set and send eviction notices to containers
// of objects.
//
void
Step_Generation::_reclaim_generation(Remembered_Set *p_root_set)
{

    assert (p_root_set);

    //
    // Start at the beginning of the root set handed to us.
    //
	p_root_set->rewind();

	Java_java_lang_Object **pp_obj_ref;
	//
	// Cycle through each entry in the supplied root set and scavenge 
	// all reachable objects. The reference can be in the heap
    // or it can be somewhere else, such as the stack.
    //
    // Can we do this without concern for the fact that this is a hash
    // table which complicates the delete logic since you have to reshash.
    // TUNING OPPORTUNITY....
	//

   	while((pp_obj_ref = p_root_set->next()) != NULL) {

#if (GC_DEBUG>1)
		root_reference_hook(pp_obj_ref);
#endif // _DEBUG 
#ifdef GC_FIXED_V1
        assert (p_global_bs->is_object_in_large_object_space(*pp_obj_ref));
#endif
        assert ((*pp_obj_ref == NULL) ||
                (p_global_bs->is_address_in_heap(*pp_obj_ref)));

        gc_trace (*pp_obj_ref, "Root reference to this object being processed by step_generation ln 1041.");
 
		//
		// Process only those entries that point to objects in the GC'd heap.
		//
        if (*pp_obj_ref == NULL) {
            // This referent holds NULL so no objects are moved.
            continue;
        }

        // RLH-TRAIN Check if it is a pointer into an object
        // that is in YOS. If so evict it if it hasn't been forwarded..
        if (is_object_in_my_generation(*pp_obj_ref)) {
    		//
			// This object is in the young generation (mine).
            // It could be in LOS. Marked objects act like forwarded objects.
            // Evict the object if not already forwarded and update pointer.
            //
            // Deal with remset issues below.
            // RLH-TRAIN -- This assignment is redundant since it is done
            //              in _p_evict_object_or_update_forwarding already.
    		*pp_obj_ref = _p_evict_object_or_update_forwarding(pp_obj_ref, false);
            if (p_global_bs->is_reference_in_heap (pp_obj_ref)) {
                update_remembered_sets_as_needed (pp_obj_ref, false);
            }
        } else {
            // The object is not in the YOS (step/nursery/LOS) generation. The slot might have previously
            // held a reference to an object into the step generation so it is possible that
            // we ended up here when the reference was overwritten with a pointer into the MOS.
            // In anycase we are not collecting the MOS so we have nothing else to do.
            // I will note that if we do have a slot with an MOS reference the slot will be in
            // the any appropriate remset already. *WE MIGHT WANT TO ASSERT THIS.*
            //

            assert (_p_mature_generation->is_object_in_my_generation(*pp_obj_ref));

#if (GC_DEBUG>0)

            /// DELETE SOMEDAY IF THE ASSERTS AREN'T HIT RLH JAN 7, 2000

            if (_p_mature_generation->is_object_in_my_generation(*pp_obj_ref)) {
                if (((Mrl_Gc_V1 *)p_gc)->p_get_focus_car() != NULL) {
                    if (((Mrl_Gc_V1 *)p_gc)->p_get_focus_car() == p_global_bs->p_get_object_car(*pp_obj_ref)) {
#ifdef GC_TRAIN_V5
                        // If we are using the train algorithm we only move objects
                        // that are pointed to from outside the train.
                        if (((Mrl_Gc_V1 *)p_gc)->p_get_focus_train() == p_global_bs->p_get_object_train(*pp_obj_ref)) {
//                            orp_cout << "step_generation.cpp ln 1075 slot in focus train with focus car ref ignored." << endl;
                            // The train algorithm moves these later.
                            continue;
                        }
#endif // GC_TRAIN_V5
                        // object is in focus car and should be moved into youngest train
                        // WE have a root pointer to an object in the focus car and we need
                        // to move it to the youngest train. I think we just evict but
                        Java_java_lang_Object *temp = 
                            ((Mrl_Gc_V1 *)p_gc)->p_get_focus_car()->p_evict_if_out_of_train_object(pp_obj_ref);
                        if (temp) {
                            // We are collecting 
                            assert (0);
                            *pp_obj_ref = temp;
                            if (p_global_bs->is_reference_in_heap (pp_obj_ref)) {
                                update_remembered_sets_as_needed (pp_obj_ref, false);
                            }
                        } else {
                            // the slot was in the focus train so the object isn't moved now
                            // since it might be moved to another train if it is referenced outside
                            // the train.....
                        } 
                    }
                }
            } else {
                assert (0); // Object isn't in YOS or MOS so where is it??
            }


            
#endif // (GC_DEBUG>0)
            // This used to be a pointer to an object in this generation 
            // but it got overwritten
            // We will do nothing with this object so no remsets need to updated.

            continue;
        }
        
    }  // end while((pp_obj_ref = p_root_set->next_delete()) != NULL)
//    assert (p_root_set->is_empty());
}



void
Step_Generation::run_all_finalizers() 
{
    Java_java_lang_Object *p_obj;

    _p_finalizable_object_list->rewind();

    while((p_obj = _p_finalizable_object_list->next()) != NULL) {
        orp_finalize_object(p_obj);
    }
}

#ifndef PER_CAR_BARRIERS // only need this for SINGLE_BARRIER
//
// The GC Interface is informing us that a change in an older
// generation (perhaps the reclamation of a car in a train generation)
// could result in some entries in my write barrier remembered set to
// become obsolete. We remove all entries in our write barrier that
// originate in the specified container.
//
void
Step_Generation::selective_flush_write_barrier(Car *p_car)
{
	//
	// WARNING: THIS IS A TEMPORARY, HEAVY-WEIGHT ROUTINE
	// THAT WILL GET REPLACED WITH A BETTER STRATEGY LATER.
	//
	Java_java_lang_Object **pp_obj_ref;

	_p_old_to_young_remembered_set->rewind();
	while ((pp_obj_ref = _p_old_to_young_remembered_set->next()) != NULL) {
		//
		// Locate the container associated with the source of this
		// remembered set entry. (The origin of the reference).
		//
		Gc_Space *p_container =
			_p_block_store->p_get_reference_container(pp_obj_ref);
		//
		// See if this matches the car that is being deleted.
		//
		if (p_container == p_car) {
			//
			// We got a match - delete it.
			//
			purge_write_barrier_remembered_set(pp_obj_ref);
		}
	}
#if (GC_DEBUG>3)
    _p_old_to_young_remembered_set->verify_no_obsolete_references();
#endif // GC_DEBUG>3
}
#endif // PER_CAR_BARRIERS

//
// This routine is called as the first order of business by the routine
// (p_reclaim_generation) that reclaims this generation. 
//
void 
Step_Generation::_set_up_cheney_spaces()
{
	// 
	// Give each step an opportunity to set up the cheney spaces.
	// They will rename all TO spaces to FROM, and create new TO spaces.
	//
	for (unsigned long step_idx = 0; step_idx < _number_of_steps; step_idx++) {
		_p_step[step_idx]->set_up_cheney_spaces();
	}
}

//
// Scan cheney spaces determines if any of the steps have objects in their
// 'to' spaces that need to be scanned. If so it scans them and returns true
// to indicate that more work might need to be done.
// Otherwise it returns false.
// If all the generations return false then there is no more cheney 
// scanning to be done.
//
bool
Step_Generation::scan_cheney_generation (bool doing_mos_collection)
{
    // Scan each of the steps until no step has any work to do.
    // Scanning MOS might lead to more work and if so we this 
    // will be called again.
    bool result = false;
    bool more = true;
    while (more) {
        more = false;
        for (unsigned long step_idx = 0; step_idx < _number_of_steps; step_idx++) {
            if (_p_step[step_idx]->cheney_scan_block_list(doing_mos_collection)) {
                result = true;
                more = true;
            }
	    }
    }
    assert (!(cheney_scan_pending()));  
    return result;
}

bool
Step_Generation::cheney_scan_pending ()
{
    bool result = false;
    for (unsigned long step_idx = 0; step_idx < _number_of_steps; step_idx++) {
        if (_p_step[step_idx]->cheney_scan_pending()) {
            result = true;
        }
    }
    return result;
}
// #endif // END of NEW CHENEY CODE.

//
// During a reclamation of an object that is live, perhaps during
// the top-level reclaim or the cheney scan of a contained space, 
// a pointer out of this generation into mature space was found.
// Let's try to be clever and capture only those pointers from young-to-
// old that are relevant to a subsequent incremental scan of mature space.
//
void 
Step_Generation::_update_into_mature_generation_ref(Java_java_lang_Object **pp_obj_ref, 
                                                    bool doing_mos_collection)
{
	//
	// Be clever and record this reference only if it is into the 
	// focus car/train.
    //
	// RLH-TRAIN we only need to note the existance of the pointer. If it is
    // to the focus car it will be moved out of the focus train (and car).
    // If it is to the focus train but not the focus car then the train
    // must be kept alive. We need to note this fact here...
	if (doing_mos_collection) {
		if (((Mrl_Gc_V1 *)p_gc)->p_get_focus_car()->is_my_object(*pp_obj_ref)) {

            // We have a slot with a pointer into the focus car, evict the object.
            *pp_obj_ref = ((Mrl_Gc_V1 *)p_gc)->p_get_focus_car()->p_evict_out_of_mos_ref(pp_obj_ref);

			//
			// Also note that this focus train is alive.
			//
			_train_is_live = true;

		} else if (((Mrl_Gc_V1 *)p_gc)->p_get_focus_train()->is_my_object(*pp_obj_ref)){
#ifndef GC_TRAIN_V5
#ifdef GC_GEN_V3
            assert (0); // For GC_GEN_V3 there is only one car in the train so we
                        // should not be here.
#endif
#endif // GC_TRAIN_V5
			//
			// Record a boolean if it is a reference into the oldest (focus) train.
			//
			_train_is_live = true;
		}
    } else { 
        // Just doing a YOS collection, so remember ref it is to the focus car.
        if (((Mrl_Gc_V1 *)p_gc)->p_get_focus_car() != NULL) {
            if (((Mrl_Gc_V1 *)p_gc)->p_get_focus_car()->is_my_object(*pp_obj_ref)) {
                add_young_to_old_ref(pp_obj_ref);
    			_train_is_live = true;
            }
        }
    }
}

void Step_Generation::enumerate_reference_queues ()
{
    if (_p_weak_object_list) {
        gc_add_root_set_entry((Java_java_lang_Object **)&_p_weak_object_list);
    }
    
    if (_p_soft_object_list) {
        gc_add_root_set_entry((Java_java_lang_Object **)&_p_soft_object_list);
    }
    
    if (_p_phantom_object_list) {
        gc_add_root_set_entry((Java_java_lang_Object **)&_p_phantom_object_list);
    }
#ifdef GC_TRACE_WEAK_REF
    orp_cout << "Step generation just enumerated the weak/soft/phantom reference queues" << endl;
#endif // GC_TRACE_WEAK_REF
}
// RACE CONDITION USE COMPARE_EXCHANGE
void Step_Generation::enqueue_soft_ref (java_lang_ref_Reference *soft_reference)
{
//    soft_reference->next = _p_soft_object_list;
    gc_heap_slot_write_ref ((Java_java_lang_Object *)soft_reference,
        (Java_java_lang_Object **)&(soft_reference->next),
        (Java_java_lang_Object *)_p_soft_object_list);
	_p_soft_object_list = soft_reference; 
}

void Step_Generation::enqueue_weak_ref (java_lang_ref_Reference *weak_reference)
{
//    weak_reference->next = _p_weak_object_list;
    gc_heap_slot_write_ref ((Java_java_lang_Object *)weak_reference,
        (Java_java_lang_Object **)&(weak_reference->next),
        (Java_java_lang_Object *)_p_weak_object_list);
	_p_weak_object_list = weak_reference; 
}

void Step_Generation::enqueue_phantom_ref (java_lang_ref_Reference *phantom_reference)
{ 
//    phantom_reference->next = _p_phantom_object_list;
    gc_heap_slot_write_ref ((Java_java_lang_Object *)phantom_reference,
        (Java_java_lang_Object **)&(phantom_reference->next),
        (Java_java_lang_Object *)_p_phantom_object_list);
	_p_phantom_object_list = phantom_reference;
}

//
// If a queue has been specified we need to move the reference into the queue.
// Similar code is in car.cpp, if there is a bug here there probable is one there.
//
void process_weak_reference (Java_java_lang_ref_Reference *p_ref_Reference)
{
    // If there is a queue enqueue the reference object.
    if (p_ref_Reference->queue) {
        // If the object has not already been enqueud do, if it has
        // been enqued don't enqueue it again.
        if (p_ref_Reference->enqueued == false) {
            java_lang_ref_ReferenceQueue *the_queue = p_ref_Reference->queue;
            if (the_queue->which == 1){
                p_ref_Reference->next = the_queue->head2;
                the_queue->head2 = p_ref_Reference;
            } else {                
                assert ((the_queue->which == 2) || (the_queue->which == 0));
                p_ref_Reference->next = the_queue->head1;
                the_queue->head1 = p_ref_Reference;
            }
            orp_notify_reference_queue((Java_java_lang_Object *)the_queue);
            p_ref_Reference->enqueued = true;
        }
    }
    p_ref_Reference->referent = NULL; // Clear the referent.
#ifdef GC_TRACE_WEAK_REF
    orp_cout << "Weak Reference has to been processed." << endl;
#endif // GC_TRACE_WEAK_REF
    return;
}

void process_soft_reference (Java_java_lang_ref_Reference *p_ref_Reference)
{
    // If there is a queue, enqueue the reference object.
    if (p_ref_Reference->queue) {
        // If the object has not already been enqueued, do it, if it has
        // been enqueued don't enqueue it again.
        if (p_ref_Reference->enqueued == false) {
            java_lang_ref_ReferenceQueue *the_queue = p_ref_Reference->queue;
            if (the_queue->which == 1){
                p_ref_Reference->next = the_queue->head2;
                the_queue->head2 = p_ref_Reference;
            } else {                
                assert ((the_queue->which == 2) || (the_queue->which == 0));
                p_ref_Reference->next = the_queue->head1;
                the_queue->head1 = p_ref_Reference;
            }
            orp_notify_reference_queue((Java_java_lang_Object *)the_queue);
            p_ref_Reference->enqueued = true;
        }
    }
    p_ref_Reference->referent = NULL; // Clear the referent.
#ifdef GC_TRACE_WEAK_REF
	orp_cout << "Soft Reference has been processed." << endl;
#endif // GC_TRACE_WEAK_REF
    return;
}

void process_phantom_reference (Java_java_lang_ref_Reference *p_ref_Reference)
{
    // If there is a queue, enqueue the reference object.
    if (p_ref_Reference->queue) {
        // If the object has not already been enqueued do it, if it has
        // been enqueued don't enqueue it again.
        if (p_ref_Reference->enqueued == false) {
            java_lang_ref_ReferenceQueue *the_queue = p_ref_Reference->queue;
            if (the_queue->which == 1){
                p_ref_Reference->next = the_queue->head2;
                the_queue->head2 = p_ref_Reference;
            } else {                
                assert ((the_queue->which == 2) || (the_queue->which == 0));
                p_ref_Reference->next = the_queue->head1;
                the_queue->head1 = p_ref_Reference;
            }
            orp_notify_reference_queue((Java_java_lang_Object *)the_queue);
            p_ref_Reference->enqueued = true;
        }
    } 
#ifdef GC_TRACE_WEAK_REF
    // The referent is not cleared in the case of a phantom reference.
	orp_cout << "Phantom Reference has been processed." << endl;
#endif // GC_TRACE_WEAK_REF
    return;
}


// This is executed for all steps and cars at each collection.
//
// We have a list of java.lang.ref.referenece objects. They each have 
// a hidden field pointing to the referent. This field is only accessable 
// via the "get" method. If the referent was in a region that was being
// collected then either it was moved indicating that it is still alive
// or it was not moved indicating that the only references to it are 
// weak references. 
//
// If it was moved (and is alive) then the java.lang.ref.reference is 
//    dequeued from the queue associated with the old location of the object
//    and queued with the new location.
// If it was not moved (is unreachable) then the java.lang.ref.reference 
//    object will be places on the java.lang.ref.ReferenceQueue if on was associated 
//    with it when it was created.
//
//
void
Step_Generation::_update_weak_references(bool doing_mos_collection)
{
	if (_p_weak_object_list) {
		java_lang_ref_Reference *p_ref_Reference = (java_lang_ref_Reference *)_p_weak_object_list;
		java_lang_ref_Reference *p_next = NULL;
		_p_weak_object_list = NULL; // This is about to be rebuilt.

	    while (p_ref_Reference != NULL) {
			p_next = p_ref_Reference->next;
		    Java_java_lang_Object *p_obj = p_ref_Reference->referent;
 
		    if (p_obj) {			
				Gc_Space *p_gc_space = p_global_bs->p_get_object_container(p_obj);
				if (p_gc_space->is_large_object_space()) {
		            if (is_object_unmarked(p_obj)) {
			            //
				        // An unmarked object in LOS is toast at the end of the
					    // gc cycle. 
						//
						// The object is only reachable via this pointer.
						// enqueue the object if that has been requested. 
						process_weak_reference (p_ref_Reference);
					}
					else {
							//LOS objects that are marked are put back on the _p_weak_ref_object_list;
						gc_register_weak_ref ((Partial_Reveal_JavaObject *)p_ref_Reference, (Partial_Reveal_JavaObject *)(get_forwarded_object(p_obj)));
					}
	            } else {	
                   
 					assert (this == p_global_bs->p_get_object_generation(p_obj));
					if (is_object_forwarded(p_obj)) { 
						// gc_register_weak_ref will always add to the front of the list
						// so we won't have to reprocess objects that have been removed and
						// then reregistered with the step_generation.
						gc_register_weak_ref ((Partial_Reveal_JavaObject *)p_ref_Reference, (Partial_Reveal_JavaObject *)(get_forwarded_object(p_obj)));
					} else { 
						//
                        // The object is only reachable via this this pointer.
						// enqueue the object if that has been requested.
						//
						process_weak_reference (p_ref_Reference); 
                 	}
				}
	        }  else {
				// We have a reference with the referent cleared. We can ignore it.
			} 
			p_ref_Reference = p_next;
		}
	}
    // Cheney scan since we might have promoted a reference object.
    Generation::execute_pending_cheney_scans(false);
} 

void
Step_Generation::_update_phantom_references(bool doing_mos_collection)
{
	// loop through all the objects on the list. All the referents have been moved
	// to new cars if they are strongly reachable (still alive). If they aren't 
	// alive then the referents are enqueued. My reading of the spec is that we
	// don't clear the referent even though we can not retrieve the referent. I assume this
	// is done so that the objects that the referent's referents are kept
	// alive so that they can be used.

	if (_p_phantom_object_list) {
		java_lang_ref_Reference *p_ref_Reference = (java_lang_ref_Reference *)_p_phantom_object_list;
		java_lang_ref_Reference *p_next = NULL;

		// Clear _p_phantom_object_list since it will be rebuilt as needed.
		_p_phantom_object_list = NULL;

	    while (p_ref_Reference != NULL) {
		    Java_java_lang_Object *p_obj;
			p_obj = p_ref_Reference->referent;
		    if (p_obj) {			
				Gc_Space *p_gc_space = p_global_bs->p_get_object_container(p_obj);
				if (p_gc_space->is_large_object_space()) {
		            if (is_object_unmarked(p_obj)) {
			            //
				        // An unmarked object in LOS is toast at the end of the
					    // gc cycle. 
						//
						// The object is only reachable via this pointer.
						// enqueue the object if that has been requested. 
						process_phantom_reference (p_ref_Reference);
						// Keep the object alive.

						// Evict the referent, treating the object as if it 
						// was reachable from a root.
						Java_java_lang_Object *temp =
							_p_evict_object_or_update_forwarding(&p_ref_Reference->referent, doing_mos_collection); 

						// gc_register_soft_ref will always add to the front of the list
						// of some other car. These pointers are scanned, not remembered.
						gc_register_phantom_ref ((Partial_Reveal_JavaObject *)p_ref_Reference, (Partial_Reveal_JavaObject *)(p_obj));
					} else { // LOS objects that are marked (alive) are re registered
						gc_register_phantom_ref ((Partial_Reveal_JavaObject *)p_ref_Reference, (Partial_Reveal_JavaObject *)(p_obj));
					}
				} else { // It is not a large object
					if (is_object_forwarded(p_obj)) { // It been moved?
						p_next = p_ref_Reference->next;
						p_ref_Reference->next = NULL;
						// gc_register_weak_ref will always add to the front of the list
						// of some other car.
						gc_register_phantom_ref ((Partial_Reveal_JavaObject *)p_ref_Reference, (Partial_Reveal_JavaObject *)(get_forwarded_object(p_obj)));
					} else { // Object has died. Save it.
	    				// Assert that p_obj is in this generation
						assert (this == p_global_bs->p_get_object_generation(p_obj)); 
						p_next = p_ref_Reference->next;
						p_ref_Reference->next = NULL;
						// Evict the referent, treating the object as if it 
						// was reachable from a root.
						Java_java_lang_Object *temp =
							_p_evict_object_or_update_forwarding(&p_ref_Reference->referent, doing_mos_collection); 
#if (GC_DEBUG>0)
						assert (temp == p_ref_Reference->referent);
#endif
						//
						// The object is only reachable via this this pointer.
						// enqueue the object if that has been requested.
						//
						process_phantom_reference (p_ref_Reference);

#ifdef GC_TRACE_WEAK_REF
						orp_cout << "Moving phantom object." << endl;
#endif // GC_TRACE_WEAK_REF
					}
               	}
	        } else {
				// How can we have a ref.Reference with out a referent?
				// The application can clear the referent leaving us with a null referent.
				// In this case the ref.Reference object can be removed from the list.
				p_next = p_ref_Reference->next;
				p_ref_Reference->next = NULL; 
			}
			p_ref_Reference = p_next;
		}
	}
    // Cheney scan the spaces that might hold promoted objects.
	
	Generation::execute_pending_cheney_scans(false);
} 

void
Step_Generation::_update_soft_references(bool low_on_space, bool doing_mos_collection)
{
	// loop through all the objects on the list. All the referents have been moved
	// if they are strongly reachable (still alive). If they aren't 
	// alive and update_soft_reference we are low on space then the referents are 
	// cleared and enqueued.
	// If we are not low on space then the object is copied to new space
	// and kept alive.

	if (_p_soft_object_list) {
		java_lang_ref_Reference *p_ref_Reference = (java_lang_ref_Reference *)_p_soft_object_list;
		java_lang_ref_Reference *p_next = NULL;

		// Clear _p_soft_object_list since it will be rebuilt as needed.
		_p_soft_object_list = NULL;

	    while (p_ref_Reference != NULL) {
		    Java_java_lang_Object *p_obj;
			p_obj = p_ref_Reference->referent;
		    if (p_obj) {
				if (is_object_forwarded(p_obj)) { // It has been moved.
					p_next = p_ref_Reference->next;
					p_ref_Reference->next = NULL;
					// gc_register_soft_ref will always add to the front of the list
					// of some other car.
					gc_register_soft_ref ((Partial_Reveal_JavaObject *)p_ref_Reference, (Partial_Reveal_JavaObject *)(get_forwarded_object(p_obj)));
				} else { // Object has died. Kill it if we are low_on_space
	    			// Assert that p_obj is in this generation

				assert (this == p_global_bs->p_get_object_generation(p_obj)); 
					if (low_on_space) {
						p_next = p_ref_Reference->next;
						p_ref_Reference->next = NULL;
						//
				        // The object is only reachable via this this pointer.
						// enqueue the object if that has been requested.
						//
						process_soft_reference (p_ref_Reference);
					} else {
						p_next = p_ref_Reference->next;
						p_ref_Reference->next = NULL;
						// Move the referent to another car, treating the object as if it 
						// was reachable from a root.				
						Java_java_lang_Object *temp =
							_p_evict_object_or_update_forwarding(&p_ref_Reference->referent, doing_mos_collection); 
#if (GC_DEBUG>0)
						assert (temp == p_ref_Reference->referent);
#endif

#ifdef GC_TRACE_WEAK_REF
						orp_cout << "Moving unreachable soft object." << endl;
#endif // GC_TRACE_WEAK_REF
					}
               	}
	        } else {
				// How can we have a ref.Reference with out a referent?
				// The application can clear the referent leaving us with a null referent.
				// In this case the ref.Reference object can be removed from the list.
				p_next = p_ref_Reference->next;
				p_ref_Reference->next = NULL; 
			}

			p_ref_Reference = p_next;
		}
	}
    // Cheney scan the spaces that might hold promoted objects.
	
	Generation::execute_pending_cheney_scans(false);
} 

#if (GC_DEBUG>3)
//
// Routine to verify the consistency of all the spaces of this generation.
//
bool Step_Generation::verify_generation()
{
	for (unsigned long idx = 0; idx < _number_of_steps; idx++) {
		if (_p_step[idx]->verify_space()) {
			continue;
		} else {
			return false;
		}
	}
	return true;
}
#endif // _DEBUG

// end file step_generation.cpp
