// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc_v2/verify_heap.cpp,v 1.2 2001/12/11 16:08:04 rlhudson Exp $
//

#include "gc_for_orp.h"
#include "finalize.h"

#include "mrl_gc_v1.h"
#include "nursery_step_gen.h"
#include "train_generation.h"

#include "los.h"
#include "gc_hooks.h"
#include "gc_consts.h"
#include "gc_plan.h"
#include "gc_globals.h"
#include "gc_debug.h"

#ifdef ORP_POSIX
#include "linux_api.h"
#endif

#ifdef ORP_NT
#include "win32_api.h"
#endif

// verify_heap
//   These routines are responsible for scanning the heap and determining if everything is
//   internally consistent. If the system gives the GC an invalid heap then this is the routine that
//   will discover this. If the GC makes the heap invalid then this is the routine that will discover that.
//   In theory if the GC gets a valid heap, does a GC then the heap should remain valid.
//   
//   The only entry point is void verify_heap(Root_List *root_set)
//

//
// Verify slot
//
void verify_slot (Java_java_lang_Object **p_slot)
{
    if (*p_slot == NULL) {
        return; // a null slot is cool
    }
    block_info *target_block_info = GC_BLOCK_INFO(*p_slot);
    Java_java_lang_Object *the_obj = *p_slot;
    
    // verify the block is in the heap and not on the free list and 
    // that the target object appears valid.
    assert (the_obj->vt);
    // The class is in the pinned single block large object space
    assert (GC_BLOCK_INFO(the_obj->vt->clss)->in_los_p);
    assert (GC_BLOCK_INFO(the_obj->vt->clss)->number_of_blocks == 1);
    if (!target_block_info->in_los_p) {
        assert (target_block_info->free > the_obj);
        if (!target_block_info->in_nursery_p) {
            assert (target_block_info->scan == target_block_info->free);
        }
    }
}
void verify_card_mark (Java_java_lang_Object *source, Java_java_lang_Object *target) 
{
    if (target == NULL) { // not interesting.
        return;
    }
    
    boolean mark = false;
    if (GC_BLOCK_INFO(source)->train_birthday > 0) {
        // source is in MOS.
        if (GC_BLOCK_INFO(source)->train_birthday > GC_BLOCK_INFO(target)->train_birthday) {
            // source is in younger train.
            mark = true;
        }
        if (GC_BLOCK_INFO(source)->train_birthday == GC_BLOCK_INFO(target)->train_birthday) {
            if (GC_BLOCK_INFO(source)->car_birthday > GC_BLOCK_INFO(target)->car_birthday) {
                // source is in same train but younger car.
                mark = true;
            }
        }
    }
    if (mark) {
        // Make conservative check, only concern is that it is set when it needs to be.
        assert (GC_BLOCK_INFO(source)->card_table[GC_CARD_INDEX(source)]);
    }
}

//
// Verify objects (including arrays).
//
void verify_object (Java_java_lang_Object *p_object)
{
    gc_trace (p_object, "Verifying this object");
    
    assert (p_object->vt); 
    Class *p_class = p_object->vt->clss;  // get_object_class(p_object);
    unsigned int *offset_scanner; 
    Java_java_lang_Object **pp_target_object;
    // Loop through slots in array or objects verify what is in the slot
    if (is_array(p_object)) {
        if (is_array_of_primitives(p_object)) {
            return;
        }
        unsigned int array_offset = init_array_scanner(p_object);
        while ((pp_target_object = p_get_array_ref(p_object, array_offset)) != NULL) {
            array_offset = next_array_ref (array_offset);
            verify_slot (pp_target_object);
            verify_card_mark (p_object, *pp_target_object);
        } // end while
        return;
    } // end while for arrays
    // It isn't an array, it is an object.
    offset_scanner = init_object_scanner (p_object);
    while ((pp_target_object = p_get_ref(offset_scanner, p_object)) != NULL) {
        // Move the scanner to the next reference.
        offset_scanner = p_next_ref (offset_scanner);
        verify_slot (pp_target_object);
        verify_card_mark (p_object, *pp_target_object);
    } // end while for objects
    // We are done.
}

//
// Verify that a block that has allocated objects linearly has only valid pointers.
//
void verify_simple_block (block_info *the_block)
{
    Object_Gc_Header *the_obj_start = (Object_Gc_Header *)GC_BLOCK_ALLOC_START(the_block);
    Object_Gc_Header *previous_obj_start = NULL;
    while (the_obj_start < the_block->free) {
        previous_obj_start = the_obj_start; // For debugging it is nice to have the previous object available.
        unsigned int size = get_real_object_size_bytes(P_OBJ_FROM_START(the_obj_start));
        verify_object (P_OBJ_FROM_START(the_obj_start));
        the_obj_start = (Object_Gc_Header *)((char *)the_obj_start + size);
    }
}

void verify_nurseries ()
{
    // First scan the nurseries in use.
    // All nurseries live int the nurseries array from index 0 up to allocated_nurseries - 1
    int i;
    for (i = 0; i < allocated_nurseries; i++) {
        assert (nurseries[i] != NULL);
        if ((nurseries[i]->nursery_status == spent_nursery) || (nurseries[i]->nursery_status == active_nursery)) {
            verify_simple_block(nurseries[i]);
            assert (nurseries[i]->train_birthday == 0);
            assert (nurseries[i]->car_birthday == 0);
        }
    }
    
    for (i = allocated_nurseries; i < MAX_NURSERIES; i++) {
        assert (nurseries[i] == NULL);
    }
    
    assert (free_nursery_hint < allocated_nurseries);
}

void verify_step ()
{
    
    // Now verify the step
    block_info *a_block = step->blocks;
    block_info *this_block;
    while (a_block) {
        this_block = a_block;
        verify_simple_block(this_block);
        assert (this_block->list_info == step);
        assert (this_block->in_step_p);
        assert (!this_block->in_nursery_p);
        assert (this_block->train_birthday == 0);
        assert (this_block->car_birthday == 0);
        a_block = a_block->next;
    }
    assert (this_block == step->alloc_block);
    assert (step->scan_block == step->alloc_block);
    assert (step->my_train == NULL);
}

void verify_all_marks_are_clear(block_info *this_block)
{
    // Make sure all the marks in this block are cleared
    int j;
    for (j = 0; j <= GC_LOS_MAX_MARK_INDEX; j++) {
        assert (!this_block->mark_table[j]);
    }
}	

void verify_single_object_blocks()
{
    block_info *a_single_object_blocks = single_object_blocks;
    block_info *this_single_object_blocks;
    while (a_single_object_blocks) {
        this_single_object_blocks = a_single_object_blocks;
        a_single_object_blocks = a_single_object_blocks->next;
        verify_all_marks_are_clear(this_single_object_blocks);
        verify_object (P_OBJ_FROM_START(GC_BLOCK_ALLOC_START(this_single_object_blocks)));        
    }
}

void verify_los ()
{
    int i;
    for (i = 0; i < GC_LOS_BUCKETS; i++) {
        block_info *this_bucket = los_buckets[i];
        block_info *a_block = this_bucket;
        block_info *this_block;
        while (a_block) {
            
            this_block = a_block;
            verify_all_marks_are_clear(this_block);
            // Check out the block info
            assert (this_block->c_area_p); // LOS blocks always have this set.
            assert (this_block->in_los_p);
            assert (this_block->train_birthday == 0);
            assert (this_block->car_birthday == 0);
            assert (this_block->number_of_blocks == 1);
            assert (this_block->los_object_size < GC_BLOCK_ALLOC_SIZE);
            assert (this_block->los_object_size >= 40); // Just to keep us honest. GC_FIXED_V1 might need to change this.            
            Object_Gc_Header *first_obj_header = (Object_Gc_Header *)GC_BLOCK_ALLOC_START(this_block);
            // There may be some space at the end that is unused, this is the last possible place to start a valid object.
            Object_Gc_Header *last_possible_obj_header = 
                (Object_Gc_Header *)((GC_BLOCK_SIZE_BYTES + (POINTER_SIZE_INT)this_block) - this_block->los_object_size);
            Object_Gc_Header *an_obj_header = first_obj_header;
            int objects_in_block = GC_BLOCK_ALLOC_SIZE / this_block->los_object_size; 
            // Create a map of the free objects.
            MARK *free_map = (MARK *)malloc (objects_in_block * sizeof(MARK));
            for (i = 0; i < objects_in_block; i++) {
                free_map[i] = false;
            }
            
            los_free_link *a_free = (los_free_link *)this_block->free;
            los_free_link *this_free;
            int count = 0;
            while (a_free) {
                this_free = a_free;
                assert ((POINTER_SIZE_INT)this_free >= (POINTER_SIZE_INT)first_obj_header);
                assert ((POINTER_SIZE_INT)this_free <= (POINTER_SIZE_INT)last_possible_obj_header);
                int free_map_index = (POINTER_SIZE_INT)this_free - (POINTER_SIZE_INT)first_obj_header;
                free_map_index = free_map_index / this_block->los_object_size;
                free_map[free_map_index] = true;
                count++;
                a_free = ((los_free_link *)a_free)->next;
            }
            if (stats_gc) {
                orp_cout << "Free count is " << count << " object size it " << ((int)this_block->los_object_size) << endl;
            }
            int used_count = 0;
            int index_check = 0;
            while ((POINTER_SIZE_INT)an_obj_header <= (POINTER_SIZE_INT)last_possible_obj_header) {
                int free_map_index = (POINTER_SIZE_INT)an_obj_header - (POINTER_SIZE_INT)first_obj_header;
                free_map_index = free_map_index / this_block->los_object_size;
                assert (free_map_index == index_check);
                index_check++;
                if (!free_map[free_map_index]) {
                    assert (GC_BLOCK_INFO(an_obj_header) == this_block);
                    verify_object(P_OBJ_FROM_START(an_obj_header));
                    used_count++;
                    count++;
                }
                an_obj_header = (Object_Gc_Header *)( (POINTER_SIZE_INT)an_obj_header + (POINTER_SIZE_INT)(this_block->los_object_size));
            }
            if (stats_gc) {
                orp_cout << "used_count is " << used_count << endl;
            }
            assert ((count * this_block->los_object_size) <= GC_BLOCK_ALLOC_SIZE);
            assert (((count + 1) * this_block->los_object_size) > GC_BLOCK_ALLOC_SIZE);
            free(free_map);
            a_block = a_block->next;
        } // while (a_block)
    }
    // Now verify the single object multi-block objects.
    
    verify_single_object_blocks();
    
}

void verify_mos()
{
    train_info *this_train;
    train_info *train_list = trains;
    car_info *this_car;
    car_info *car_list;
    block_info *block_list;
    block_info *this_block;
    int train_count = 0;
    while (train_list) {
        this_train = train_list;
        train_count++;
        int car_count = 0;
        car_list = this_train->cars;
        unsigned int car_birthday = car_list->birthday; // Cars could have been deleted so birthday != 0.
        while (car_list) {
            this_car = car_list;
            car_count++;
            assert (car_birthday == this_car->birthday);
            block_list = this_car->blocks;
            int block_count = 0;
            while (block_list) {
                block_count++;
                this_block = block_list;
                verify_simple_block(this_block);
                assert (this_train->birthday == this_block->train_birthday);
                assert (this_car->birthday == this_block->car_birthday);
                block_list = block_list->next;
            }
            
            if (verbose_gc) {
                orp_cout << "Car " << (int)this_car->birthday << " in train " << (int)this_train->birthday << " has " << block_count << " blocks." << endl;
            }
            car_birthday++;
            car_list = car_list->next;
        }
        if (verbose_gc) {
            orp_cout << "Car count for train " << ((int)this_train->birthday) << " is " << ((int)car_count) << endl;
        }
        train_list = train_list->next;
    }
    if (verbose_gc) {
        orp_cout << "Train count is " << train_count << endl;
    }
}



void verify_roots (Root_List *root_set)
{        
    int temp_root_count = 0;
    if (root_set != NULL ) {
        p_verify_root_set->rewind();
        for (Java_java_lang_Object **pp_slot = p_verify_root_set->pp_next(); pp_slot; pp_slot = p_verify_root_set->pp_next()) {
            temp_root_count++;
            verify_object (*pp_slot);
        }
    }
}

void
verify_all_blocks_list()
{    
    block_group_link *the_group_list = p_global_bs->block_group_list;
    int free_count = 0;
    int used_count = 0;
    while (the_group_list) {
        block_info *the_blocks = the_group_list->block_list;
        while (the_blocks) {
            if (the_blocks->in_free_p) {
                free_count = free_count + the_blocks->number_of_blocks;
            } else {
                used_count = used_count + the_blocks->number_of_blocks;
            }
            the_blocks = the_blocks->all_blocks_next;
        }
        the_group_list = the_group_list->next;
    }
}


//
// This can be called at any point and it will ask the VM for roots and it will then
// walk the heap verifying that the is valid in every respect that it can. This is done
// by linearly scanning each block currently in the heap making sure that it points to 
// a legal object. It also scans the root slot and make sure that they point to legal objects.
//
int verify_stop = 0x0;
int verify_count = 0;
void verify_heap(Root_List *root_set)
{
    
#ifndef _DEBUG
    return;
#endif
    verify_count++;
    if (verify_count == verify_stop) {
        // To help with debugging you can break here...
    }
    
    verify_all_blocks_list();
    
    verify_nurseries();
    
    verify_step();
    // Now scan the multiple object LOS blocks as well as the single object LOS blocks
    
    verify_los();
    
    // Now scan the blocks in the cars in the train
    // Check to make sure that if there are card marks for all appropriate objects 
    verify_mos();
    // Now scan the roots to make sure they all point to legal objects.
    
    verify_roots (root_set);
    // Scan the free blocks making sure they are free.
    if (stats_gc) {
        orp_cout << "Heap verified" << endl;
    }
}
