#include <stdio.h>
#include <string.h>
#include <jvmti.h>
#include <config.h>

#include <tijmp.h>
#include <vm_init.h>
#include <vm_death.h>
#include <gc.h>

jvmtiEnv* jvmti;

void handle_global_error (jvmtiError err) {
    char* err_name;
    (*jvmti)->GetErrorName(jvmti, err, &err_name);
    fprintf (stderr, "global_error: %d: %s\n", err, err_name);
    (*jvmti)->Deallocate(jvmti, (unsigned char*)err_name);
}

static jint is_path_separator (char* c) {
    return *c == ':' || *c == ';';
}

/** Try to find the given file from a base path, if found an entry will be 
 *  added to the system class loader search. 
 *
 * @param start the first char of the file path to test
 * @param end the character after the last char included in the test (: or \0).
 */
static jint try_path_file (char* start, char* end, char* filename) {
    unsigned char* file;
    FILE* fp;
    int flen = strlen (filename);
    jint len = end - start;
    (*jvmti)->Allocate (jvmti, len + flen + 1, &file);
    memcpy (file, start, len);
    memcpy (file + len, filename, flen);
    file[len + flen] = '\0';
    fp = fopen ((char*)file, "r"); // test if file exists...
    if (fp != NULL) {
	fclose (fp);
	fprintf (stdout, "found tijmp.jar as: %s\n", file);
	(*jvmti)->AddToSystemClassLoaderSearch (jvmti, (const char*)file);
    }
    (*jvmti)->Deallocate(jvmti, (unsigned char*)file);
    return fp == NULL ? -1 : 0;
}

void change_to_backslash (char* c) {
    while (*c != '\0') {
	if (*c == '/')
	    *c = '\\';
	c++;
    }
}

/** Try to find tijmp.jar given a base path, if found an entry will be 
 *  added to the system class loader search. 
 *
 * @param start the first char of the file path to test
 * @param end the character after the last char included in the test (: or \0).
 */
static jint try_path (char* start, char* end) {
    char* short_name = "/tijmp.jar";
    char* long_name = "/../share/java/tijmp.jar";
    
    /* TODO: improve this... */
    if (*start != '/') {
	change_to_backslash (short_name);
	change_to_backslash (long_name);
    }

    /* try same path first, may be easier for windows systems... */
    if (try_path_file (start, end, short_name))
	/* try standard linux/unix path */
	return try_path_file (start, end, long_name);
    return 0;
}

static jint setup_classpath (void) {
    jvmtiError err;
    char* path;
    char* start;
    char* end;

    /* Check if the user has specified a certain jar, then use that. */
    err = (*jvmti)->GetSystemProperty (jvmti, "tijmp.jar", &path);
    if (err == JVMTI_ERROR_NONE) {
	fprintf (stdout, "tijmp.jar system property found: %s\n", path);
	(*jvmti)->AddToSystemClassLoaderSearch (jvmti, (const char*)path);
	(*jvmti)->Deallocate(jvmti, (unsigned char*)path);
	return 0;
    }

    /* I am not sure if java.library.path is set on windows, but 
     * for now we use that
     */
    err = (*jvmti)->GetSystemProperty (jvmti, "java.library.path", &path);
    if (err != JVMTI_ERROR_NONE) {
	handle_global_error (err);
	return -1;
    }
    
    if (path == NULL) {
	fprintf (stderr, "java.library.path is not set\n");
	return -1;
    }
    
    /* ok while we can find a separator, try that dir */    
    start = path;
    do {
	end = start;
	while (*end != '\0' && !is_path_separator (end))
	    end++;
	if (!try_path (start, end)) { // non inclusive [)
	    (*jvmti)->Deallocate(jvmti, (unsigned char*)path);
	    return 0;
	}       
	start = end + 1;
    } while (*end != '\0');
    (*jvmti)->Deallocate(jvmti, (unsigned char*)path);
    return -1;
}

JNIEXPORT jint JNICALL 
Agent_OnLoad (JavaVM* jvm, char* options, void* reserved) {
    jint res;
    jvmtiError err;
    jvmtiCapabilities potential_capa;
    jvmtiCapabilities wanted_capa;
    jvmtiEventCallbacks callbacks;
    
    fprintf (stdout, "tijmp Agent_OnLoad: options: '%s'\n", options);
    res = (*jvm)->GetEnv (jvm, (void**)&jvmti, JVMTI_VERSION_1_1);
    if (res != JNI_OK) {
	fprintf (stderr, 
		 "tijmp: error in obtaining jvmti interface pointer\n");
	return JNI_ERR;
    }

    memset (&wanted_capa, 0, sizeof(wanted_capa));
    wanted_capa.can_generate_garbage_collection_events = 1;
    wanted_capa.can_tag_objects = 1;
    err = (*jvmti)->GetPotentialCapabilities (jvmti, &potential_capa);

    if (err == JVMTI_ERROR_NONE) {
	if (!potential_capa.can_generate_garbage_collection_events) {
	    fprintf (stderr, "tijmp: gc events not possible!\n");
	    return JNI_ERR;
	}
	if (!potential_capa.can_tag_objects) {
	    fprintf (stderr, "tijmp: object tagging not possible!\n");
	    return JNI_ERR;
	}
	err = (*jvmti)->AddCapabilities(jvmti, &wanted_capa);
	if (err != JVMTI_ERROR_NONE) {
	    handle_global_error (err);
	    return JNI_ERR;
	}
    } else {
	handle_global_error (err);
	return JNI_ERR;
    }

    memset (&callbacks, 0, sizeof(callbacks));
    callbacks.VMInit = VMInit;
    callbacks.VMDeath = VMDeath;
    callbacks.GarbageCollectionStart = gc_start;
    callbacks.GarbageCollectionFinish = gc_finish;

    (*jvmti)->SetEventCallbacks (jvmti, &callbacks, sizeof(callbacks));
    (*jvmti)->SetEventNotificationMode(jvmti, JVMTI_ENABLE, 
				       JVMTI_EVENT_VM_INIT, NULL);
    (*jvmti)->SetEventNotificationMode(jvmti, JVMTI_ENABLE, 
				       JVMTI_EVENT_VM_DEATH, NULL);
    (*jvmti)->SetEventNotificationMode(jvmti, JVMTI_ENABLE, 
				       JVMTI_EVENT_GARBAGE_COLLECTION_START, NULL);
    (*jvmti)->SetEventNotificationMode(jvmti, JVMTI_ENABLE, 
				       JVMTI_EVENT_GARBAGE_COLLECTION_FINISH, NULL);
    
    return setup_classpath ();    
}

JNIEXPORT void JNICALL 
Agent_OnUnload (JavaVM* vm) {
    jvmtiError err;

    fprintf (stdout, "tijmp Agent_OnUnload\n");
    err = (*jvmti)->DisposeEnvironment (jvmti);
    if (err != JVMTI_ERROR_NONE)
	handle_global_error (err);
}

