-- `Topal': GPG/Pine integration
--
-- Copyright (C) 2001,2002  Phillip J. Brooke
--
--     This program is free software; you can redistribute it and/or modify
--     it under the terms of the GNU General Public License as published by
--     the Free Software Foundation; either version 2 of the License, or
--     (at your option) any later version.
--
--     This program is distributed in the hope that it will be useful,
--     but WITHOUT ANY WARRANTY; without even the implied warranty of
--     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--     GNU General Public License for more details.
--
--     You should have received a copy of the GNU General Public License
--     along with this program; if not, write to the Free Software
--     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

with Ada.IO_Exceptions;
with Ada.Strings.Fixed;
with Ada.Strings.Maps.Constants;
with Ada.Text_IO;
with Expanding_Array;
with Externals;                  use Externals;
with Externals.GPG;
with Externals.Mail;
with Externals.Ops;
with Externals.Simple;           use Externals.Simple;
with Globals;                    use Globals;
with Menus;                      use Menus;
with Misc;                       use Misc;

package body Receiving is

   -- A local subprogram common to both Display and Mime_Display to sort
   -- out whether to process the block, and whether to use cache and how.
   procedure Process_Check (Decrypt       : in     Boolean;
                            Cached        : in     Boolean;
                            Process_Block :    out Boolean;
                            Use_Cache     :    out Boolean;
                            Write_Cache   :    out Boolean) is
   begin
      Debug("+Receiving.Process_Check");
      Process_Block := False;
      Use_Cache := False;
      Write_Cache := False;
      if Decrypt then
         -- Decrypt?
         if Cached then
            if Config.Decrypt_Cached = 1 then
               Ada.Text_IO.Put_Line("Cache file exists, using it instead of decrypting");
               Process_Block := True;
            elsif Config.Decrypt_Cached = 2 then
               Process_Block := YN_Menu("Cache file exists; decrypt? ") = Yes;
            end if;
            if Process_Block then
               if Config.Decrypt_Cached_Use_Cache = 1 then
                  Ada.Text_IO.Put_Line("Using existing cache file");
                  Use_Cache := True;
               elsif Config.Decrypt_Cached_Use_Cache = 2 then
                  Ada.Text_IO.Put_Line("Replacing existing cache file");
                  Write_Cache := True;
               elsif Config.Decrypt_Cached_Use_Cache = 3 then
                  Ada.Text_IO.Put_Line("Replacing existing cache file");
                  Write_Cache := YN_Menu("Replace existing cache file? ") = Yes;
                  -- Config.Decrypt_Cached_Use_Cache = 4: Change nothing.
               elsif Config.Decrypt_Cached_Use_Cache = 5 then
                  declare
                     Selection : URI_Index;
                  begin
                     Selection := URI_Menu;
                     if Selection = UUse then
                        Use_Cache := True;
                     elsif Selection = Replace then
                        Write_Cache := True;
                     end if;
                  end;
               end if;
            end if;
         else
            if Config.Decrypt_Not_Cached = 1 then
               Ada.Text_IO.Put_Line("No cache file exists, decrypting");
               Process_Block := True;
            elsif Config.Decrypt_Not_Cached = 2 then
               Process_Block := YN_Menu("No cache file exists; decrypt? ") = Yes;
            end if;
            if Process_Block then
               if Config.Decrypt_Not_Cached_Use_Cache = 1 then
                  Ada.Text_IO.Put_Line("Will write cache file");
                  Write_Cache := True;
               elsif Config.Decrypt_Not_Cached_Use_Cache = 2 then
                  Write_Cache := YN_Menu("Write cache file? ") = Yes;
               end if;
            end if;
         end if;
      else
         -- Verify?
         if Cached then
            if Config.Verify_Cached = 1 then
      Ada.Text_IO.Put_Line("Cache file exists, using it instead of verifying");
               Process_Block := True;
            elsif Config.Verify_Cached = 2 then
               Process_Block := YN_Menu("Cache file exists; verify? ") = Yes;
            end if;
            if Process_Block then
               if Config.Verify_Cached_Use_Cache = 1 then
                  Ada.Text_IO.Put_Line("Using existing cache file");
                  Use_Cache := True;
               elsif Config.Verify_Cached_Use_Cache = 2 then
                  Ada.Text_IO.Put_Line("Replacing existing cache file");
                  Write_Cache := True;
               elsif Config.Verify_Cached_Use_Cache = 3 then
                  Ada.Text_IO.Put_Line("Replacing existing cache file");
                  Write_Cache := YN_Menu("Replace existing cache file? ") = Yes;
                  -- Config.Verify_Cached_Use_Cache = 4: Change nothing.
               elsif Config.Verify_Cached_Use_Cache = 5 then
                  declare
                     Selection : URI_Index;
                  begin
                     Selection := URI_Menu;
                     if Selection = UUse then
                        Use_Cache := True;
                     elsif Selection = Replace then
                        Write_Cache := True;
                     end if;
                  end;
               end if;
            end if;
         else
            if Config.Verify_Not_Cached = 1 then
               Ada.Text_IO.Put_Line("No cache file exists, verifying");
               Process_Block := True;
            elsif Config.Verify_Not_Cached = 2 then
               Process_Block := YN_Menu("No cache file exists; verify? ") = Yes;
            end if;
            if Process_Block then
               if Config.Verify_Not_Cached_Use_Cache = 1 then
                  Ada.Text_IO.Put_Line("Will write cache file");
                  Write_Cache := True;
               elsif Config.Verify_Not_Cached_Use_Cache = 2 then
                  Write_Cache := YN_Menu("Write cache file? ") = Yes;
               end if;
            end if;
         end if;
      end if;
      Debug("-Receiving.Process_Check");
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Receiving.Process_Check");
         raise;
   end Process_Check;

   Block_Array_Initial_Size : constant Positive := 100;

   -- We want to verify or decrypt a message.
   procedure Display (Tmpfile : in String) is
      Cache_Dir : UBS := ToUBS(ToStr(Topal_Directory)
                               & "/cache");

      -- We are going to break the message up into a sequence of blocks.
      -- Each block has a record associated with it.  Of the three fields,
      -- Used indicates if that block has, funnily enough, been used. PGP
      -- is True if we're going to send it to GPG (either decrypt or
      -- verify).  Decrypt True if the message starts -----BEGIN PGP
      -- MESSAGE----- (i.e., we need to decrypt it) and is False if the
      -- message starts -----BEGIN PGP SIGNED MESSAGE----- (i.e., we won't
      -- need a passphrase).
      type Block_Record is
         record
            Used : Boolean := False;
            PGP  : Boolean := False;
            Decrypt : Boolean := False;
         end record;

      package Block_Array_Package
      is new Expanding_Array (Item => Block_Record);

      use Block_Array_Package;

      Blocks : Big_Array;

      -- Flag indicating if we should wait at the end.
      Do_Wait : Boolean := False;
   begin
      Debug("+Receiving.Display");
      -- Initialize Blocks.
      Create(Blocks, Block_Array_Initial_Size);
      -- Make sure that the cache directory exists.
      Mkdir_P(ToStr(Cache_Dir));
      -- Turn the received message in Tmpfile into blocks.
      -- Block names are of the form Temp_File_Name("blocknnn").
      declare
         use Ada.Text_IO;
         -- The handle for the tmpfile:
         TF : File_Type;
         -- The handle for the current block:
         BF : File_Type;
         -- Block count is the index into the arrays.
         Block_Count : Positive := 1;
         -- The state machine is fairly noddy.
         type State_Machine_Modes is (Starting, Verbatim, Message, Signed);
         State : State_Machine_Modes := Starting;
         -- The strings we're interested in....
         Message_Start : constant String := "-----BEGIN PGP MESSAGE-----";
         Signed_Start  : constant String := "-----BEGIN PGP SIGNED MESSAGE-----";
         Message_End   : constant String := "-----END PGP MESSAGE-----";
         Signed_End    : constant String := "-----END PGP SIGNATURE-----";
      begin
         -- Open the file.  Let exceptions here propogate.
         begin
            Open(File => TF,
                 Mode => In_File,
                 Name => Tmpfile);
         exception
            when others =>
               Error("Exception generated while opening input file "
                     & "for receive.");
               raise;
         end;
     Tmp_Read_Loop:
         loop
            begin
               declare
                  The_Line : String := ToStr(Unbounded_Get_Line(TF));

                  function Block_Name return String is
                  begin
                     return Temp_File_Name("block"
                           & Trim_Leading_Spaces(Integer'Image(Block_Count)));
                  exception
                     when others =>
                        Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                             "Exception raised in Receiving.Display.Block_Name");
                        raise;
                  end Block_Name;

                  procedure Set_BR (Used    : Boolean := False;
                                    PGP     : Boolean := False;
                                    Decrypt : Boolean := False) is
                  begin
                     Set(Blocks,
                         Block_Count,
                         Block_Record'(Used    => Used,
                                       PGP     => PGP,
                                       Decrypt => Decrypt));
                  exception
                     when others =>
                        Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                             "Exception raised in Receiving.Display.Set_BR");
                        raise;
                  end Set_BR;

               begin
                  case State is
                     when Starting =>
                        if The_Line'Length >= Message_Start'Length
                          and then The_Line(The_Line'First..
                                    The_Line'First+Message_Start'Length-1)
                                     = Message_Start then
                           -- This line starts a PGP message.
                           -- Open a new block.
                           begin
                              Create(File => BF,
                                     Mode => Out_File,
                                     Name => Block_Name);
                           exception
                              when others =>
                                 Error("Exception generated when opening block"
                                       & " to write 1: `" & Block_Name & "'");
                           end;
                           -- Write this line.
                           Put_Line(BF, The_Line);
                           -- Set the arrays and state appropriately.
                           Set_BR(Used    => True,
                                  PGP     => True,
                                  Decrypt => True);
                           State := Message;
                        elsif The_Line'Length >= Signed_Start'Length
                          and then The_Line(The_Line'First..
                                    The_Line'First+Signed_Start'Length-1)
                                     = Signed_Start then
                           -- This line starts a PGP signed message.
                           -- Open a new block.
                           begin
                              Create(File => BF,
                                     Mode => Out_File,
                                     Name => Block_Name);
                           exception
                              when others =>
                                 Error("Exception generated when opening block"
                                       & " to write 2: `" & Block_Name & "'");
                           end;
                           -- Write this line.
                           Put_Line(BF, The_Line);
                           -- Set the arrays and state appropriately.
                           Set_BR(Used => True,
                                  PGP  => True);
                           State := Signed;
                        else
                           -- This line is verbatim.
                           -- Open a new block.
                           begin
                              Create(File => BF,
                                     Mode => Out_File,
                                     Name => Block_Name);
                           exception
                              when others =>
                                 Error("Exception generated when opening block"
                                       & " to write 3: `" & Block_Name & "'");
                           end;
                           -- Write this line.
                           Put_Line(BF, The_Line);
                           -- Set the arrays and state appropriately.
                           Set_BR(Used => True);
                           State := Verbatim;
                        end if;
                     when Verbatim =>
                        if The_Line'Length >= Message_Start'Length
                          and then The_Line(The_Line'First..
                                    The_Line'First+Message_Start'Length-1)
                                     = Message_Start then
                           -- This line starts a PGP message.
                           -- Close the existing block.
                           Close(BF);
                           Block_Count := Block_Count + 1;
                           -- Open a new block.
                           begin
                              Create(File => BF,
                                     Mode => Out_File,
                                     Name => Block_Name);
                           exception
                              when others =>
                                 Error("Exception generated when opening block"
                                       & " to write 4: `" & Block_Name & "'");
                           end;
                           -- Write this line.
                           Put_Line(BF, The_Line);
                           -- Set the arrays and state appropriately.
                           Set_BR(Used    => True,
                                  PGP     => True,
                                  Decrypt => True);
                           State := Message;
                        elsif The_Line'Length >= Signed_Start'Length
                          and then The_Line(The_Line'First..
                                    The_Line'First+Signed_Start'Length-1)
                                     = Signed_Start then
                           -- This line starts a PGP signed message.
                           -- Close the existing block.
                           Close(BF);
                           Block_Count := Block_Count + 1;
                           -- Open a new block.
                           begin
                              Create(File => BF,
                                     Mode => Out_File,
                                     Name => Block_Name);
                           exception
                              when others =>
                                 Error("Exception generated when opening block"
                                       & " to write 5: `" & Block_Name & "'");
                           end;
                           -- Write this line.
                           Put_Line(BF, The_Line);
                           -- Set the arrays and state appropriately.
                           Set_BR(Used => True,
                                  PGP  => True);
                           State := Signed;
                        else
                           -- This line is verbatim.
                           -- Write it in the current block.
                           -- Write this line.
                           Put_Line(BF, The_Line);
                        end if;
                     when Message =>
                        if The_Line'Length >= Message_End'Length
                          and then The_Line(The_Line'First..
                                    The_Line'First+Message_End'Length-1)
                                     = Message_End then
                           -- This line ends the current PGP message.
                           -- Write this line.
                           Put_Line(BF, The_Line);
                           -- Close the existing block.
                           Close(BF);
                           Block_Count := Block_Count + 1;
                           -- Set the arrays and state appropriately.
                           State := Starting;
                        else
                           -- Write this into the current block.
                           Put_Line(BF, The_Line);
                        end if;
                     when Signed =>
                        if The_Line'Length >= Signed_End'Length
                          and then The_Line(The_Line'First..
                                    The_Line'First+Signed_End'Length-1)
                                     = Signed_End then
                           -- This line ends the current PGP message.
                           -- Write this line.
                           Put_Line(BF, The_Line);
                           -- Close the existing block.
                           Close(BF);
                           Block_Count := Block_Count + 1;
                           -- Set the arrays and state appropriately.
                           State := Starting;
                        else
                           -- Write this into the current block.
                           Put_Line(BF, The_Line);
                        end if;
                  end case;
               end;
            exception
               when Ada.IO_Exceptions.End_Error =>
                  -- Run out of lines to read.  We're done.
                  exit Tmp_Read_Loop;
            end;
         end loop Tmp_Read_Loop;
         -- Tidy up -- close the existing files.
         if Is_Open(BF) then
            Close(BF);
         end if;
         if Is_Open(TF) then
            Close(TF);
         end if;
      exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Receiving.Display (Tmpfile -> Blocks)");
         raise;
      end; -- declare: The Tmpfile -> Blocks bit.

      -- Delete the Tmpfile.  We're going to append to it instead.
      Rm_File(Tmpfile);

      -- Now, run through the entire array.  Stop when we run out of blocks
      -- or find one which says `Block_Used = False'.
  Block_Use_Loop:
      for I in 1 .. Last(Blocks) loop
         exit when not Value(Blocks, I).Used;
         if Value(Blocks, I).PGP then
            declare
               Process_Block   : Boolean;
               Cached          : Boolean;
               Cached2         : Boolean;
               Write_Cache     : Boolean;
               Use_Cache       : Boolean;
               This_Block_Wait : Boolean := True;
               -- Get temp filenames.
               Com_File   : String := Temp_File_Name("com"
                            & Trim_Leading_Spaces(Integer'Image(I)));
               Out_File   : String := Temp_File_Name("out"
                            & Trim_Leading_Spaces(Integer'Image(I)));
               Err_File   : String := Temp_File_Name("err"
                            & Trim_Leading_Spaces(Integer'Image(I)));
               Blk_File   : String := Temp_File_Name("blk"
                            & Trim_Leading_Spaces(Integer'Image(I)));
               -- We'll need to read in the cache line.
               MD5        : UBS;
            begin
               -- Is this file cached?
               -- Get the cache filename.
               MD5 := Ops.Get_MD5Sum_Of_File(Temp_File_Name("block"
                                  & Trim_Leading_Spaces(Integer'Image(I))));
               -- So the cache filename is ToStr(Cache_Dir) & "/" & ToStr(MD5).
               -- Does it exist?
               Cached := Test_R(ToStr(Cache_Dir) & "/" & ToStr(MD5));
               Cached2 := Test_R(ToStr(Cache_Dir) & "/" & ToStr(MD5) & ".out");
               if Cached and (not Cached2) then
                  Ada.Text_IO.Put_Line("Hmm. Missing .out file from cache....");
               end if;
               -- At this point, we know whether or not the file is cached
               -- (Cached) and also if this is a decrypt or verify
               -- (Block_Decrypt(I)).  Now check what we need to do in
               -- terms of Process_Block, Use_Cache and Write_Cache.
               Process_Check(Value(Blocks, I).Decrypt,
                             Cached,
                             Process_Block,
                             Use_Cache,
                             Write_Cache);
               -- Now we know if we're going to process the block, and
               -- whether or not we'll read from a cache, or write to a
               -- cache.
               if Process_Block then
                  -- Run GPG or get stuff from cache..
                  if Use_Cache then
                     -- It's cached, copy the cache file.
                     -- Append the cache file to the tmpfile.
                     Echo_Append("----- Topal: Using cache file `"
                                 & ToStr(Cache_Dir) & "/" & ToStr(MD5)
                                 & "'-----",
                                 Com_File);
                     Cat_Append(ToStr(Cache_Dir) & "/" & ToStr(MD5), Com_File);
                     if Config.Inline_Separate_Output then
                        Pager(Com_File);
                        if Cached2 then
                           Cat_Append(ToStr(Cache_Dir) & "/" & ToStr(MD5) & ".out",
                                      Tmpfile);
                        else
                           Error("Need .out!  Have you cleared the cache since upgrading to Topal >=0.7.8??");
                        end if;
                     else
                        Cat_Append(Com_File, Tmpfile);
                        if Value(Blocks, I).Decrypt then
                           Echo_Append("----- Topal: Decrypted output starts ----- ", Tmpfile);
                        if Cached2 then
                           Cat_Append(ToStr(Cache_Dir) & "/" & ToStr(MD5) & ".out",
                                      Tmpfile);
                        else
                           Error("Need .out!  Have you cleared the cache since upgrading to Topal >=0.7.8?");
                        end if;
                           Echo_Append("----- Topal: Decrypted output ends ----- ", Tmpfile);
                        else
                           Echo_Append("----- Topal: Original message starts ----- ", Tmpfile);
                           Cat_Append(Temp_File_Name("block"
                                                     & Trim_Leading_Spaces(Integer'Image(I))), Tmpfile);
                           Echo_Append("----- Topal: Original message ends ----- ", Tmpfile);
                        end if;

                     end if;
                     -- Sort out the This_Block_Wait.
                     if Value(Blocks, I).Decrypt then
                        This_Block_Wait := not Config.Decrypt_Cached_Fast_Continue;
                     else
                        This_Block_Wait := not Config.Verify_Cached_Fast_Continue;
                     end if;
                  else
                     -- Run GPG!
                     declare
                        GPG_Return_Value : Integer;
                        Include_Out_File : Boolean := True;
                     begin
                        GPG_Return_Value
                          := GPG.GPG_Tee(Temp_File_Name("block"
                                                        & Trim_Leading_Spaces(Integer'Image(I))),
                                         Out_File,
                                         Err_File);
                        -- Now, according to the GPG man page: ``The
                        -- program returns 0 if everything was fine, 1 if
                        -- at least a signature was bad, and other error
                        -- codes for fatal errors.''
                        -- So, we're not bothered here about bad signatures
                        -- (it's up to the user to decide what to do), but
                        -- anything else means that it's broken and we can't
                        -- do much else.
                        if GPG_Return_Value > 1 then
                           -- Certainly don't cache this thing.
                           Write_Cache := False;
                           -- Omit anything that deals with the output file
                           -- (because it probably doesn't exist).
                           Include_Out_File := False;
                           Ada.Text_IO.Put_Line("Topal: GPG return value > 1; was = "
                                                & Trim_Leading_Spaces(Integer'Image(GPG_Return_Value))
                                                & "; this is not good -- omitting output.");
                        end if;
                        -- Append GPG's stdout, stderr, some wrappers to block
                        -- result file.
                        Echo_Out_N("----- Topal: Output generated on ",
                                   Blk_File);
                        Date_Append(Blk_File);
                        Echo_Append("----- Topal: GPG output starts ----- ",
                                    Blk_File);
                        Cat_Append(Err_File, Blk_File);
                        Echo_Append("----- Topal: GPG output ends ----- ",
                                    Blk_File);
                        -- Copy GPG's block result file and the separate
                        -- output to cache files.
                        -- But only if we really want to.
                        if Write_Cache then
                           Cat_Out(Blk_File,
                                   ToStr(Cache_Dir) & "/" & ToStr(MD5));
                           Cat_Out(Out_File,
                                   ToStr(Cache_Dir) & "/" & ToStr(MD5) & ".out");
                        end if;
                        if Config.Inline_Separate_Output then
                           Pager(Blk_File);
                           Cat_Append(Out_File, Tmpfile);
                        else
                           if Value(Blocks, I).Decrypt
                             and Include_Out_File then
                              -- Copy the processed output with wrappers.
                              Echo_Append("----- Topal: Decrypted output starts ----- ",
                                          Blk_File);
                              Cat_Append(Out_File, Blk_File);
                              Echo_Append("----- Topal: Decrypted output ends ----- ",
                                          Blk_File);
                           else
                              -- Copy the original file, with wrappers.
                              Echo_Append("----- Topal: Original message starts ----- ",
                                          Blk_File);
                              Cat_Append(Temp_File_Name("block"
                                                        & Trim_Leading_Spaces(Integer'Image(I))),
                                         Blk_File);
                              Echo_Append("----- Topal: Original message ends ----- ",
                                          Blk_File);
                           end if;
                           -- Append block result file to Pine tmpfile..
                           Cat_Append(Blk_File, Tmpfile);
                        end if;
                        -- Sort out the This_Block_Wait.
                        if not Value(Blocks, I).Decrypt then
                           This_Block_Wait
                             := not Config.Verify_Not_Cached_Fast_Continue;
                        end if;
                     end;
                  end if;
               else
                  -- Don't process the block; just append it.
                  Cat_Append(Temp_File_Name("block"
                                            & Trim_Leading_Spaces(Integer'Image(I))),
                             Tmpfile);
               end if;
               -- Sort out the wait part.
               Do_Wait := Do_Wait or This_Block_Wait;
            end; -- Of declare block.
         else
            -- Verbatim block; just append it.
            Cat_Append(Temp_File_Name("block"
                                      & Trim_Leading_Spaces(Integer'Image(I))),
                       Tmpfile);
            -- Verbatim blocks don't affect the Do_Wait setting.
         end if;
      end loop Block_Use_Loop;

      if Do_Wait then
         Debug("Receiving.Display: Waiting...");
         Wait;
      else
         Debug("Receiving.Display: Not waiting...");
      end if;
      Debug("-Receiving.Display");
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Receiving.Display");
         raise;
   end Display;

   procedure Mime_Display (Infile       : in String;
                           Content_Type : in String) is
      Cache_Dir : UBS := ToUBS(ToStr(Topal_Directory)
                               & "/cache");
      Boundary : UBS;
      -- If this subprogram has been called, we expect to be decoding a
      -- MIME RFC2015 message.  This will be split into two parts, possibly
      -- with some junk at the head and tail.  We use the boundary to get
      -- the two parts.

      -- The temp filenames.
      Part_One      : String := Temp_File_Name("part1");
      Part_Two      : String := Temp_File_Name("part2");
      Out_File      : String := Temp_File_Name("out");
      Err_File      : String := Temp_File_Name("err");
      Blk_File      : String := Temp_File_Name("blk");
      -- Other similar stuff to the previous subprogram.
      Decrypt       : Boolean;
      Process_Block : Boolean;
      Cached        : Boolean;
      Write_Cache   : Boolean;
      Use_Cache     : Boolean;
      -- We'll need to read in the cache line.
      MD5        : UBS;
      -- Messing with GPG.
      GPG_Return_Value    : Integer;
      Include_Out_File    : Boolean := True;
   begin
      Debug("+Receiving.Mime_Display");
      Debug("Infile=" & Infile);
      Debug("Content_Type=" & Content_Type);
      -- We need to figure out what the boundary is.
      Boundary := Mail.Find_Mime_Boundary(Infile);
      Debug("Boundary=" & ToStr(Boundary));
      -- Make sure that the cache directory exists.
      Mkdir_P(ToStr(Cache_Dir));
      -- Is this file cached?
      -- Get the cache filename.
      MD5 := Ops.Get_MD5Sum_Of_File(Infile);
      -- So the cache filename is ToStr(Cache_Dir) & "/" & ToStr(MD5).
      -- Does it exist?
      Cached := Test_R(ToStr(Cache_Dir) & "/" & ToStr(MD5));
      -- Is this a verify or decrypt?
      -- Need to use Ada.Strings.Maps.Constants.Lower_Case_Map
      declare
         CT : String := Ada.Strings.Fixed.Translate(Content_Type,
                                    Ada.Strings.Maps.Constants.Lower_Case_Map);
      begin
         if CT = "multipart/signed"
           or CT = "application/x-topal-signed" then
            Decrypt := False;
         elsif CT = "multipart/encrypted"
           or CT = "application/x-topal-encrypted" then
            Decrypt := True;
         else
            -- Have no idea what to do -- bail out.
            Error("Don't know what to do with Content-Type: "
                  & CT);
         end if;
      end;
      -- Get the two parts.
      Mail.Split_Two_Parts(Infile, Part_One, Part_Two, Boundary);
      -- Now, we have the two parts.  We need to check if it is cached (but
      -- we'll ignore this for now).  So, sort out what to do with this block.
      Process_Check(Decrypt,
                    Cached,
                    Process_Block,
                    Use_Cache,
                    Write_Cache);
      -- And now we either use the cache, or process the block.
      if Process_Block then
         if Use_Cache then
            -- It's cached, copy the cache file.
            -- Append the cache file to the tmpfile.
            Echo_Append("----- Topal: Using cache file `"
                        & ToStr(Cache_Dir) & "/" & ToStr(MD5)
                        & "'----- ",
                        Blk_File);
            Cat_Append(ToStr(Cache_Dir) & "/" & ToStr(MD5), Blk_File);
            -- This is MIME, so we also retrieve the out file.
            -- But verifying a MIME message means we ignore the outfile....
            if Decrypt then
               Cat_Out(ToStr(Cache_Dir) & "/" & ToStr(MD5) & ".out", Out_File);
            else
               -- Verify case: test and warn.
               if Test_R(ToStr(Cache_Dir) & "/" & ToStr(MD5) & ".out") then
                  Ada.Text_IO.Put_Line("Hmm. Unexpected .out file in cache....");
                  Cat_Out(ToStr(Cache_Dir) & "/" & ToStr(MD5) & ".out",
                          Out_File);
               end if;
            end if;
         else
            if Decrypt then
               -- Just pass the second part to GPG.
               GPG_Return_Value := GPG.GPG_Tee(Part_Two, Out_File, Err_File);
            else
               -- We'll use both parts.  But we have to turn part one into
               -- canonical line-end first.
               Dos2Unix_U(Part_One);
               -- Use both parts....
               GPG_Return_Value := GPG.GPG_Verify_Tee(Input_File  => Part_One,
                                                      Sig_File    => Part_Two,
                                                      Output_File => Out_File,
                                                      Err_File    => Err_File);
            end if;
            -- Now, according to the GPG man page: ``The
            -- program returns 0 if everything was fine, 1 if
            -- at least a signature was bad, and other error
            -- codes for fatal errors.''
            -- So, we're not bothered here about bad signatures
            -- (it's up to the user to decide what to do), but
            -- anything else means that it's broken and we can't
            -- do much else.
            if GPG_Return_Value > 1 then
               -- Certainly don't cache this thing.
               Write_Cache := False;
               -- Omit anything that deals with the output file
               -- (because it probably doesn't exist).
               Include_Out_File := False;
               Ada.Text_IO.Put_Line("Topal: GPG return value > 1; was = "
                                    & Trim_Leading_Spaces(Integer'Image(GPG_Return_Value))
                                    & "; this is not good -- omitting output.");
            end if;
            -- Append GPG's stdout, stderr, some wrappers to block
            -- result file.
            Echo_Out_N("----- Topal: Output generated on ",
                       Blk_File);
            Date_Append(Blk_File);
            Echo_Append("----- Topal: GPG output starts ----- ",
                        Blk_File);
            Cat_Append(Err_File, Blk_File);
            Echo_Append("----- Topal: GPG output ends ----- ",
                        Blk_File);
            -- Copy GPG's block result file to cache file.
            -- But only if we really want to.
            if Write_Cache then
               Cat_Out(Blk_File, ToStr(Cache_Dir) & "/" & ToStr(MD5));
               -- For MIME decrypt stuff, we also save the `out' file.
               if Decrypt then
                  Cat_Out(Out_File,
                          ToStr(Cache_Dir) & "/" & ToStr(MD5) & ".out");
               else
                  -- Check for the existence of an out file for verify....
                  if Test_R(Out_File) then
                     Ada.Text_IO.Put_Line("Hmm. Unexpected output file exists....");
                     Cat_Out(Out_File,
                             ToStr(Cache_Dir) & "/" & ToStr(MD5) & ".out");
                  end if;
               end if;
            end if;
         end if;
         -- Output the results.
         Pager(Blk_File);
         -- If we've verified a signature, we need to then show the nice
         -- version of the file via metamail.  If it was decrypted, then we
         -- show off the Out_File.  Note that if the next object in is also
         -- a MIME object, we might be invoked again.
         if Decrypt then
            if Include_Out_File then
               -- If this was False, then GPG failed.  So there's nothing for
               -- View_MIME to deal with.
               Dos2Unix(Out_File);
               View_MIME(Out_File);
            end if;
         else
            Dos2Unix(Part_One);
            View_MIME(Part_One);
         end if;
      else
         -- If we're not to process the block, then just dump out original
         -- input.
         Pager(Infile);
      end if;
      Debug("-Receiving.Mime_Display");
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Receiving.Mime_Display");
         raise;
   end Mime_Display;

   -- Our approach to displaying application/pgp content-types is to hand
   -- off the input file to Display and get it to do the work.  We simply
   -- display it.
   procedure Mime_Display_APGP (Infile       : in String;
                                Content_Type : in String) is
      CT : String := Ada.Strings.Fixed.Translate(Content_Type,
                                   Ada.Strings.Maps.Constants.Lower_Case_Map);
   begin
      Debug("+Receiving.Mime_Display_APGP");
      if CT = "application/pgp" then
         -- We handle all of this nasty, deprecated mode here.
         declare
            Tempfile : String := Temp_File_Name("apgptemp");
         begin
            -- Copy the input to the temporary file.
            Cat_Out(Infile, Tempfile);
            -- Invoke Display on the temporary file.
            Display(Tempfile);
            -- Then display it using less.
            Pager(Tempfile);
         end;
      else
         -- Have no idea what to do -- bail out.
         Error("Don't know what to do with Content-Type: "
               & CT);
      end if;
      Debug("-Receiving.Mime_Display_APGP");
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Receiving.Mime_Display_APGP");
         raise;
   end Mime_Display_APGP;

end Receiving;
