/*
	**
	** limit-report.c
	**
	** Routines to limit the number of hosts and/or peers in a report
	**
	** Copyright 1998-1999 Damien Miller <dmiller@ilogic.com.au>
	**
	** This software is licensed under the terms of the GNU General 
	** Public License (GPL). Please see the file COPYING for details.
	** 
	** $Id: limit-report.c,v 1.4 1999/02/10 12:29:20 dmiller Exp $
	**
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>

#include <glib.h>

#include "report.h"
#include "limit-report.h"

static char rcsid[] = "$Id: limit-report.c,v 1.4 1999/02/10 12:29:20 dmiller Exp $";

/* Prototypes */
void limit_peer_list_by_number(peer_t **root, unsigned int n_peers);

void limit_hosts_by_number(host_t **root, unsigned int n_hosts)
{
	host_t	*h;
	host_t	*last_h;
	
	/* Count down through the hosts */
	last_h = NULL;
	h = *root;
	while ((h != NULL) && (n_hosts > 0))
	{
		last_h = h;
		h = h->next;
		n_hosts--;
	}
	
	/* If we ran out of hosts, return */
	if (h == NULL)
		return;
		
	/* If we didn't find any hosts -or- if n_hosts was 0 */
	/* then set root to NULL, otherwise terminate list at last host */
	if (last_h == NULL)
		*root = NULL;
	else
		last_h->next = NULL;
	
	/* Free all hosts after last host */
	free_hosts(h);
}

void limit_peers_by_number(host_t *h, unsigned int n_peers)
{
	while (h != NULL)
	{
		limit_peer_list_by_number(&(h->peers), n_peers);
		h = h->next;
	}
}

void limit_peer_list_by_number(peer_t **root, unsigned int n_peers)
{
	peer_t	*p;
	peer_t	*last_p;
	
	/* Count down through the peers */
	last_p = NULL;
	p = *root;
	while ((p != NULL) && (n_peers > 0))
	{
		last_p = p;
		p = p->next;
		n_peers--;
	}
	
	/* If we ran out of peers, return */
	if (p == NULL)
		return;
		
	/* If we didn't find any peers -or- if n_peers was 0 */
	/* then set root to NULL, otherwise terminate list at last peer */
	if (last_p == NULL)
		*root = NULL;
	else
		last_p->next = NULL;
	
	/* Free all peers after last peer */
	free_peers(p);
}

void trim_peers(host_t *root)
{
	GHashTable *hosts;
	host_t *h;
	peer_t *p;
	peer_t *last_p;
	peer_t *delete_p;
	
	/* Create hosts hash table, keyed by IP address */
	hosts = g_hash_table_new((GHashFunc)host_hash, (GCompareFunc)host_compare);
	if (hosts == NULL)
	{
		fprintf(stderr, "\nOut of memory.\n");
		raise(SIGSEGV);
	}
	
	/* Add each host to hash table */
	h = root;
	while (h != NULL)
	{
		g_hash_table_insert(hosts, (gpointer)&(h->ip_addr), (gpointer)h);
		h = h->next;
	}

	/* For each host, check peer list */
	h = root;
	while (h != NULL)
	{
		p = h->peers;
		last_p = NULL;
		while (p != NULL)
		{
			if ((NULL == g_hash_table_lookup(hosts, (gpointer)&(p->src_addr))) ||
				 (NULL == g_hash_table_lookup(hosts, (gpointer)&(p->dst_addr))))
			{
				if (last_p == NULL)
				{
					delete_p = p;
					p = h->peers = p->next;
					last_p = NULL;
					free(delete_p);
				} else
				{
					delete_p = p;
					p = last_p->next = p->next;
					free(delete_p);
				}
			} else
			{
				last_p = p;
				p = p->next;
			}
		}

		h = h->next;
	}
	
	g_hash_table_destroy(hosts);
}

void limit_hosts_by_address(report_t *r, int mode, u_int32_t address, u_int32_t mask)
{
	host_t *last_h;
	host_t *old_h;
	host_t *h;
	
	if (mode == EXCLUDE_NONE)
		return;

	last_h = NULL;
	h = r->hosts;
	
	/* For each host */
	while(h != NULL)
	{
		/* Check whether address matches */
		if (((mode == EXCLUDE_SPECIFIED) && ((h->ip_addr & mask) == address)) ||
			 ((mode == EXCLUDE_UNSPECIFIED) && ((h->ip_addr & mask) != address)))
		{
			/* Check for deletion at head of list */
			if (last_h != NULL)
				last_h->next = h->next;
			else
				r->hosts = h->next;

			old_h = h;
			h = h->next;

			/* Delete host */
			free_host(old_h);
		} else
		{					
			/* Otherwise, move onto next */
			last_h = h;
			h = h->next;
		}
	}
	
	trim_peers(r->hosts);
}
