/*
  Copyright (C) 1997,1998,1999  Dimitrios P. Bouras

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   For author contact information, look in the README file.
*/

/* ASCII resource control file I/O routines and local data */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <varargs.h>
#include <string.h>
#include <math.h>
#include <sys/time.h>
#include "common.h"
#include "logs.h"
#include "version.h"

#ifdef SUNOS41x
#include <memory.h>
extern int sys_nerr;
extern char *sys_errlist[];
extern int errno;
extern int sscanf(), fputs(), vfprintf(), fscanf(), fprintf(), fclose();
extern long strtol();
#endif

/* Local storage */

static char *rcfn;							/* pointer to RC file name str */
static glob_t reset_global = GLOB_DEFAULT;	/* for resetting global struct */

/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                        Local Utility routines                           |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

/* Convert binary password to ASCII printable characters */

static char *passwd2ascii(unsigned char *pwd)
{
	static unsigned char apwd[2*MAXLEN_PASSWD+1];
	unsigned char msh, lsh, *p;
	int i;

	for (i=0, p=apwd; i<MAXLEN_PASSWD; i++, pwd++) {
		msh = (*pwd & 0xF0) >> 4;
		lsh = *pwd & 0x0F;
		*p++ = lsh + 'A';
		*p++ = msh + 'A';
	}
	*p = 0;
	return apwd;
}

/* Convert IP address bytes to ASCII string */

static char *IP2str(unsigned char *ip)
{
	static char IPstr[16];

	sprintf(IPstr, "%u.%u.%u.%u",
			ip[0],ip[1],ip[2],ip[3]);
	return IPstr;
}

/* Convert speed bit-mask to ASCII string */

static char *speed2str(unsigned char speed)
{
	static char sstr[8][8] = {"1200","2400","4800","9600",
							  "19200","38400","57600","115200"};
	static int snum[8] = {0x01,0x02,0x04,0x08,0x10,0x20,0x40,0x80};
	int i;

	for (i=0; i<8; i++)
		if (speed & snum[i])
			return sstr[i];
	return NULL;
}

/* Convert ASCII password to binary */

static char *ascii2passwd(char *buf)
{
	static unsigned char pwd[MAXLEN_PASSWD+1];
	unsigned char msh, lsh, *p;
	int i = 0;

	if (*buf) {
		for (p=pwd; i<MAXLEN_PASSWD; i++) {
			lsh = (*buf++ - 'A') & 0x0F;
			msh = (*buf++ - 'A') & 0x0F;
			pwd[i] = (msh << 4) | lsh;
		}
	}
	pwd[i] = 0;
	return pwd;
}

/* Convert ASCII speed string to bit-mask */

static unsigned int str2speed(char *buf)
{
	static char sstr[8][8] = {"1200","2400","4800","9600",
							  "19200","38400","57600","115200"};
	static int snum[8] = {0x01,0x02,0x04,0x08,0x10,0x20,0x40,0x80};
	int i;

	for (i=0; i<8; i++)
		if (strcmp(buf, sstr[i]) == 0)
			return snum[i];
	return 0;
}

/* Convert ASCII string to IP address */

static unsigned char *str2IP(char *buf)
{
	static unsigned char ip[4];
	unsigned int iip[4];
	int i;

	i = sscanf(buf, "%3u.%3u.%3u.%3u", &iip[0],
			   &iip[1], &iip[2], &iip[3]);
	if (i < 4)
		return NULL;
	for (i=0; i<4; i++)
		ip[i] = iip[i];
	return ip;
}

/* Print message together with system error message and exit. Note the
   implicit total length of MSGLEN_ERR bytes for the resulting string. */

#define MSGLEN_ERR 128

static void doErr(char *msg)
{
	char emsg[MSGLEN_ERR+1];

#ifdef HAVE_STRERROR
	sprintf(emsg, "xISP: %s: %s\n", msg, strerror(errno));
#else
	if (errno < sys_nerr)
		sprintf(emsg, "xISP: %s: %s\n", msg, sys_errlist[errno]);
	else
		sprintf(emsg, "xISP: %s: error #%d\n", msg, errno);
#endif
	fputs(emsg, stderr);
	exit(1);
}

/* Function for writing strings in the .xisprc resource control file */

static int rcPrint(va_alist) va_dcl
{
	int bw;
	va_list ap;
	char *fmt;
	FILE *fp;

	va_start(ap);
	fp = va_arg(ap, FILE*);
	fmt = va_arg(ap, char*);
	bw = vfprintf(fp, fmt, ap);
	va_end(ap);
	if (bw <= 0)
		doErr("outputAllXisprc");
	return bw;
}

/* Function for writing out the entire .xisprc file */

static void outputAllXisprc(xisprc_t *p, FILE *rcfp, glob_t *global)
{
	int i, n;

	rcPrint(rcfp, "xISP-%s\n", Version);
	rcPrint(rcfp, "NUMISPS: %d\n", global->numISPs);
	rcPrint(rcfp, "DEFAULTISP: %d\n", global->dfltISP);
	rcPrint(rcfp, "NUMPTTS: %d\n", global->numPTTs);
	rcPrint(rcfp, "QTHPTT: %d\n", global->dfltPTT);
	rcPrint(rcfp, "QTHZONE: %d\n", global->dfltZone);
	rcPrint(rcfp, "LOGOPTS: %hX\n", global->logOpts);
	rcPrint(rcfp, "PPPDPATH: %s\n", global->pppdPath);
	rcPrint(rcfp, "RUNPATH: %s\n", global->runPath);
	rcPrint(rcfp, "CHATPATH: %s\n", global->chatPath);
	rcPrint(rcfp, "UTILSPATH: %s\n", global->utilsPath);
	rcPrint(rcfp, "PIPEPATH: %s\n", global->pipePath);
	for (i=0; p!=NULL && i<global->numISPs; i++, p++) {
		rcPrint(rcfp, "-%d-\n", i);
		rcPrint(rcfp, "DESCR: %s\n", p->descr);
		rcPrint(rcfp, "ACCOUNT: %s\n", p->account);
		if (*(p->account))
			rcPrint(rcfp, "PASSWD: %s\n", passwd2ascii(p->passwd));
		else
			rcPrint(rcfp, "PASSWD: \n");
		rcPrint(rcfp, "NAME: %s\n", p->name);
		rcPrint(rcfp, "REMOTENAME: %s\n", p->rname);
		rcPrint(rcfp, "MAXATTEMPTS: %d\n", p->maxAttempts);
		rcPrint(rcfp, "SLEEPDELAY: %d\n", p->sleepDelay);
		rcPrint(rcfp, "CONNECTWAIT: %d\n", p->connectWait);
		rcPrint(rcfp, "NUMPHONES: %d\n", p->numPhones);
		for (n=0; n<p->numPhones; n++)
			rcPrint(rcfp, " PHONE%d: %s\n", n, p->phone[n]);
		rcPrint(rcfp, "NUMSLINES: %d\n", p->numSlines);
		for (n=0; n<p->numSlines; n++) {
			rcPrint(rcfp, " SLINE%dE: %s\n", n, p->sline[n]);
			rcPrint(rcfp, " SLINE%dS: %s\n", n, p->sline[MAXNUM_SLINES+n]);
		}
		rcPrint(rcfp, "CBDELAY: %d\n", p->CBDelay);
		rcPrint(rcfp, "NUMCBSLINES: %d\n", p->numCBSlns);
		for (n=0; n<p->numCBSlns; n++) {
			rcPrint(rcfp, " CBSLINE%dE: %s\n", n, p->CBsln[n]);
			rcPrint(rcfp, " CBSLINE%dS: %s\n", n, p->CBsln[MAXNUM_SLINES+n]);
		}
		rcPrint(rcfp, "TERMW: %d\n", p->termW);
		rcPrint(rcfp, "TERMH: %d\n", p->termH);
		rcPrint(rcfp, "CBTERMW: %d\n", p->CBtermW);
		rcPrint(rcfp, "CBTERMH: %d\n", p->CBtermH);
		rcPrint(rcfp, "CBPHONE: %s\n", p->CBphone);
		rcPrint(rcfp, "MODEMDEVICE: %s\n", p->modemDevice);
		rcPrint(rcfp, "MODEMSPEED: %s\n", speed2str(p->modemSpeed));
		rcPrint(rcfp, "MODEMRESET: %s\n", p->modemReset);
		rcPrint(rcfp, "MODEMINIT: %s\n", p->modemInit);
		rcPrint(rcfp, "MODEMCONNECT: %s\n", p->modemConnect);
		rcPrint(rcfp, "OPEROPTS: %lX\n", p->operOpts);
		rcPrint(rcfp, "DIALEXTRA: %s\n", p->dialExtra);
		rcPrint(rcfp, "COMPLEVEL: %d\n", p->compLevel);
		rcPrint(rcfp, "ASYNCMAP: %s\n", p->asyncmap);
		rcPrint(rcfp, "ESCAPE: %s\n", p->escape);
		rcPrint(rcfp, "LOCALIP: %s\n", IP2str(p->localIP));
		rcPrint(rcfp, "REMOTEIP: %s\n", IP2str(p->remoteIP));
		rcPrint(rcfp, "NETMASK: %s\n", IP2str(p->netmask));
		rcPrint(rcfp, "DNS1: %s\n", IP2str(p->dns1));
		rcPrint(rcfp, "DNS2: %s\n", IP2str(p->dns2));
		rcPrint(rcfp, "MTU: %u\n", p->mtu);
		rcPrint(rcfp, "MRU: %u\n", p->mru);
	}
}

/* Function for retrieving the .xisprc version. Note that it
   must be called after fopen() and before all other input */

static char *rcVersion(FILE *fp)
{
	static char version[8] = {0};

	if (fscanf(fp, "xISP-%7[.0-9] ", version) < 1) {
		fputs("xISP: rcVersion: error reading file version!\n", stderr);
		exit(1);
	}
	return version;
}

/* Function for retrieving the ISP sequence number. Note that
   it must be called at the beginning of each ISP record. */

static void rcISP(FILE *fp, int nISP)
{
	int n;

	if (fscanf(fp, "-%3d- ", &n) < 1) {
		fprintf(stderr, "xISP: rcISP: error reading ISP number in %s\n",
				rcfn);
		exit(1);
	}
	if (n != nISP) {
		fprintf(stderr, "xISP: rcISP: ISP sequence number wrong in %s\n",
				rcfn);
		fprintf(stderr, "xISP: rcISP: expected %d, got %d\n", nISP, n);
		exit(1);
	}
}

/* Prints error message if error occurs while
   processing .xisprc. Prints offending line */

static void rcLineError(char *function, char *line)
{
	fprintf(stderr, "xISP: %s: error reading %s\n", function, rcfn);
	fprintf(stderr, "xISP: %s: offending line: [%s]\n", function, line);
	exit(1);
}

/* Reads parameters from .xisprc file, doing error checking. NOTE: it
   assumes a line less than 512 bytes and a parameter less than 256 */

#define MAXLEN_LINE 512
#define MAXLEN_PARAM 256

static char *readParam(FILE *fp, char *name, char type,
					   int llimit, int ulimit, void *data)
{
	static char line[MAXLEN_LINE];
	char sparam[MAXLEN_PARAM] = {0}, *p, *endp;
	long iparam;
	int len, hex = 0;

	if (fgets(line, MAXLEN_LINE, fp) == NULL) {
		fprintf(stderr, "xISP: readParam: %s: premature EOF\n", rcfn);
		exit(1);
	}
	line[strlen(line)-1] = 0;
	if ((p=strchr(line,':')) == NULL)
		rcLineError("readParam", line);
	if (strncmp(name, line, (int)(p-line))) {
		fprintf(stderr, "xISP: readParam: expected [%s], got [%s]\n",
				name, line);
		exit(1);
	}
	for (++p; *p==' '; p++);
	strncpy(sparam, p, sizeof(sparam)-1);
	switch (type) {
		case 'S':
			len = strlen(sparam);
			if (ulimit<len || len<llimit)
				rcLineError("readParam", line);
			strcpy((char *)data, sparam);
			break;

		case 'X':
		case 'L':
			hex = 1;
		case 'B':
		case 'I':
			iparam = strtol(sparam, &endp, (hex)? 16:10);
			if (hex) {
				if (type == 'X') *((unsigned short *)data) = iparam;
				else *((unsigned long *)data) = iparam;
			}
			else {
				if (ulimit<iparam || iparam<llimit || endp == sparam || *endp)
					rcLineError("readParam", line);
				if (type == 'B') *((unsigned char *)data) = iparam;
				else *((unsigned int *)data) = iparam;
			}
			break;

		default: break;
	}
	return(line);
}

/* Function for reading in the global section from the .xisprc file */

static void inputGlobals(FILE *rcfp, glob_t *global)
{
	readParam(rcfp, "NUMISPS", 'B', 0, 255, &(global->numISPs));
	readParam(rcfp, "DEFAULTISP", 'B', 0, 255, &(global->dfltISP));
	readParam(rcfp, "NUMPTTS", 'B', 1, 255, &(global->numPTTs));
	readParam(rcfp, "QTHPTT", 'B', 0, (global->numPTTs)-1, &(global->dfltPTT));
	readParam(rcfp, "QTHZONE", 'B', 0, MAXNUM_ZONES-1, &(global->dfltZone));
	readParam(rcfp, "LOGOPTS", 'X', 0, 0, &(global->logOpts));
	readParam(rcfp, "PPPDPATH", 'S', 0, MAXLEN_PATH, global->pppdPath);
	readParam(rcfp, "RUNPATH", 'S', 0, MAXLEN_PATH, global->runPath);
	readParam(rcfp, "CHATPATH", 'S', 0, MAXLEN_PATH, global->chatPath);
	readParam(rcfp, "UTILSPATH", 'S', 0, MAXLEN_PATH, global->utilsPath);
	readParam(rcfp, "PIPEPATH", 'S', 0, MAXLEN_PATH, global->pipePath);
}

/* Function for reading in the ISP sections from the .xisprc file */

static void inputAllXisprc(xisprc_t *p, FILE *rcfp, glob_t *global)
{
	int i, j, n;
	char buf[2*MAXLEN_PASSWD+1], pname[32] = {0}, *line;
	unsigned char *ipp;

	for (i=0; i<global->numISPs; i++, p++) {
		memset(p, 0, sizeof(xisprc_t));
		rcISP(rcfp, i);
		readParam(rcfp, "DESCR", 'S', 0, MAXLEN_DESCR, p->descr);
		readParam(rcfp, "ACCOUNT", 'S', 0, MAXLEN_ACCOUNT, p->account);
		if (! *(p->account))
			n = 0;
		else
			n = 2*MAXLEN_PASSWD;
		readParam(rcfp, "PASSWD", 'S', n, n, buf);
		memcpy(p->passwd, ascii2passwd(buf), MAXLEN_PASSWD);
		readParam(rcfp, "NAME", 'S', 0, MAXLEN_UNR, p->name);
		readParam(rcfp, "REMOTENAME", 'S', 0, MAXLEN_UNR, p->rname);
		readParam(rcfp, "MAXATTEMPTS", 'B', 0, 255, &(p->maxAttempts));
		readParam(rcfp, "SLEEPDELAY", 'B', 0, 255, &(p->sleepDelay));
		readParam(rcfp, "CONNECTWAIT", 'B', 0, 255, &(p->connectWait));
		readParam(rcfp, "NUMPHONES", 'B', 0, MAXNUM_TELS, &(p->numPhones));
		for (j=0; j<p->numPhones; j++) {
			sprintf(pname, " PHONE%d", j);
			readParam(rcfp, pname, 'S', 0, MAXLEN_PHONE, p->phone[j]);
		}
		readParam(rcfp, "NUMSLINES", 'B', 0, MAXNUM_SLINES, &(p->numSlines));
		for (j=0; j<p->numSlines; j++) {
			sprintf(pname, " SLINE%dE", j);
			readParam(rcfp, pname, 'S', 0, MAXLEN_SLINE, p->sline[j]);
			sprintf(pname, " SLINE%dS", j);
			readParam(rcfp, pname, 'S', 0, MAXLEN_SLINE,
					  p->sline[MAXNUM_SLINES+j]);
		}
		readParam(rcfp, "CBDELAY", 'B', 0, 255, &(p->CBDelay));
		readParam(rcfp, "NUMCBSLINES", 'B', 0, MAXNUM_SLINES, &(p->numCBSlns));
		for (j=0; j<p->numCBSlns; j++) {
			sprintf(pname, " CBSLINE%dE", j);
			readParam(rcfp, pname, 'S', 0, MAXLEN_SLINE, p->CBsln[j]);
			sprintf(pname, " CBSLINE%dS", j);
			readParam(rcfp, pname, 'S', 0, MAXLEN_SLINE,
					  p->CBsln[MAXNUM_SLINES+j]);
		}
		readParam(rcfp, "TERMW", 'B', MINCHAR_TERMW, 255, &(p->termW));
		readParam(rcfp, "TERMH", 'B', MINCHAR_TERMH, 255, &(p->termH));
		readParam(rcfp, "CBTERMW", 'B', MINCHAR_TERMW, 255, &(p->CBtermW));
		readParam(rcfp, "CBTERMH", 'B', MINCHAR_TERMH, 255, &(p->CBtermH));
		readParam(rcfp, "CBPHONE", 'S', 0, MAXLEN_PHONE, p->CBphone);
		readParam(rcfp, "MODEMDEVICE", 'S', 0, MAXLEN_DEVICE, p->modemDevice);
		line = readParam(rcfp, "MODEMSPEED", 'S', 4, 6, buf);
		if (! (n=str2speed(buf)))
			rcLineError("inputAllXisprc", line);
		p->modemSpeed = n;
		readParam(rcfp, "MODEMRESET", 'S', 0, MAXLEN_MDMCMD, p->modemReset);
		readParam(rcfp, "MODEMINIT", 'S', 0, MAXLEN_MDMCMD, p->modemInit);
		readParam(rcfp, "MODEMCONNECT", 'S',0, MAXLEN_MDMSTR, p->modemConnect);
		readParam(rcfp, "OPEROPTS", 'L', 0, 0, &(p->operOpts));
		readParam(rcfp, "DIALEXTRA", 'S', 0, MAXLEN_DIALEXTRA, p->dialExtra);
		readParam(rcfp, "COMPLEVEL", 'B', 9, 15, &(p->compLevel));
		readParam(rcfp, "ASYNCMAP", 'S', 2, MAXDIG_ASYNCMAP, p->asyncmap);
		readParam(rcfp, "ESCAPE", 'S', 0, MAXLEN_ESCAPE, p->escape);
		line = readParam(rcfp, "LOCALIP", 'S', 7, MAXLEN_IP, buf);
		if ((ipp=str2IP(buf)) == NULL)
			rcLineError("inputAllXisprc", line);
		memcpy(p->localIP, ipp, 4);
		line = readParam(rcfp, "REMOTEIP", 'S', 7, MAXLEN_IP, buf);
		if ((ipp=str2IP(buf)) == NULL)
			rcLineError("inputAllXisprc", line);
		memcpy(p->remoteIP, ipp, 4);
		line = readParam(rcfp, "NETMASK", 'S', 7, MAXLEN_IP, buf);
		if ((ipp=str2IP(buf)) == NULL)
			rcLineError("inputAllXisprc", line);
		memcpy(p->netmask, ipp, 4);
		line = readParam(rcfp, "DNS1", 'S', 7, MAXLEN_IP, buf);
		if ((ipp=str2IP(buf)) == NULL)
			rcLineError("inputAllXisprc", line);
		memcpy(p->dns1, ipp, 4);
		line = readParam(rcfp, "DNS2", 'S', 7, MAXLEN_IP, buf);
		if ((ipp=str2IP(buf)) == NULL)
			rcLineError("inputAllXisprc", line);
		memcpy(p->dns2, ipp, 4);
		readParam(rcfp, "MTU", 'I', 128, 2048, &(p->mtu));
		readParam(rcfp, "MRU", 'I', 128, 2048, &(p->mru));
	}
}

void RCSq(void)
{
	char *rcsp;
	rcsp = RCSid;
	rcsp = PatchLevel;
}

/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                           Interface routines                            |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

/* Initializes the xisprc data structure. */

void initXisprc(xisprc_t *p)
{
	int i;

	p->descr[0] = 0;
	p->account[0] = 0;
	p->passwd[0] = 0;
	p->name[0] = 0;
	p->rname[0] = 0;
	p->maxAttempts = MAXNUM_RETRY;
	p->sleepDelay = MAXSEC_DELAY;
	p->connectWait = MAXSEC_CNWAIT;
	p->numPhones = 0;
	for (i=0; i<MAXNUM_TELS; i++)
		p->phone[i][0] = 0;
	p->numSlines = p->numCBSlns = 0;
	for (i=0; i<(2*MAXNUM_SLINES); i++)
		p->sline[i][0] = p->CBsln[i][0] = 0;
	p->CBDelay = MAXSEC_CBDELAY;
	p->termW = TERMW;
	p->termH = TERMH;
	p->CBtermW = TERMW;
	p->CBtermH = TERMH;
	p->CBphone[0] = 0;
	strcpy(p->modemDevice, MODEM_DEVICE);
	p->modemSpeed = MODEM_SPEED;
	strcpy(p->modemReset, MODEM_RESET);
	strcpy(p->modemInit, MODEM_INIT);
	strcpy(p->modemConnect, MODEM_CONNECT);
	p->operOpts = OPER_OPTS;
	strcpy(p->dialExtra, DIAL_EXTRA);
	p->compLevel = COMP_LEVEL;
	strcpy(p->asyncmap, PPPD_HASYNCMAP);
	strcpy(p->escape, PPPD_ESCAPE);
	memcpy(p->localIP, LOCAL_IP, 4);
	memcpy(p->remoteIP, REMOTE_IP, 4);
	memcpy(p->netmask, NETMASK, 4);
	memcpy(p->dns1, DNS, 4);
	memcpy(p->dns2, DNS, 4);
	p->mtu = MTU;
	p->mru = MRU;
}

/* Read all xisprc data records from the user file, or create
   a new file if it's the first time the program is run. */

void readXisprc(char *rcfname, xisprc_t **rcdataptr, glob_t *global)
{
	FILE *rcfp;
	extern void outofMem();

	rcfn = rcfname;
	rcfp = fopen(rcfn, "r");
	if (rcfp != NULL) {
		if (strcmp(rcVersion(rcfp), Version)) {
			fprintf(stderr, "xISP: %s is not a v%s file!\n", rcfn, Version);
			fprintf(stderr, "xISP: Please run xisprccv with no arguments.\n");
			exit(1);
		}
		*global = reset_global;
		inputGlobals(rcfp, global);
		*rcdataptr = (xisprc_t *)malloc((1+global->numISPs)*sizeof(xisprc_t));
		if (*rcdataptr == NULL)
			outofMem();
		initXisprc(*rcdataptr);	/* initialize default record */
		inputAllXisprc(&((*rcdataptr)[1]), rcfp, global);
		fclose(rcfp);
	}
	else {
		*global = reset_global;
		*rcdataptr = (xisprc_t *)malloc((1+global->numISPs)*sizeof(xisprc_t));
		if (*rcdataptr == NULL)
			outofMem();
		initXisprc(*rcdataptr);	/* initialize default record */
		rcfp = fopen(rcfn, "w");
		if (rcfp == NULL)
			doErr("readAllXisprc: creating: fopen");
		outputAllXisprc(NULL, rcfp, global);
		fclose(rcfp);
	}
}

/* Write all xisprc data records to the user file. */

void writeXisprc(char *rcfname, xisprc_t *rcdata, glob_t *global)
{
	FILE *rcfp;

	rcfn = rcfname;
	rcfp = fopen(rcfn, "w");
	if (rcfp != NULL) {
		outputAllXisprc(&rcdata[1], rcfp, global);
		fclose(rcfp);
	}
	else
		doErr("writeXisprc: fopen");
}

