/* ------------------------------------------------------------------------
 * $Id: client.cc,v 1.5 2001/08/22 12:23:18 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-07-31 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- System Includes
#include <time.h>
#include <iostream>
#include <unistd.h>

// -- 3Dwm Includes
#include <Nobel++/Nobel.hh>
#include <Nobel/Solid.hh>
#include <Nobel/SolidKit.hh>
#include <Nobel/SolidContainer.hh>
#include <Nobel/Appearance.hh>
#include <Nobel/GeometryKit.hh>
#include <Nobel/PrimitiveKit.hh>

using namespace Nobel;

// -- Class Declarations

class Clock {
public:
    
    /**
     * Constructor.
     **/
    Clock(int &argc, char **argv) : _client("Clock", argc, argv) {
	
	// Resolve the necessary kits
	_geo_kit = _client.resolve<GeometryKit>(Nobelxx::name<GeometryKit>());
	_solid_kit = _client.resolve<SolidKit>(Nobelxx::name<SolidKit>());
	_prim_kit = _client.resolve<PrimitiveKit>
	    (Nobelxx::name<PrimitiveKit>());
	
	// Build the clock geometry
	buildClock(Node_var(_client.root()));
    }
    
    /**
     * Destructor.
     **/
    ~Clock() {
	// empty
    }
    
    void run() {

	while (true) {
	    
	    // Retrieve a time value
	    time_t abstime = time(0);
	    tm *t = localtime(&abstime);

	    // Update the clock hand positions
	    setHands(t->tm_hour, t->tm_min, t->tm_sec);
	    
	    // Sleep for a while
	    sleep(5);
	}
    }

private:

    void buildClock(Node_ptr root) {
	MaterialAttributes material;

	// Create some primitives to use
	Primitive_var cylinder = _prim_kit->createCylinder(32);
	Primitive_var box = _prim_kit->createCuboid();
	
	// Create the clock face
	Appearance_var app1 = _geo_kit->createAppearance();
	Solid::Primitive_var face = _solid_kit->createPrimitive(cylinder);
	Solid::Appearance_var a_face = _solid_kit->createAppearance(app1);
	Solid::Transform_var t_face = _solid_kit->createTransform(a_face);
	
	t_face->scale(Nobelxx::Vertex3D(10.0, 1.0, 10.0));
	t_face->rotate(90, Nobelxx::xaxis);

	material.diffuse = Nobelxx::Color(0.3, 0.41, 0.51);
	app1->material(material);
	a_face->body(face);

	// Create the clock hands
	Appearance_var app2 = _geo_kit->createAppearance();
	Solid::Primitive_var hand = _solid_kit->createPrimitive(box);
	Solid::Appearance_var a_hand = _solid_kit->createAppearance(app2);
	Solid::Transform_var m_hand = _solid_kit->createTransform(a_hand);
	Solid::Transform_var h_hand = _solid_kit->createTransform(a_hand);
	_minutes = _solid_kit->createTransform(m_hand);
	_hours = _solid_kit->createTransform(h_hand);
	
	m_hand->scale(Nobelxx::Vertex3D(0.5, 8, 0.1));
	m_hand->translate(Nobelxx::Vertex3D(0, 3, 1.3));

	h_hand->scale(Nobelxx::Vertex3D(0.75, 4, 0.1));
	h_hand->translate(Nobelxx::Vertex3D(0, 1, 1.3));

	material.diffuse = Nobelxx::Color(0.8, 0.8, 0.8);
	app2->material(material);
	a_hand->body(hand);

	// Create an appropriate transform
	Solid::Transform_var tick = _solid_kit->createTransform(a_hand);
	tick->scale(Nobelxx::Vertex3D(0.4, 1, 0.2));
	tick->translate(Nobelxx::Vertex3D(0, 9, 1.1));

	// Now, combine it with the face
	Solid::Binary_var clock = _solid_kit->createUnion(tick, t_face);

	// Create the rest of the face
	for (int i = 1; i < 12; i++) {
	    
	    // Create an appropriate transform
	    tick = _solid_kit->createTransform(hand);

	    // Set the scale correctly
	    tick->scale(Nobelxx::Vertex3D(0.2, 0.5, 0.2));
	    tick->translate(Nobelxx::Vertex3D(0, 9, 1.1));
	    tick->rotate(360.0f * i / 12.0f, Nobelxx::zaxis);
	    
	    // Now, combine it with the face
	    clock = _solid_kit->createUnion(tick, clock);
	}

	// Combine the hands and the face
	Solid::Binary_var hands = _solid_kit->createUnion(_minutes, _hours);
	clock = _solid_kit->createUnion(hands, clock);
	
	// Create solid container to insert into the scene graph
	_node = _solid_kit->createContainer();

	// Set CSG trees into the nodes
	_node->setTree(clock);

	// Finally, insert them into the scene graph
	root->insert(_node);
    }
    
    void setHands(int hours, int minutes, int seconds) {
	
	// Compute values for the hours and minutes
	int min_in_sec = minutes * 60 + seconds;
	int hour_in_sec = hours * 60 * 60 + minutes * 60 + seconds;
	
	// Compute angles for the clock hands
	float min_angle = 360.0f * min_in_sec / (60.0f * 60.0f);
	float hour_angle = 360.0 * hour_in_sec / (12.0 * 60.0 * 60.0f);
	
	// Set transforms
	_minutes->identity();
	_hours->identity();
	_minutes->rotate(-min_angle, Nobelxx::zaxis);
	_hours->rotate(-hour_angle, Nobelxx::zaxis);
	
	// Re-evaluate the solid node
	_node->evaluate();
    }
    
    Nobelxx::Client _client;
    SolidKit_var _solid_kit;
    PrimitiveKit_var _prim_kit;
    GeometryKit_var _geo_kit;
    Solid::Transform_var _minutes, _hours;
    SolidContainer_var _node;
};

// -- Code Segment

int main(int argc, char *argv[]) {
    
    try {
	// Initialize application
	Clock clock(argc, argv);
	
	// Run it! (Never terminates)
	clock.run();
    }
    catch (...) {
	return EXIT_FAILURE;
    }
    
    return EXIT_SUCCESS;
}
