/* ------------------------------------------------------------------------
 * $Id: Matrix3D.hh,v 1.7 2001/07/30 15:18:55 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-08-11 by Niklas Elmqvist.
 *
 * Copyright (c) 2000, 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _Matrix3D_hh_
#define _Matrix3D_hh_

// -- System Includes
#include <iostream>

// -- Local Includes
#include "Math.hh"

// -- Class Declarations

/**
 * 4x4 3D graphics matrix class. This is _not_ a general matrix class!
 * Many of the operations (notably matrix inversion) here are specific
 * to 3D transformation matrices. For general, MxN-dimensional matrices,
 * we should have a look at MTL, the Matrix Template Library, at
 * <http://lsc.nd.edu/research/mtl/>.
 **/
class Matrix3D {

public:
    
    /**
     * Constructor.
     **/
    Matrix3D() { clear(); }
    
    /**
     * Constructor.
     **/
    Matrix3D(const float m[4][4]) { load(m); }

    /**
     * Destructor.
     **/
    ~Matrix3D() { }
    
    /**
     * Retrieve the number of rows.
     *
     * @return the number of rows.
     **/
    int rows() const { return _dim; }
    
    /**
     * Retrieve the number of columns.
     *
     * @return the number of columns.
     **/
    int columns() const { return _dim; }
    
    /**
     * Clear the matrix.
     *
     * @param f the value to fill in the matrix with.
     **/
    void clear(float f = 0);
    
    /**
     * Build the identity matrix.
     **/
    void identity();

    /**
     * Transpose the matrix.
     **/
    void transpose();

    /**
     * Invert the matrix. Note that this is NOT a general matrix
     * inversion operation! Using this operation for non-3D-like
     * transformation matrices will produce erratic results (notably,
     * the 4th row will always be [0 0 0 1]).
     **/
    void invert();

    /**
     * Compute the matrix inverse and return it.
     * 
     * @return inverse matrix.
     **/
    Matrix3D inverse() const;

    /**
     * Compute the matrix determinant.
     **/
    float determinant() const;
    
    /**
     * Element access operator.
     *
     * @param r the row number.
     * @param c the column number.
     * @return the value.
     **/
    inline float &operator () (int r, int c) {
	return val(r, c);
    }
    
    /**
     * Element access operator (constant).
     *
     * @param r the row number.
     * @param c the column number.
     * @return the value.
     **/
    inline float operator () (int r, int c) const {
	return val(r, c);
    }
    
    /**
     * Matrix multiplication operator.
     *
     * @param m the matrix to multiply with. 
     * @return the resulting matrix.
     **/
    Matrix3D operator * (const Matrix3D &m) const;
    
    /**
     * Matrix addition operator.
     *
     * @param m the matrix to add with. 
     * @return the resulting matrix.
     **/
    Matrix3D operator + (const Matrix3D &m) const;

    /**
     * Matrix subtraction operator.
     *
     * @param m the matrix to subtract with.
     * @return the resulting matrix.
     **/
    Matrix3D operator - (const Matrix3D &m) const;

    /**
     * Scalar multiplication operator.
     * 
     * @param f the scalar.
     * @return the resulting matrix.
     **/
    Matrix3D operator * (float f) const;

    /**
     * Scalar multiplication operator. (friend function)
     * 
     * @param f the scalar.
     * @param m the matrix.
     * @return the resulting matrix.
     **/
    friend Matrix3D operator * (float f, const Matrix3D &m) {
	return m * f;
    }

    /**
     * Scalar addition operator.
     * 
     * @param f the scalar.
     * @return the resulting matrix.
     **/
    Matrix3D operator + (float f) const;

    /**
     * Scalar subtraction operator.
     * 
     * @param f the scalar.
     * @return the resulting matrix.
     **/
    Matrix3D operator - (float f) const;

    /**
     * Scalar subtraction operator.
     * 
     * @param f the scalar.
     * @return the resulting matrix.
     **/
    Matrix3D operator / (float f) const;
    
    /**
     * Load the matrix with external data.
     *
     * @param m a 4x4 matrix of float values to load.
     **/
    void load(const float m[4][4]) {
	for (int i = 0; i < _dim; i++) 
	    for (int j = 0; j < _dim; j++) 
		val(i, j) = m[i][j];
    }
    
    /**
     * Store internal matrix data to an array.
     * 
     * @param m a 4x4 matrix of float values to store to.
     **/
    void store(float m[4][4]) const {
	for (int i = 0; i < _dim; i++) 
	    for (int j = 0; j < _dim; j++) 
		m[i][j] = val(i, j);
    }

    /**
     * Retrieve the matrix data as a raw array.
     * 
     * @return const pointer to array of floats representing matrix.
     **/
    const float *data() const {
	return (float *) _data;
    }
    
    /**
     * Retrieve matrix dimension (always 4).
     *
     * @return matrix dimension.
     **/
    static int dim() {
	return _dim; 
    }

    /**
     * Matrix output operator.
     **/
    friend std::ostream &operator << (std::ostream &f, const Matrix3D &v);
    
private:
    
    /// Element access utility function
    inline float &val(int i, int j) { return _data[i * _dim + j]; }

    /// Element access utility function (constant)
    inline float val(int i, int j) const { return _data[i * _dim + j]; }

    /// Element access utility function (general array)
    inline float &val(float *f, int i, int j) { return f[i * _dim + j]; }

    /// Matrix dimension
    static const int _dim = 4;

    /// Matrix data
    float _data[_dim * _dim];
};
    
/**
 * Matrix equality operator (uses an epsilon precision tolerance).
 *
 * @param m the matrix to compare with.
 * @return true if the matrices are equal. 
 **/
inline bool operator == (const Matrix3D &a, const Matrix3D &b)
{
    // @@@Use epsilon tolerance here!
    for (int i = 0; i < Matrix3D::dim(); i++) 
	for (int j = 0; j < Matrix3D::dim(); j++) 
	    if (Math::equal(a(i, j), b(i, j)) == false)
		return false;
    return true;
}

/**
 * Matrix inequality operator (uses an epsilon precision tolerance).
 *
 * @param m the matrix to compare with.
 * @return false if the matrices are equal. 
 **/
inline bool operator != (const Matrix3D &a, const Matrix3D &b)
{
    return !(a == b);
}

#endif /* Matrix3D.hh */
