/* ------------------------------------------------------------------------
 * $Id: SolidBuilder.hh,v 1.1 2001/08/01 15:09:00 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-08-01 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _SolidBuilder_hh_
#define _SolidBuilder_hh_

// -- System Includes
#include <vector>

// -- 3Dwm Includes
#include "Celsius/Vector2D.hh"
#include "Celsius/Vector3D.hh"
#include "Solid/Face.hh"

// -- Struct Declarations

/**
 * 3D solid representation struct.
 **/
struct SolidMesh {
    std::vector<TriangleFace> faces;
    std::vector<Vector3D> vertices;
    std::vector<Vector3D> normals;
    std::vector<Vector2D> texCoords;
};

// -- Class Declarations

/**
 * SolidMesh building class. Static class for constructing solid 3D
 * primitives of various kinds, including spheres, cylinders, cones,
 * cuboids, tori, etc. All solids are unit-sized, i.e. with 
 **/
class SolidBuilder {
public:

    /**
     * Build a 3D cube solid. By default, the cube will be
     * unit-sized.
     *
     * @return cube solid.
     **/
    static SolidMesh *buildCuboid();

    /**
     * Build a 3D plane. By default, the plane will be unit-sized and
     * aligned with the XZ plane.
     *
     * @return plane solid.
     **/
    static SolidMesh *buildPlane();
    
    /**
     * Build a 3D cylinder solid. By default, the cylinder will be
     * unit-sized and with its top and bottom aligned with the XZ
     * plane.
     *
     * @param segments number of segments in solid.
     * @return cylinder solid.
     **/
    static SolidMesh *buildCylinder(int segments);
    
    /**
     * Build a 3D sphere solid. By default, the sphere will be
     * unit-sized.
     *
     * @param latSegments number of latitudinal segments in solid.
     * @param longSegments number of longitudinal segments in solid.
     * @return sphere solid.
     **/
    static SolidMesh *buildSphere(int latSegments, int longSegments);

    /**
     * Build a 3D cone solid. By default, the cone will be unit-sized
     * and its bottom will be aligned with the XZ plane. This method
     * can also be used to generate pyramids by varying the segment
     * parameter.
     *
     * @param segments number of segments in solid.
     * @return cone solid.
     **/
    static SolidMesh *buildCone(int segments);

    /**
     * Build a 3D torus solid. By default, the torus will be
     * unit-sized and aligned with the XZ plane. 
     *
     * @param slices number of circular segments (slices) in solid.
     * @param segments number of segments in the "tube" of the solid. 
     * @return torus solid.
     **/
    static SolidMesh *buildTorus(int slices, int segments);

private:

    /**
     * Build a solid mesh given geometry data.
     **/
    static SolidMesh *buildSolidMesh(const std::vector<float> &v_data,
				     const std::vector<float> &tc_data,
				     const std::vector<int> &vi_data,
				     const std::vector<int> &tci_data);
};

#endif /* SolidBuilder.hh */
