/* ------------------------------------------------------------------------
 * $Id: BSPTree.cc,v 1.2 2001/08/02 14:22:11 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-07-30 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- 3Dwm Includes
#include "Solid/BSP.hh"
#include "Solid/BSPTree.hh"

// Undefine the following #def to disable vertex sharing and speed up
// BSP generation (with the result that there will be no attempts at
// vertex sharing when generating new vertices).
//#define SHARE_GEOMETRY_DATA

// -- Code Segment

int BSPTree::addVertex(const Vector3D &vertex)
{
#ifdef SHARE_GEOMETRY_DATA
    
    // Look for the vertex in the existing vertex list
    unsigned int index = 0; 
    for (std::vector<Vector3D>::iterator i = _vertices.begin();
	 i != _vertices.end(); i++, index++) {
	if (*i == vertex) break;
    }

    // If we didn't find it, just add it to the end of the list
    if (index == _vertices.size())
	_vertices.push_back(vertex);
    return index;
#else
    
    // Simply add the vertex to the end of the list
    _vertices.push_back(vertex);
    return _vertices.size() - 1;
#endif
}

int BSPTree::addTexCoord(const Vector2D &texCoord) 
{
#ifdef SHARE_GEOMETRY_DATA

    // Look for the vertex in the existing vertex list
    unsigned int index = 0; 
    for (std::vector<Vector2D>::iterator i = _texCoords.begin();
	 i != _texCoords.end(); i++, index++) {
	if (*i == texCoord) break;
    }
    
    // If we didn't find it, just add it to the end of the list
    if (index == _texCoords.size())
	_texCoords.push_back(texCoord);
    return index;
#else
    
    // Simply add the vertex to the end of the list
    _texCoords.push_back(texCoord);
    return _texCoords.size() - 1;
#endif
}

int BSPTree::addNormal(const Vector3D &normal)
{
#ifdef SHARE_GEOMETRY_DATA

    // Look for the vertex in the existing vertex list
    unsigned int index = 0; 
    for (std::vector<Vector3D>::iterator i = _normals.begin();
	 i != _normals.end(); i++, index++) {
	if (*i == normal) break;
    }

    // If we didn't find it, just add it to the end of the list
    if (index == _normals.size())
	_normals.push_back(normal);
    return index;
#else
    
    // Simply add the vertex to the end of the list
    _normals.push_back(normal);
    return _normals.size() - 1;
#endif
}

void BSPTree::complement() 
{    
    // Complement the actual BSP tree
    _tree->complement();
    
    // Now, flip the normals
    for (std::vector<Vector3D>::iterator i = _normals.begin(); 
	 i != _normals.end(); i++) {
	*i = - *i;
    }
}

void BSPTree::transform(const Matrix3D &trafo)
{
    // Apply the transformation to all vertices
    for (std::vector<Vector3D>::iterator i = _vertices.begin();
	 i != _vertices.end(); i++) {
	*i = *i * trafo;
    }

    // Invert and transpose the transformation for applying to
    // normals. (See Turkowski, Ken, "Properties of Surface-Normal
    // Transformations", in Andrew Glassner (editor), Graphics Gems,
    // Academic Press, Inc., pp. 539-547,
    // 1990. http://www.worldserver.com/turk/computergraphics/index.html)
    Matrix3D inv_trafo = trafo.inverse();
    inv_trafo.transpose();
    
    // Now, apply transformation to all normals as well
    for (std::vector<Vector3D>::iterator i = _normals.begin();
	 i != _normals.end(); i++) {
	
	// Make sure the w() component is zero so that the normal is
	// only affected by rotations and not translations. 
	i->w() = 0.0f;
	
	// Then multiply!
	*i = *i * inv_trafo;
	
	// ...and renormalize to unit length
	i->normalize();
    }
    
    // Finally, we need to transform the planes of the BSP tree itself
    _tree->transform(inv_trafo);
}
