/*--------------------------------------------------------------------*/
/*--- Callgrind                                                    ---*/
/*---                                                   ct_jumps.c ---*/
/*--------------------------------------------------------------------*/

/*
   This file is part of Callgrind, a Valgrind tool for call tracing.

   Copyright (C) 2002-2004, Josef Weidendorfer (Josef.Weidendorfer@gmx.de)

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/

#include "global.h"

#define N_JCC_INITIAL_ENTRIES  4437

/*------------------------------------------------------------*/
/*--- Jump Cost Center (JCC) operations, including Calls   ---*/
/*------------------------------------------------------------*/

#define N_JCC_INITIAL_ENTRIES  4437

jcc_hash current_jccs;

void SK_(init_jcc_hash)(jcc_hash* jccs)
{
   Int i;

   CT_ASSERT(jccs != 0);

   jccs->size    = N_JCC_INITIAL_ENTRIES;
   jccs->entries = 0;
   jccs->table = (jCC**) VG_(malloc)(jccs->size * sizeof(jCC*));
   jccs->spontaneous = 0;

   for (i = 0; i < jccs->size; i++)
     jccs->table[i] = 0;
}


void SK_(copy_current_jcc_hash)(jcc_hash* dst)
{
  CT_ASSERT(dst != 0);

  dst->size        = current_jccs.size;
  dst->entries     = current_jccs.entries;
  dst->table       = current_jccs.table;
  dst->spontaneous = current_jccs.spontaneous;
}

void SK_(set_current_jcc_hash)(jcc_hash* h)
{
  CT_ASSERT(h != 0);

  current_jccs.size        = h->size;
  current_jccs.entries     = h->entries;
  current_jccs.table       = h->table;
  current_jccs.spontaneous = h->spontaneous;
}

__inline__
static UInt jcc_hash_val(BBCC* from, BBCC* to, UInt size)
{
    return ( (UInt)from + 7* (UInt)to ) % size;
} 

/* double size of jcc table  */
static void resize_jcc_table()
{
    Int i, new_size, conflicts1 = 0, conflicts2 = 0;
    jCC** new_table;
    UInt new_hash;
    jCC *curr_jcc, *next_jcc;

    new_size  = 2* current_jccs.size +3;
    new_table = (jCC**) VG_(malloc)(new_size * sizeof(jCC*));
 
    if (!new_table) return;
 
    for (i = 0; i < new_size; i++)
      new_table[i] = NULL;
 
    for (i = 0; i < current_jccs.size; i++) {
	if (current_jccs.table[i] == NULL) continue;
 
	curr_jcc = current_jccs.table[i];
	while (NULL != curr_jcc) {
	    next_jcc = curr_jcc->next_hash;

	    new_hash = jcc_hash_val(curr_jcc->from, 
				    curr_jcc->to, new_size);

	    curr_jcc->next_hash = new_table[new_hash];
	    new_table[new_hash] = curr_jcc;
	    if (curr_jcc->next_hash) {
		conflicts1++;
		if (curr_jcc->next_hash->next_hash)
		    conflicts2++;
	    }

	    curr_jcc = next_jcc;
	}
    }

    VG_(free)(current_jccs.table);


    CT_DEBUG(0, "Resize JCC Hash: %d => %d (entries %d, conflicts %d/%d)\n",
	     current_jccs.size, new_size,
	     current_jccs.entries, conflicts1, conflicts2);

    current_jccs.size  = new_size;
    current_jccs.table = new_table;
    SK_(stat).jcc_hash_resizes++;
}



/* new jCC structure: a call was done to a BB of a BBCC 
 * for a spontaneous call, from is 0 (i.e. caller unknown)
 */
static jCC* new_jcc(BBCC* from, BBCC* to)
{
   jCC* new;
   UInt new_hash;

   /* check fill degree of jcc hash table and resize if needed (>80%) */
   current_jccs.entries++;
   if (10 * current_jccs.entries / current_jccs.size > 8)
       resize_jcc_table();

   new = VG_(malloc)(sizeof(jCC));

   new->from      = from;
   new->to        = to;
   new->jmpkind   = JmpCall;
   new->call_counter = 0;
   new->cost = 0;

   /* insert into JCC chain of calling BBCC.
    * This list is only used at dumping time */

   if (from) {
       new->next_from = from->jcc_list;
       from->jcc_list = new;
   }
   else {
       new->next_from = current_jccs.spontaneous;
       current_jccs.spontaneous = new;
   }

   /* insert into JCC hash table */
   new_hash = jcc_hash_val(from, to, current_jccs.size);
   new->next_hash = current_jccs.table[new_hash];
   current_jccs.table[new_hash] = new;

   SK_(stat).distinct_jccs++;

   CT_DEBUGIF(3) {
     VG_(printf)("  new_jcc (now %d): 0x%x\n",
		 SK_(stat).distinct_jccs, new);
   }

   return new;
}


/* get the jCC for a call arc (BBCC->BBCC) */
jCC* SK_(get_jcc)(BBCC* from, BBCC* to)
{
    jCC* jcc;
    Int hash;

    CT_DEBUG(5, "+ get_jcc(bbcc %p => bbcc %p)\n",
		from, to);

    /* first check last recently used JCC */
    jcc = to->lru_to_jcc;
    if (jcc && (jcc->from == from)) {
	CT_ASSERT(to == jcc->to);
	CT_DEBUG(5,"- get_jcc: [LRU to] jcc %p\n", jcc);
	return jcc;
    }

    jcc = from->lru_from_jcc;
    if (jcc && (jcc->to == to)) {
	CT_ASSERT(from == jcc->from);
	CT_DEBUG(5, "- get_jcc: [LRU from] jcc %p\n", jcc);
	return jcc;
    }

    SK_(stat).jcc_lru_misses++;

    hash = jcc_hash_val(from, to, current_jccs.size);
    jcc = current_jccs.table[hash];

    while(jcc) {
	if ((jcc->from == from) && 
	    (jcc->to == to)) break;
	jcc = jcc->next_hash;
    }

    if (!jcc)
	jcc = new_jcc(from, to);

    /* set LRU */
    from->lru_from_jcc = jcc;
    to->lru_to_jcc = jcc;

    CT_DEBUG(5, "- get_jcc(bbcc %p => bbcc %p)\n",
		from, to);

    return jcc;
}

