/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2004 The Caudium Group
 * Based on camas_main  Stefan Wallstrm and Bosse Lincoln.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas_actions.pike,v 1.148.2.10 2004/05/10 15:42:19 vida Exp $
 */

#include <module.h>
inherit "module";
inherit "caudiumlib";

#include <camas/globals.h>      // Global definitions
#include <camas/utf.h>          // UTF8 conversion macros
#include <camas/language.h>     // Language sym names M_*
#include <camas/msg.h>          // MSG() macros
#include <camas/smtp.h>         // SMTP client
#include <camas/pmods.h>        // Local or normal camas pike modules
#include <camas/addressbook.h>  // Address book defines

constant cvs_version   = "$Id: camas_actions.pike,v 1.148.2.10 2004/05/10 15:42:19 vida Exp $";
constant module_type   = MODULE_PROVIDER;
constant module_name   = "CAMAS: Actions";
constant module_doc    = "This module handles the so called 'actions' in Camas. Actions are "
                         "the actions taken by Camas each time a user make a query. Examples "
                         "of actions are login to IMAP server, launching the spellchecker "
                         "when a user press the spellcheck button,...<br />"
                         "You can have several Actions module, the internal priority of these "
                         "modules can be used to specify the order in which the actions are "
                         "executed in these modules. See the Actions: NULL module to see how to"
                         " create Camas actions modules."
			 "<br /><b>This module is automatically "
			 "selected if you select \"CAMAS: Main module\".</b>";
constant thread_safe   = 1;
constant module_unique = 1;

array(string) query_provides()
{
  return ({ "camas_actions" });
}

void create()
{
#ifdef CAMAS_DEBUG
   defvar("debug", 0, "Debug", TYPE_FLAG,
     "When on, debug messages will be logged in Caudium's debug logfile. "
     "This information is very useful to the developers when fixing bugs.");
#endif
}

int camas_feature (int f) {
  object features = my_configuration ()->get_provider ("camas_features");
  if (objectp (features))
    return features->feature (f);
}

string status()
{
  string out = "<table border=\"1\"><tr><th>Actions provided by this module</th></tr>";
  array(string) actions = sort(indices(Actions()));
  foreach(actions, string action)
    out += "<tr><td>" + action + "</td></tr>";
  out += "</table>";
  return out;
}

//! class: Actions
//!  The class that contains actions methods
//
class Actions {

  private object features;
  private array imap_commands = ({ });
  private object camas_main;
  private object sessobj;
  private object id;
  private object handler;
  
  void getselected()
  {
    sessobj->selected = ({ });
    sessobj->selectedfiles = ({ });
    sessobj->selectedattachments = ({ });
    
    array ind_variables = indices(id->variables);
    foreach (ind_variables, string var) {
      if (id->variables[var] == "1") {
        int n, nr, tonr = -1;
        if (n = sscanf (var, "msg%d-%d", nr, tonr)) {
          switch (n) {
          case 1: sessobj->selected += ({ nr }); break;
          case 2: sessobj->selected += ({ nr + ":" + tonr }); break;
          default: break;
          }
        }
        else {
          if (sscanf (var, "file%d", nr))
            sessobj->selectedfiles += ({ nr });
        }
      }

      if (var == "attachments") {
        foreach (id->variables[var] / "\0", string s)
	    if ((int)s >= 0)
	      sessobj->selectedattachments += ({ (int) s });
      }
    }
  }
  
  void create(object _id, object _handler)
  {
    if(_id)
    {
	      id = _id;
      	features = id->conf->get_provider("camas_features");
      	camas_main = id->conf->get_provider("camas_main");
        handler = _handler;
        sessobj = CSESSION;
      	CDEBUG(sprintf("replytoidx=%O\n", sessobj->replytouid));
    }
  }

  void destroy()
  {
    if(handler)
      handler->add_imap_commands(imap_commands);
  }

  // return 1 if successfully authentificated, 0 otherwise 
  int handle_auth_module(object auth_module, int is_last_call)
  {
    string|int auth_login;
    object camas_adminif = id->conf->get_provider("camas_adminif");

    if(!auth_module)
      report_fatal("No Auth modules not found, please install at least one of them\n");
    camas_main->QUERY(cadmif);
    if (sessobj->login != (camas_main->QUERY(cadmif)?camas_adminif->QUERY(adminlogin):"")) 
    {
      CDEBUG("user login: ");
      CDEBUG(sprintf ("checking for auth_error() ... %s found.\n",
		        (auth_module->auth_error) ? "" : "not"));
      CDEBUG("sessobj->login= " + sessobj->login + "\n");
   
      auth_login = auth_module->getlogin (sessobj->login);

      if (intp (auth_login)) { // error (:
        if(camas_main->QUERY(allowfailauth)) {
      	  CDEBUG("auth failed but not fatal\n");
          // auth fail but the admin don't want it to be a fatal error
          // so fill with default values
          // Avoid email addresses with foo@foo.com@defaultdomain.com
          if(!has_value(sessobj->login,"@"))
            sessobj->address = sessobj->login + "@" + camas_main->QUERY (smtpmaildomain);
          else 
            sessobj->address = sessobj->login;
        }
        else if (auth_module->auth_error) {
      	  CDEBUG("failed: [" + auth_module->auth_error (auth_login) + "] (" + auth_login + ")\n");
          sessobj->loginerror = auth_module->auth_error (auth_login);
          sessobj->login = "";
          sessobj->status = LOGINFAILED;
          id->misc->status = LOGINFAILED;
          return 0;
        }
      }
      else { // ok :)
        array|int auth_result = auth_module->getfullnames (auth_login);
        CDEBUG("ok: " + auth_login + "\n");
        CDEBUG(sprintf ("auth_result is: %O\n", auth_result));
        sessobj->login = id->variables->login = auth_login;
        if (!intp(auth_result)) {
          sessobj->name = String.capitalize (auth_result[0]) + " " + String.capitalize (auth_result[1]);
          sessobj->address = auth_result[2];
          CDEBUG("address = " + sessobj->address + "\n");
        }
      }
    }

    // Figure out imap-server and domainname
    // If user entered user@domain, use @domain instead of the chosen value in the dialog
    int version;
    version = auth_module->version();
    CDEBUG(sprintf("Auth module version : %d \n",version));
    if(version >= 2) {
      CDEBUG("Version is >= 2, getimapserver() function should be here... ");
      CDEBUG(sprintf("%s\n",(auth_module->getimapserver) ? "Yes!":"No"));
      if(!auth_module->getimapserver) version = 1;
    }
    // sets the IMAP server and the stuff if the user sucessfully authentificated
    // or if no IMAP server has been set and we are the last called
    if(version >= 2 && (!intp(auth_login) || (!sessobj->imapserver && is_last_call))) {
      // Now trying to find the imapserver
      mapping|int imapsrv = auth_module->getimapserver(sessobj->login);
      if(!imapsrv)
        report_fatal("Can't find IMAP server from auth module!");
      if(mappingp(imapsrv)) {
        array ind_imapsrv = indices(imapsrv);
        foreach(ind_imapsrv, string indice) {
          if(imapsrv[indice])
            sessobj[indice] = imapsrv[indice];
        }
      }
      if(imapsrv)
      {
        array(string) loginsplit = sessobj->login / "@";
        // don't login with email address
        if(sizeof(loginsplit) == 2 && !auth_module->getemaillogin())
          sessobj->login = loginsplit[0];
      }
    }
    return (!intp(auth_login));
  }

  void login()
  {
    if (sessobj->status > 0 || id->misc->camas->status > 0)
      return 0;
    sessobj->logintime = time ();
    sessobj->passwd = utf8_to_string(id->variables->passwd || ""); // FIX For Stupid 8bit passwords !
    sessobj->login = id->variables->login || "";
    if(id->method == "GET")
    {
      sessobj->passwd = HTTP_DECODE_STRING(sessobj->passwd) || "";
      sessobj->login = HTTP_DECODE_STRING(sessobj->login) || "";
    }
    // Finish to handle Camas Runtime AIF --Xavier
    object camas_adminif = id->conf->get_provider("camas_adminif");
   
    mapping(int:array(object)) auth_modules = CAMAS.Tools.get_providers_bypriority("camas_auth", id);
    int i = 0;
    int nb_modules = 0;
    array val_auth_modules = values(auth_modules);
    foreach(val_auth_modules, array(object) module)
      nb_modules += sizeof(module);
    array rev_sort_ind_auth_modules = reverse(sort(indices(auth_modules)));
    foreach(rev_sort_ind_auth_modules, int priority)
      {
    	foreach(auth_modules[priority], object module)
  	  {
  	    if(module && objectp(module))
	      {
      		CDEBUG(sprintf("checking auth module %O\n", module));
      		if(handle_auth_module(module, (++i == nb_modules)))
      		  break;
	      }
      }
    }

    CAMAS.Tools.init_temp_vars(id);

    CDEBUG("CAMAS: get imapserver : "+sessobj->imapserver+":"+sessobj->imapport +"\n");
    sessobj->status = LOGINFAILED;
    id->misc->camas->status = LOGINFAILED;
    // Finish to handle Camas RAIF --Xavier
    if (camas_main->QUERY(cadmif) && (sessobj->login == (camas_adminif->QUERY (adminlogin)||"")) 
        && crypt (sessobj->passwd, (camas_adminif->QUERY (adminpasswd)||""))) {
      sessobj->address="ADMINISTRATOR";
      sessobj->administrator = 1;
      sessobj->administrator_addressbook = 1;
      sessobj->status = ADMINMAIN;
      id->misc->camas->status = ADMINMAIN;
      return;
    }
    else {
      array lcoms = ({ });
     
      if(camas_main->QUERY(allowmailpathoverwrite))
        lcoms += ({ CAMAS.IMAPTools.imap_cmd("low_namespace") });

      lcoms += ({ CAMAS.IMAPTools.imap_cmd("low_list",
					   "mailpath", sessobj->mailpath,
					   "sharedpath", sessobj->sharedpath) });

      if (camas_feature (FEAT_MAILBOXES)) {
        array folders = ({ });
        if (sizeof(features->QUERY (newbiecreatefolder)))
          folders = features->QUERY (newbiecreatefolder)/",";
        if(features->QUERY(newbiecreatesentfolder))
          folders += ({ sessobj->sentfolder });
        if(features->QUERY(newbiecreatetrashfolder))
          folders += ({ sessobj->trashfolder });
        if(features->QUERY(newbiecreatedraftsfolder))
          folders += ({ sessobj->draftsfolder });
        if(features->QUERY(newbiecreateansweredfolder))
          folders += ({ sessobj->answeredfolder });
        foreach (folders, string folder) {
          string tocreate;
          tocreate = CAMAS.Tools.strip(folder);
          if(sizeof(tocreate))
            lcoms += ({
               	      CAMAS.IMAPTools.imap_cmd("create",
         	 		       "newmailbox", tocreate, "addmailpath", 1,
                     "createonlyifnotexists", 1)
        	  });
        }
      }

      // the sessobj->prefsbox string is used even if we use other things than 
      // IMAP preferences module. For example saving addressbook
      array(object) prefs_modules = id->conf->get_providers("camas_preferences");
      foreach(prefs_modules, object pref_module)
        if(pref_module->defaultprefsbox)
        {
          sessobj->baseprefsbox = pref_module->defaultprefsbox;
          sessobj->prefsbox = sessobj->mailpath + sessobj->baseprefsbox;
        }
      mapping(int:array(object)) preferences_modules = 
      	CAMAS.Tools.get_providers_bypriority("camas_preferences", id);
      array rev_sort_ind_preferences_modules = reverse(sort(indices(preferences_modules)));
      // try to login so that we reload the preferences
      // at each time the user try to login so that
      // we are sure to always have good preferences even if
      // the user type a wrong login/password
      sessobj->prefsloaded = 0;
      foreach(rev_sort_ind_preferences_modules, int priority)
      {
        foreach(preferences_modules[priority], object module)
          if(module && objectp(module) &&!sessobj->prefsloaded)
          {
            mixed res = module->get_preferences(id);
            if(arrayp(res))
              lcoms += res;
  	      }
      }

      lcoms += ({ 
          CAMAS.IMAPTools.imap_cmd("check_mailboxes"),
          CAMAS.IMAPTools.imap_cmd("status","mailbox",sessobj->mailbox[MB_FOLDERNAME_IDX]),
          CAMAS.IMAPTools.imap_cmd("scan_folders"),
          CAMAS.IMAPTools.imap_cmd("get_headers",
				   "output",CAMAS.IMAPTools.imap_cmd_var("mails"),
				   "setsortcolumn",CAMAS.IMAPTools.imap_cmd_var("mailssortcolumn")),
	        CAMAS.IMAPTools.imap_cmd("apply_mail_filters", "updatembox", CAMAS.IMAPTools.imap_cmd_var ("mails")),
      });
      sessobj->firstvisiblemail = -1; // eq to updatelistpos
      imap_commands += lcoms;
    }
  }

  void logout()
  {
    sessobj->status = LOGOUT;
    id->misc->camas->status = LOGOUT;
    foreach(id->conf->get_providers("camas_preferences"), object pref_module)
    {
     	mixed res = pref_module->save_on_logout(id);
     	if(arrayp(res))
   	  imap_commands += res;
    }
    if (sessobj->imapclient) {
      CAMAS.Log.log (my_configuration (), "logout", ([ "login" : sessobj->address ]));
      imap_commands += ({ CAMAS.IMAPTools.imap_cmd("low_logout", "noselect", 1) });
    }
  }

  void compose()
  {
    CAMAS.Tools.init_temp_vars(id);
    sessobj->originalmessage = "";
    if (id->variables->to)
      sessobj->to = TO_UNICODE(id->variables->to);
    else
      sessobj->to = "";
    sessobj->replytoidx = -1;
    sessobj->cc = "";
    if(sessobj->autobcc && sessobj->autobcc != "0") 
      sessobj->bcc = CAMAS.Tools.fix_coding(sessobj->autobcc);
    else
      sessobj->bcc = "";
    sessobj->subject = "";
    sessobj->messageid = 0;
    sessobj->messageids = 0;
    sessobj->uids = 0;
    sessobj->attachments = ({ });
    sessobj->tos = 0;
    if(id->variables->message)
      sessobj->message = TO_UNICODE(id->variables->message);
    else
      sessobj->message = "\n\n" + sessobj->signature;
    sessobj->status = COMPOSE;
    id->misc->camas->status = COMPOSE;
  }

	//! action : gotoaddattachemebt
	//!  Takes the user to the ATTACHMENT screen
	//!  Invoqued from COMPOSE screen
  void gotoaddattachment()
  {
		// Save the fields filled in in the session 
		handler->add_action("savecomposefields");
		
    sessobj->status = ATTACHMENTS;
    id->misc->camas->status = ATTACHMENTS;
  }

	//! action : removeattachment
	//!  Removes attachments selected by select multiple
  void removeattachment()
  {
    getselected();
    array deletes = ({ });
    foreach(sessobj->selectedattachments, int i)
      deletes += ({ sessobj->attachments[i] });
    sessobj->attachments -= deletes;

		if(id->misc->camas->status == COMPOSE)
		{
			handler->add_action("savecomposefields");
		}
  }

	//! action : removeattachment2
	//!  Removes attachments selected by the checkbox
  void removeattachment2()
  {
    array ind_variables = indices(id->variables);
		sessobj->selectedattachments = ({ });
    foreach (ind_variables, string var)
		{
      if (var[0..sizeof("selectattachment_")-1] == "selectattachment_") {
      	int the_attachment;
      	if (sscanf (var, "selectattachment_%d", the_attachment) == 1)
    	  if (sessobj->attachments[the_attachment])
	        sessobj->selectedattachments += ({ the_attachment });
      }
    }

    if (sessobj->selectedattachments)
		{
      array deletes = ({ });
      foreach(sessobj->selectedattachments, int i)
      	deletes += ({ sessobj->attachments[i] });
      sessobj->attachments -= deletes;
    }

		if(id->misc->camas->status == COMPOSE)
		{
			handler->add_action("savecomposefields");	
		}
		
  }

  void continuecompose()
  {
    sessobj->status = COMPOSE;
    id->misc->camas->status = COMPOSE;
  }

  void continueeditfilter()
  {
    sessobj->status = EDITADDRESSFILTER;
    id->misc->camas->status = EDITADDRESSFILTER;
  }

  void addfileattachment()
  {
    getselected();
    foreach(sessobj->selectedattachments, int i)
      sessobj->attachments += ({ sessobj->files[i] });
    sessobj->status = COMPOSE;
    id->misc->camas->status = COMPOSE;
  }

	//! action savecomposefields()
	//!  When you are in compose fields and you want to do some action, call this one to save the fields filled in the session
	void savecomposefields()
	{
		array fieldstosave = ({ "to", "cc", "bcc", "message", "subject" });
		
		foreach(fieldstosave, string field)
		{
			if(stringp(id->variables[field]))
				sessobj[field] = TO_UNICODE(id->variables[field]);
		}
	}

	//! action uploadattachment()
	//!  Upload specified filepath and stay in the current screen.
	//!  Used for both attachement and compose screen
  void uploadattachment()
  {
    if(camas_feature (FEAT_ATTACHMENTS) && id->variables->file && sizeof(id->variables->file))
		{
      string fname = (id->variables->fixedfilename && sizeof(id->variables->fixedfilename)) ? id->variables->fixedfilename : id->variables["file.filename"];
      fname=((((fname)/"/")[-1])/"\\")[-1];
      sessobj->attachments +=
				({
	  			([
						"fname" : TO_UNICODE(fname),
	     			"size" : sizeof(id->variables->file),
	     			"type" : CAMAS.Tools.filetype(fname, my_configuration()),
	     			"data": id->variables->file
	  			])
				});
    }

		// if we upload attachments from the COMPOSE field, save all the forms filled in
		if(id->misc->camas->status == COMPOSE)
		{
			handler->add_action("savecomposefields");	
		}

  }

  void addressbook_to_unicode()
  {
    sessobj->to=TO_UNICODE(id->variables->to || "");
    sessobj->cc=TO_UNICODE(id->variables->cc || "");
    sessobj->bcc=TO_UNICODE(id->variables->bcc || "");
    sessobj->subject=TO_UNICODE(id->variables->subject || "");
    sessobj->message=TO_UNICODE(id->variables->message || "");
  }

  void addressbook_content()
  {
    if(id->variables->selectaddressbook)
      sessobj->selectedaddrbook = TO_UNICODE(id->variables->selectaddressbook);
    sessobj->addrbook_contents = CAMAS.AddressBook2.get_currentaddrbook(id)->get_all(id);
  }
 
  // same as addressbookto but don't erase sessobj->to, cc and so on
  void addressbookto_without_erase()
  {
    sessobj->recipientfield = "to";
    sessobj->status = ADDRESSBOOK;
    id->misc->camas->status = ADDRESSBOOK;
    addressbook_content();
  }
  void addressbookto()
  {
    sessobj->recipientfield = "to";
    sessobj->status = ADDRESSBOOK;
    id->misc->camas->status = ADDRESSBOOK;
    addressbook_to_unicode();
    addressbook_content();
  }
  void addressbookcc()
  {
    sessobj->recipientfield = "cc";
    sessobj->status = ADDRESSBOOK;
    id->misc->camas->status = ADDRESSBOOK;
    addressbook_to_unicode();
    addressbook_content();
  }
  void addressbookbcc()
  {
    sessobj->recipientfield = "bcc";
    sessobj->status = ADDRESSBOOK;
    id->misc->camas->status = ADDRESSBOOK;
    addressbook_to_unicode();
    addressbook_content();
  }

  void addressbooktoccbcc()
  {
    addressbook_to_unicode();
    sessobj->status = ADDRESSBOOK2;
    id->misc->camas->status = ADDRESSBOOK2;
    sessobj->recipientfield = "toccbcc";
    addressbook_content();
  }

  void addressbookselect()
  {
    sessobj->editaddress = 0;
    addressbook_content();
  }

/* Compat stuff: will be removed in 1.3 */
#if constant(Protocols.LDAP) && constant(Protocols.LDAP.client)
  void ldapto()
  {
    sessobj->recipientfield = "to";
    handler->add_action("ldaptoccbcc");
  }
  void ldapcc()
  {
    sessobj->recipientfield = "cc";
    handler->add_action("ldaptoccbcc");
  }
  void ldapbcc()
  {
    sessobj->recipientfield = "bcc";
    handler->add_action("ldaptoccbcc");
  }
  void ldaptoccbcc()
  {
    sessobj->recipientfield = "toccbcc";
    sessobj->to=TO_UNICODE(id->variables->to);
    sessobj->cc=TO_UNICODE(id->variables->cc);
    sessobj->bcc=TO_UNICODE(id->variables->bcc);
    sessobj->subject=TO_UNICODE(id->variables->subject);
    sessobj->message=TO_UNICODE(id->variables->message);
    sessobj->status = LDAPSEARCH;
    id->misc->camas->status = LDAPSEARCH;
  }

  void searchldap()
  {
    sessobj->ldapadd = CAMAS.LDAP.getldapaddr(TO_UNICODE(id->variables->namecont), features->QUERY(ldapserver),
					       features->QUERY(ldapsearchroot), (int)features->QUERY(ldapversion),
					       features->QUERY(ldapuser), features->QUERY(ldappass), features->QUERY(ldapshowou),
					       features->QUERY(ldapmail));
    sessobj->status = LDAPRESULT;
    id->misc->camas->status = LDAPRESULT;
  }

  void searchldaptoccbcc()
  {
    sessobj->ldapadd = CAMAS.LDAP.getldapaddr(TO_UNICODE(id->variables->namecont), features->QUERY(ldapserver),
					       features->QUERY(ldapsearchroot), (int)features->QUERY(ldapversion),
					       features->QUERY(ldapuser), features->QUERY(ldappass), features->QUERY(ldapshowou),
					       features->QUERY(ldapmail));
    sessobj->status = LDAPRESULT2;
    id->misc->camas->status = LDAPRESULT2;
  }
#endif
  /* end compat stuff */

  void addressbook()
  {
    sessobj->recipientfield = 0;
    sessobj->status = ADDRESSBOOK;
    id->misc->camas->status = ADDRESSBOOK;
    addressbook_content();
  }

  void addressbook2()
  {
    sessobj->recipientfield = 0;
    sessobj->status = ADDRESSBOOK2;
    id->misc->camas->status = ADDRESSBOOK2;
    addressbook_content();
  }

  /* By default search all fields */
  void searchaddrbook()
  {
    if(id->variables->selectaddressbook)
      sessobj->selectedaddrbook = TO_UNICODE(id->variables->selectaddressbook);
    string searchwhat = TO_UNICODE(id->variables->search);
    object addressbook = CAMAS.AddressBook2.get_currentaddrbook(id);
    array(string) searchfields = addressbook->get_all_attributes();
    mapping variables2search = mkmapping(searchfields, allocate(sizeof(searchfields), searchwhat));
    // or'ed search with wildcards
    sessobj->addrbook_contents = CAMAS.AddressBook2.get_currentaddrbook(id)->get(variables2search, id, 1, 1);
    if(addressbook->get_lastprotocol_error(id))
    {
      sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
    	sessobj->dialogactions = ({ "actionaddressbook2" });
      sessobj->dialogtext = MSG(M_ADDRBOOKERRROR);
      if(CSESSION->displayaddrbookerrors)
        sessobj->dialogtext += "\n" + addressbook->get_lastprotocol_error(id);
      sessobj->dialogtarget = id->misc->camas->frame;
    	sessobj->status = DIALOGBOX;
      id->misc->camas->status = DIALOGBOX;
      return;
    }
  }

  /* edit an entry in the address book from the readmail screen */
  void editaddressfromreadmail()
  {
    // make sure camas doesn't believe we come from compose screen
    sessobj->recipientfield = 0;
    string searchwhat = HTML_DECODE_STRING(TO_UNICODE(id->variables->address));
    string name;
    // extract the email address
    sscanf(searchwhat, "%s<%s>", name, searchwhat);
    mapping(int:string) variables2search = ([ AD_MAIL: searchwhat ]);
    object current_abook = CAMAS.AddressBook2.get_currentaddrbook(id);
    // search in current address book first
    // if it fails then search other address book
    array(object) addressbooks = id->conf->get_providers("camas_addressbook") -
      ({ current_abook });
    addressbooks = ({ current_abook }) + addressbooks;
    foreach(addressbooks, object addressbook)
    {
      array(mapping(int:string)) content = addressbook->get(variables2search, id);
      if(sizeof(content) > 0)
      {
        if(addressbook->get_lastprotocol_error(id))
        {
          sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
        	sessobj->dialogactions = ({ "actionindex" });
          sessobj->dialogtext = MSG(M_ADDRBOOKERRROR);
          if(CSESSION->displayaddrbookerrors)
            sessobj->dialogtext += "\n" + addressbook->get_lastprotocol_error(id);
          sessobj->dialogtarget = id->misc->camas->frame;
        	sessobj->status = DIALOGBOX;
          id->misc->camas->status = DIALOGBOX;
         return;
        }
      }
    }
    // this address is not in the address books, add a new one
    handler->add_action("newaddress2", searchwhat, name);
  }
  
  void filterbook()
  {
    sessobj->recipientfield = 0;
    sessobj->status = MAILFILTER;
    id->misc->camas->status = MAILFILTER;
  }

  void gotoimportaddress()
  {
    sessobj->status = IMPORTADDRESS;
    id->misc->camas->status = IMPORTADDRESS;
  }
  
  void oldimportaddress()
  {
    //FIXME: coding of files?
    if(sizeof(id->variables->file)) 
    {
      string newbook = CAMAS.AddressBook.impabook(id->variables->file);
      if (sizeof(newbook) > 0) 
      {
      	array (string) addresses = (sessobj->addressbook - "\r")/"\n";
      	addresses += newbook / "\n";

       	//addresses[0] = ' '; //added to index addressbook correctly.

       	sessobj->addressbook = sort (Array.uniq (addresses)) * "\n";
      	sessobj->status = ADDRESSBOOK;
        id->misc->camas->status = ADDRESSBOOK;
        array|void imp_command = CAMAS.Tools.save_preferences(id);
        if(imp_command && arrayp(imp_command))
         imap_commands += imp_command;
      } 
      else
      {
      	sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
      	sessobj->dialogactions = ({ "actionaddressbook" });
      	sessobj->dialogtext = MSG(M_CANNOTIMPORT);
      	sessobj->dialogtarget = id->misc->camas->frame;
      	sessobj->status = DIALOGBOX;
        id->misc->camas->status = DIALOGBOX;
      }
    }
    sessobj->status = ADDRESSBOOK;
    id->misc->camas->status = ADDRESSBOOK;
  }

  void importaddress()
  {
    if(id->variables->selectaddressbook)
      sessobj->selectedaddrbook = TO_UNICODE(id->variables->selectaddressbook);
    if(id->variables->file)
    {
      array(string) typeofabook = CAMAS.AddressBook2.typeofabook(id->variables->file);
      if(typeofabook)
      {
        sessobj->typeofabook = typeofabook[0];
        if(sessobj->typeofabook == "csv") 
          sessobj->imported_addrbook = CAMAS.AddressBook2.CSV.imp(id->variables->file, typeofabook[1]);
        if(sessobj->typeofabook == "ldif")
          sessobj->imported_addrbook = CAMAS.AddressBook2.LDIF.imp(id->variables->file);
      }
    }
    if(!sessobj->imported_addrbook || !sizeof(sessobj->imported_addrbook))
    {
      sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
    	sessobj->dialogactions = ({ "actionaddressbook" });
      sessobj->dialogtext = MSG(M_CANNOTIMPORT);
      sessobj->dialogtarget = id->misc->camas->frame;
      sessobj->status = DIALOGBOX;
      id->misc->camas->status = DIALOGBOX;
    }
    else
    {
      if(sessobj->typeofabook == "csv")
      {
        sessobj->status = IMPORTADDRESS2;
        id->misc->camas->status = IMPORTADDRESS2;
      }
      else
        handler->add_action("importaddress2", 1);
    }
  }

  // indirectcall is 1 if we don't come from the IMPORTADDRESS2 screen (used to import CSV files)
  void importaddress2(int indirectcall)
  {
    object addressbook = CAMAS.AddressBook2.get_currentaddrbook(id);
    if(!addressbook->writeable())
    {
      report_warning("importaddress2 called on a read only address book\n");
      return;
    }

    mapping(int:int) colnb2internalcolname = ([ ]);
    int n = !indirectcall?sizeof(sessobj->imported_addrbook[0]):sizeof(sessobj->imported_addrbook);
    if(!indirectcall)
    {
      foreach(indices(id->variables), string var)
      {
        int column_number;
        int internal_colname = (int) id->variables[var];
        if(internal_colname && sscanf(var, "%*s_%d", column_number) == 2)
          colnb2internalcolname += ([ column_number - 1: internal_colname ]);
      }
    }
    mapping(int:string) variables2insert;
    // the number of entries who fails to be added
    int error_status = 0;
    for(int i = 0; i < n; i++)
    {
      if(!indirectcall)
      {
        variables2insert = ([ ]);
        for(int j = 0; j < sizeof(sessobj->imported_addrbook); j++)
        {
          if(colnb2internalcolname[j])
            variables2insert += ([ colnb2internalcolname[j]: sessobj->imported_addrbook[j][i] ]);
        }
      }
      else
        // ldif already contains the good structure
        variables2insert = sessobj->imported_addrbook[i];
      CDEBUG(sprintf("importing %O to addressbook %O\n", 
            variables2insert, addressbook));
      mixed missing_fields;
      missing_fields = addressbook->add(variables2insert, id);
      if(addressbook->get_lastprotocol_error(id))
      {
        error_status++;
        sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
      	sessobj->dialogactions = ({ "actionimportaddress" });
        sessobj->dialogtext = MSG(M_ADDRBOOKERRROR);
        if(CSESSION->displayaddrbookerrors)
          sessobj->dialogtext += "\n" + addressbook->get_lastprotocol_error(id);
        sessobj->dialogtarget = id->misc->camas->frame;
      	sessobj->status = DIALOGBOX;
        id->misc->camas->status = DIALOGBOX;
        report_error("Error importing to addressbook %O: %s\n", CAMAS.AddressBook2.get_currentaddrbook(id),
            addressbook->get_lastprotocol_error(id));
      }
      if(missing_fields)
      {
        sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
      	sessobj->dialogactions = ({ "actionimportaddress" });
      	sessobj->dialogtext = MSG(M_MUSTSELECTFIELD) + " " + 
          MSG(internal_attributenames2translated[missing_fields]);
      	sessobj->dialogtarget = id->misc->camas->frame;
      	sessobj->status = DIALOGBOX;
        id->misc->camas->status = DIALOGBOX;
        return;
      }
    }
    // display an error only if all fails
    if(error_status > 0 && error_status == n - 1)
    {
      sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
     	sessobj->dialogactions = ({ "actionaddressbook" });
     	sessobj->dialogtext = MSG(M_ADDRESSBOOKWRITEPROBLEM) + sessobj->dialogtext || "";
     	sessobj->dialogtarget = id->misc->camas->frame;
     	sessobj->status = DIALOGBOX;
      id->misc->camas->status = DIALOGBOX;
      return;
    }
    sessobj->status = ADDRESSBOOK;
    id->misc->camas->status = ADDRESSBOOK;
    // free address book content from memory
    sessobj->imported_addrbook = 0;
    // update display of address book content
    addressbook_content();
    // write preferences to preferences module (used when the user uses the preferences address book module)
    array|void imp_command = CAMAS.Tools.save_preferences(id);
    if(imp_command && arrayp(imp_command))
      imap_commands += imp_command;
  }

  // can be called by deletealladdress2
  void deleteaddress2(array(string) _addresses2delete)
  {
    array ind_variables = indices(id->variables);
    array(string) addresses2delete = ({ });
    if(_addresses2delete)
      addresses2delete = _addresses2delete;
    string address;
    if(id->variables->selectaddressbook)
      sessobj->selectedaddrbook = TO_UNICODE(id->variables->selectaddressbook);
    object addressbook = CAMAS.AddressBook2.get_currentaddrbook(id);
    if(!addressbook->writeable())
    {
      report_warning("deleteaddress2 called on a read only address book\n");
      return;
    }

    sessobj->status = ADDRESSBOOK;
    id->misc->camas->status = ADDRESSBOOK;
    if(id->variables->address)
      addresses2delete += ({ id->variables->address });
    else
    {
      foreach (ind_variables, string var) 
      {
        if (id->variables[var] == "1")
        {
          if (sscanf (var, "msg%s", address) == 1)
          {
            addresses2delete += ({ address });
          }
        }
      }
    }
    foreach(addresses2delete, string address2delete)
      addressbook->delete(address2delete, id);
    addressbook_content();
    if(addressbook->get_lastprotocol_error(id))
    {
      sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
    	sessobj->dialogactions = ({ "actionaddressbook2" });
      sessobj->dialogtext = MSG(M_ADDRBOOKERRROR);
      if(CSESSION->displayaddrbookerrors)
        sessobj->dialogtext += "\n" + addressbook->get_lastprotocol_error(id);
      sessobj->dialogtarget = id->misc->camas->frame;
    	sessobj->status = DIALOGBOX;
      id->misc->camas->status = DIALOGBOX;
    }
    array|void imp_command = CAMAS.Tools.save_preferences(id);
    if(imp_command && arrayp(imp_command))
      imap_commands += imp_command;
  }

  void deletealladdress2()
  {
    array(string) addresses2delete = ({ });
    array(mapping(int:array(string))) all_entries = CAMAS.AddressBook2.get_currentaddrbook(id)->get_all(id);
    foreach(all_entries, mapping(int:array(string)) entry)
    {
      addresses2delete += ({ entry[AD_MAIL][0] });
    }
    deleteaddress2(addresses2delete);
  }


  void newaddress2(void|string address, void|string name)
  {
    object addressbook = CAMAS.AddressBook2.get_currentaddrbook(id);
    if(!addressbook->writeable())
    {
      report_warning("newaddress2 called on a read only address book\n");
      return;
    }
    sessobj->status = EDITADDRESS;
    id->misc->camas->status = EDITADDRESS;
    if(address)
      sessobj->editaddress = address;
    else
    {
      if(id->variables->address)
        sessobj->editaddress = id->variables->address;
      else
        sessobj->editaddress = 0;
    }
    sessobj->editname = name;
    sessobj->editaddressmode = "new";
  }
 
  void editaddress2()
  {
    if (id->variables->address) 
    {
      sessobj->editaddress = id->variables->address;
      sessobj->status = EDITADDRESS;
      id->misc->camas->status = EDITADDRESS;
      sessobj->editaddressmode = "modify";
    }
  }

  void editaddressdone2()
  {
    array(mapping(int:array(string))) contact;
    if(id->variables->selectaddressbook)
      sessobj->selectedaddrbook = TO_UNICODE(id->variables->selectaddressbook);
    object addressbook = CAMAS.AddressBook2.get_currentaddrbook(id);
    if(!addressbook->writeable())
    {
      report_warning("editaddressdone2 called on a read only address book\n");
      return;
    }
    
    if(id->variables->address)
    {
      // modify
      contact = addressbook->get(([ AD_MAIL: id->variables->address ]), id);
      mapping(int:string) modifiedentries = ([ ]);
      foreach(indices(ldapattributes2input), string form_input_name)
      {
        if(id->variables[form_input_name] != contact[0][ldapattributes2input[form_input_name]])
        {
          modifiedentries += ([ ldapattributes2input[form_input_name]: TO_UNICODE(id->variables[form_input_name]) ]);
        }
      }
      addressbook->modify(modifiedentries, id->variables->address, id); 
      if(addressbook->get_lastprotocol_error(id))
        sessobj->dialogactions = ({ "actionaddressbook2" });
    }
    else
    {
      // add
      mixed missing_fields;
      mapping(string:int) variables2insert = ([ ]);
      foreach(indices(ldapattributes2input), string form_input_name)
        variables2insert += ([ ldapattributes2input[form_input_name]: TO_UNICODE(id->variables[form_input_name]) ]);
      missing_fields = addressbook->add(variables2insert, id);
      if(missing_fields)
      {
        sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
      	sessobj->dialogactions = ({ "actionnewaddress2" });
      	sessobj->dialogtext = MSG(M_MUSTSELECTFIELD) + " " + 
          MSG(internal_attributenames2translated[missing_fields]);
      	sessobj->dialogtarget = id->misc->camas->frame;
      	sessobj->status = DIALOGBOX;
        id->misc->camas->status = DIALOGBOX;
        return;
      }
      if(addressbook->get_lastprotocol_error(id))
        sessobj->dialogactions = ({ "actionnewaddress2" });
    }
    sessobj->status = ADDRESSBOOK;
    id->misc->camas->status = ADDRESSBOOK;
    addressbook_content();
    if(addressbook->get_lastprotocol_error(id))
    {
      sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
      sessobj->dialogtext = MSG(M_ADDRBOOKERRROR);
      if(CSESSION->displayaddrbookerrors)
        sessobj->dialogtext += "\n" + addressbook->get_lastprotocol_error(id);
      sessobj->dialogtarget = id->misc->camas->frame;
    	sessobj->status = DIALOGBOX;
      id->misc->camas->status = DIALOGBOX;
    }
    array|void imp_command = CAMAS.Tools.save_preferences(id);
    if(imp_command && arrayp(imp_command))
      imap_commands += imp_command;
  }
  
  // compat stuff obsoleted by new address book in 1.2
  void editaddress()
  {
    if (id->variables->address) 
    {
      sessobj->editaddress = ((int) id->variables->address);
      sessobj->status = EDITADDRESS;
      id->misc->camas->status = EDITADDRESS;
    }
    sessobj->editaddressmode = "old";
  }
  
  void editaddressfilter()
  {
    if (id->variables->namefield) 
    {
      sessobj->editaddressfilter = ((int) id->variables->namefield);
      sessobj->status = EDITADDRESSFILTER;
      id->misc->camas->status = EDITADDRESSFILTER;
    }
    sessobj->editaddressfiltermode = "old";
  }
  
  // compat stuff obsoleted by new address book in 1.2
  void newaddress()
  {
    string name="", address="";
    if (id->variables->address)
      address = CAMAS.Tools.fix_coding(id->variables->address);
    if (id->variables->name)
      name = id->variables->name;
    if (id->variables->take)
      sessobj->editaddressmode = "take";
    else
      sessobj->editaddressmode = "new";
    if (sizeof(sessobj->addressbook) > 0)
      sessobj->addressbook += "\n"+replace(name,":","")+":"+address;
    else
      sessobj->addressbook = replace(name,":","")+":"+address;

    sessobj->status = EDITADDRESS;
    id->misc->camas->status = EDITADDRESS;
    sessobj->editaddress = sizeof(sessobj->addressbook/"\n")-1;
  }
  
  void newaddressfilter()
  {
    string namefield="", filterexpression="", filterfolder="";

    if (id->variables->namefield)
      namefield = CAMAS.Tools.fix_coding(id->variables->namefield);
    if (id->variables->filterfolder)
      filterfolder = CAMAS.Tools.fix_coding(id->variables->filterfolder);
    if (id->variables->filterexpression)
      filterexpression = id->variables->filterexpression;
    if (id->variables->take)
      sessobj->editaddressfiltermode = "take"; //oliv3: never used ?
    else
      sessobj->editaddressfiltermode = "new";

    sessobj->newaddressfilter = CAMAS.Filters.encode_filter(namefield) 
      + ":" + CAMAS.Filters.encode_filter(filterexpression)
      + ":" + CAMAS.Filters.encode_filter(filterfolder);
    sessobj->status = EDITADDRESSFILTER;
    id->misc->camas->status = EDITADDRESSFILTER;
  }
  
  // compat stuff obsoleted by new address book in 1.2
  void editaddressdone()
  {
    if (id->variables->address && id->variables->name) 
    {
      array (string) addresses = sessobj->addressbook/"\n";
      addresses[sessobj->editaddress] = replace(TO_UNICODE(id->variables->name),":", "")+":"+TO_UNICODE(id->variables->address);
      sessobj->addressbook = sort(addresses)*"\n";
      if (sessobj->editaddressmode == "take")
      {
      	sessobj->status = READMAIL;
        id->misc->camas->status = READMAIL;
      }
      else
      {
      	sessobj->status = ADDRESSBOOK;
        id->misc->camas->status = ADDRESSBOOK;
      }
      if (camas_feature (FEAT_EXTENDEDABOOK) && sessobj->extendedabook) 
      {
      	string lval;
       	array(string) fn = sessobj->extendedabook->get_fields();
      	foreach(indices(fn), int i) {
       	  lval = id->variables[sprintf("ebook%d",i)];
      	  if (lval) 
          {
       	    sessobj->extendedabook->set_entry(id->variables->name, fn[i],
					       TO_UNICODE(lval));
          }

      	  if (sessobj->debug)
      	    CDEBUG(sprintf("Handling ebook field %s: %s\n", fn[i], lval||"NONE"));
      	}
      }
    }
    array|void imp_command = CAMAS.Tools.save_preferences(id);
    if(imp_command && arrayp(imp_command))
      imap_commands += imp_command;
  }

  int _editaddressfilterdone(object id, int autofilter)
  {
    string namefield = TO_UNICODE(id->variables->namefield);
    string filterexpression = TO_UNICODE(id->variables->filterexpression);
    string filterfolder = TO_UNICODE(id->variables->filterfolder);
    // autofilter created from move mail action: check that there is not the same entry for 2 differents mailboxes
    if(autofilter)
      CAMAS.Filters.delete_filter(id, namefield, filterexpression);
    if(sessobj->editaddressfiltermode == "old")
      CAMAS.Filters.replace_filter_by_filternumber(id, sessobj->editaddressfilter, 
          namefield, filterexpression, filterfolder);
    else
      CAMAS.Filters.add_filter(id, namefield, filterexpression, filterfolder);
    return 0;
  }

  void editaddressfilterdone()
  {
    if (id->variables->namefield && id->variables->filterfolder && id->variables->filterexpression) 
    {
      _editaddressfilterdone(id, 0);
      if (sessobj->editaddressfiltermode == "take")
      {
      	sessobj->status = READMAIL;
        id->misc->camas->status = READMAIL;
      }
      else
      {
      	sessobj->status = MAILFILTER;
        id->misc->camas->status = MAILFILTER;
      }
      array|void imp_command = CAMAS.Tools.save_preferences(id);
      if(imp_command && arrayp(imp_command))
        imap_commands += imp_command;
    }
    else
    {
      sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
      sessobj->dialogactions = ({ "actioncontinueeditfilter" });
      sessobj->dialogtext = MSG(M_EMPTYADDRESSFILTER);
      sessobj->status = DIALOGBOX;
      id->misc->camas->status = DIALOGBOX;
    }
  }
 
  // compat stuff obsoleted by new address book in 1.2
  void deleteaddress()
  {
    array (string) addresses = sessobj->addressbook/"\n";
    if (camas_feature (FEAT_EXTENDEDABOOK) && features->QUERY (deleteinebook)) 
    {
      string badguy = (addresses[sessobj->editaddress] / ":")[0];
      sessobj->extendedabook->del_entry(badguy);
    }
    addresses -= ({ addresses[sessobj->editaddress] });
    sessobj->addressbook = addresses * "\n";
    sessobj->status = ADDRESSBOOK;
    id->misc->camas->status = ADDRESSBOOK;
    array|void imp_command = CAMAS.Tools.save_preferences(id);
    if(imp_command && arrayp(imp_command))
      imap_commands += imp_command;
  }
  
  // compat stuff obsoleted by new address book in 1.2
  void deletealladdress()
  {
    sessobj->addressbook = ({ }) * "\n";
    if (camas_feature (FEAT_EXTENDEDABOOK) && features->QUERY (deleteinebook) && sessobj->extendedabook)
      sessobj->extendedabook->del_entry("*");

    sessobj->status = ADDRESSBOOK;
    id->misc->camas->status = ADDRESSBOOK;
    array|void imp_command = CAMAS.Tools.save_preferences(id);
    if(imp_command && arrayp(imp_command))
      imap_commands += imp_command;
  }
 
  void canceleditaddress()
  {
    if (sessobj->editaddressmode != "old") 
    {
      array (string) addresses = sessobj->addressbook/"\n";
      addresses -= ({ addresses[-1] });
      sessobj->addressbook = addresses*"\n";
    }
    if (sessobj->editaddressmode == "take")
    {
      sessobj->status = READMAIL;
      id->misc->camas->status = READMAIL;
    }
    else
    {
      sessobj->status = ADDRESSBOOK;
      id->misc->camas->status = ADDRESSBOOK;
    }
  }
  
  array search_in_memory(string what, array(string) fields, array all_mails, mapping memory_fields, 
			 int fix_coding)
  {
    CDEBUG("search_in_memory started\n");
    if(!all_mails)
      return 0;
    if(sizeof(all_mails) == 0)
      return 0;
    what = lower_case(what);
    array filtered_mails = 0;
    // OK, it's a little complex but it's Pikeish and fast enough ;) /vida
    // for every mails
    filtered_mails = Array.filter(all_mails, lambda(mapping mail) 
    {
       int retcode = 0, n = sizeof(fields), i = 0;
       // for every fields
       foreach(fields, string field)
       {
      	 int index = memory_fields[field];
      	 mixed searchin = mail->imap->ENVELOPE[index];
      	 // make the string from an array is searching in fields like to or from
      	 if(arrayp(searchin))
      	 {
      	   searchin = searchin[0];
      	   searchin = searchin[0] + " <" + searchin[2..3] * "@" + ">";
      	 }
      	 if(searchin)
      	 {
           if(fix_coding)
             searchin = CAMAS.Tools.fix_coding(searchin);
           CDEBUG(sprintf("field=%s, searchin=%O\n", field, searchin));
      	   // 1 == this mail match the search
      	   // 0 == this mail doesn't match
      	   retcode = has_value(lower_case(searchin), what);
      	 }
      	 // returns if this mail matches or we scanned all the fields
      	 if(retcode || ++i == n)
      	 {
      	   CDEBUG(sprintf("returning code %d\n", retcode));
      	   return retcode;
      	 }
       }
       return 0;
    });
    return filtered_mails;
  }

  void searchmail()
  {
    getselected();
    sessobj->nothingfound = 0;
    sessobj->mailssortcolumn = "";
    string text2search;
    if(id->variables->text1)
      text2search = HTTP_DECODE_STRING(id->variables->text1);
    array mboxes = ({ });
    int first_search = 1;
    mapping(string:int) memory_fields = ([
      "to": TO_IDX,
      "from": FROM_IDX,
      "subject": SUBJECT_IDX,
      "header message-id": MESSAGEID_IDX,
    ]);

    if(id->variables->mbox)
      mboxes = ({ id->variables->mbox });
    else
    {
      array ind_variables = indices(id->variables);
      foreach(ind_variables, string index)
      {
      	if(index[..2] == "msg" && id->variables[index] == "1")
          mboxes += ({ index[3..] });
      }
    }
    // if the user didn't provide any mailboxes, try to find a better match case
    if(sizeof(mboxes) == 0)
    {
      // first case, the cache is here and we can search in it -> search in all mboxes
      if(sessobj->cachefeeded && has_value(indices(memory_fields), id->variables->searchfield1))
        mboxes = map(sessobj->mailboxes, lambda(array mailbox) { return mailbox[MB_FOLDERNAME_IDX]; });
      // search in current mailbox
      else
        mboxes = ({ sessobj->mailbox[MB_FOLDERNAME_IDX] });
    }
    if (text2search && id->variables->searchfield1 && mboxes && sizeof(mboxes))
    {
      if (sessobj->searchstring && sizeof(sessobj->searchstring) > 0 && !id->variables->nohistsearch)
      {
        first_search = 0;
      	sessobj->searchstring+= " "+ MSG(M_SEARCHAND)+" ";
      }
      else
      	sessobj->searchstring = "";
      sessobj->searchstring += "\""+TO_UNICODE(text2search)+"\" ";

      if(id->variables->stickyfilter)
      {
      	sessobj->stickyfilter = 1;
      	sessobj->searchstringtext1 = TO_UNICODE(text2search);
      	sessobj->searchfield1 = id->variables->searchfield1;
      }
      else
      	sessobj->stickyfilter = 0;

      switch(id->variables->searchfield1)
      {
        case "text":
        	sessobj->searchstring += MSG(M_SEARCHANYWHERE);
        	break;
        case "to":
        	sessobj->searchstring += MSG(M_SEARCHTOFIELD);
        	break;
        case "from":
        	sessobj->searchstring += MSG(M_SEARCHFROMFIELD);
        	break; 
        case "subject":
        	sessobj->searchstring += MSG(M_SEARCHSUBJECT);
        	break;
        case "body":
        	sessobj->searchstring += MSG(M_SEARCHBODY);
        	break;
        case "all headers":
        	sessobj->searchstring += MSG(M_SEARCHHEADERS);
        	break;
      }
      sessobj->status = MAILINDEX;
      id->misc->camas->status = MAILINDEX;

      if(id->variables->history)
      {
        sessobj->historysearch = 1;
        sessobj->searchstring = TO_UNICODE(text2search);
        // the mail boxes to search in
        array history_mboxes = ({ });
        string absolute_sent_path = CAMASFOLDERTOOLS.getmailboxroot(sessobj->mailbox, sessobj) 
          + sessobj->sentfolder;
        mboxes = map(sessobj->mailboxes, lambda(array mailbox) { return mailbox[MB_FOLDERNAME_IDX]; });
        // now search for absolute_sent_path or sub-directories of it
        foreach(mboxes, string mbox)
        {
          if(search(mbox, absolute_sent_path) != -1)
            history_mboxes += ({ mbox });
        }
        if(sizeof(history_mboxes))
        	mboxes = history_mboxes;
      }

      if(id->variables->tracking)
      {
      	// We try to extract the specified tracking pattern from the given searchstring
        array(string) tags = Regexp(features->QUERY(threadsearchafter))->split(text2search);
        string searchstring;
        if(tags)
          searchstring = tags[0];
        if(searchstring) {
          sessobj->searchstring = TO_UNICODE(searchstring);
          text2search = searchstring;
        }
      }
      CDEBUG(sprintf("searching for %O\n", sessobj->searchstring));

      // we need to search using the IMAP server in this case, very slow:
      // . the user want to search on body
      // . the cache is not feeded entirely yet
      if(id->variables->searchfield1 == "body" || id->variables->searchfield1 == "text"
         || (!sessobj->cachefeeded)) 
      {
        CDEBUG("searching using IMAP server\n");
      	string cmbox = mboxes[0];
      	/* stolen from mailindex() */
      	int mbox_idx = Array.search_array(sessobj->mailboxes,
					  lambda(array a,string mbox)
					  {
					    if(a[MB_FOLDERNAME_IDX] == mbox)
					      return 1;
					    return 0;
					  },
					  cmbox);
        if (cmbox && mbox_idx != -1)
        {
          // if mbox is different from current mailbox
          if (sessobj->mailbox[MB_FOLDERNAME_IDX] != cmbox)
          {
            // set current mailbox to mbox
            sessobj->mailbox = sessobj->mailboxes[mbox_idx];
            // as the imap search command filters sessobj->mails, "status" and "get_headers" are used to set 'em with values of the new mbox
            imap_commands += ({
	      CAMAS.IMAPTools.imap_cmd("status","mailbox",sessobj->mailbox[MB_FOLDERNAME_IDX]),
	      CAMAS.IMAPTools.imap_cmd("get_headers","output",CAMAS.IMAPTools.imap_cmd_var("mails"),
				       "setsortcolumn",CAMAS.IMAPTools.imap_cmd_var("mailssortcolumn")),
	      CAMAS.IMAPTools.imap_cmd("search", "searchtext", TO_UNICODE(text2search),
				       "searchfield", id->variables->searchfield1, "headers", 
				       CAMAS.IMAPTools.imap_cmd_var("mails"))
	    });
	    sessobj->firstvisiblemail = -1; // eq to updatelistpos
          }
        }
        // if mbox is not different from actual mbox or not set, just search
        imap_commands += ({
	  CAMAS.IMAPTools.imap_cmd("search", "searchtext", TO_UNICODE(text2search), 
				   "searchfield", id->variables->searchfield1, "headers", CAMAS.IMAPTools.imap_cmd_var("mails"))
	});
	return;       
      }
      else
      {
        CDEBUG("Searching in my memory\n");
        int fix_coding = 0;
        string what = TO_UNICODE(text2search);
        if(id->variables->searchfield1 == "subject" && !CAMAS.Tools.is_7bitsafe(what))
          fix_coding = 1;
        array(string) fields = ({ });
        if(id->variables->searchfield1 == "all headers")
          fields = ({ "from", "to", "subject", "header messageid" });
        else
          fields = ({ id->variables->searchfield1, 
            id->variables->searchfield2 }) - ({ 0 });
        // the mails to search for
        array mails = ({ });
        // if the user already search for something, search only in resuslts
        if(first_search)
	  foreach(mboxes, string cmbox)
	  {
            CDEBUG(sprintf("First search, searching in %s mbox\n", cmbox));
            // don't search in mailboxes that don't contain mails like "shared"
            object cache;
#ifdef CAMAS_DEBUG
            cache = camas_main->imapclient()->Cache (cmbox, sessobj, camas_main);
#else
            cache = CAMASIMAPCLIENT.Cache (cmbox, sessobj, camas_main);
#endif
            // don't search in mailboxes that don't contain mails like "shared"
            if(cache->mcache)
              mails += cache->mcache->mails;
          }
        else
        {
          CDEBUG(sprintf("Not first search, searching in available mails (%d)\n", 
                sizeof(sessobj->mails)));
          mails = sessobj->mails;
        }
        sessobj->mails = ({ });
        array res = search_in_memory(what, fields, mails, memory_fields, fix_coding);
      	if(res && sizeof(res)) 
      	  sessobj->mails += res;
      }
      if(sizeof(sessobj->mails) == 0)
        sessobj->nothingfound = 1;
    }
    else
    {
      sessobj->status = SEARCHMAIL;
      id->misc->camas->status = SEARCHMAIL;
    }
  }

  void addrecipient()
  {
    if (id->variables->address) 
    {
      if (sessobj->recipientfield == "to" ||
    	  sessobj->recipientfield == "cc" ||
    	  sessobj->recipientfield == "bcc") {
    	if (sizeof(sessobj[sessobj->recipientfield]) > 0)
    	  sessobj[sessobj->recipientfield] += ", "+TO_UNICODE(id->variables->address);
    	else
    	  sessobj[sessobj->recipientfield] = TO_UNICODE(id->variables->address);
      }
    }
    sessobj->status = COMPOSE;
    id->misc->camas->status = COMPOSE;
  }
  
  void addrecipients()
  {
    array ind_variables = indices(id->variables);
    foreach (ind_variables, string k) {
      int no = 0;
      if (sscanf (k, "recipient_to_%d", no) == 1) { // found a "to" recipient
        string key2 = "address_" + no; // the address key
        string value = (id->variables[key2] / "\0")[0];
        if (sizeof (sessobj->to) > 0)
          sessobj->to += ", "+TO_UNICODE(value);
        else
          sessobj->to = TO_UNICODE(value);
      }
      if (sscanf (k, "recipient_cc_%d", no) == 1) { // found a "cc" recipient
      	string key2 = "address_" + no; // the address key
        string value = (id->variables[key2] / "\0")[0];
        if (sizeof (sessobj->cc) > 0)
          sessobj->cc += ", "+TO_UNICODE(value);
        else
          sessobj->cc = TO_UNICODE(value);
      }
      if (sscanf (k, "recipient_bcc_%d", no) == 1) { // found a "bcc" recipient
      	string key2 = "address_" + no; // the address key
        string value = (id->variables[key2] / "\0")[0];
        if (sizeof (sessobj->bcc) > 0)
          sessobj->bcc += ", "+TO_UNICODE(value);
        else
          sessobj->bcc = TO_UNICODE(value);
      }
    }
    sessobj->status = COMPOSE;
    id->misc->camas->status = COMPOSE;
  }

  void deleteaddressfilter()
  {
    CAMAS.Filters.delete_filter_by_filternumber(id, sessobj->editaddressfilter);
    if (sessobj->editaddressfiltermode == "take")
    {
      sessobj->status = READMAIL;
      id->misc->camas->status = READMAIL;
    }
    else
    {
      sessobj->status = MAILFILTER;
      id->misc->camas->status = MAILFILTER;
    }
    array|void imp_command = CAMAS.Tools.save_preferences(id);
    if(imp_command && arrayp(imp_command))
      imap_commands += imp_command;
  }
  
  void canceleditaddressfilter()
  {
    if (sessobj->editaddressfiltermode != "old")
      sessobj->newaddressfilter = "";

    if (sessobj->editaddressfiltermode == "take")
    {
      sessobj->status = READMAIL;
      id->misc->camas->status = READMAIL;
    }
    else
    {
      sessobj->status = MAILFILTER;
      id->misc->camas->status = MAILFILTER;
    }
  }

  void createfolder()
  {
    sessobj->status = FOLDERLIST;
    id->misc->camas->status = FOLDERLIST;
    if (id->variables->foldername && (sizeof (id->variables->foldername) > 0)
	&& (camas_main->QUERY (foldernamemaxlength) ? (strlen (id->variables->foldername) <= camas_main->QUERY (foldernamemaxlength)) : 1)
	&& ((id->variables->foldername = TO_UNICODE (id->variables->foldername)) != MSG(M_NEWMBOXNAME)))
    {
      string path=id->variables->path || "";
      int mbox_idx=-1;
      string newmbox="";
      int ok=0;

      if(path=="")
      {
       	newmbox=sessobj->mailpath+id->variables->foldername;
	ok=1;
      }
      else
      {
	// looking after current mailbox index
	mbox_idx=Array.search_array(sessobj->mailboxes,
				    lambda(array a,string path)
				    {
				      if(a[MB_FOLDERNAME_IDX] == path)
					return 1;
				      return 0;
				    },
				    path);
       	if(mbox_idx != -1) 
	{
	  newmbox = sessobj -> mailboxes[mbox_idx][MB_FOLDERNAME_IDX] + sessobj -> mailboxes[mbox_idx][MB_SEPARATOR_IDX] + id -> variables -> foldername;
	  ok = 1;
	}
      }
      if(ok) 
      {
       	if(sessobj -> mboxencode)
	  newmbox = CAMAS.Tools.encode_mboxname(newmbox);
	imap_commands += ({ CAMAS.IMAPTools.imap_cmd("create", "newmailbox", newmbox, "error", MSG(M_CREATEMBOXERROR)),
			    CAMAS.IMAPTools.imap_cmd("updatelist", 
						     "mailpath", sessobj->mailpath,
						     "sharedpath", sessobj->sharedpath,
						     "why", "create",
						     "newmailbox", newmbox )
			    /*CAMAS.IMAPTools.imap_cmd("low_list")*/ });
      }
    }
    else
    {
      sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
      sessobj->dialogactions = ({ "actionfolderlist" });
      sessobj->dialogtext = MSG(M_NEWMBOXNONAME);
      sessobj->dialogtarget = id->misc->camas->frame;
      sessobj->status = DIALOGBOX;
      id->misc->camas->status = DIALOGBOX;
    }
  }
  
  void deletefolder()
  {
    array (string) mboxes = ({});
    string mboxtext = "";
    array ind_variables = indices(id->variables);
    foreach(ind_variables,string var) 
    {
      if(id->variables[var] == "1" && sscanf(var,"delf_%s", string mbox)) 
      {
        mboxes += ({ mbox });
      	mboxtext += CAMASFOLDERTOOLS.mb_displayname_from_name(mbox,sessobj->mailboxes)+"\n";
      }
    }

    if (sizeof(mboxes) > 0) 
    {
      sessobj->dialogstrings = ({ MSG(M_DIALOGOK),
				   MSG(M_DIALOGCANCEL) });
      sessobj->deletemboxes = mboxes;
      sessobj->dialogactions = ({ "actiondeletefoldersure",
				   "actionfolderlist" });
      sessobj->dialogtext = MSG(M_MBOXREMOVEP)+"\n"+mboxtext;
    } 
    else 
    {
      sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
      sessobj->dialogactions = ({ "actionfolderlist" });
      sessobj->dialogtext = MSG(M_MBOXMARKONE);
    }
    sessobj->dialogtarget = id->misc->camas->frame;
    sessobj->status = DIALOGBOX;
    id->misc->camas->status = DIALOGBOX;
  }
  
  void deletefoldersure()
  {
    sessobj->status = FOLDERLIST;
    id->misc->camas->status = FOLDERLIST;
    imap_commands += ({ CAMAS.IMAPTools.imap_cmd("delete_folders", "folders", 
						 CAMAS.IMAPTools.imap_cmd_var("deletemboxes")),
			CAMAS.IMAPTools.imap_cmd("updatelist", 
						 "mailpath", sessobj->mailpath,
						 "sharedpath", sessobj->sharedpath,
						 "why", "delete",
						 "folders", 
						 CAMAS.IMAPTools.imap_cmd_var("deletemboxes"))
			/*CAMAS.IMAPTools.imap_cmd("low_list")*/ });
    array|void imp_command = CAMAS.Tools.save_preferences(id);
      if(imp_command && arrayp(imp_command))
        imap_commands += imp_command;
  }

  void folderlist()
  {
    sessobj->status = FOLDERLIST;
    id->misc->camas->status = FOLDERLIST;
    imap_commands += ({ CAMAS.IMAPTools.imap_cmd("low_list",
						 "mailpath", sessobj->mailpath,
						 "sharedpath", sessobj->sharedpath) });
    if (features->QUERY (foldersinfo))
      imap_commands += ({ CAMAS.IMAPTools.imap_cmd("scan_folders", "updatequota", 1) });
  }
  
  void folderlistshow()
  {
    sessobj->status = FOLDERLIST;
    id->misc->camas->status = FOLDERLIST;
  }

  void cancel()
  {
    if(sessobj->replytoidx == -1)
    {
      sessobj->status = MAILINDEX;
      id->misc->camas->status = MAILINDEX;
    }
    else
    {
      sessobj->status = READMAIL;
      id->misc->camas->status = READMAIL;
    }
    sessobj->attachments = ({ });
    CAMAS.Tools.init_temp_vars(id);
  }
  
  //! method: void spellcheck()
  //!  Launches the spellchecker specified in the CIF
  void _spellcheck ()
  {
    sessobj->spelling = ({ });
    sessobj->misspelled = ({ });
    sessobj->checkword = 0;

    object file1=Stdio.File();
    object file2=file1->pipe();
    object file3=Stdio.File();
    object file4=file3->pipe();
    string res;

    Process.create_process(({camas_main->QUERY (ispell),"-a","-d",(sessobj->ispelldict / ":")[0]})+ ((camas_main->QUERY (speller)=="aspell") ? ({ "--ignore-repl" }):({ })),(["stdin":file2,"stdout":file4 ]) );
    //add a space before each line to prevent ispell from interpreting single-word lines as commands
    //FIXME: how should charsets be treated?
    string stripped_message=Locale.Charset.encoder("iso-8859-1","")->feed(sessobj->message)->drain();
    file1->write((Array.map((stripped_message)/"\n",lambda(string s){return " "+s;}))*"\n");
    file1->close();
    file2->close();
    file4->close();
    res=file3->read();
    file3->close();

    array ispell_data=res/"\n";
    array input_rows=(stripped_message)/"\n";

    if(sizeof(ispell_data)>1) {
      int i,row=0,pos=0,pos2;
      string word,suggestions;
      for(i=1;i<sizeof(ispell_data)-1 && row<sizeof(input_rows);i++) {
	if(!sizeof(ispell_data[i])){ // next row
	  (sessobj->spelling)+=({ input_rows[row][pos..]+"\n" });
	  row++;
	  pos=0;
	}
	else {
	  // pos2 will be a position ahead original data, due to the extra space.
	  switch(ispell_data[i][0]) {
	  case '&': // misspelled, suggestions
	    sscanf(ispell_data[i],"& %s %*d %d:%s",word,pos2,suggestions);
	    (sessobj->spelling)+=({ input_rows[row][pos..pos2-2] ,({ word , word })+Array.map(suggestions/",",String.trim_whites) });
	    (sessobj->misspelled)+=({ sizeof(sessobj->spelling)-1 });
	    pos=pos2-1+sizeof(word);
	    break;
	  case '#': //misspelled
	    sscanf(ispell_data[i],"# %s %d",word,pos2);
	    (sessobj->spelling)+=({ input_rows[row][pos..pos2-2] ,({ word , word }) });
	    (sessobj->misspelled)+=({ sizeof(sessobj->spelling)-1 });
	    pos=pos2-1+sizeof(word);
	    break;

	  }
	}
      }
    }
  }

  void checkword()
  {
    sessobj->checkword = (int)id->variables->word;
  }
  
  void spellprev()
  {
    sessobj->checkword--;
  }
  
  void spellnext()
  {
    sessobj->checkword++;
  }
  
  void spellreplace()
  {
    (sessobj->spelling)[(sessobj->misspelled)[sessobj->checkword]][1]=TO_UNICODE(id->variables->newword);
    sessobj->checkword++;
  }
  
  void spellselect()
  {
    if( (((int)id->variables->selectedword) >= 0) && (((int)id->variables->selectedword) < sizeof((sessobj->spelling[(sessobj->misspelled)[sessobj->checkword]]))-2)) {
      (sessobj->spelling)[(sessobj->misspelled)[sessobj->checkword]][1]=(sessobj->spelling)[(sessobj->misspelled)[sessobj->checkword]][(int)id->variables->selectedword + 2];
    }
    sessobj->checkword++;
  }
  
  void spelldone()
  {
    sessobj->message="";
    foreach(sessobj->spelling,mixed foo) {
      if(arrayp(foo))
	      sessobj->message+=foo[1];
      else
      	sessobj->message+=foo;
    }
    sessobj->status = COMPOSE;
    id->misc->camas->status = COMPOSE;
  }
  
  void spellcheck2()
  {
    id->variables->ispelldict = id->variables->ispelldict2;
    handler->add_action("spellcheck");
  }
  
  void spellcheck()
  {
    if(sizeof(camas_main->QUERY (ispell))) {
			handler->add_action("savecomposefields");
      sessobj->ispelldict = id->variables->ispelldict;
      _spellcheck();
      if(sizeof(sessobj->spelling))
      {
      	sessobj->status = SPELLCHECK;
        id->misc->camas->status = SPELLCHECK;
      }
    }
  }
 
  /* here follow various helper functions used by the send() actions */
  
  string send_fixheaders()
  {
    array fixedaddr;
    string brokenaddr = "";
    if (id->variables->to)
    {
      fixedaddr = CAMAS.Tools.fixaddresses(TO_UNICODE(id->variables->to));
      sessobj->to=fixedaddr[0];
      if (sizeof(brokenaddr) == 0)
      	brokenaddr = fixedaddr[1];
    }
    if (id->variables->cc)
    {
      fixedaddr = CAMAS.Tools.fixaddresses(TO_UNICODE(id->variables->cc));
      sessobj->cc=fixedaddr[0];
      if (sizeof(brokenaddr) == 0)
      	brokenaddr = fixedaddr[1];
    }
    if (id->variables->bcc)
    {
      fixedaddr = CAMAS.Tools.fixaddresses(TO_UNICODE(id->variables->bcc));
      sessobj->bcc=fixedaddr[0];
      if (sizeof(brokenaddr) == 0)
      	brokenaddr = fixedaddr[1];
    }

    if (id->variables->bcc2admin && (int)sessobj->bcc2admin && sizeof(sessobj->bcc2admin))
    {
      sessobj->bcc += sessobj->bcc2admin;
    }

    if (id->variables->subject)
      sessobj->subject = TO_UNICODE (id->variables->subject);

    if (id->variables->message)
      sessobj->message = TO_UNICODE (id->variables->message);

    if (id->variables->from && features->QUERY (userfrom))
    {
      string name, address;
      sscanf(TO_UNICODE(id->variables->from), "%s<%s>", name, address);
      if(name)
      {
        name = String.trim_whites(name);
	sessobj->name = name;
      }
      if(address)
        sessobj->address = address;
    }
    
    if (sizeof (features->QUERY (sitesig)))
      sessobj->message += "\n" + features->QUERY (sitesig);
    return brokenaddr;
  }
  
  string send_reallysendmail(string to, string cc, string bcc, string subject, string messageid, int uid)
  {
    if (id->variables->dsnsuccess)
      sessobj->dsnsuccess = ((int) id->variables->dsnsuccess);
    else
      sessobj->dsnsuccess = 0;

    if (id->variables->dsndelay)
      sessobj->dsndelay = ((int) id->variables->dsndelay);
    else
      sessobj->dsndelay = 0;

    if (id->variables->mdn)
      sessobj->mdn = ((int) id->variables->mdn);
    else
      sessobj->mdn = 0;

    string maildata;
    mapping args = ([ 
      "includeipnumber": features->QUERY(includeipnumber),
      "siteorg": features->QUERY(siteorg),
      "addmaildomain": camas_main->QUERY(addmaildomain),
      "morecharsets": camas_main->QUERY(morecharsets),
      "shareddepth": camas_main->QUERY(defaultshareddepth),
      "pref_extraheader": camas_main->QUERY(pref_extraheader),
      "mdntype": camas_main->QUERY(mdntype),
      "uploaddir": camas_main->QUERY(uploaddir),
      "lang_charsets": CAMAS_LANGUAGE->lang_charsets,
      "camas_version": camas_main->camas_version
    ]);
    if(id->variables->from)
      args += ([ "from": HTTP_DECODE_STRING(id->variables->from) ]);
    array recipients;
    array res = sendmail (sessobj, to, cc, bcc, messageid, subject, sessobj->message, 0, args);
    maildata = res[0];
    recipients = res[1];
    CAMAS.Log.log (my_configuration (), "send_mail", ([ "from" : sessobj->address,
     "to" : to,
     "cc" : cc,
       /* "bcc" : bcc, */
     "size" : sizeof (maildata) ]));
    string user_date = "Unknown date";
    // uid = 0 or - 1 if we come from compose screen
    if(uid > 0)
      user_date = get_infos_from_imap_uid(uid)->INTERNALDATE || user_date;
    CAMAS.Log.user_log (my_configuration (), "send_mail", 
        ([
           "sentmaildata": maildata,
           "to": to,
           "cc": cc,
           "subject": subject,
           "messageid": messageid,
           "date": user_date
           ]), sessobj);

    if (camas_main->QUERY (sendmethod) == "SMTP")
    {
       string err = camas_main->smtp_send (sessobj, recipients, 
          maildata, camas_main->QUERY (smtpserver),
        (int)camas_main->QUERY (smtpport), camas_main->QUERY (smtpmaildomain));
      if(err && sizeof(err) > 0)
        throw(err);
    }
    else if(camas_main->QUERY (sendmethod) == "sendmail")
      low_sendmail(camas_main->QUERY(sendmail), sessobj->address, maildata);
      
    return res[2];
  }

	//! method: send_move_mail2imap(string location, array(int) uid)
	//!  Move given uids to the given location
	//! args: string location
	//!  The location where to copy the mails to move
	//!  Will be appended to the current mailbox root
	//! args: array(int) uid
	//!  The list of UIDS to move
  void send_move_mail2imap(string location, array(int) uid)
  {
    sessobj->copytobox =  CAMASFOLDERTOOLS.getmailboxroot(sessobj->mailbox, sessobj)
       + location;

    if(sessobj->mboxencode)
      sessobj->copytobox = CAMAS.Tools.encode_mboxname(sessobj->copytobox);

    if(camas_main->QUERY(buggyimap))
    {
      CDEBUG("buggyimap => creating: " + sessobj->copytobox + "\n");
      imap_commands += ({ CAMAS.IMAPTools.imap_cmd ("create", "newmailbox", sessobj->copytobox) });
    }

    imap_commands += ({ CAMAS.IMAPTools.imap_cmd("move", "uids", uid, "updatembox", CAMAS.IMAPTools.imap_cmd_var("mails") ) });
  }
 
 	//! method: send_notify(int replytoidx, int replytouid)
  void send_notify(int replytoidx, int replytouid)
  {
		// Don't go to READMAIL if the answered mail has been moved
    if(replytoidx == -1 || sessobj->usermove)
      sessobj->dialogactions = ({ "actionmailindex" });
    else
    {
      sessobj->dialogactions = ({ "actionread" });
      sessobj->dialogoptions = ({
				    "msguid=" + replytouid,
				    "mbox=" + HTTP_ENCODE_URL(sessobj->mailbox[MB_FOLDERNAME_IDX])
				  });
    }

    sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
    sessobj->dialogtarget = id->misc->camas->frame;
    sessobj->status = DIALOGBOX;
    id->misc->camas->status = DIALOGBOX;
  }
  
  void send_copytosent(string maildata, string to, string cc, string bcc, string subject)
  {
    if (camas_feature (FEAT_MAILBOXES) &&
  ((camas_feature (FEAT_SAVEMAILCOPY) && id->variables->nosavemail == 0) ||
   (!camas_feature (FEAT_SAVEMAILCOPY) && !(id->variables->dosavemail==0))) &&
    sizeof(sessobj->sentfolder))
    {
      string appendbox = CAMASFOLDERTOOLS.getmailboxroot(sessobj->mailbox, sessobj)
      + sessobj->sentfolder;

      if(sessobj->mboxencode)
        appendbox=CAMAS.Tools.encode_mboxname(appendbox);

      if (camas_main->QUERY (buggyimap))
      {
        CDEBUG("buggyimap => creating: " + appendbox + "\n");
        imap_commands += ({ CAMAS.IMAPTools.imap_cmd ("create", "newmailbox", appendbox) });
      }
      imap_commands+= ({CAMAS.IMAPTools.imap_cmd("low_append","tobox",appendbox,"data",
        maildata,"error",MSG(M_SENDSAVEFAILED)) });
    }
  }

  void send_move(int replytouid)
  {
	
    /* In int we are sending a mail that is a reply to another one */
    int replytoidx = CAMAS.Tools.find_mail(sessobj->mails, replytouid);
    CDEBUG("new sessobj->replytoidx: "+replytoidx+"\n");

    if(replytoidx != -1)
    {
      sessobj->mails[replytoidx]->imap->FLAGS += ({ "\\Answered" });
      imap_commands += ({ CAMAS.IMAPTools.imap_cmd("add_flag","uid",replytouid,"flag","\\Answered" )});
      if(features->QUERY(moveanswered) && sessobj->usermove)
        send_move_mail2imap(sessobj->sentfolderaction, ({ replytouid }));
    }

    /* In int we are sending a forward */

    CDEBUG("sessobj->forwardidx: "+sessobj->forwardidx+"\n");
    CDEBUG("sessobj->forwarduid: "+sessobj->forwarduid+"\n");
    // Fool proof
    sessobj->forwardidx = CAMAS.Tools.find_mail(sessobj->mails, sessobj->forwarduid);
    CDEBUG("new sessobj->forwardidx: "+sessobj->forwardidx+"\n");

    if(sessobj->forwardidx != -1 && sessobj->usermove)
      send_move_mail2imap(sessobj->sentfolderaction, ({ sessobj->forwarduid }));

    /* In int we are sending a mail that is a draft */
    CDEBUG("sessobj->draftidx: "+sessobj->draftidx+"\n");
    CDEBUG("sessobj->draftuid: "+sessobj->draftuid+"\n");
    // Fool proof
    sessobj->draftidx = CAMAS.Tools.find_mail(sessobj->mails, sessobj->draftuid);
    CDEBUG("new sessobj->draftidx: "+sessobj->draftidx+"\n");

    if(sessobj->draftidx != -1 && sessobj->usermove)
      // TODO: features->QUERY(movedraft)
      send_move_mail2imap(sessobj->sentfolderaction, ({ sessobj->draftuid }));
  }

  int send_failed(string brokenaddr)
  {
    sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
    sessobj->dialogactions = ({ "actioncontinuecompose" });
    if (sizeof(sessobj->to) == 0)
      sessobj->dialogtext = MSG(M_SENDNORECV);
    else
      sessobj->dialogtext = MSGA(M_SENDBROKENADDR, ({ CAMAS.Tools.fix_header(brokenaddr) }));
    sessobj->status = DIALOGBOX;
    id->misc->camas->status = DIALOGBOX;
    return 0;
  }

  void send()
  {
    string brokenaddr = send_fixheaders();
    mixed err = 0;
    array(string) good_rcpts, bad_rcpts;
    good_rcpts = ({ });
    bad_rcpts = ({ });
    array(string) messageids;
    array(int) uids;
    int recipients;
    array alreadysended = ({ });
    if(sizeof(sessobj->to) == 0 || sizeof(brokenaddr) > 0)
    {
      // bad recipient(s)
      send_failed(brokenaddr);
      return;
    }

    // if we reply from mailindex screen and not readmail
    if((int)id->variables->blindsend && features->QUERY(userblindsend))
    {
      messageids = sessobj->messageids;
      uids       = sessobj->uids;
      recipients = sizeof(CAMAS.Tools.address_split(sessobj->to));
    }
    else
    {
      messageids = ({ sessobj->messageid });
      uids       = ({ sessobj->replytouid });
      recipients = 1;
    }
    sessobj->dialogtext = "";

    for(int i = 0; i < recipients; i++)
    {
      string cc = sessobj->cc, bcc = sessobj->bcc, 
        subject = sessobj->subject;
      string to = sessobj->to;
      string messageid = sessobj->messageid;
      int uid;
      if(messageids && sizeof(messageids) > 0)
      {
        messageid  = messageids[0];
        messageids = messageids[1..];
      }
      if(uids && sizeof(uids) > 0)
      {
        uid  = uids[0];
        uids = uids[1..];
      }
      if(id->variables->blindsend && features->QUERY(userblindsend))
      {
        // we reply from mailindex to maybe several mails at once
        if(sessobj->tos && sessobj->tos[messageid])
        {
           to = sessobj->tos[messageid];
        }
        else
        {
          // the user added an entry manually and he have 
          // to find it
          array tos = CAMAS.Tools.address_split(sessobj->to);
      	  tos = Array.map(tos, lambda(string _to) {
            sscanf(_to, "%*s<%s>%*s", _to);
      	    return String.trim_all_whites(_to); });
          tos = Array.filter(tos, lambda(string _to) {
            if(search(alreadysended, _to) == -1) 
              return 1; });
          to = tos[0];
        }
      }
      string tmp = to;
      sscanf(tmp, "%*s<%s>%*s", tmp);
      alreadysended += ({ String.trim_all_whites(tmp) });
      string maildata;
      CDEBUG(sprintf("sending mail to=%O, cc=%O, bcc=%O in_reply_to=%O\n", to, cc, bcc, messageid));
      err =
        catch 
        {
          maildata = send_reallysendmail(to, cc, bcc, subject, messageid, uid);
        };
      CDEBUG(sprintf("error from sending mail=%s\n", describe_backtrace(err)));
      string all_recipients = to;
      if(cc != "")
        all_recipients += ", " + cc;
      if(bcc != "")
        all_recipients += ", " + bcc;
      if(!err)
      {
        send_copytosent(maildata, to, cc, bcc, subject);
        good_rcpts += ({ all_recipients });
      }
      else
      {
        if (sessobj->displaysmtperrors)
          sessobj->dialogtext += "\n" + err[0];
        bad_rcpts += ({ all_recipients });
      }
    } // for
    // some mails were not sent
    if (sizeof(bad_rcpts) > 0)
    {
      sessobj->status = DIALOGBOX;
      id->misc->camas->status = DIALOGBOX;
      sessobj->dialogactions = ({ "actioncontinuecompose" });
      sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
      sessobj->dialogtext += MSGA(M_SMTPERROR, ({ bad_rcpts * ", "}));
      CAMAS.Log.log (my_configuration (), "smtp_errors", ([ "login" : sessobj->address,
         "command" : "smtp_send",
         "line" : error ]));
    }
    // all mails were sent successfully
    else if(sizeof(good_rcpts) > 0)
    {
      sessobj->attachments = ({ }); // Remove all attachments from mem if already sent
      if(sessobj->uids && sizeof(sessobj->uids) > 0)
        sessobj->replytoidx = -1;
      array(int) replytouids;
      if(sessobj->uids && sizeof(sessobj->uids))
        replytouids = sessobj->uids;
      else
        replytouids = ({ sessobj->replytouid });
      foreach(replytouids, int replytouid)
        send_move(replytouid); 
      
      if (camas_main->QUERY (displaysmtpok))
      {
        sessobj->dialogtext += MSGA (M_SMTPOK, ({ sessobj->subject, good_rcpts * ", " }));
        send_notify(sessobj->replytoidx, sessobj->replytouid);
      }
      else
      {
        if (sessobj->replytoidx == -1)
        {
          sessobj->status = MAILINDEX;
          id->misc->camas->status = MAILINDEX;
        }
        else
        {
          sessobj->status = READMAIL;
          id->misc->camas->status = READMAIL;
        }
      }
    }
    if(sizeof(imap_commands)) 
    {
      CDEBUG(sprintf("imap_commands = %O\n", imap_commands));
      CAMAS.Tools.init_temp_vars(id);
    }
  }
  
  void savedraft(string savedraftto)
  {
    send_fixheaders();

    if(!savedraftto || !sizeof((string)savedraftto))
    {
      savedraftto =  CAMASFOLDERTOOLS.getmailboxroot(sessobj->mailbox, sessobj)
        + sessobj->draftsfolder;
      CDEBUG("savedraftto: "+savedraftto+"\n");
    }

    sessobj->status = MAILINDEX;
    id->misc->camas->status = MAILINDEX;

    sessobj->selected = ({ });
    mapping args = ([ 
      "includeipnumber": features->QUERY(includeipnumber),
      "siteorg": features->QUERY(siteorg),
      "addmaildomain": camas_main->QUERY(addmaildomain),
      "morecharsets": camas_main->QUERY(morecharsets),
      "shareddepth": camas_main->QUERY(defaultshareddepth),
      "pref_extraheader": camas_main->QUERY(pref_extraheader),
      "mdntype": camas_main->QUERY(mdntype),
      "uploaddir": camas_main->QUERY(uploaddir),
      "lang_charsets": CAMAS_LANGUAGE->lang_charsets,
      "camas_version": camas_main->camas_version
    ]);
    string maildata = sendmail (sessobj, sessobj->to, sessobj->cc, 
      sessobj->bcc, sessobj->messageid, sessobj->subject, sessobj->message, 1, args)[0];
    sessobj->attachments = ({ }); // Remove all attachments from mem
    imap_commands += ({
			CAMAS.IMAPTools.imap_cmd ("low_append",
				  "tobox", savedraftto,
				  "data", maildata),
		      });

    if ((sessobj->mailbox[MB_FOLDERNAME_IDX] == 
      (sessobj->mailpath+sessobj->draftsfolder)) && sessobj->draftuid)
    {
      imap_commands += ({
			  CAMAS.IMAPTools.imap_cmd ("delete",
				    "mailbox", (sessobj->mailpath + sessobj->draftsfolder),
				    "uids", ({ sessobj->draftuid })),
			  CAMAS.IMAPTools.imap_cmd ("status","mailbox",sessobj->mailbox[MB_FOLDERNAME_IDX]),
			  CAMAS.IMAPTools.imap_cmd ("get_headers",
				    "output", CAMAS.IMAPTools.imap_cmd_var("mails"),
				    "setsortcolumn",CAMAS.IMAPTools.imap_cmd_var("mailssortcolumn"))
			});
      sessobj->firstvisiblemail = -1; // eq to updatelistpos
    }

    // Foolproof: recalculate idx from uid
    sessobj->replytoidx = CAMAS.Tools.find_mail(sessobj->mails, sessobj->replytouid);

    /* If we are saving a draft that is a reply to another one */
    if(sessobj->replytoidx != -1)
    {
      sessobj->mails[sessobj->replytoidx]->imap->FLAGS += ({ "\\Answered" });
      imap_commands += ({ CAMAS.IMAPTools.imap_cmd("add_flag","uid",sessobj->mails[sessobj->replytoidx]->imap->UID,"flag","\\Answered" )});
      if(features->QUERY(moveanswered) && sessobj->usermove)
        send_move_mail2imap(sessobj->answeredfolder, ({ sessobj->mails[sessobj->replytoidx]->imap->UID }));
    }

    // TODO: verify forward
    // Foolproof: recalculate idx from uid
    sessobj->forwardidx = CAMAS.Tools.find_mail(sessobj->mails, sessobj->forwarduid);

    /* If we are saving a draft that is a forward to another one */
    if(sessobj->forwardidx != -1)
    {
      // TODO: flag \\Forwarded?
      //sessobj->mails[sessobj->replytoidx]->imap->FLAGS += ({ "\\Answered" });
      //imap_commands += ({ CAMAS.IMAPTools.imap_cmd("add_flag","uid",sessobj->mails[sessobj->replytoidx]->imap->UID,"flag","\\Answered" )});
      if(sessobj->usermove)
        send_move_mail2imap(sessobj->answeredfolder, ({ sessobj->forwarduid }));
    }

    CAMAS.Tools.init_temp_vars(id);
  }
  
  void deleteall()
  {
    array uids = Array.map (sessobj->mails, lambda (mapping m) { return m->imap->UID; });
    if (sizeof (uids)) {
      array cmds = ({ });
      for (int i = 0; i < sizeof (uids); ) {
	array uids2 = ({ });
	for (int k = 0; (k < 100) && (i < sizeof (uids)); k++)
	  uids2 += ({ uids[i++] });
	  imap_commands += ({ CAMAS.IMAPTools.imap_cmd ("delete", "uids", uids2 + ({ }), "updatembox", CAMAS.IMAPTools.imap_cmd_var ("mails")) });
      }
      imap_commands += ({ CAMAS.IMAPTools.imap_cmd ("get_headers", "output", CAMAS.IMAPTools.imap_cmd_var("mails"),
			   "setsortcolumn", CAMAS.IMAPTools.imap_cmd_var ("mailssortcolumn")),
		 CAMAS.IMAPTools.imap_cmd ("low_close"),    CAMAS.IMAPTools.imap_cmd ("status", "mailbox", "INBOX") /*CAMAS.IMAPTools.imap_cmd ("low_close")*/ });
      sessobj->firstvisiblemail = -1; // eq to updatelistpos	
    }
  }
  
  void delete()
  {
    getselected();
    if (sizeof (sessobj->selected))
      imap_commands += ({ CAMAS.IMAPTools.imap_cmd("delete","uids",sessobj->selected,"updatembox",CAMAS.IMAPTools.imap_cmd_var("mails")) });
  }
  
  void deleteask()
  {
    getselected();
    if (sizeof (sessobj->selected))
    {
      sessobj->dialogstrings = ({ MSG(M_DIALOGOK), MSG(M_DIALOGCANCEL) });
      sessobj->deletemails = sessobj->selected;
      sessobj->dialogactions = ({ "actiondeletesure", "actionindex" });
      sessobj->dialogtext = MSGA(M_DELETEMARKEDP, ({ sizeof(sessobj->selected) }));
    }
    else
    {
      sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
      sessobj->dialogactions = ({ "actionindex" });
      sessobj->dialogtext = MSG(M_DELETEMARKEDNONE);
    }
    sessobj->dialogtarget = id->misc->camas->frame;
    sessobj->status = DIALOGBOX;
    id->misc->camas->status = DIALOGBOX;
  }
  
  void deletesure()
  {
    //FIXME: move from sessobj to url
    sessobj->status = MAILINDEX;
    id->misc->camas->status = MAILINDEX;
    imap_commands += ({ CAMAS.IMAPTools.imap_cmd("delete", "uids",sessobj->deletemails+({ }),"updatembox",CAMAS.IMAPTools.imap_cmd_var("mails")) });
  }

  void trash()
  {
    getselected();
    if (sizeof (sessobj->selected) > 0) {
      send_move_mail2imap(sessobj->trashfolder, sessobj->selected+({ }));

      mapping messageids = ([ ]); // mapping to store messageids of selecting mails

      int   i = 0,      // loop counter
            j = 0;      // counter for mails put into messageids

      CDEBUG(sprintf("uids of selected mails: %O\n\n",sessobj->selected));

      for(i=0; i<sizeof(sessobj->mails); i++){

   	    if(has_value(sessobj->selected, sessobj->mails[i]->imap->UID))
       	{
           messageids += ([ j:sessobj->mails[i]->imap->ENVELOPE[MESSAGEID_IDX] ]);
           j++;
       	}
      }
      CDEBUG(sprintf("all messagesids of selected mails: %O\n",messageids));

      return;
    }
    sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
    sessobj->dialogactions = ({ "actionindex" });
    sessobj->dialogtarget = id->misc->camas->frame;
    sessobj->dialogtext = MSG(M_DELETEMARKEDNONE);
    sessobj->status = DIALOGBOX;
    id->misc->camas->status = DIALOGBOX;
  }

	//! method: movethis()
	//!  Moves current displayed mail to selected mailbox
	//!  Used from READMAIL
	void movethis()
	{
		handler->add_action("move");

		// If there's a next mail in the list, prepare to display it
		// Otherwise, go back to MAILINDEX
		if((int)id->variables->nextuid != -1)
		{
			imap_commands += 
				({
					CAMAS.IMAPTools.imap_cmd("get_mail", "mailbox", id->variables->mbox||0, "uid", (int)id->variables->nextuid, "output", CAMAS.IMAPTools.imap_cmd_var("cmail")),
					CAMAS.IMAPTools.imap_cmd("check")
				});

			mapping cmail = ([ ]);

			if(sizeof(sessobj->mails))
			{
				cmail = sessobj->mails[sessobj->cmailidx];
				sessobj->cmailuid = cmail->imap->UID;
				sessobj->selected = ({ cmail->imap->UID });
			}
		}
		else
		{
			id->misc->camas->status = MAILINDEX;
			sessobj->status = MAILINDEX;
		}
	}

	//! method: movemarked()
	//!  Moves marked mails to selected mailbox
	//!  Used from MAILINDEX
	void movemarked()
	{
		handler->add_action("move");
	}

	//! method: move()
	//!  Move mails
  void move()
  {
    getselected();

    string mboxtomoveto;

    // if user has chosen a mailbox to move to and mails to move
    if(id->variables->mboxmove && id->variables->mboxmove!="imhonomailbox" && sizeof(sessobj->selected))
		{
      mboxtomoveto = id->variables->mboxmove;
   
    	// if the chosen mailbox is different from the actual mailbox
    	if(sessobj->mailbox[MB_FOLDERNAME_IDX] != mboxtomoveto)
    	{
      	sessobj->copytobox=mboxtomoveto;
			
				// if user can do autofiltering 
      	if(features->QUERY(filterbook) && features->QUERY(autofilter) && features->QUERY(userautofilter))
      	{
					id->variables->namefield = "From";
      		id->variables->filterfolder = mboxtomoveto;
      		int mails = sizeof (sessobj->mails);
       		int selected = sizeof (sessobj->selected);
					if (mails)
       		{
       	  	for (int i = 0; i < mails; i++)
      	  	{
      	    	object mail = sessobj->mails[i];
      	    	for (int j = 0; j < selected; j++)
      	    	{
       	      	if (mail->imap->UID == sessobj->selected[j])
      	      	{
									id->variables->filterexpression = mail->imap->ENVELOPE[FROM_IDX][0][0];
									if (id->variables->namefield && id->variables->filterfolder && 
		    						id->variables->filterexpression)
									{
		  							sessobj->editaddressfiltermode = "new";
		  							_editaddressfilterdone(id, 1);
									}
	      				}
	    				}
	  				}
      		}
        	array|void imp_command = CAMAS.Tools.save_preferences(id);
        	if(imp_command && arrayp(imp_command))
          	imap_commands += imp_command;
				}
				
      	imap_commands += ({ CAMAS.IMAPTools.imap_cmd("move","uids",sessobj->selected+({ }),"updatembox",CAMAS.IMAPTools.imap_cmd_var("mails") ) });
      }
    }
  }
  
  void reload()
  {
    sessobj->status=MAILINDEX;
    id->misc->camas->status = MAILINDEX;
    // Resetting all search stuff
    sessobj->searchstring = "";
    sessobj->nothingfound = 0;
    sessobj->stickyfilter = 0;
    sessobj->historysearch = 0;

    // In order to sync the IMAP cache for headers we need to get headers on
    // every mailboxes
    // Note that this code will not be executed if you don't use the background
    // IMAP client at startup
    if (sessobj->cachefeeded)
    {
      foreach (sessobj->mailboxes, array mailbox)
      {
      	if (mailbox[MB_FOLDERNAME_IDX] != sessobj->mailbox[MB_FOLDERNAME_IDX]
    	   && !(mailbox[MB_FLAGS_IDX] & MB_NOSELECT) 
         && !(camas_main->notreloadmailboxes && 
               Array.search_array(camas_main->notreloadmailboxes, 
                 lambda(string mbox) { return has_value(mailbox[MB_FOLDERNAME_IDX], mbox); }) != -1))
      	{
      	  imap_commands += ({ CAMAS.IMAPTools.imap_cmd("status","mailbox",mailbox[MB_FOLDERNAME_IDX]), 
			      CAMAS.IMAPTools.imap_cmd("get_headers", "mailbox",mailbox[MB_FOLDERNAME_IDX], "output",CAMAS.IMAPTools.imap_cmd_var("dummy")) });
      	}
      }
    }
    else
      imap_commands += ({ CAMAS.IMAPTools.imap_cmd("scan_folders", "reload", 1) });
    imap_commands += ({ CAMAS.IMAPTools.imap_cmd("status","mailbox", sessobj->mailbox[MB_FOLDERNAME_IDX]), 
			CAMAS.IMAPTools.imap_cmd("get_headers", "mailbox", sessobj->mailbox[MB_FOLDERNAME_IDX], "output",CAMAS.IMAPTools.imap_cmd_var("mails"),"setsortcolumn",CAMAS.IMAPTools.imap_cmd_var("mailssortcolumn")) }); 
    sessobj->firstvisiblemail = -1; // not here will update only if asnewmail -- pb with showallafter search
  }
  
  void checkactivemailboxes()
  {
    // Just check, don't reload
    imap_commands += ({ CAMAS.IMAPTools.imap_cmd ("check_mailboxes"/*, "noselect", 1*/) });
  }
  
  void reloadactivemailboxes()
  {
    sessobj->searchstring = "";
    imap_commands += ({
					CAMAS.IMAPTools.imap_cmd ("check_mailboxes"),
					CAMAS.IMAPTools.imap_cmd ("status", "mailbox", sessobj->mailbox[MB_FOLDERNAME_IDX]),
					CAMAS.IMAPTools.imap_cmd ("get_headers", "output", CAMAS.IMAPTools.imap_cmd_var ("mails"),
								  "setsortcolumn", CAMAS.IMAPTools.imap_cmd_var ("mailssortcolumn")),
					CAMAS.IMAPTools.imap_cmd ("apply_mail_filters", "updatembox", CAMAS.IMAPTools.imap_cmd_var ("mails")) });
    sessobj->firstvisiblemail = -1; // not here will update only if asnewmail -- pb with showallafter search
  }
  
  void togglesortorder()
  {
    if (!sessobj->sessionsortorder)
      sessobj->sessionsortorder = sessobj->sortorder;
    if (sessobj->sessionsortorder == "forward")
      sessobj->sessionsortorder = "backward";
    else
      sessobj->sessionsortorder = "forward";
    sessobj->sortordertoggled = 1;
  }
  
  void changesort()
  {
    if (id->variables->col) {
      sessobj->sortcolumn = id->variables->col;
    }
  }
  
  void trashthis()
  {
    int prevuid=(int)id->variables->prevuid || -1;
    int nextuid=(int)id->variables->nextuid || -1;
    int uid=(nextuid==-1?prevuid:nextuid);
    getselected();
    sessobj->cmailidx = CAMAS.Tools.find_mail(sessobj->mails, uid);
    sessobj->cmailuid = uid;
    if (uid ==-1)
    {
      sessobj->status = MAILINDEX;
      id->misc->camas->status = MAILINDEX;
      send_move_mail2imap(sessobj->trashfolder, sessobj->selected + ({ }));
    }
    else
    {
      send_move_mail2imap(sessobj->trashfolder, sessobj->selected + ({ }));
      imap_commands += ({ CAMAS.IMAPTools.imap_cmd("get_mail","uid",uid,"output",CAMAS.IMAPTools.imap_cmd_var("cmail")), CAMAS.IMAPTools.imap_cmd ("check") });
    }
    CDEBUG(sprintf("       uid selected: %O\n", sessobj->selected ));
    if(!((sessobj->status == READMAIL || id->misc->camas->status == READMAIL) && (int)id->variables->nextuid != -1))
    {
      sessobj->status = MAILINDEX;
      id->misc->camas->status = MAILINDEX;
    }
  }
  
  void deletethis()
  {
    getselected();
    int prevuid=(int)id->variables->prevuid || -1;
    int nextuid=(int)id->variables->nextuid || -1;
    int uid=(nextuid==-1?prevuid:nextuid);
    sessobj->cmailidx = CAMAS.Tools.find_mail(sessobj->mails, uid);
    sessobj->cmailuid = uid;
    if (uid == -1) {
      sessobj->status = MAILINDEX;
      id->misc->camas->status = MAILINDEX;
      imap_commands += ({ CAMAS.IMAPTools.imap_cmd("delete","uids",sessobj->selected+({ }),"updatembox",CAMAS.IMAPTools.imap_cmd_var("mails") ) });
    } else {
       imap_commands += ({ CAMAS.IMAPTools.imap_cmd("delete","uids",sessobj->selected+({ }),"updatembox",CAMAS.IMAPTools.imap_cmd_var("mails") ), CAMAS.IMAPTools.imap_cmd("get_mail","uid",uid,"output",CAMAS.IMAPTools.imap_cmd_var("cmail")), CAMAS.IMAPTools.imap_cmd ("check") });
    }
  }
  
  void readprev()
  {
    read((int)id->variables->prevuid);
  }
  void readnext()
  {
    read((int)id->variables->nextuid);
  }
  void read(int uid)
  {
    if(sessobj->status == MDNDIALOG)
      removemdn();
    sessobj->status = READMAIL;
    id->misc->camas->status = READMAIL;

    if(!uid)
      uid=(int)id->variables->msguid;

    if(uid)
    {
      sessobj->cmailidx = CAMAS.Tools.find_mail(sessobj->mails, uid);
      sessobj->cmailuid = uid;
    }
    if (sessobj->cmailidx != -1)
    {
      imap_commands += ({ CAMAS.IMAPTools.imap_cmd("get_mail","mailbox",id->variables->mbox||0,"uid",uid,"output",CAMAS.IMAPTools.imap_cmd_var("cmail")), CAMAS.IMAPTools.imap_cmd ("check") });
    }
    else
    {
      sessobj->status = MAILINDEX;
      id->misc->camas->status;
    }
  }
   
  void showheaders()
  {
    sessobj->showheaders = 1;
  }
  
  void hideheaders()
  {
    sessobj->showheaders = 0;
  }

  //! action : replytoall_formatmessage(object mess)
  //!  Sets up the message the body of the answer, like original message, charset and so on
  void replytoall_formatmessage(object mess)
  {
    if(mess->type == "text" && mess->subtype != "html")
    {
      mixed err=0;
      string msg;
      err=catch {msg=Locale.Charset.decoder(mess->charset)->feed(mess->getdata()||"")->drain();};
      if(err)
       	msg=mess->getdata()||"";
      sessobj->originalmessage = msg;
      sessobj->message = sessobj->replymsgprefix + replace(msg,"\n","\n"+sessobj->replymsgprefix);
      sessobj->replytocharset = err?"iso-8859-1":mess->charset;
    }
    else
    {
      CDEBUG("mess->type: "+mess->type+" mess->subtype: "+mess->subtype+"\n");
      mixed err=0;
      string msg;
      sessobj->message = sessobj->replymsgprefix;
      err=catch {msg=Locale.Charset.decoder(mess->charset)->feed(mess->getdata()||"")->drain();};
      if(err)
      {
      	msg=mess->getdata()||"";
      	CDEBUG("error in decoding message\n");
      }
      string message = CAMAS.Tools.html2text(msg);
      // Fix for stupid ASPMail that mixes encodings... -- Bertrand
      message = replace(message,"&eacute;","<E9>");
      sessobj->originalmessage = msg;
      sessobj->message += replace(message,"\n","\n"+sessobj->replymsgprefix);
      sessobj->replytocharset = err?"iso-8859-1":mess->charset;
    }
  }
  
  //! action : replymove()
  //!  Reply to an email and move it to the answered-mail directory
  void replymove()
  {
    sessobj->usermove = 1;
    sessobj->sentfolderaction = sessobj->answeredfolder;
    handler->add_action("reply", 1);
  }

  //! action: replytoallmove()
  //!  Reply to all the recipients of an email and move it to the answered-mail directory
  void replytoallmove()
  {
    CAMAS.Tools.init_temp_vars(id);
    sessobj->usermove = 1;
    sessobj->sentfolderaction = sessobj->answeredfolder;
    handler->add_action("replytoall", 1);
  }

  //! action : replytoall()
  //!  Reply to all the recipients of an email, ie all addresses in "To:" and "Cc:"
  void replytoall(int indirectcall)
  {
    handler->add_action("reply", indirectcall);
    object mail = sessobj->cmail;
    sessobj->cc = CAMAS.Tools.fixaddresses(CAMAS.Tools.fix_coding(mail->headers->to))[0];
    if(mail->headers->cc)
      sessobj->cc += ", " + CAMAS.Tools.fixaddresses(CAMAS.Tools.fix_coding(mail->headers->cc))[0];
  }

  //! action : reply (int indirectcall)
  //!  Reply to a email
  //!  If a "reply-to" field is set, send en email there
  //!  Else, send an email to "from"
	//! args: int indirectcall
	//!  0 if method called by the user
	//!  1 if called by another method
  void reply(int indirectcall)
  {
    if(!indirectcall)
      CAMAS.Tools.init_temp_vars(id);

    object mail = sessobj->cmail;
    sessobj->cc = "";

    if(mail->headers["reply-to"] && sizeof(mail->headers["reply-to"]))
      sessobj->to = CAMAS.Tools.fixaddresses(CAMAS.Tools.fix_coding(mail->headers["reply-to"]))[0];
    else
      sessobj->to = CAMAS.Tools.fixaddresses(CAMAS.Tools.fix_coding(mail->headers->from))[0];
    sessobj->subject = CAMAS.Tools.fix_coding(mail->headers->subject);

    if (lower_case(sessobj->subject[0..2]) != "re:")
      sessobj->subject = "Re: "+sessobj->subject;
    if (mail->headers["message-id"])
      sessobj->messageid = mail->headers["message-id"];

    sessobj->message = "";
    sessobj->messageids = 0;
    sessobj->tos = 0;
    sessobj->uids = 0;
    if (sessobj->replyincludemsg == "1")
    {
      array(object) msgs = ({ mail });
      while(sizeof(msgs) > 0)
      {
      	object mess = msgs[0];
      	msgs = msgs[1..];
      	if (mess->body_parts)
        {
      	  msgs = mess->body_parts + msgs;
        }
      	else
      	{
      	  replytoall_formatmessage(mess);
          msgs = ({ });
      	}
      }
			if(sessobj->replyontop == "0")
      	sessobj->message = "\n" + MSG(M_FWD_REP_SEP) + "\n" + sessobj->message + "\n\n" + sessobj->signature;
			else
				sessobj->message = "\n" + sessobj->signature + "\n" + MSG(M_FWD_REP_SEP) + "\n" + sessobj->message;
    }
    sessobj->bcc = sessobj->cc = "";
    if(sessobj->autobcc && sessobj->autobcc != "0")
      sessobj->bcc = CAMAS.Tools.fix_coding(sessobj->autobcc);
    else
      sessobj->bcc = "";
    sessobj->replytoidx = sessobj->cmailidx;
    sessobj->replytouid = sessobj->cmailuid;
    sessobj->status = COMPOSE;
    id->misc->camas->status = COMPOSE;
    sessobj->attachments = ({ });
  }

  //! action: replytoallmarked
  //!  Reply to all poeple in To: and CC: in several mails 
  void replytoallmarked()
  {
    CAMAS.Tools.init_temp_vars(id);
    handler->add_action("replymarked", 1, 1);
  }

  //! action: replytoallmarked
  //!  Reply to all poeple in To: and CC: in several mails 
  //!  and move the original mail in the answered folder
  void replytoallmovemarked()
  {
    CAMAS.Tools.init_temp_vars(id);
    sessobj->usermove = 1;
		sessobj->sentfolderaction = sessobj->answeredfolder;
    handler->add_action("replymarked", 1, 1);
  }

  mapping get_infos_from_imap_uid(int uid)
  {
    foreach(sessobj->mails->imap, mapping imap)
    {
      int imapuid = imap->UID;
      if(uid == imapuid)
        return imap;
    }
    // if we are here we didn't find any matching mails and that's bad
    throw(({ "Can't find any mails with uid (" + uid + ")", backtrace() }));
  }

  //! action : replymovemarked()
  //!  Mass reply many marked emails and move them in the answered-mail directory
  void replymovemarked()
  {
    CAMAS.Tools.init_temp_vars(id);
    sessobj->usermove = 1;
		sessobj->sentfolderaction = sessobj->answeredfolder;
    handler->add_action("replymarked", 1);
  }
 
  //! action : replymarked
  //!  Mass reply many marked emails
 	//! args: int indirectcall
	//!  0 if method called by the user
	//!  1 if called by another method
  //!  void|int replytoall
  //!  0 if we reply only to the poeple in the To: header
  //!  1 if we reply to poeple in the To: and CC: headers
  //!  Note: doesn't use the reply-to header
  void replymarked(int indirectcall, void|int replytoall)
  {
    if(!indirectcall)
      CAMAS.Tools.init_temp_vars(id);
			
    getselected();
    if(!sessobj->message)
      sessobj->message = "";

		if(sessobj->session)
		{
      // if no messages has been selected, display the page again
      if(!sessobj->selected || sizeof(sessobj->selected) == 0)
        return;
			// If only one message selected for which we have the contents (that is
      // the user is reading it ) then we can revert to normal mode
      // which is safer
			if(sizeof(sessobj->selected) == 1 && sessobj->cmail
          && sessobj->cmailuid == sessobj->selected[0])
			{
				sessobj->uids = ({ });
        sessobj->replytoidx = sessobj->cmailidx;
        sessobj->replytouid = sessobj->cmailuid;
        if(replytoall)
          handler->add_action("replytoall", 1);
        else
				  handler->add_action("reply", 1);
			}
			// If several messages selected, mass reply them
			else
			{
      	sessobj->uids = ({ });
      	array addrs = ({ });
      	array imap;
        array ccs = ({ });
      	sessobj->messageids = ({ });
      	sessobj->tos = ([ ]);
        sessobj->cc = "";
      	foreach(sessobj->selected, int uid)
      	{
        	int idx = CAMAS.Tools.find_mail(sessobj->mails, uid);
          // convert imap array of addresses shit generated by the 
          // imapclient parser to a real formatted address
          function get_addresses = lambda(array imap_addr) {
            array result = ({ });
            if(imap_addr)
              foreach(imap_addr, array recipients)
              {
                string real_name =  recipients[0] || ""; 
                result += ({ CAMAS.Tools.fix_coding(real_name +
                             " <" + replace(recipients[2] + "@" + recipients[3],
                            ({ "@.MISSING-HOST-NAME.", "@0" }), ({ "", "" })) + ">")
                           });
              }
            return result;
          };
        	if (idx != -1)
        	{
          	imap = get_infos_from_imap_uid(uid)->ENVELOPE;
            string addr;
            if(imap[REPLAYTO_IDX])
              addr = get_addresses(imap[REPLAYTO_IDX])[0];
            else
              addr = get_addresses(imap[FROM_IDX])[0];
            if(replytoall)
            {
              ccs += get_addresses(imap[TO_IDX]);
              ccs += get_addresses(imap[CC_IDX]);
            }
          	sessobj->uids += ({ uid });
      	  	sessobj->messageids += ({ imap[MESSAGEID_IDX] });
      	  	sessobj->tos += ([ imap[MESSAGEID_IDX]: addr ]);
            addrs += ({ addr });
        	}
      	}
        // if we reply to all, then put everybody in CC: except the sender
        sessobj->cc += CAMAS.Tools.fixaddresses(Array.uniq(ccs) * ", ")[0];
      	sessobj->to = CAMAS.Tools.fixaddresses(addrs * ", ")[0];
      	if(sessobj->autobcc && sessobj->autobcc != "0") 
        	sessobj->bcc = CAMAS.Tools.fix_coding(sessobj->autobcc);
      	else
        	sessobj->bcc = "";
				if(sizeof(sessobj->selected) == 1 && imap)
     		{
     	  	sessobj->subject = CAMAS.Tools.fix_coding("Re: " + imap[SUBJECT_IDX]);
     	 	}
     	 	else 
     	  	sessobj->subject = "Re: ";
     	 	sessobj->attachments = ({ });
        if(sessobj->replyontop == "0")
        	sessobj->message = "\n" + MSG(M_FWD_REP_SEP) + "\n" + sessobj->message + "\n\n" + sessobj->signature;
  			else
		  		sessobj->message = "\n" + sessobj->signature + "\n" + MSG(M_FWD_REP_SEP) + "\n" + sessobj->message;

     	 	sessobj->selected = 0;
     	 	sessobj->status = COMPOSE;
    	  id->misc->camas->status = COMPOSE;
			}
		}
  }
    
  //! action : composedraftmove
  //!  Edit a previously saved draft and move it to sent-mails when really sent
  int composedraftmove()
  {
    CAMAS.Tools.init_temp_vars(id);
    sessobj->usermove=1;
    sessobj->sentfolderaction = sessobj->sentfolder;
    handler->add_action("composedraft", 1);
  }
  //! action : composedraft
  //!  Edit a previously saved draft
  void composedraft(int indirectcall)
  {
    if(!indirectcall)
      CAMAS.Tools.init_temp_vars(id);
    object mail = sessobj->cmail;
    sessobj->draftuid = sessobj->cmailuid;
    sessobj->to = mail->headers->to?CAMAS.Tools.fix_coding(mail->headers->to):"";
    sessobj->cc = mail->headers->cc?CAMAS.Tools.fix_coding(mail->headers->cc):"";
    sessobj->bcc = mail->headers->bcc?CAMAS.Tools.fix_coding(mail->headers->bcc):"";

    CDEBUG(sprintf("mail->headers: %O\n",mail->headers));
    sessobj->in_reply_to = "0";
    if(mail->headers["in-reply-to"]!="0")
      sessobj->in_reply_to = mail->headers["in-reply-to"];
    CDEBUG(sprintf("mail->headers[\"in-reply-to\"]: %O\n",mail->headers["in-reply-to"]));
    sessobj->replytoidx = -1;
    sessobj->status = COMPOSE;
    id->misc->camas->status = COMPOSE;
    sessobj->attachments = ({ });
    sessobj->subject = CAMAS.Tools.fix_coding(mail->headers->subject);
    sessobj->message = "";

    if(mail->body_parts){
      int messageadded=0;
      foreach(mail->body_parts, object mess) {
      	if(!messageadded &&
    	    mess->type == "text" &&
  	    mess->subtype != "html") {
	      sessobj->message += mess->getdata()||"";
    	  messageadded=1;
    	} else {
      	  string name=mess->disp_params->filename || mess->params->name || "unknown";
      	  name = MIME.decode_words_text_remapped(name);
      	  mapping file=([ ]);
      	  file->fname=name;
      	  file->data=mess->getdata()||"";
      	  file->size=sizeof(file->data);
      	  file->type=CAMAS.Tools.filetype(string_to_utf8(name), my_configuration());
      	  sessobj->attachments += ({ file });
      	}
      }
    }
    else
      sessobj->message += mail->getdata()||"";
  }
  
  //! action: sendselecteddraftsmove
  //!  Sends all the draft selected and move them into sent-mail
  void sendmarkeddraftsmove()
  {
    CAMAS.Tools.init_temp_vars(id);
    sessobj->usermove=1;
    handler->add_action("sendmarkeddrafts", 1);
  }
  //! action: sendselecteddraftsmove
  //!  Sends all the draft selected and move them into sent-mail
  void sendmarkeddrafts(int indirectcalls)
  {
    if(!indirectcalls)
      CAMAS.Tools.init_temp_vars(id);
    getselected();
    if(sizeof(sessobj->selected) > 0)
    {
      imap_commands += ({ CAMAS.IMAPTools.imap_cmd("get_mail","mailbox",id->variables->mbox||0,
              "uid",sessobj->selected[0],
              "output",CAMAS.IMAPTools.imap_cmd_var("cmail")), 
              CAMAS.IMAPTools.imap_cmd("check") });
    }
    else
    {
      CDEBUG("sendmarkeddrafts: No drafts selected\n");
    }
  }

  void forwardmove()
  {
    CAMAS.Tools.init_temp_vars(id);
    sessobj->usermove=1;
    sessobj->sentfolderaction = sessobj->sentfolder;
    handler->add_action("forward", 1);
  }
  void forward(int indirectcall)
  {
    if(!indirectcall)
    {
      CAMAS.Tools.init_temp_vars(id);
    }
    object mail = sessobj->cmail;
    if(!mail)
    {
      report_warning("CAMAS actions: warning: Can't forward mail, it doesn't exist\n");
      return;
    }
    sessobj->message = MSG(M_FWD_SEP) + "\n";
    sessobj->to = "";
    sessobj->cc = "";
    if(sessobj->autobcc && sessobj->autobcc != "0") 
      sessobj->bcc = CAMAS.Tools.fix_coding(sessobj->autobcc);
    else
      sessobj->bcc = "";
    sessobj->replytoidx = -1;
    sessobj->status = COMPOSE;
    id->misc->camas->status = COMPOSE;
    sessobj->attachments = ({ });
    sessobj->subject = CAMAS.Tools.fix_coding(mail->headers->subject);
    if (sessobj->subject[0..3] != "Fwd:")
      sessobj->subject = MSG(M_FWD_SUBJECT)+" "+sessobj->subject;
    sessobj->message += MSG(M_FWD_FROM)+" "+CAMAS.Tools.fix_coding(mail->headers->from)+"\n";
    if(mail->headers["reply-to"])
      sessobj->message += MSG(M_FWD_REPLYTO)+" "+ CAMAS.Tools.fix_coding(mail->headers["reply-to"]);
    sessobj->message += MSG(M_FWD_TO)+" "+CAMAS.Tools.fix_coding(mail->headers->to)+"\n";
    if(mail->headers->cc)
      sessobj->message += MSG(M_FWD_CC)+" "+ CAMAS.Tools.fix_coding(mail->headers->cc )+"\n";
    sessobj->message += MSG(M_FWD_DATE)+" "+ CAMAS.Tools.fix_coding(mail->headers->date || "");
    sessobj->message+="\n\n";


    if(mail->body_parts){
      int messageadded=0;
      foreach(mail->body_parts, object mess) {
      	if(!messageadded &&
    	    mess->type == "text" &&
    	    mess->subtype != "html") {
        	  sessobj->message += mess->getdata()||"";
        	  messageadded=1;
        	} else if(mess->type == "multipart"
        	  && mess->subtype == "alternative") {
            foreach( mess->body_parts, object altmess ) {
              if( altmess->type == "text" &&
             		altmess->subtype == "plain" ) {
                sessobj->message += altmess->getdata()||"";
              } else {
                mapping file=([]);
                file->fname="BODY-alternative";
                file->data=altmess->getdata()||"";
                file->size=sizeof(file->data);
                file->type=( altmess->type && altmess->subtype
          		     ?  altmess->type+"/"+altmess->subtype
          		     :  "application/octet-stream" );
                sessobj->attachments += ({ file });
        	   }
      	  } /* foreach */
  
	    } else {
        string name=mess->disp_params->filename || mess->params->name || "unknown";
    	  name=MIME.decode_words_text_remapped(name);
    	  mapping file=([ ]);
    	  file->fname=name;
    	  file->data=mess->getdata()||"";
    	  file->size=sizeof(file->data);
    	  file->type=CAMAS.Tools.filetype(string_to_utf8(name), my_configuration());
    	  sessobj->attachments += ({ file });
      	}
      }
    }
    else
      if( mail->type == "text" ) {
      	sessobj->message += mail->getdata()||"";
      } else {
        	string name=mail->get_filename() || mail->params->name || "unknown";
        	name=MIME.decode_words_text_remapped(name);
        	mapping file=([ ]);
          file->fname=name;
        	file->data=mail->getdata()||"";
        	file->size=sizeof(file->data);
        	file->type=CAMAS.Tools.filetype(string_to_utf8(name), my_configuration());
        	sessobj->attachments += ({ file });
      }
    sessobj->message += "\n" + sessobj->signature;
    sessobj->forwardidx = sessobj->cmailidx;
    sessobj->forwarduid = sessobj->cmailuid;
  }
  
  void backtomailindex()
  {
    sessobj->status = MAILINDEX;
    id->misc->camas->status = MAILINDEX;
  }
  
  void index()
  {
    mailindex();
  }
  void mailindex()
  {
    sessobj->status = MAILINDEX;
    id->misc->camas->status = MAILINDEX;
    // If user comes from compose mail, free memory
    sessobj->message = sessobj->to = sessobj->cc = sessobj->bcc = "";

		sessobj->mailssortcolumn = "";

    if(features->QUERY(scan_folders_call_out) > 0 && 
        sessobj->nb_mailindex_visits++ > features->QUERY(scan_folders_call_out))
    {
      imap_commands += ({ CAMAS.IMAPTools.imap_cmd("scan_folders", "reload", 1) });
      sessobj->nb_mailindex_visits = 0;
    }
    int mbox_idx;
    if (id->variables->mbox &&
	   (mbox_idx=Array.search_array(sessobj->mailboxes,
				     lambda(array a,string mbox)
				   {
				     if(a[MB_FOLDERNAME_IDX] == mbox)
					     return 1;
				      return 0;
				     },
				     id->variables->mbox)) != -1 ) {
        if (sessobj->mailbox[MB_FOLDERNAME_IDX] != id->variables->mbox) {
          sessobj->mailbox = sessobj->mailboxes[mbox_idx];
        CDEBUG ("\nchecking headers (" + sessobj->mailbox[MB_FOLDERNAME_IDX] + ")...\n");
        //sessobj->searchstring = "";
        imap_commands += ({
           CAMAS.IMAPTools.imap_cmd("status","mailbox",sessobj->mailbox[MB_FOLDERNAME_IDX]),
           CAMAS.IMAPTools.imap_cmd("get_headers", "mailbox", sessobj->mailbox[MB_FOLDERNAME_IDX], "output",CAMAS.IMAPTools.imap_cmd_var("mails"),"setsortcolumn",CAMAS.IMAPTools.imap_cmd_var("mailssortcolumn"))
	   });
	sessobj->firstvisiblemail = -1;
	
        if (sessobj->mailbox[MB_FOLDERNAME_IDX] == "INBOX")
          imap_commands += ({ CAMAS.IMAPTools.imap_cmd("apply_mail_filters", "updatembox", CAMAS.IMAPTools.imap_cmd_var ("mails")) });
        if (sessobj->stickyfilter)
        {
          imap_commands += ({ CAMAS.IMAPTools.imap_cmd("search", "searchtext", TO_UNICODE(sessobj->searchstringtext1), "searchfield", sessobj->searchfield1, "headers", CAMAS.IMAPTools.imap_cmd_var("mails")) });
        }
        else
        {
          // If !sessobj->stickyfilter, reset all search stuff
          sessobj->searchstring = "";
          sessobj->nothingfound = 0;
          sessobj->historysearch = 0;
        }
      }
    }
  }
  
  void gotoblock()
  {
    int from;
    if (id->variables->msg && (sscanf (id->variables->msg, "%d", from) == 1))
      sessobj->firstvisiblemail = from;
    sessobj->status = MAILINDEX;
    id->misc->camas->status = MAILINDEX;
  }
  
  void nextblock()
  {
    sessobj->firstvisiblemail += (int)sessobj->visiblemail; /* MI handles ovfl */
    sessobj->status = MAILINDEX;
    id->misc->camas->status = MAILINDEX;
  }
  
  void prevblock()
  {
    sessobj->firstvisiblemail -= (int)sessobj->visiblemail; /* MI handles underfl */
    sessobj->status = MAILINDEX;
    id->misc->camas->status = MAILINDEX;
  }
  
  void files()
  {
    if(sizeof(camas_main->QUERY (uploaddir)))
    {
      sessobj->status = FILES;
      id->misc->camas->status = FILES;
    }
    else
    {
      sessobj->status = MAILINDEX;
      id->misc->camas->status = MAILINDEX;
    }
  }
  
  void upload()
  {
    if (id->variables->file && sizeof(id->variables->file)) 
    {
      int totsize=0;
      int allowupload=1;
      foreach(sessobj->files,mapping file)
      {	
      	totsize+=file->size;
      }
      
      if((int)camas_main->QUERY (uploadquota)) {
      	if ((((((int)camas_main->QUERY (uploadquota)+ (int)camas_main->QUERY (uploadsoftquota))*1024)
	      - totsize - sizeof(id->variables->file))) <= 0)
	  allowupload=0;
      	if ((((((int)camas_main->QUERY (uploadquota))*1024)-totsize)) <= 0)
      	  allowupload=0;
      }

      if(allowupload) 
      {
      	string fname=(id->variables->fixedfilename && 
		      sizeof(id->variables->fixedfilename))?id->variables->fixedfilename:id->variables["file.filename"];
	fname=((((fname)/"/")[-1])/"\\")[-1];
      	string encfname=camas_main->QUERY (uploaddir)+"/"+lower_case(sessobj->login)
    	  +"_"+CAMAS.Tools.encode_fname(fname);
      	object file=open(encfname,"wct");
	
      	if(file) 
	{
      	  file->write(id->variables->file);
      	  file->close();

          sessobj->files+=
	    ({
	      ([ 
		"fname" : TO_UNICODE(fname),
		"size" : sizeof(id->variables->file),
		"type" : CAMAS.Tools.filetype(fname, my_configuration())
	      ])
	    });
      	}
      	else
      	  report_warning("CAMAS: Could not write uploaded file: "+encfname+"\n");
      }
    }
  }
  
  void deletefiles()
  {
    getselected();
    array (mapping) files_to_delete = ({ });
    foreach(sessobj->selectedfiles,int i)
    if(i < sizeof(sessobj->files))
      files_to_delete += ({ sessobj->files[i] });
    foreach(files_to_delete,mapping file)
      rm(camas_main->QUERY (uploaddir)+"/"+sessobj->login+"_"+CAMAS.Tools.encode_fname(file->fname));
    sessobj->files -= files_to_delete;
  }

  void setup()
  {
    if(camas_feature (FEAT_USEREDITSETUP))
    {
      sessobj->status = SETUP;
      id->misc->camas->status = SETUP;
    }
  }
  
  void savesetup()
  {
    array ind_prefproperties = indices(camas_main->prefproperties);
    foreach(ind_prefproperties, string prop)
    {
      if (id->variables[prop] && sessobj["usersetup"+prop])
      {
        sessobj[prop] = CAMAS.Tools.strip(TO_UNICODE(id->variables[prop]));
        CDEBUG(sprintf("savesetup: sessobj[%s]=%O", prop, sessobj[prop]));
      }
    }
    
    if(sessobj->language!="english" && !CAMAS_LANGUAGE->lang_progs[sessobj->language]) {
      // fuzzy select instead
      string lang=Array.filter(sessobj->language/"",lambda(string s) { return (s[0]>='a' && s[0]<='z');} )*"";
      string lang2=0;
      array langs=indices(CAMAS_LANGUAGE->lang_progs);
      foreach(langs, string l) {
        if(Array.filter(l/"",lambda(string s) { return (s[0]>='a' && s[0]<='z');} )*"" == lang) {
     	  lang2=l;
    	  break;
      	}
      }
      if(lang2)
      	sessobj->language=lang2;
    }

    if (sessobj->signature)
      sessobj->signature = sessobj->signature;
    if (sessobj->sortorder)
      sessobj->sessionsortorder = sessobj->sortorder;
    if (sessobj->trashfolder)
      sessobj->trashfolder = CAMAS.Tools.strip_path(sessobj->trashfolder);
    if (sessobj->sentfolder)
      sessobj->sentfolder = CAMAS.Tools.strip_path(sessobj->sentfolder);
    if (sessobj->draftsfolder)
      sessobj->draftsfolder = CAMAS.Tools.strip_path(sessobj->draftsfolder);
    if (sessobj->answeredfolder)
      sessobj->answeredfolder = CAMAS.Tools.strip_path(sessobj->answeredfolder);
    sessobj->status = MAILINDEX;
    id->misc->camas->status = MAILINDEX;
    
    array|void imp_command = CAMAS.Tools.save_preferences(id);
    if(imp_command && arrayp(imp_command))
      imap_commands += imp_command;
  }
  
  // once the user saw the MDN request, we remove the flag from the mail
  // so that he's not asked again
  void removemdn()
  {
    sessobj->mails[sessobj->cmailidx]->imap->FLAGS |= ({ IMAP_MDN_FLAG });
    imap_commands += ({ CAMAS.IMAPTools.imap_cmd ("add_flag", "uid", sessobj->mails[sessobj->cmailidx]->imap->UID, "flag", IMAP_MDN_FLAG) });
  }
  
  void sendmdn()
  {
    removemdn();
    object mail = sessobj->cmail;
    sessobj->to = CAMAS.Tools.fix_coding (mail->headers->from);
    sessobj->subject = MSG (M_MDNHEADER) + CAMAS.Tools.fix_coding(mail->headers->subject);
    if (mail->headers["message-id"])
      sessobj->messageid = mail->headers["message-id"];
    sessobj->message = MSGA (M_MDNMESSAGE, ({ CAMAS.Tools.fix_coding (mail->headers->to) }));
    sessobj->cc = "";
    sessobj->bcc = "";
    sessobj->attachments = ({ });
    sessobj->dsnsuccess = 0;
    sessobj->dsndelay = 0;
    sessobj->mdn = 0;
    mapping args = ([ 
      "includeipnumber": features->QUERY(includeipnumber),
      "siteorg": features->QUERY(siteorg),
      "addmaildomain": camas_main->QUERY(addmaildomain),
      "morecharsets": camas_main->QUERY(morecharsets),
      "shareddepth": camas_main->QUERY(defaultshareddepth),
      "pref_extraheader": camas_main->QUERY(pref_extraheader),
      "mdntype": camas_main->QUERY(mdntype),
      "uploaddir": camas_main->QUERY(uploaddir),
      "lang_charsets": CAMAS_LANGUAGE->lang_charsets,
      "camas_version": camas_main->camas_version
    ]);
    array res = sendmail (sessobj, sessobj->to, sessobj->cc, sessobj->bcc,
        sessobj->messageid, sessobj->subject, sessobj->message, 1, args);
    string maildata = res[0];
    array recipients = res[1];
    void|string error = camas_main->smtp_send (sessobj, recipients, maildata,
    camas_main->QUERY (smtpserver), (int)camas_main->QUERY (smtpport),
    camas_main->QUERY (smtpmaildomain));

    if (error && sizeof(error) > 0)
    {
      sessobj->status = DIALOGBOX;
      id->misc->camas->status = DIALOGBOX;
      sessobj->dialogtext = MSG(M_SMTPERROR);
      sessobj->dialogactions = ({ "actioncontinuecompose" });
      sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
      if (sessobj->displaysmtperrors)
      	sessobj->dialogtext += "\n" + error;
      CAMAS.Log.log (my_configuration (), "smtp_errors", ([ "login" : sessobj->address,
	     "command" : "smtp_send",
	     "line" : error ]));
    }
    sessobj->status = READMAIL;
    id->misc->camas->status = READMAIL;
  }
  
  void renamefolder()
  {
    string newfoldername, oldfoldername;

    sessobj->status = FOLDERLIST;
    id->misc->camas->status = FOLDERLIST;
    if (id->variables->newfoldername && sizeof(id->variables->newfoldername)>0 &&
    	id->variables->oldfoldername && sizeof(id->variables->oldfoldername)>0 &&
    	((id->variables->newfoldername=TO_UNICODE(id->variables->newfoldername)) != MSG(M_NEWMBOXNAME)) &&
    	(id->variables->oldfoldername != "imhonomailbox")) {

      string path=id->variables->path || "";
      int mbox_idx=-1;
      int ok=0;
      if(path!="")
      	mbox_idx=Array.search_array(sessobj->mailboxes,
				    lambda(array a,string path)
				      {
					if(a[MB_DISPLAYNAME_IDX] == path)
					  return 1;
					return 0;
				      },
				    path);
      if(path=="") 
      {
      	newfoldername=sessobj->mailpath+id->variables->newfoldername;
      	ok=1;
      }
      else
      	if(mbox_idx!=-1) 
	      {
      	  newfoldername=sessobj->mailboxes[mbox_idx][MB_FOLDERNAME_IDX]+sessobj->mailboxes[mbox_idx][MB_SEPARATOR_IDX]+id->variables->newfoldername;
	  ok=1;
      	}

      if (ok) 
      {
      	oldfoldername=id->variables->oldfoldername;
      	if(sessobj->mboxencode)
      	  newfoldername=CAMAS.Tools.encode_mboxname(newfoldername);
      	imap_commands += ({ CAMAS.IMAPTools.imap_cmd("rename_folder", 
						     "mailpath", sessobj->mailpath,
						     "sharedpath", sessobj->sharedpath,
						     "newfoldername", newfoldername,
						     "oldfoldername", oldfoldername),
			    CAMAS.IMAPTools.imap_cmd("updatelist", 
						     "mailpath", sessobj->mailpath,
						     "sharedpath", sessobj->sharedpath,
						     "why", "rename",
						     "newfoldername", newfoldername,
						     "oldfoldername", oldfoldername)
			    /*CAMAS.IMAPTools.imap_cmd("low_list")*/ });
        array|void imp_command = CAMAS.Tools.save_preferences(id);
        if(imp_command && arrayp(imp_command))
          imap_commands += imp_command;
      }
    }
    else {
      sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
      sessobj->dialogactions = ({ "actionfolderlist" });
      sessobj->dialogtext = MSG(M_NOMBOXERROR);
      sessobj->dialogtarget = id->misc->camas->frame;
      sessobj->status = DIALOGBOX;
      id->misc->camas->status = DIALOGBOX;
    }
  }
};

/* START AUTOGENERATED DEFVAR DOCS */

//! defvar: debug
//! When on, debug messages will be logged in Caudium's debug logfile. This information is very useful to the developers when fixing bugs.
//!  type: TYPE_FLAG
//!  name: Debug
//

/*
 * If you visit a file that doesn't contain these lines at its end, please
 * cut and paste everything from here to that file.
 */

/*
 * Local Variables:
 * c-basic-offset: 2
 * End:
 *
 * vim: softtabstop=2 tabstop=2 expandtab autoindent formatoptions=croqlt smartindent cindent shiftwidth=2
 */

