/**
 * @file kontros.c
 * Transfer routines for the 8-bit/8-bit C64/C128/Vic-20 to PC cable
 * designed by Frank Kontros (frank@kontros.uzhgorod.ua)
 * @author Marko Mkel (msmakela@nic.funet.fi)
 */

/*
 * Copyright  1994-1997 Marko Mkel and Olaf Seibert
 * Copyright  2001,2002 Marko Mkel
 * Original Linux and Commodore 64/128/Vic-20 version by Marko Mkel
 * Ported to the PET and the Amiga series by Olaf Seibert
 * Restructured by Marko Mkel
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef COMM_PC
# include "info.h"
# include "kontros.h"
# include "pcpar.h"

/** Open the communication channel
 * @param dev		name of the communication device
 * @param hostinfo	(output) the computer model information
 * @return		zero on success, nonzero on failure
 */
int
kontros_init (const char* dev, struct hostinfo* hostinfo)
{
  unsigned char detectbuf[5];
  if (!open_port (dev))
    return 1;
  out_ctrl (0); /* A/B=1, STROBE=1 */
  if (kontros_write ("", 1))
    return 1;
  kontros_sr ();
  if (kontros_read (detectbuf, sizeof detectbuf))
    return 2;
  kontros_rs ();
  hostinfo->host = detectbuf[0];
  hostinfo->driver = (unsigned) detectbuf[1] | (unsigned) detectbuf[2] << 8;
  hostinfo->basic = (unsigned) detectbuf[3] | (unsigned) detectbuf[4] << 8;
  return 0;
}

/** Close the communication channel */
void
kontros_close (void)
{
  out_ctrl (0);
  close_port ();
}

/** Switch the data direction from receive to send */
void
kontros_rs (void)
{
  while (!(in_stat () & 0x80));
}

/** Switch the data direction from send to receive */
void
kontros_sr (void)
{
  out_ctrl (1); /* -STROBE=0 */
  out_ctrl (0); /* -STROBE=1 */
}

#if !defined __BCC__ && (!defined __GNUC__ || defined USE_PPDEV)
/** Send data (unoptimised)
 * @param buf		the data to be sent
 * @param len		length of the data in bytes
 */
static void
kontros_write_slow (const void* buf, unsigned len)
{
  register const unsigned char* buffer = buf;

  while (len--) {
    register unsigned char byte = *buffer++;
    out_data (byte);
    out_ctrl (9); /* STROBE=0, SEL=0 (change port direction) */
    out_ctrl (8); /* STROBE=1 */
    while (!(0x80 & in_stat ()));
  }
}

/** Receive data (unoptimised)
 * @param buf		the data to be received
 * @param len		length of the data in bytes
 */
static void
kontros_read_slow (void* buf, unsigned len)
{
  register unsigned char data1, data;
  register unsigned char* buffer = buf;

  while (len--) {
    /* get 5 bits (higher 4 bits) and wait for handshaking */
    while (!(0x80 & (data1 = in_stat ()))); /* wait for -BUSY to change */

    out_ctrl (2); /* A/B=0 (lower bits from LS257) */
    data = (in_stat () >> 3) & 0x0f; /* get lower 4 bits */
    data |= (data1 << 1) & 0xf0; /* combine the bits */

    out_ctrl (1); /* -STROBE=0 */
    out_ctrl (0); /* -STROBE=1 */

    *buffer++ = data;
  }
}
#endif /* !__BCC__ && (!__GNUC__ || USE_PPDEV) */

/** Send data
 * @param buf		the data to be sent
 * @param len		length of the data in bytes
 * @return		zero on success, nonzero on failure
 */
int
kontros_write (const void* buf, unsigned len)
{
# ifdef __GNUC__
#  ifdef USE_PPDEV
  if (baseaddr == (unsigned) -1) {
    kontros_write_slow (buf, len);
    return 0;
  }
#  endif /* USE_PPDEV */
  __asm__ volatile ("\n0:\n\t"
		    "movb (%%ebx),%%al\n\t"
		    "outb %%al,%%dx\n\t"
		    "addw $2,%%dx\n\t"
		    "movb $9,%%al\n\t"
		    "outb %%al,%%dx\n\t"
		    "decb %%al\n\t"
		    "outb %%al,%%dx\n\t"
		    "decw %%dx\n"
		    "1:\n\t"
		    "inb %%dx,%%al\n\t"
		    "rolb $1,%%al\n\t"
		    "jnc 1b\n\t"
		    "decw %%dx\n\t"
		    "inc %%ebx\n\t"
		    "loop 0b"
		    : /* no outputs */
		    : "b" (buf), "c" (len), "d" (baseaddr)
		    : "ax");
  return 0;
# else /* BCC does not recognize #elif */
# ifdef __BCC__
#  asm
    push bp
    mov bp,sp
    push es
    mov dx,_baseaddr
    mov cx,len_
    mov bx,buf_
    mov ax,I 0x40
    mov es,ax

.sloop:
    mov al,[bx]
    out dx,al
    add dx,I 2
    mov al,I 9
    out dx,al
    dec al
    out dx,al
    dec dx

.owait:
    in al,dx
    rol al,I 1
    jc .nowait
    eseg
    mov al,byte [0x71]	; is the keyboard interrupt flag set?
    test al,I 0x80
    jz .owait
    jmp near _abreak
.nowait:
    dec dx

    inc bx
    loop .sloop

    pop es
    pop bp
    xor ax,ax
#  endasm
# else
  kontros_write_slow (buf, len);
  return 0;
# endif
# endif /* BCC does not recognize #elif */
}

/** Receive data
 * @param buf		the data to be received
 * @param len		length of the data in bytes
 * @return		zero on success, nonzero on failure
 */
int
kontros_read (void* buf, unsigned len)
{
# ifdef __GNUC__
#  ifdef USE_PPDEV
  if (baseaddr == (unsigned) -1) {
    kontros_read_slow (buf, len);
    return 0;
  }
#  endif /* USE_PPDEV */
  __asm__ volatile ("0:\t"
		    "movb $2,%%ah\n"
		    "1:\t"
		    "inb %%dx,%%al\n\t"
		    "rolb $1,%%al\n\t"
		    "jnc 1b\n\t"
		    "incw %%dx\n\t"
		    "xchgb %%ah,%%al\n\t"
		    "outb %%al,%%dx\n\t"
		    "decw %%dx\n\t"
		    "inb %%dx,%%al\n\t"
		    "rorb $3,%%al\n\t"
		    "orb $16,%%al\n\t"
		    "andb %%ah,%%al\n\t"
		    "movb %%al,(%%ebx)\n\t"
		    "incw %%dx\n\t"
		    "movb $1,%%al\n\t"
		    "outb %%al,%%dx\n\t"
		    "xorb %%al,%%al\n\t"
		    "outb %%al,%%dx\n\t"
		    "decw %%dx\n\t"
		    "inc %%ebx\n\t"
		    "loop 0b"
		    : /* no outputs */
		    : "b" (buf), "c" (len), "d" (stataddr)
		    : "ax");
  return 0;
# else /* BCC does not recognize #elif */
# ifdef __BCC__
#  asm
    push bp
    mov bp,sp
    push es
    mov dx,_baseaddr
    mov cx,len_
    mov bx,buf_
    mov ax,I 0x40
    mov es,ax

    inc dx
.rloop:
    mov ah,I 2
.iwait:
    in al,dx
    rol al,I 1
    jc .niwait
    eseg
    mov al,byte [0x71]	; is the keyboard interrupt flag set?
    test al,I 0x80
    jz .iwait
    jmp near _abreak

.niwait:
    inc dx
    xchg al,ah
    out dx,al
    dec dx
    in al,dx
    ror al,I 3
    or al,I 16
    and al,ah
    mov [bx],al
    inc dx
    mov al,I 1
    out dx,al
    xor al,al
    out dx,al
    dec dx
    inc bx
    loop .rloop

    pop es
    pop bp
    xor ax,ax
#  endasm
# else
  kontros_read_slow (buf, len);
  return 0;
# endif
# endif /* BCC does not recognize #elif */
}

#endif /* COMM_PC */
