/*
 * libmsn
 *
 * Copyright (C) 1999, Shane P. Brady <shane@jxie.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>

#include "libmsn.h"
#include "parse_utils.h"
#include "msn_commands.h"

extern MSN_Conn mainMSNConnection;
extern vector<MSN_Conn *> msn_connections;

void (*msn_event[MSN_NUM_EVENTS])(void *data);

/*
 * String representations of commands codes
 */

char CommandString[NUM_OF_COMMANDS][COMMAND_LENGTH] = {
    "ACK", "ADD", "ANS", "BLP", "BYE", "CAL", "CHG", "FLN", "GTC", "INF",
    "ILN", "IRO", "JOI", "LST", "MSG", "NAK", "NLN", "OUT", "REM", "RNG",
    "SYN", "USR", "VER", "XFR"
};

/*
** State strings
*/

char StateStrings[NUM_OF_STATES][4] = {
    "NLN", "FLN", "HDN", "BSY", "IDL", "BRB", "AWY", "PHN", "LUN"
};

void MSN_Log(const char *fmt, ...) {
    char buf[512];
    va_list ap;

    if(msn_event[MSN_LOG]) {
	va_start(ap, fmt);
	vsprintf(buf, fmt, ap);
	va_end(ap);
	(*msn_event[MSN_LOG])(buf);
    }
}

/*
** Name:    MSN_RegisterCallback
** Purpose: This function regsiters the call backs for the proper event
** Input:   eventType  - type of server event
**          func       - function pointer
** Output:  none
*/

void MSN_RegisterCallback(int eventType, MSN_CALLBACK func)
{
    msn_event[eventType] = func;
}

/*
** Name:    InitializeMSNConnection
** Purpose: This function initializes a MSN_Conn structure to default values
** Input:   conn - msn connection structure
** Output:  0 on success, -1 on failure
*/

int InitializeMSNConnection(MSN_Conn *conn)
{
    conn->serverType = DISPATCH_CONN;
    conn->cookie[0] = '\0';
    conn->commonName[0] = '\0';
    conn->unreadMail = 0;
    return 0;
}

/*
** Name:    MSN_Login
** Purpose: This function encapsulates the login process to MSN
** Input:   handle   - user's handle
**          passwd   - user's password
**          host     - notification server to use
**          port     - port number of notifcation server
** Output:  0 on success, -1 on failure
*/

int MSN_Login(const char *handle, const char *passwd, const char *host, int port)
{
    int  lID;
    char sp[10];
    int retry = 0;

    InitializeMSNConnection(&mainMSNConnection);

    strcpy(mainMSNConnection.passwd, passwd);
    strcpy(mainMSNConnection.handle, handle);

LOGIN_RETRY:
    retry++;

    alarm(3);
    int r = ConnectToServer(&mainMSNConnection, host, port);
    alarm(0);

    if (r != 0) {
	return -1;
    }

    if (SetProtocol(&mainMSNConnection, DEFAULT_PROTOCOL) != 0) {
	if(retry <= 20)
	{
		goto LOGIN_RETRY;
	}
	else
	{
	    return -1;
	}
	
    }
    if (GetServerPolicyInfo(&mainMSNConnection, sp) != 0) {
	return -1;
    }

    if (AuthenticateUserMD5(&mainMSNConnection, handle, passwd) != 0) {
/*
	if(MSN_ErrorOut)
	    (*MSN_ErrorOut)("Unable to authenticate user", "Authentication Error");
*/
	return -1;
    }
   
    Synchronize(&mainMSNConnection);
    ChangeState(&mainMSNConnection, "NLN");

    return 0;
}

/*
** Name:    MSN_SendMessage
** Purpose: This function encapuslates the sending of an instant message
** Input:   handle - user's handle who is receiving the message
**          message - the actual message to send
** Output:  0 on success, -1 on failure
*/

int MSN_SendMessage(const char *handle, const char *message)
{
    char     *newHandle;
    MSN_Conn *conn;

    AddHotmail(handle, &newHandle);
    conn = FindMSNConnectionByHandle(newHandle);

    if (conn == NULL) {
    /* if a connection isn't found try to establish a new connection */
	if (RequestSwitchboardSession(&mainMSNConnection, newHandle) == -1) {
	    free(newHandle);
	    return -1;
	}

	conn = FindMSNConnectionByHandle(newHandle);

	if (conn == NULL) {
	    free(newHandle);
	    return -1;
	}
    } 

    SendMessage(conn, message);
    free(newHandle);

    return 0;
}

/*
** Name:    MSN_AddContact
** Purpose: This function sends a add to forward contact list message to the
**          server
** Input:   handle  - handle to add to the list
** Output:  0 on success, -1 on failure
*/

int MSN_AddContact(const char *handle)
{
    if (handle == NULL)
	return -1;

    AddContact(&mainMSNConnection, handle);
    return 0;    
}

/*
** Name:    MSN_RemoveContact
** Purpose: This function sends a remove to forward contact list message to
**          the server
** Input:   handle - handle to remove from the list
** Output:  0 on success, -1 on failure
*/

int MSN_RemoveContact(const char *handle)
{
    if (handle == NULL)
	return -1;

    RemoveContact(&mainMSNConnection, handle);
    return 0;
}

/*
** Name:    MSN_AuthorizeContact
** Purpose: This function sends an authorize message to the server
** Input:   conn   - MSN connection structure
**          handle - handle to authorize
** Output:  0 on success, -1 on failure
*/

int MSN_AuthorizeContact(MSN_Conn *conn, const char *handle)
{
    return AuthorizeContact(conn, handle);
}

/*
** Name:    MSN_EndChat
** Purpose: This function sends an OUT mesage to the server to end a 
**          chat with a user
** Input:   handle - handle to end chat with 
** Output:  0 on success, -1 on failure
*/

int MSN_EndChat(const char *handle)
{
    MSN_Conn *conn;

    conn = FindMSNConnectionByHandle(handle);
    if (conn == NULL)
	return -1;
    return SendSBYE(conn);
}

/*
** Name:    MSN_Logout
** Purpose: This function properly logs out of the MSN service
** Input:   none
** Output:  0 on success, -1 on failure
*/

int MSN_Logout()
{
    return SendBYE(&mainMSNConnection);
}

/*
** Name:    MSN_ChangeState
** Purpose: This function changes the current state of the user
** Input:   state - integer representation of the state
** Output:  0 on success, -1 on failure
*/

int MSN_ChangeState(int state)
{
    if (state > (NUM_OF_STATES-1))
	return -1;

    return ChangeState(&mainMSNConnection, StateStrings[state]);
}

void MSN_Main() {
    fd_set fds;
    int high, fd, ssize;
    vector<int> sockets;
    vector<int>::iterator is;
    vector<MSN_Conn *>::iterator ic;
    struct timeval tv;

    high = 0;
    sockets = MSN_GetSockets();

    for(is = sockets.begin(); is != sockets.end(); is++) {
	high = max(high, *is);
	FD_SET(*is, &fds);
    }

    tv.tv_sec = tv.tv_usec = 0;

    if(select(high+1, &fds, 0, 0, &tv) >= 0) {
	if(FD_ISSET(mainMSNConnection.fd, &fds)) {
	    ParseForCommand(&mainMSNConnection);
	}

	sockets.clear();
	ic = msn_connections.begin();

	while(ic != msn_connections.end()) {
	    fd = (*ic)->fd;

	    if(find(sockets.begin(), sockets.end(), fd) == sockets.end()) {
		if(FD_ISSET(fd, &fds)) {
		    sockets.push_back(fd);
		    ssize = msn_connections.size();

		    ParseForCommand(*ic);

		    if(ssize != msn_connections.size()) {
			ic = msn_connections.begin();
			continue;
		    }
		}
	    }

	    ic++;
	}
    }
}

vector<int> MSN_GetSockets() {
    vector<int> r;
    vector<MSN_Conn *>::iterator ic;

    r.push_back(mainMSNConnection.fd);

    for(ic = msn_connections.begin(); ic != msn_connections.end(); ic++) {
	r.push_back((*ic)->fd);
    }

    return r;
}
