/*
 * Rgb.hpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright 2005  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
    Rgb.hpp  --  Rgb-Templates.

    Hartmut Sbosny  <hartmut.sbosny@gmx.de>

    (Helpful suggestions from the Standard C++ Library <complex> 
    (/usr/include/g++/complex)).


    Wahlweise mit typreinen oder typmischenden arithm. Operatoren
    (Schalter RGB_MIXED_ARITHMETIC)

    SEMANTIK arithmetischer Vec3-Operationen (@ \in {+, -, *, /} ):

        Rgb(r,g,b) @ Rgb(_r,_g,_b) = Rgb(r@_r, g@_g, b@_b).

    Semantik von (reintypigen) Ausdruecken der Art

        Rgb<T> @= T   sowie   Rgb<T> @ T   und   T @ Rgb<T>:

      Das einzelne T t wird stets interpretiert als Rgb<T>-Objekt mit
      den Komponenten (t,t,t), also z.B.

        2 - Rgb(r,g,b) = Rgb(2,2,2) - Rgb(r,g,b = Rgb(2-r, 2-g, 2-b).


    *Mischtypige* Operationen:

    Eistellige Operationen der Art  Rgb<T> @= Rgb<U_>:

      Diese sind wie in <complex> generell (dh auch bei ausgeschaltetem
      MIXED-Schalter) durch Templates erfasst und stets definiert durch
      elementweises
            t @= u   (mit t,u=r,g,b).
      Dh die eingebaute Automatik wuerde mglw. zunaecht auf den groesseren der
      beiden Typen aufblaehen, das Ergebnis aber in jedem Fall auf
      den linken Typ zurechtstutzen. Wenn T der kleinere Typ,
      gibt das Verluste, und davor warnt der Compiler vermtl. nicht.

    PROBLEM mischttypiger arithm. Operationen der Art

        Rgb<T> @ Rgb<U_>,   Rgb<T> @ U_,  T @ Rgb<U_>:

      Es waere sicherzustellen, dass analog zu eingebauten Typen jeweils auf
      die hoehere Genauigkeit, den groesseren Typ transformiert wird, also
          Rgb<float> * Rgb<int> --> Rgb<float>
          Rgb<int> / Rgb<float> --> Rgb<float>
          int(2) / Rgb<float>   --> Rgb<float> etc.
      Aber kann man das abstrakt erreichen?
      Selbst wenn alles auf Operatoren der eingebauten Typen zurueckgefuehrt
      wird und die eingebaute Automatik greift, wie kann denn der eingebaute
      Rueckgabetyp zum offiziellen Rueckgabetyp werden?
      Das geht anscheinend nur durch explizte Spezialisierung aller
      fraglichen Kombination.
      Auch <complex> hat keine Operatoren fuer complex<int> @ complex<float>,
      nur fuer complex<float> @ complex<double> und die sind expliziert!
*/
#ifndef Rgb_hpp
#define Rgb_hpp


//#define RGB_MIXED_ARITHMETIC


template <class T>
struct Rgb {

    T   r,g,b;

    //--------------
    // Konstruktoren:
    //--------------
    
    Rgb () {}                       // Default Ctor, no initialization

    explicit Rgb (T t)              // set all components to value t;
        : r(t), g(t), b(t) {}       // using Copy Ctor of the T's

    Rgb (T _r, T _g, T _b)
        : r(_r), g(_g), b(_b) {}    // using Copy Ctor of the T's

    template <typename U_>          // Konvertierung Rgb<T> <== Rgb<U_>
    Rgb (const Rgb<U_>& z)
        : r(z.r), g(z.g), b(z.b) {}


    //-------------------
    // arithm. Operatoren:
    //-------------------

    template <typename U_>
    Rgb<T>& operator+= (const Rgb<U_>& z)
        {    r += z.r; g += z.g; b += z.b;
            return *this;
        }

    template <typename U_>
    Rgb<T>& operator-= (const Rgb<U_>& z)
        {    r -= z.r; g -= z.g; b -= z.b;
            return *this;
        }

    template <typename U_>
    Rgb<T>& operator*= (const Rgb<U_>& z)
        {    r *= z.r; g *= z.g; b *= z.b;
            return *this;
        }

    template <typename U_>
    Rgb<T>& operator/= (const Rgb<U_>& z)
        {    r /= z.r; g /= z.g; b /= z.b;
            return *this;
        }

    // Ausdruecke der Art ``Rgb<T> @= T''
    // Hier koennte 'T t' ebenfalls ein anderer Typ sein.
    // FRAGE: als Argument `T' oder `const T&'? Fuer grosse Objekte zweifellos
    //   Referenz besser, fuer kleine weiss ich nicht. <complex> benutzt
    //   Referenzen.

    Rgb<T>& operator+= (T t)
        {    r += t; g += t; b += t;
            return *this;
        }

    Rgb<T>& operator-= (T t)
        {    r -= t; g -= t; b -= t;
            return *this;
        }

    Rgb<T>& operator*= (T t)
        {    r *= t; g *= t; b *= t;
            return *this;
        }

    Rgb<T>& operator/= (T t)
        {    r /= t; g /= t; b /= t;
            return *this;
        }

    // Zuweisung ``Rgb<T> = T'':

    Rgb<T>& operator= (T t)
        {    r=t; g=t; b=t;
            return *this;
        }
};


// Nichtelement-Funktionen
// =======================
// Vergleiche

template <class T> inline
bool operator== (const Rgb<T>& a, const Rgb<T>& b)
{
    return (a.r==b.r && a.g==b.g && a.b==b.b);
}

template <class T> inline
bool operator!= (const Rgb<T>& a, const Rgb<T>& b)
{
    return (a.r!=b.r || a.g!=b.g || a.b!=b.b);
}

// Vorzeichen (einstellige (+/-)-Operatoren)

template <class T> inline            // einstelliges `+'
Rgb<T> operator+ (const Rgb<T>& a)
{
    return a;
}

template <class T> inline            // einstelliges `-'
Rgb<T> operator- (const Rgb<T>& a)
{
    return Rgb<T>(-a.r, -a.g, -a.b);
}

//--------------------------------
// zweistellige arithm. Operatoren:
//--------------------------------

#ifndef RGB_MIXED_ARITHMETIC
//----------------------
// none mixed arithmetic
//----------------------

template <class T> inline
Rgb<T> operator+ (const Rgb<T>& a, const Rgb<T>& b)
{
    return Rgb<T>(a) += b;
}

template <class T> inline
Rgb<T> operator- (const Rgb<T>& a, const Rgb<T>& b)
{
    return Rgb<T>(a) -= b;
}

template <class T> inline
Rgb<T> operator* (const Rgb<T>& a, const Rgb<T>& b)
{
    return Rgb<T>(a) *= b;
}

template <class T> inline
Rgb<T> operator/ (const Rgb<T>& a, const Rgb<T>& b)
{
    return Rgb<T>(a) /= b;
}

// Ausdruecke der Art 'Rgb<T> @ T'  und  'T @ Rgb<T>'

template <class T> inline
Rgb<T> operator+ (const Rgb<T>& a, T t)
{
    return Rgb<T>(a) += t;
}

template <class T> inline
Rgb<T> operator+ (T t, const Rgb<T>& a)
{
    return Rgb<T>(a) += t;
}

template <class T> inline
Rgb<T> operator- (const Rgb<T>& a, T t)
{
    return Rgb<T>(a) -= t;
}

template <class T> inline
Rgb<T> operator- (T t, const Rgb<T>& a)
{
    return Rgb<T>(t - a.r, t - a.g, t - a.b);
}

template <class T> inline
Rgb<T> operator* (const Rgb<T>& a, T t)
{
    return Rgb<T>(a) *= t;
}

template <class T> inline
Rgb<T> operator* (T t, const Rgb<T>& a)
{
    return Rgb<T>(a) *= t;
}

template <class T> inline
Rgb<T> operator/ (const Rgb<T>& a, T t)
{
    return Rgb<T>(a) /= t;
}

// "t/a" hat das Aussehen einer Division eines Skalars durch einen Vektor
// und koennte verwirrlich wirken. Bei komplexen Zahlen ist gleichwohl ein
// solcher Ausdruck im Gebrauch und z.B "2/z" wird dort gelesen als
// die komplexe Division (2, 0) / (z.re, z.im), also der Skalar "2" nur dem
// Realteil des ersten Arguments zugedacht. Hier, bei Rgb's, kann man passend
// zur Semantik der Multiplikation "t*a" analog "t/a" festsetzen als denjenigen
// Rgb mit den Komponenten (t/a.r, t/a.g, t/a.b), also als die Rgb-Division
// (t,t,t) / (a.r, a.g, a.b). So ist's hier gemacht.

template <class T> inline
Rgb<T> operator/ (T t, const Rgb<T>& a)
{
    return Rgb<T>(t) /= a;
}


#else
//-----------------------------
// RGB_MIXED_ARITHMETIC defined
//-----------------------------

template <class T, class U_> inline
Rgb<T> operator+ (const Rgb<T>& a, const Rgb<U_>& b)
{
    return Rgb<T>(a) += b;
}

template <class T, class U_> inline
Rgb<T> operator- (const Rgb<T>& a, const Rgb<U_>& b)
{
    return Rgb<T>(a) -= b;
}

template <class T, class U_> inline
Rgb<T> operator* (const Rgb<T>& a, const Rgb<U_>& b)
{
    return Rgb<T>(a) *= b;
}

template <class T, class U_> inline
Rgb<T> operator/ (const Rgb<T>& a, const Rgb<U_>& b)
{
    return Rgb<T>(a) /= b;
}

// Spezialisierungen fuer all die Faelle, wo erster Operand ein "kleinerer"
// Typ als der zweite und Ergebnis vom zweiten "groesseren" Typ sein sollte.
// Das gibt aber einen Schwarm an Kombinationen!

// int - float

inline Rgb<float> operator+ (const Rgb<int>& a, const Rgb<float>& b)
{
    return Rgb<float>(a) += b;
}
inline Rgb<float> operator- (const Rgb<int>& a, const Rgb<float>& b)
{
    return Rgb<float>(a) -= b;
}
inline Rgb<float> operator* (const Rgb<int>& a, const Rgb<float>& b)
{
    return Rgb<float>(a) *= b;
}
inline Rgb<float> operator/ (const Rgb<int>& a, const Rgb<float>& b)
{
    return Rgb<float>(a) /= b;
}


// Ausdruecke der Art ``Rgb<T> @ T'' und ``T @ Rgb<T>''

// Zur Zeit typrein, koennte ebenfalls typmischend definiert werden,
// mit der Notwendigkeit dann aber auch zu Spezialisierungen.

template <class T> inline
Rgb<T> operator+ (const Rgb<T>& a, T t)
{
    return Rgb<T>(a) += t;
}

template <class T> inline
Rgb<T> operator+ (T t, const Rgb<T>& a)
{
    return Rgb<T>(a) += t;
}

template <class T> inline
Rgb<T> operator- (const Rgb<T>& a, T t)
{
    return Rgb<T>(a) -= t;
}

template <class T> inline
Rgb<T> operator- (T t, const Rgb<T>& a)
{
    return Rgb<T>(t - a.r, t - a.g, t - a.b);
}

template <class T> inline
Rgb<T> operator* (const Rgb<T>& a, T t)
{
    return Rgb<T>(a) *= t;
}

template <class T> inline
Rgb<T> operator* (T t, const Rgb<T>& a)
{
    return Rgb<T>(a) *= t;
}

template <class T> inline
Rgb<T> operator/ (const Rgb<T>& a, T t)
{
    return Rgb<T>(a) /= t;
}

template <class T> inline
Rgb<T> operator/ (T t, const Rgb<T>& a)
{
    return Rgb<T>(t) /= a;
}

#endif  // RGB_MIXED_ARITHMETIC

#endif  // Rgb_hpp

// END OF FILE
