;; -*- Mode: Lisp -*-

(in-package :burp)

(locally-enable-sql-reader-syntax)

(define-wm burp-categories ((element t))
  (session-instance 'burp-categories))

(define-wm burp-category-edit ((element t) (category bug-category))
  (session-instance 'burp-category-editor :category category))

(define-wm burp-category-delete ((element t) (category bug-category))
  (session-instance 'burp-category-deleter :category category))

(define-wm process-category-editor ((element t))
  (with-slots (category)
    element
    (setf (slot-value category 'category-name)
          (element-value (child-element element "name")))
    (setf (slot-value category 'category-description)
          (element-value (child-element element "description")))
    (update-records-from-instance category)
    (session-instance 'burp-categories)))

(define-wm burp-category-subcat ((element t) (category bug-category))
  (let ((sc (make-instance 'bug-category
                           :parent-id (category-id category)
                           :name (concatenate 'string "Unnamed subcategory of " (category-name category)))))
    (session-instance 'burp-category-editor :category sc)))


(defclass burp-bug-selector (html-form)
  ((message
    :initform "")))

(defmethod shared-initialize ((form burp-bug-selector) slots &rest)
  (call-next-method)
  (setf (slot-value form 'method) "process-bug-selector")
  (instantiate-children
   form
   `(("message"        static-string
      :value ,(lambda () (slot-value form 'message)))
     ("bugid" text-field)
     ("submit"          submit-button))))

(define-wm process-bug-selector ((element t))
  (let ((id (element-value (child-element element "bugid"))))
    (handler-case
        (setq id (parse-integer id))
      (error ()
        (setf (slot-value element 'message)
              (format nil
                      "'~a' doesn't look like a valid bug id (all IDs are integers, hint)" id))))
    (if (integerp id)
        (let ((bug (car (select 'burp-bug :where [= [bug_id] id]))))
          (if bug
              (session-instance 'burp-bug-detail :bug bug)
              (progn
                (setf (slot-value element 'message)
                      (format nil "Can't find a bug with Id# '~d'" id))
                nil)))
        nil)))


(defclass burp-category-editor (burp-page html-form)
  ((category
    :initarg :category
    :documentation
    "The category being edited"))
  )

(defmethod shared-initialize ((form burp-category-editor) slots &rest)
  (call-next-method)
  (setf (slot-value form 'method) "process-category-editor")
  (instantiate-children
   form
   `(("category"        static-string
      :value ,(lambda () (slot-value (slot-value form 'category) 'category-name)))
     ("submit"          submit-button)
     ("name"            text-field
     :value ,(lambda () (slot-value (slot-value form 'category) 'category-name)))
     ("description"            text-area
     :value ,(lambda () (slot-value (slot-value form 'category) 'category-description)))
     ("categories"      hyperlink
      :reference ,(refer-wm burp-categories)
      :value "Go back to the category list."))))


(defclass burp-categories (burp-page html-element)
  )

(defmethod shared-initialize ((form burp-categories) slots &rest)
  (call-next-method)
  (instantiate-children
   form
   `(("new-toplevel"      hyperlink
      :reference ,(refer-wm burp-category-subcat (base-category))
      :value "Make a new top-level category.")))
  (setf (child-element form "bug-selector") (session-instance 'burp-bug-selector)))


(defun html-color (triple)
  (format nil "#~2x~2x~2x" (car triple) (cadr triple) (caddr triple)))

(defun lighten (triple)
  (mapcar (lambda (x) (+ x 20)) triple))

(defmethod render-html ((cats burp-categories) stream)
  (labels ((draw-row (cat indent color)
             (let ((color (lighten color)))

               (with-tag (:stream stream :tag "TR" :attr `(("BGCOLOR" . ,(html-color color))))
                 (with-tag (:stream stream :tag "TD" :attr '(("WIDTH" . "200")))
                   (format stream (slot-value cat 'category-name))
                   (format stream "&nbsp;-&nbsp;~a" (category-description cat)))

                 (with-tag (:stream stream :tag "TD" :attr '(("WIDTH" . "200")))
                   (with-tag (:stream stream
                                      :tag "A"
                                      :attr `(("HREF" . ,(element-url cats :method (refer-wm burp-new-bug cat)))))
                     (format stream "Report a Bug")))

                 (with-tag (:stream stream :tag "TD" :attr '(("ALIGN" . "LEFT")))
                   (let ((number (length (category-bugs cat))))
                     (format stream "There are ")
                     (with-tag (:stream stream
                                        :tag "A"
                                        :attr `(("HREF" . ,(element-url cats :method (refer-wm burp-category-report cat)))))
                       (format stream "~d bugs in this category." number))))
                 
                 (with-tag (:stream stream :tag "TD" :attr '(("ALIGN" . "RIGHT")))
                   (with-tag (:stream stream
                                      :tag "A"
                                      :attr `(("HREF" . ,(element-url cats :method (refer-wm burp-category-edit cat)))))
                     (format stream "Edit Category"))
                   (format stream "<br>")
                   (with-tag (:stream stream
                                      :tag "A"
                                      :attr `(("HREF" . ,(element-url cats :method (refer-wm burp-category-subcat cat)))))
                     (format stream "Make Subcategory"))
                   (format stream "<br>")
                   (with-tag (:stream stream
                                      :tag "A"
                                      :attr `(("HREF" . ,(element-url cats :method (refer-wm burp-category-delete cat)))))
                     (format stream "Delete this Category"))
                   )
                 )
               (dolist (cat (slot-value cat 'category-children))
                 (draw-row cat (+ 4 indent) color)))))

    (let ((base-cat (base-category)))
      (with-tag (:stream stream :tag "P")
        (render-html (child-element cats "new-toplevel") stream)
        )
      (with-tag (:stream stream :tag "TABLE" :attr '(("WIDTH" . "100%")
                                                     ("BORDER" . "1")
                                                     ("CELLSPACING" . "0")
                                                     ("CELLPADDING" . "6")))
        (with-tag (:stream stream :tag "TR" :attr `(("BGCOLOR" . ,(html-color '(32 32 128)))))
          (with-tag (:stream stream :tag "TD")
            (with-tag (:stream stream :tag "FONT" :attr '(("COLOR" . "#ffffff")
                                                          ("SIZE" . "5")))
              (format stream "Category")))
          (with-tag (:stream stream :tag "TD")
            (with-tag (:stream stream :tag "FONT" :attr '(("COLOR" . "#ffffff")))
              (format stream "&nbsp;")))
          (with-tag (:stream stream :tag "TD")
            (with-tag (:stream stream :tag "FONT" :attr '(("COLOR" . "#ffffff")))
              (format stream "Status and Reports")))
          (with-tag (:stream stream :tag "TD")
            (with-tag (:stream stream :tag "FONT" :attr '(("COLOR" . "#ffffff")))
              (format stream "Modify Category"))))
          
        (let ((color '(120 120 120)))
          (dolist (cat (slot-value base-cat 'category-children))
            (draw-row cat 0 color))))
      (render-html (child-element cats "bug-selector") stream))))

(defun base-category ()
  (car (select 'bug-category :where [= [category_id] 0])))

(def-view-class bug-category ()
  ((category-id
    :reader category-id
    :initarg :id
    :db-kind :key
    :type (integer 6))
   (category-parent
    :db-kind :join
    :db-info (:join-class bug-category
                          :home-key category-parent-id
                          :foreign-key category-id
                          :set nil))
   (category-children
    :db-kind :join
    :db-info (:join-class bug-category
                          :home-key category-id
                          :foreign-key category-parent-id
                          :set t))
   (category-bugs
    :reader category-bugs
    :db-kind :join
    :db-info (:join-class burp-bug
                          :home-key category-id
                          :foreign-key bug-category-id
                          :set t))
   (category-parent-id
    :initarg :parent-id
    :db-kind :base
    :type (integer 6))
   (category-name
    :reader category-name
    :initarg :name
    :db-kind :base
    :type (string 50))
   (category-description
    :reader category-description
    :initarg :name
    :db-kind :base
    :type (string 200))))

(defmethod shared-initialize ((cat bug-category) slot &rest)
  (call-next-method)
  (if (not (slot-boundp cat 'category-id))
      (setf (slot-value cat 'category-id) (sequence-next 'bug-category-sequence))))

(defmethod extern-ref ((category bug-category))
  (format nil "~d" (slot-value category 'category-id)))

(defmethod intern-ref ((category (eql 'bug-category)) arg)
  (if arg
      (car (select 'bug-category :where [= [category_id] (parse-integer arg)]))))

