;;; -*- Mode: Lisp -*-
;;; $Id: local-time-funcs.lisp,v 1.35 2002/04/05 02:40:49 craig Exp $
;;;
;;; Copyright (c) 2000, 2001 onShore Development, Inc.
;;;
;;; Higher level convenience functions

(in-package :local-time)

;;; integer:integer{am pm}
;;; parses time like "12:23am" and "9:45 pm"
;;; returns time as an integer number of minutes after midnight or nil if it can't parse it

(defun current-year ()
  (nth 6 (multiple-value-list (decode-local-time (get-local-time)))))

(defun current-month ()
  (nth 5 (multiple-value-list (decode-local-time (get-local-time)))))

(defun current-day ()
  (nth 4 (multiple-value-list (decode-local-time (get-local-time)))))

;;;
;;; parses date like 08/08/01, 8.8.2001, eg
;;;
(defun parse-date-time (string)
  (flet ((zone-offset ()
           (parse-integer (subseq (format-timestring nil (get-local-time))
                                  19 22))))
    (when (> (length string) 1)
      (destructuring-bind (&optional (m (current-month))
                                     (d (current-day))
                                     (y (current-year)))
          (handler-case (mapcar #'parse-integer (hork-integers string))
            (error (c)
              (declare (ignore c))
              (return-from parse-date-time nil))) 
        (when (< y 100)
          (incf y 2000))
        (parse-timestring (format nil "~4,'0d-~2,'0d-~2,'0d 12:00:00~A" y m d
                                  (zone-offset)))))))

(defun hork-integers (input)
  (dotimes (x (length input))
    (unless (<= 48 (char-code (aref input x)) 57)
      (setf (aref input x) #\Space)))
  (split input #\Space))

(defun split (string delim)
  "Splits a string into a list of substrings: (split 'foo:bar:gaz' #\\:) returns ('foo' 'bar' 'gaz')
Removes empty strings from result"
  (when (not (characterp delim))
    (error "Delimiter is not a character"))
  (let ((div (position delim string)))
    (cond
     ((null div) ;; no delimiter
      (list string))
     ((= div 0) ;; delimiiter at beginning of string
      (split (subseq string (1+ div) (length string)) delim))
     ((= (1+ div) (length string)) ;; delimiter at end of string
      (list (subseq string 0 div)))
     (t
      (cons (subseq string 0 div)
	    (split (subseq string (1+ div) (length string)) delim))))))



(defun parse-schedule-time (string)
  (and (> (length string) 1)
       (multiple-value-bind (hours pos-col) (parse-integer string :junk-allowed t)
	 (let ((minutes (parse-integer string :start (1+ pos-col) :junk-allowed t)))
	   (when (and (numberp hours) (numberp minutes))
	     (if (= hours 12)
		 (decf hours 12))
	     (if (search "pm" string :test #'string-equal)
		 (parse-timestring (format nil "P~dM" (+ (* 60 (+ 12 hours)) minutes)))
		 (parse-timestring (format nil "P~dM" (+ (* 60 hours) minutes)))))))))

(defun ymd-string (local-time)
  (multiple-value-bind (ms ss mm hh day month year)
      (local-time::decode-local-time-for-display local-time)
    (declare (ignore ms ss mm hh))
    (format nil "~2,'0d/~2,'0d/~4,'0d" month day year)))

(defun lt-round-to-minute (local-time)
  (local-time::%make-local-time
   :day (local-time::local-time-day local-time)
   :sec (* 60 (floor (local-time::local-time-sec local-time) 60))))

;; to sidestep race conditions caused when trying to check out a resource
;; that's just been created or brought back online after being taken offline
(defun one-minute-ago (&optional (local-time (get-local-time)))
  (lt-round-to-minute
   (local-time- local-time (make-duration :sec 60))))

;;;;

(defun time-for-ymd (string)
  (%time-for-ymd (parse-integer (subseq string 0 4))
                 (parse-integer (subseq string 5 7))
                 (parse-integer (subseq string 8 10))))

(defun adjust-for-zone (t1 t2)
  (roll t2 :sec (- (local-time::local-time-gmt-offset t1)
                   (local-time::local-time-gmt-offset t2))))

(defun make-your-time (y m d min &optional (sec 0))
  (let ((new-time (parse-timestring (format nil "~4,'0d-~2,'0d-~2,'0d ~2,'0d:~2,'0d:~2,'0d"
                                           y m d (floor min 60) (mod min 60) sec))))
    (local-time::adjust-for-zone (get-local-time) new-time)))



(defun merged-time (day time-of-day)
  (multiple-value-bind (ms ss mm hh day month year)
      (decode-local-time-for-display day)
    (declare (ignore ms ss mm hh))
    (multiple-value-bind (ms ss mm hh)
	(decode-local-time-for-display time-of-day)
      (declare (ignore ms))
      (make-your-time year (1+ month) day (+ (* 60 hh) mm) ss))))

(defun %time-for-ymd (y m d)
  (local-time :unix (maketime y m d)))

(defun time-meridian (hours)
  (cond ((= hours 0)
         (values 12 "AM"))
        ((= hours 12)
         (values 12 "PM"))
        ((< 12 hours)
         (values (- hours 12) "PM"))
        (t
         (values hours "AM"))))

(defun print-date (local-time &optional (style :daytime) (zone nil))
  (multiple-value-bind (ms ss mm hh day month year dow)
      (if (eql zone :utc)
          (local-time::decode-local-time local-time)
          (local-time::decode-local-time-for-display local-time))
    (declare (ignore ms ss))
    (multiple-value-bind (hours meridian)
        (time-meridian hh)
      (ecase style
        (:time-of-day
         ;; 2:00 PM
         (format nil "~d:~2,'0d ~a" hours mm meridian))
        (:long-day
         ;; October 11th, 2000
         (format nil "~a ~d, ~d" (month-name month) day year))
        (:month
         (month-name month))
        (:month-year
         ;; October 2000
         (format nil "~a ~d" (month-name month) year))
        (:full
         (format nil "~d:~2,'0d ~a, ~a ~d, ~d"
                 hours mm meridian (month-name month) day year))
        (:full+weekday
         (format nil "~d:~2,'0d ~a ~a, ~a ~d, ~d"
                 hours mm meridian (nth dow *day-names*) (month-name month) day year))
        (:daytime
         (local-time-to-string local-time :short-pretty t))
        (:day
         (format nil "~d/~d/~d" (1+ month) day year))))))

(defun date-element (local-time element &optional (zone nil))
  (multiple-value-bind (ms ss mm hh day month year dow)
      (if (eql zone :utc)
          (local-time::decode-local-time local-time)
          (local-time::decode-local-time-for-display local-time))
    (ecase element
      (:milliseconds
       ms)
      (:seconds
       ss)
      (:minutes
       mm)
      (:hours
       hh)
      (:day-of-month
       day)
      (:integer-day-of-week
       dow)
      (:day-of-week
       (nth dow '(:sunday :monday :tuesday :wednesday :thursday :friday :saturday)))
      (:month
       month)
      (:year
       year))))

(defun print-duration (local-time &optional (style :hours))
  (if (null local-time)
      "Unknown"
      (multiple-value-bind (ms ss mm hh days)
          (local-time::decode-duration local-time)
        (ecase style
          (:reduce
           (if (= 0 days hh mm)
               "0 minutes"
               (format nil "~@[~d days ~]~@[~d hours ~]~@[~d mins.~]"
                       (and (< 0 days) days)
                       (and (< 0 hh) hh)
                       (and (< 0 mm) mm))))
          (:seconds
           (format nil "~d.~d" (+ (* 24 60 60 days) (* 60 60 hh) (* 60 mm) ss) ms))
          (:hours
           (format nil "~d hours" (+ hh (* 24 days) (if (or (< 0 mm) (< 0 ss) (< 0 ms)) 1 0))))))))

(let ((one-day (parse-timestring "P1D")))
  (defun next-day (time)
    (local-time+ time one-day))
  
  (defun previous-day (time)
    (local-time- time one-day)))

(defun month-first-day (time)
  (multiple-value-bind (ms ss mm hh day month year)
      (decode-local-time time)
    (declare (ignore ms ss hh mm day))
    (parse-timestring
     (format nil "~2,'0D-~2,'0D-~2,'0D ~2,'0D:~2,'0D:~2,'0D+00"
             year month 1 0 0 0))))

(defun month-different-p (time1 time2)
  (local-time/= (month-first-day time1)
                (month-first-day time2)))

;; 0 is Sunday
(defun local-time-day-index (time)
  (declare (type local-time time))
  (let* ((days (local-time-day time))
         (adjusted (cond ((< days 0)
                          (if (< days -7)
                              (- 7 (mod (- days) 7))  ;; -8 = 6, -13 = 1
                              (+ days 7)))
                         (t (mod days 7)))))
    ;; March 1 2000 is a Wednesday
    (mod (+ adjusted 3) 7)))

(defun sunday-before (time)
  (declare (type local-time time))
  (let ((duration (parse-timestring (format nil "P~DD" (local-time-day-index time)))))
    (local-time- time duration)))

(defun first-day-next-month (time)
  (let* ((twenty-days (parse-timestring "P20D"))
         (forty-days (parse-timestring "P40D"))
         (candidate-one (local-time+ time twenty-days))
         (candidate-two (local-time+ time forty-days)))
    (if (month-different-p time candidate-one)
        (month-first-day candidate-one)
        (month-first-day candidate-two))))

;; given a date and a duration, return an inclusive list
;; of the months present between start-time and start-time + duration
(defun month-span (start-time duration)
  (let ((last-time (local-time+ start-time duration)))
    (do* ((time (month-first-day start-time) (first-day-next-month time))
          (result (list time) (cons time result)))
         ((local-time> time last-time)
          (reverse (cdr result))))))

;; truncate hours, minutes and seconds.

(defmethod midnight ((self local-time))
  (multiple-value-bind (ms ss mm hh day month year dow dst-p tz tzabbr)
      (decode-local-time-for-display self)
    (declare (ignore ms ss mm hh dow dst-p tz tzabbr))
    (make-your-time year (1+ month) day 0)))

(defmethod utc-midnight ((self local-time))
  (multiple-value-bind (ms ss mm hh day month year)

      (decode-local-time-for-display self)
    (declare (ignore ms ss mm hh))
    (parse-timestring
     (format nil "~2,'0D-~2,'0D-~2,'0D 00:00:00-00" year (1+ month) day 0))))

(defmethod just-seconds ((self local-time))
  (%make-local-time :sec (local-time-sec self)))

(defun roll (date &key (month 0) (day 0) (sec 0) (hour 0) (minute 0) (preserve-time nil) &aux new-date)
  (when (< 0 month)
    (dotimes (x month)
      (setq date (roll date
                       :day (days-in-month (1+ (date-element date :month))
                                           (date-element date :year))))))
  (when (< month 0)
    (dotimes (x (- month))
      (let ((month (date-element date :month))
            (year (date-element date :year)))
        (decf month)
        (when (= month -1)
          (setq month 11)
          (decf year))
        (setq date (roll date
                         :day (- (days-in-month (1+ month) year)))))))
  
  (setq new-date (local-time+ date (make-duration :day day :sec (+ sec
                                                                   (* 60 minute)
                                                                   (* 60 60 hour)))))
  (if (and preserve-time (not (local-time= (midnight date)
                                           (midnight new-date))))
      (let ((adjustment (- (local-time-gmt-offset new-date)
                           (local-time-gmt-offset date))))
        (cond ((> 0 adjustment)
               (local-time- new-date (make-duration :sec adjustment)))
              ((< 0 adjustment)
               (local-time- new-date (make-duration :sec adjustment)))
              (t new-date)))
      new-date))

(defun roll-to (date size position)
  (ecase size
    (:month
     (ecase position
       (:beginning
        (roll date :day (+ 1
                           (- (date-element date :day-of-month)))))
       (:end
        (roll date :day (+ (days-in-month (date-element date :month)
                                          (date-element date :year))
                           (- (date-element date :day-of-month)))))
        
       ))
    ))

(defun week-containing (time)
  (let* ((midn (midnight time))
         (dow (date-element midn :integer-day-of-week)))
    (list (roll midn :day (- dow) :preserve-time t)
          (roll midn :day (- 7 dow) :preserve-time t))))
  