;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: wcof -*-
;;; $Id: objects.lisp,v 1.15 2002/02/27 19:24:14 craig Exp $
;;;
;;; Copyright (c) 2000, 2001, 2002 onShore Development, Inc.

(in-package :odcl)

(defvar *propertied-objects* (make-hash-table))

;; ------------------------------------------------------------
;; Display Attributes

(defgeneric get-property (object attr)
  (:documentation
   "Returns the specified value of the given object.  it provides a
unified interface to slots and metadata properties, as well as
computed value."))

(defmethod get-property ((self integer) attr)
  (ecase attr
    (:name
     (princ-to-string self))))

(defun clear-class-properties (class)
  (remhash class *propertied-objects*))

(defstruct obj-property
  type
  reader
  writer
  caption)

(defun %register-property (class property type reader writer caption)
  (unless reader
    (setq reader (lambda (self)
                   (error "property ~s is write-only for ~s" property self))))
  (unless writer
    (setq writer (lambda (self value)
                   (declare (ignore value))
                   (error "property ~s is read-only for ~s" property self))))
  (let ((phash (or (gethash class *propertied-objects*)
                   (setf (gethash class *propertied-objects*)
                         (make-hash-table)))))
    (setf (gethash property phash)
          (make-obj-property :type type :reader reader
                             :writer writer :caption caption))))

(defmacro defproperties (class props)
  `(progn
    ,@(mapcar (lambda (prop)
                (destructuring-bind (property &key reader writer type caption)
                    prop
                  `(%register-property ',class ,property ',type ,reader ,writer ,caption)))
              props)
    (values)))
      
(defmethod get-property ((object standard-object) (attr symbol))
  (if-bind (phash (gethash (type-of object) *propertied-objects*))
    (if-bind (pobj (gethash attr phash))
      (values (funcall (obj-property-reader pobj) object)
              (obj-property-type pobj)
              (obj-property-caption pobj))
      (error "No property ~s defined for object ~s" attr object))
    (error "No properties are defined for objects of type ~s" object)))

(defun store-property (object attr value)
  (if-bind (phash (gethash (type-of object) *propertied-objects*))
    (if-bind (pobj (gethash attr phash))
      (funcall (obj-property-writer pobj) object value)
      (error "No property ~s defined for object ~s" attr object))
    (error "No properties are defined for objects of type ~s" object)))

(defun property-metadata (class property-name)
  "Returns the metadata for the specified property as a list containing: reader, writer, type and caption."
  (when-bind (phash (gethash class *propertied-objects*))
    (when-bind (pobj (gethash property-name phash))
      (list (obj-property-reader pobj)
            (obj-property-writer pobj)
            (obj-property-type pobj)
            (obj-property-caption pobj)))))

(defun properties-for-class (class)
  "Returns the metadata for all properties of the specified class."
  (when-bind (phash (gethash class *propertied-objects*))
    (let ((results nil))
      (maphash (lambda (pname pobj)
		 (push (list pname
			     (obj-property-reader pobj)
			     (obj-property-writer pobj)
			     (obj-property-type pobj)
			     (obj-property-caption pobj)) results))
	       phash)
      results)))
    
(defun object-properties (self)
  (when-bind (phash (gethash (type-of self) *propertied-objects*))
    (let ((results nil))
      (maphash (lambda (pname pobj)
		 (push (list pname
			     (funcall (obj-property-reader pobj) self)
			     (obj-property-caption pobj)
			     (obj-property-type pobj))
		       results))
	       phash)
      results)))

(defmethod get-property ((self standard-object) (property list))
  (case (length property)
    (0
     (error "nil is not a valid property"))
    (1
     (funcall #'get-property self (car property)))
    (2
     (when-bind (foo (get-property self (car property)))
       (funcall #'get-property foo
                (second property))))
    (t
     (when-bind (foo (get-property self (car property)))
       (funcall #'get-property foo
                (cdr property))))))

