(in-package :odcl)
  
(defvar *default-capacity* 11)

(defclass binheap ()
  ((size       :initform 0)
   (compare-fn :initarg :test
               :initform #'<)
   (order-ok   :initform t)
   (array      :initform (make-array *default-capacity* :initial-element nil))))

(defun binheap-insert (heap item)
  "insert into the priority queue, maintaining heap order. duplicates allowed"
  (with-slots (size compare-fn order-ok array)
    heap
    (if (not order-ok)
        (binheap-toss heap item)
        (progn
          (when (= size (1- (length array)))
            (setf array (adjust-array array (1+ (* size 2)))))
          (let ((hole (incf size)))
            (while (binheap-compare compare-fn item (aref array (floor hole 2)))
              (setf (aref array hole)
                    (aref array (floor hole 2)))
              (setf hole (floor hole 2)))
            (setf (aref array hole) item))))))    

(defun binheap-toss (heap item)
  "insert into the priority queue, without maintaining heap order. Duplicates are allowed"
  (with-slots (size compare-fn order-ok array)
    heap
    (when (= size (1- (length array)))
      (setf array (adjust-array array (1+ (* size 2)))))
    (setf (aref array (incf size)) item)
    (when (binheap-compare compare-fn item (aref array (floor size 2)))
      (setf order-ok nil))))

(defun binheap-remove (heap item &optional (test #'equal))
  (with-slots (size compare-fn order-ok array)
    heap
    (when-bind (pos (position item array :test test))
               (setf (aref array pos) (decf size))
               (setf order-ok nil))))
  
(defun binheap-find-min (heap)
  "find the smallest item in the priority queue"
  (with-slots (order-ok array)
    heap
    (unless (binheap-empty heap)
      (unless order-ok
        (binheap-fix-heap heap))
      (aref array 1))))

(defun binheap-delete-min (heap)
  "remove the smallest item from the priority queue"
  (with-slots (size array)
    heap
    (when-bind (min (binheap-find-min heap))
      (setf (aref array 1)
            (aref array size))
      (decf size)
      (percolate-down heap 1)
      min)))

(defun binheap-fix-heap (heap)
  "reestablish heap order property after a series of toss operations"
  (with-slots (size order-ok)
    heap
    (do ((i (floor size 2) (1- i)))
        ((= 0 i)
         (setf order-ok t))
      (percolate-down heap i))))

(defun binheap-empty (heap)
  "test if the priority queue is logically empty"
  (= 0 (slot-value heap 'size)))

(defun percolate-down (heap hole &aux child)
  "internal method to percolate down in the heap"
  (with-slots (size compare-fn array)
    heap
    (let ((tmp (aref array hole)))
      (loop
       (unless (<= (* 2 hole) size)
         (return))
       (setq child (* 2 hole))
       (when (and (not (= child size))
                  (binheap-compare compare-fn (aref array (1+ child))
                           (aref array child)))
         (incf child))
       (if (binheap-compare compare-fn (aref array child) tmp)
           (setf (aref array hole)
                 (aref array child))
           (return))
       (setq hole child))
      (setf (aref array hole) tmp))))

(defun binheap-compare (fn a b)
  (cond ((and a b)
         (funcall fn a b))
        (b
         t)
        (t
         nil)))


