;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: profile -*-
;;; $Id: profile.lisp,v 1.6 2002/02/22 17:02:08 jesse Exp $
;;;
;;; Copyright (c) 2000, 2001, 2002 onShore Development, Inc.

(in-package :profile)

(defun profile-pkg (package)
  (let ((package (if (packagep package)
		     package
		     (find-package package))))
    (do-symbols (symbol package (values))
      (when (and (eq (symbol-package symbol) package)
		 (fboundp symbol)
                 (not (macro-function symbol))
                 (not (equal #\( (aref (symbol-name symbol) 0))))
	(profile-1-function symbol nil)))))

(defun %report-times-2 (count &key sorter)
  (declare (optimize (speed 0)))
  (unless (boundp '*call-overhead*)
    (compute-time-overhead))
  (let ((info ()))
    (dolist (name *timed-functions*)
      (let ((pinfo (profile-info-or-lose name)))
	(unless (eq (fdefinition name)
		    (profile-info-new-definition pinfo))
	  (warn "Function ~S has been redefined, so times may be inaccurate.~@
	         PROFILE it again to record calls to the new definition."
		name))
	(multiple-value-bind
              (calls time consing profile callers)
	    (funcall (profile-info-read-time pinfo))
	  (if (not (zerop calls))
	      (push (make-time-info name calls
				    (compensate-time calls time profile)
				    consing
				    (sort (copy-seq callers)
					  #'>= :key #'cdr))
		    info)))))
    (if sorter
        (setq info (subseq (sort info sorter) 0 count))
        (setq info (subseq (sort info #'>= :key #'time-info-time) 0 count)))
    (format *trace-output*
	    "~&  Seconds  |  Consed   |  Calls  |  Sec/Call  |  Name:~@
	       ------------------------------------------------------~%")
    (let ((total-time 0.0)
	  (total-consed 0)
	  (total-calls 0))
      (dolist (time info)
	(incf total-time (time-info-time time))
	(incf total-calls (time-info-calls time))
	(incf total-consed (time-info-consing time))
	(format *trace-output*
		"~10,3F | ~9:D | ~7:D | ~10,5F | ~S~%"
		(time-info-time time)
		(time-info-consing time)
		(time-info-calls time)
		(/ (time-info-time time) (float (time-info-calls time)))
		(time-info-name time))
	(let ((callers (time-info-callers time)))
	  (when callers
	    (dolist (x (subseq callers 0 (min (length callers) 5)))
	      (format *trace-output* "~10:D: " (cdr x))
	      (print-caller-info (car x) *trace-output*)
	      (terpri *trace-output*))
	    (terpri *trace-output*))))
      (format *trace-output*
	      "------------------------------------------------------~@
	      ~10,3F | ~9:D | ~7:D |            | Total~%"
	      total-time total-consed total-calls)
      (format *trace-output*
	      "~%Estimated total profiling overhead: ~4,2F seconds~%"
	      (* *total-profile-overhead* (float total-calls))))
    (values)))

(defun record-package (package)
  (do-external-symbols (s package)
    (when (fboundp s)
      (make-function-recordable s))))

(defun unrecord-package (package)
  (do-external-symbols (s package)
    (when (fboundp s)
      (make-function-nonrecordable s))))

(defun call-record (package)
  (let ((calls nil))
    (do-external-symbols (s package)
      (when (fboundp s)
        (let ((record (get s :call-record)))
          (dolist (call record)
            (push (cons (car call) (cons s (cdr call)))
                  calls)))))
    (sort calls #'(lambda (x y) (< (car x) (car y))))))

(defun get-utime ()
  (multiple-value-bind (x sec usec)
      (unix:unix-gettimeofday)
    (+ (* 1000000 sec) usec)))

(defun recordable-function (function-name)
  "Takes a function NAME and returns a function OBJECT that does what #'NAME
   did, except also keeps track of the number of times it has been called"
  (let ((function (symbol-function function-name)))
    (setf (get function-name :call-record) nil)
    (setf (get function-name :non-recording-function) function)
    #'(lambda (&rest args)
        (let ((start-time (get-utime))
              (res nil))
          (setq res (multiple-value-list (apply function args)))
          (push (list (- (get-utime) start-time) res args)
                (get function-name :call-record))
          (values-list res)))))

(defun make-function-recordable (function-name)
  "given a function name changes it into equivalent version that records
   function calls"
  (setf (symbol-function function-name)
        (recordable-function function-name)) )

(defun report-calls (s &key (return nil))
  (let (calls)
    (when (fboundp s)
      (let ((record (get s :call-record)))
        (dolist (call record)
          (push (if return
                    call
                    (third call))
                calls))))
    calls))

(defun report-calls-2 (s)
  (let ((record (get s :call-record)))
    (sort record #'(lambda (x y) (> (car x) (car y))))))
  

(defun make-function-nonrecordable (function-name)
  "Returns the function to its original (non-recording) state"
  (let ((original (get function-name :non-recording-function)))
    (cond      
      (original
       (setf (symbol-function function-name) original)
       (remf (symbol-plist function-name) :non-recording-function)
       (remf (symbol-plist function-name) :call-record)
       original)
      (t
       (format nil "~%function ~s wasn't recordable to begin with: unchanged."
                   function-name)))
    ))

(defun mem-meter (&key (string "Hello World") (font "fixed"))
  ;; CLX demo, says STRING using FONT in its own window on HOST
  (let ((host "localhost")
        (display nil)
	(abort t))
    (unwind-protect
	(progn 
	  (setq display (open-display host))
	  (multiple-value-prog1
	   (let* ((screen (display-default-screen display))
		  (black (screen-black-pixel screen))
		  (white (screen-white-pixel screen))
		  (font (open-font display font))
		  (border 1)			; Minimum margin around the text
		  (width 200)
		  (height 300)
		  (x (truncate (- (screen-width screen) width) 2))
		  (y (truncate (- (screen-height screen) height) 2))
		  (window (create-window :parent (screen-root screen)
					 :x x :y y :width width :height height
					 :background black
					 :border white
					 :border-width 1
					 :colormap (screen-default-colormap screen)
					 :bit-gravity :center
					 :event-mask '(:exposure :button-press)))
		  (gcontext (create-gcontext :drawable window
					     :background black
					     :foreground white
					     :font font)))
	     ;; Set window manager hints
	     (set-wm-properties window
				:name 'hello-world
				:icon-name string
				:resource-name string
				:resource-class 'hello-world
				:command (list* 'hello-world '(foo))
				:x x :y y :width width :height height
				:min-width width :min-height height
				:input :off :initial-state :normal)
	     (map-window window)		; Map the window

             ;; process
             (let ((ct 0))
               (labels ((profile-reporter ()
                          (loop
                           (sleep 0.1)
                           (incf ct)
                           (profile-update)))
                        (profile-update ()
                          (with-state (window)
                            (let ((x 10)
                                  (y 30))
                              ;; Draw text centered in widnow
                              (clear-area window)
                              (unless (boundp 'profile::*call-overhead*)
                                (profile::compute-time-overhead))
                              (let ((info ()))
                                (dolist (name profile::*timed-functions*)
                                  (let ((pinfo (profile::profile-info-or-lose name)))
                                    (multiple-value-bind
                                          (calls time consing profile callers)
                                        (funcall (profile::profile-info-read-time pinfo))
                                      (if (not (zerop calls))
                                          (push (profile::make-time-info name calls
                                                                (profile::compensate-time calls time profile)
                                                                consing
                                                                (sort (copy-seq callers)
                                                                      #'>= :key #'cdr))
                                                info)))))
                                (setq info (subseq (sort info #'>= :key #'profile::time-info-time) 0 100))
                                (draw-glyphs window gcontext 2 10 "  Seconds  |  Consed   |  Calls  |  Sec/Call  |  Name:")
                                (draw-glyphs window gcontext 2 20 "------------------------------------------------------")
                                (let ((total-time 0.0)
                                      (total-consed 0)
                                      (total-calls 0))
                                  (dolist (time info)
                                    (incf total-time (profile::time-info-time time))
                                    (incf total-calls (profile::time-info-calls time))
                                    (incf total-consed (profile::time-info-consing time))
                                    (draw-glyphs window gcontext 2 y
                                                 (format nil "~10,3F | ~9:D | ~7:D | ~10,5F | ~S~%"
                                                         (profile::time-info-time time)
                                                         (profile::time-info-consing time)
                                                         (profile::time-info-calls time)
                                                         (/ (profile::time-info-time time) (float (profile::time-info-calls time)))
                                                         (profile::time-info-name time)))
                                    (setf y (+ y 10)))
                                  (draw-glyphs window gcontext 2 y
                                               "------------------------------------------------------")
                                  (setf y (+ y 10))
                                  (draw-glyphs window gcontext 2 y
                                               (format nil "~10,3F | ~9:D | ~7:D |            | Total"
                                                       total-time total-consed total-calls))
                                  (setf y (+ y 10))
                                  (draw-glyphs window gcontext 2 y
                                               (format nil "Estimated total profiling overhead: ~4,2F seconds"
                                                       (* profile::*total-profile-overhead* (float total-calls)))))
                                (values))))))
                 (let ((proc (mp:make-process #'profile-reporter)))
                   ;; Handle events
                   (event-case (display :discard-p t :force-output-p t)
                               (exposure  ;; Come here on exposure events
                                (window count)
                                (when (zerop count) ;; Ignore all but the last exposure event
                                  (profile-update)
                                  nil))
                               (button-press () t)))  ;; Pressing any mouse-button exits
                 (setq abort nil)))
             ))
          ;; Ensure display is closed when done
          (when display
            (close-display display :abort abort))))))
  

(in-package :odcl)

(defun iterate-plot (fn &key (init-fn nil) (times 100) &aux data)
  (when init-fn
    (funcall init-fn))
  (dotimes (x times)
    (let (start end)
      (setq start (get-utime))
      (funcall fn)
      (setq end (get-utime))
      (push (cons x (- end start)) data)))
  (let ((temp (temp-file "plot"))
        (ctrl (temp-file "control"))
        (ps (temp-file "psdata")))
    (with-open-file (stream temp :direction :output :if-does-not-exist :create :if-exists :error)
      (dolist (pair (nreverse data))
        (format stream "~&~d ~d~%" (car pair) (cdr pair))))
    (with-open-file (stream ctrl :direction :output :if-does-not-exist :create :if-exists :error)
      (format stream "set terminal postscript~%")
      (format stream "set output \"~a\"~%" ps)
      (format stream "plot [0:~d] '~a' with lines~%" times temp))
    (ext:run-program "gnuplot" (list ctrl))
    (ext:run-program "gv" (list ps))))

(defun iterate-profile (fn &key (init-fn nil) (times 100) &aux data max)
  (when init-fn
    (funcall init-fn))
  (profile:reset-time)
  (dotimes (x times)
    (funcall fn)
    (push (profile::%report-time-list) data))
  (setq max (car data))
  (setq data (nreverse data))
  (let ((ctrl (temp-file "control")))
    (with-open-file (stream ctrl :direction :output :if-does-not-exist :create :if-exists :error)
      (dotimes (x (length data))
        (dolist (fun max)
          (let ((function (car fun)))
            (cmsg "Dumping ~s" function)
            (let ((calls (temp-file "plot"))
                  (bytes (temp-file "plot"))
                  (timing (temp-file "plot")))
              (with-open-file (calls calls :direction :output :if-does-not-exist :create :if-exists :error)
                (with-open-file (bytes bytes :direction :output :if-does-not-exist :create :if-exists :error)
                  (with-open-file (timing timing :direction :output :if-does-not-exist :create :if-exists :error)
                    (dotimes (x times)
                      (destructuring-bind (c b tt)
                          (get-alist function (nth x data))
                        (format calls "~&~d ~d~%" x c)
                        (format bytes "~&~d ~d~%" x b)
                        (format timing "~&~d ~d~%" x tt))))))
              (format stream "plot [0:~d] '~a' t \"~a: number of calls\" with lines~%" times calls (symbol-name function))
              (format stream "pause -1 \"Hit return to continue\"~%")
              (format stream "plot [0:~d] '~a' t \"~a: bytes consed\" with lines~%" times bytes (symbol-name function))
              (format stream "pause -1 \"Hit return to continue\"~%")
              (format stream "plot [0:~d] '~a' t \"~a: timing\" with lines~%" times timing (symbol-name function))
              (format stream "pause -1 \"Hit return to continue\"~%"))))))
    ctrl))
              
