;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: wcof -*-
;;; $Id: string.lisp,v 1.12 2002/02/20 05:29:48 jesse Exp $
;;;
;;; Copyright (c) 2000, 2001 onShore Development, Inc.

(in-package :odcl)


(defun random-string (&key (length 16))
  "Returns a random alphabetic string."
  (let ((id (make-string length)))
    (do ((x 0 (incf x)))
	(( = x length))
      (setf (aref id x) (code-char (+ 97 (random 26)))))
    id))


(defun filter-list (pred list)
  "return elts of list which match pred"
  (labels ((rec (lst accum)
             (if (null lst)
                 (reverse accum)
                 (if (funcall pred (car lst))
                     (rec (cdr lst) (cons (car lst) accum))
                     (rec (cdr lst) accum)))))
    (rec list nil)))

(defun char-count (string char)
  (let ((count 0))
    (dotimes (x (length string))
      (if (equal (aref string x) char)
          (incf count)))
    count))

(defun split-string (char string)
  (let ((index (position char string)))
    (if index
        (values (subseq string 0 index)
                (subseq string (1+ index)))
        (values string nil))))


(defun split (string delim)
  "Splits a string into a list of substrings: (split 'foo:bar:gaz' #\\:) returns ('foo' 'bar' 'gaz')
Removes empty strings from result"
  (when (not (characterp delim))
    (error "Delimiter is not a character"))
  (let ((div (position delim string)))
    (cond
     ((null div) ;; no delimiter
      (list string))
     ((= div 0) ;; delimiiter at beginning of string
      (split (subseq string (1+ div) (length string)) delim))
     ((= (1+ div) (length string)) ;; delimiter at end of string
      (list (subseq string 0 div)))
     (t
      (cons (subseq string 0 div)
	    (split (subseq string (1+ div) (length string)) delim))))))

(defun split-counted (string delim count)
  "Splits a string into a list of substrings: (split 'foo:bar:gaz' #\\:) returns ('foo' 'bar' 'gaz')
Removes empty strings from result"
  (when (not (characterp delim))
    (error "Delimiter is not a character"))
  (labels ((%internal (string delim count)
             (let ((div (position delim string :from-end t)))
               (cond
                 ((or (null div) (= count 0)) ;; no delimiter
                  (list string))
                 ((= div 0) ;; delimiiter at beginning of string
                  (split (subseq string (1+ div) (length string)) delim))
                 ((= (1+ div) (length string)) ;; delimiter at end of string
                  (list (subseq string 0 div)))
                 (t
                  (cons (subseq string (1+ div) (length string))
                        (%internal (subseq string 0 div) delim (1- count))))))))
    (reverse (%internal string delim count))))
    
(defun string-join (string-list &optional (delim " "))
  (typecase delim
    (base-char
     (setq delim (char-to-string delim))))
  (etypecase string-list
    (list
     (let ((s (make-string-output-stream)))
       (write-string (coerce-to-string (car string-list)) s)
       (dolist (x (cdr string-list))
         (write-string delim s)
         (write-string (coerce-to-string x) s))
       (get-output-stream-string s)))
    (string
     string-list)))

(defun coerce-to-string (seq)
  (etypecase seq
    (string seq)
    (sequence
     (let ((string (make-string (length seq))))
       (dotimes (x (length seq))
         (setf (aref string x)
               (code-char (aref seq x))))
       string))))

(defun char-to-string (char)
  (let ((string (make-string 1)))
    (setf (aref string 0) char)
    string))

(defun char-replace (char char2 string)
  (let ((string (copy-seq string)))
    (dotimes (i (length string))
      (when (char= char (aref string i))
        (setf (aref string i) char2)))
    string))

(defun overlay (string-a string-b offset)
  (dotimes (i (length string-b))
    (setf (aref string-a (+ offset i))
          (aref string-b i)))
  string-a)

(defun char-count (char string)
  (let ((count 0))
    (dotimes (i (length string))
      (when (char= char (aref string i))
        (incf count)))
    count))

(defun string-or-nil (string)
  (typecase string
    (string
     (let ((string (string-trim " " string)))
       (when (< 0 (length string))
         string)))))

(defun strcat (ids)
  (let ((str ""))
    (when ids
      (dolist (id (reverse ids))
        (setf str (concatenate 'string " " id str))))
    (string-trim " " str)))

(defun trimstring (string len)
  (if (< len (length string))
      (concatenate 'string (subseq string 0 len) "...")
      string))

(defun write-padded (count string stream)
  (write-space stream count)
  (write-string string stream))

(defun write-space (stream &optional (count 1))
  (dotimes (x count)
    (write-string "&nbsp;" stream)))


(defun string-replace (char replace-text string)
  (let ((count (char-count char string)))
    (if (= 0 count)
        (copy-seq string)
        (let* ((orig-length (length string))
               (repl-length (length replace-text))
               (new-length (+ orig-length (* (- repl-length 1) count)))
               (new-string (make-string new-length))
               (j 0))
          (do ((i 0 (incf i)))
              ((= i orig-length) new-string)
            (let ((test-char (aref string i)))
              (cond ((char= test-char char)
                     (overlay new-string replace-text j)
                     (incf j repl-length))
                    (t
                     (setf (aref new-string j) test-char)
                     (incf j)))))))))

(defun replace-char (in a b)
  (let ((out (make-string (length in))))
    (dotimes (i (length in))
      (let ((char (aref in i)))
        (if (char= char a)
            (setf (aref out i) b)
            (setf (aref out i) char))))
    out))

;; like search but allows for an escape character to nullify matches.
;; this needs to be brought more in line with the param list for
;; search.  for now it only accepts the :start2 keyword param
;; also needs to have more robust handling of escaped escape characters
;; and, what happens if the search sequence begins with the escape char.
(defun search-escaped (seq1 seq2 &key (start2 0) (escape-char #\\))
  (let ((index (search seq1 seq2 :start2 start2)))
    (cond ((or (null index) (eq 0 index)) index)
          ((char-equal (elt seq2 (- index 1)) escape-char)
           (if (eql (- 1 (length seq2)) index)
               nil
               (search-escaped seq1 seq2 :start2 (+ 1 index) :escape-char escape-char)))
          (t index))))

;; takes a template and an alist of symbol->value and returns a filled-in
;; instance of the template

(defun fill-template (values template
                      &key
                      (sub-delim "$")
                      (escape-char #\\)
                      (template-resolver))
  (let* ((start-index (search-escaped sub-delim template :escape-char escape-char))
         (end-index (cond ((null start-index) nil)
                          ((eql (- (length template) 1) start-index) start-index)
                          (t (search-escaped sub-delim template
                                             :start2 (+ 1 start-index)
                                             :escape-char escape-char)))))
    (cond ((null start-index) template)
          ((null end-index) (error "null end index"))
          ((eq start-index end-index) (error" start = end"))
          (t (concatenate 'string (subseq template 0 start-index)
                          (let ((value (subseq template (+ start-index 1) end-index)))
                            (string-join
                             (if template-resolver
                                 (funcall template-resolver values value)
                                 (or (cdr (assoc value values :test #'equalp))
                                     (error "unknown tempate key ~A" value)))
                             ", "))
                          (fill-template values (subseq template (+ 1 end-index))
                                         :sub-delim sub-delim
                                         :template-resolver template-resolver
                                         :escape-char escape-char))))))

(defun stringify (symbol)
  (etypecase symbol
    (symbol (symbol-name symbol))
    (string symbol)))


(defun string-to-array (string)
  (let* ((length (length string))
         (array  (make-array length)))
    (do ((i 0 (incf i)))
        ((= i length))
      (setf (aref array i)
            (char-code (aref string i))))
    array))

(defun array-to-string (string)
  (let* ((length (length string))
         (array  (make-string length)))
    (do ((i 0 (incf i)))
        ((= i length))
      (setf (aref array i)
            (code-char (aref string i))))
    array))

;;;
;;; These should be string-munging functions and macros.
;;;

;;
;; 't' if a string is non-nil and non-zero-length
;;

(defun nonempty (string)
  (and string (< 0 (length string))))

;;
;; Return a string with a space tacked on the end.
;;

(defun padstring (string)
  (cond ((null string)
	 "")
	(t
	 (concatenate 'string string " "))))

;;
;; String character replacement
;;

(defun char-to-char-in-string (str char1 char2)
  "Replace CHAR1 with CHAR2 throughout STR."
  (do ((pos (position char1 str) (position char1 str)))
      ((not pos) str)
    (setf (char str pos) char2)))

(defun string-replacing-char (str char1 char2)
  "Return a copy of STR with CHAR1 replaced everywhere by CHAR2."
  (char-to-char-in-string (copy-seq str) char1 char2))

;;
;; Generic stringy thingies
;;

(defmacro string-empty-p (str)
  (list 'string= str ""))

(defmacro string-nil-or-empty-p (str)
  (list 'or (list 'not str) (list 'string= str "")))

(defun y-or-n-string (value)
  (if value "Yes" "No"))




