/*
	$Id: netgame.h,v 1.1.1.1 2000/04/09 12:17:59 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------

	File purpose:
		Network component header file.


*/

//! component="Network"

#ifndef header_netgame
#define header_netgame

#define ACCESS_CHANNEL_READ 1
#define ACCESS_CHANNEL_WRITE 2

class CL_NetComputer;
class CL_NetGroup;
class CL_NetMessage;

class CL_NetGame
//: CL_NetGame is the interface to communcation with a network game.
{
public:
	virtual ~CL_NetGame() { ; }

	virtual const CL_NetComputer *get_server() const =0;
	//: The server of the network game.

	virtual const CL_NetGroup *get_all() const =0;
	//: A group of all the computers in the netgame.
	
	virtual bool peek(int channel) const =0;
	//: Returns true if a message is available on the specified channel.
	//!param: channel - Channel to look for a message.
	//!retval: True if a message is available on the specified channel.

	virtual CL_NetMessage receive(int channel, int timeout=-1)=0;
	//: Receives a message from the specified channel. The call is blocking, so
	//: don't call it if you aren't completly sure it will arrive (unless you set
	//: a timeout).
	//!param: channel - Channel to look for a message.
	//!param: timeout - Timeout, -1 means block forever.
	//!retval: The message received from the specified channel.

	virtual void send(
		const int dest_channel,
		const CL_NetComputer *dest,
		const CL_NetMessage &message,
		bool reliable = true)=0;
	//: Sends a message to the computer 'dest' at netchannel 'dest_port'.
	//!param: dest - Computer in the netgame to receive the message.
	//!param: dest_channel - Netchannel the message is sent to.
	//!param: message - Data in the message sent. All the data from 'message' is duplicated before this function returns.
	//!param: reliable - Arrival reliability. True guarantees arrival (use TCP/Pony express), false doesn't (use UDP/US Mail).

	virtual void send(
		const int dest_channel,
		const CL_NetGroup *dest,
		const CL_NetMessage &message,
		bool reliable = true)=0;
	//: Sends a message to a group of computers at netchannel 'dest_channel'.
	//!param: dest - The group of computers to receive the message.
	//!param: dest_channel - Netchannel the message is sent to.
	//!param: message - Data in the message sent. All the data from 'message' is duplicated before this function returns.
	//!param: reliable - Arrival reliability. True guarantees arrival (use TCP/Pony express), false doesn't (use UDP/US Mail).

	virtual CL_NetComputer *receive_computer_leave()=0;
	//: Returns the first (if more) computer that has left the game, or NULL if
	//: none. The next time the function is called, it returns the next computer,
	//: etc.<br>
	//: It is up the game to delete the netcomputer, if it do want its object to be
	//: removed from the all group. Note that a computer not deleted will rejoin
	//: with the same computer object, if it isn't deleted.
	//!retval: The netcomputer that left the game - delete it to remove it from the
	//!retval: all group.

	virtual const CL_NetComputer *receive_computer_join()=0;
	//: Returns the first (if more) computer that is joining the game, or NULL if
	//: none. The next time the function is called, it returns the next computer,
	//: etc.
	//!retval: The computer joining the game.

	virtual const CL_NetComputer *receive_computer_rejoin()=0;
	//: Returns the first (if more) computer that is re-joining the game, or NULL
	//: if none. The next time the function is called, it returns the next computer,
	//: etc. <br>
	//: A re-joining computer is simply a computer joining the game, that still
	//: havn't had its CL_NetComputer object deleted. This provides a way to
	//: recognize a computer that was in the game.
	//!retval: The computer re-joining the game.

	virtual bool receive_game_closed()=0;
	// Returns true if the game was closed.
	//!retval: True if game is closed, false otherwise.

	virtual int access_status(int channel) const =0;
	virtual bool is_writable(int channel) const =0;
	virtual bool is_readable(int channel) const =0;

	virtual int receive_access_changed()=0;
	//:Client side only

	virtual void set_access(
		int channel,
		const CL_NetComputer *computer,
		int access_rights=ACCESS_CHANNEL_READ|ACCESS_CHANNEL_WRITE)=0;
	//:Server side only
	virtual void set_access(
		int channel,
		const CL_NetGroup *group,
		int access_rights=ACCESS_CHANNEL_READ|ACCESS_CHANNEL_WRITE)=0;
	//:Server side only
};

#endif
