/*
	$Id: soundbuffer.h,v 1.1.1.1 2000/04/09 12:17:59 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------
*/

//! component="Sound"

#ifndef header_soundbuffer
#define header_soundbuffer

#include "soundbuffer_session.h"
#include "../Resources/resource_manager.h"

class CL_StaticSoundProvider;
class CL_StreamSoundProvider;
class CL_SoundCard;

class CL_SoundBuffer
//: Sample interface in ClanLib.
// The CL_SoundBuffer class represents a sample in ClanLib. It can either be
// static or streamed. The soundbuffer gets its sample data from a
// soundprovider, that is passed during construction.
// <br>
// A static sample are normally sound effects and other sounds, that do not
// change. ClanLib will always load the entire sample into memory, and
// possibly upload it to the soundcard.
// <br>
// Streamed samples are either large sample that should be loaded a bit at a
// time (music for instance), or sounds that change from playback to
// playback (a microphone, speech over the net, etc).
//!also: CL_StaticSoundProvider - Static sample data source.
//!also: CL_StreamSoundProvider - Streamed sample data source.
//!also: CL_SoundBuffer_Session - Sample playback control.
{
public:
	virtual ~CL_SoundBuffer() { ; }

	static CL_SoundBuffer *load(
		const char *res_id,
		CL_ResourceManager *manager);

	static CL_SoundBuffer *create(
		CL_StaticSoundProvider *provider,
		bool delete_provider=false);
	//: Creates a static soundbuffer. If 'delete_provider' is true, the provider
	//: will be deleted when the soundbuffer is deleted.
	//!retval: The created soundbuffer.
	
	static CL_SoundBuffer *create(
		CL_StreamSoundProvider *provider,
		bool delete_provider=false);
	//: Creates a streamed soundbuffer. If 'delete_provider' is true, the provider
	//: will be delete when the soundbuffer is deleted.
	//!retval: The created soundbuffer.

	virtual CL_StaticSoundProvider *get_static_provider() const =0;
	//: Returns the static soundprovider attached or NULL if it is a streamed
	//: sound buffer.
	//!retval: The static soundprovider or NULL if none.

	virtual CL_StreamSoundProvider *get_stream_provider() const =0;
	//: Return the streamed soundprovider attached or NULL if it is a static
	//: sound buffer.
	//!retval: The streamed soundprovider of NULL if none.

	virtual int get_length() const =0;
	//: Returns the length of the soundbuffer. 
	//: That is num_samples * bytes_per_sample.
	//!retval: Length of soundbuffer.
	
	virtual int get_num_samples() const =0;
	//: Returns the number of samples in the soundbuffer.
	//!retval: Number of samples in soundbuffer.

	virtual int get_frequency() const =0;
	//: Returns the start frequency used when the buffer is played.
	//!retval: Default/start frequency.
	
	virtual bool set_frequency(int new_freq)=0;
	//: Sets the start frequency used when the buffer is played.
	//!param: new_freq - The new default frequency.
	//!retval: True if successful, false otherwise.

	virtual float get_volume() const =0;
	//: Returns the start/default volume used when the buffer is played.
	//!retval: The volume.
	
	virtual bool set_volume(float new_volume)=0;
	//: Sets the default volume used when the buffer is played.
	//!param: new_volume - The new default volume.
	//!retval: True if successful, false otherwise.

	virtual float get_pan() const =0;
	//: Returns the default panning position when the buffer is played.
	//!retval: The panning position.

	virtual bool set_pan(float new_pan)=0;
	//: Sets the default panning position when the buffer is played.
	//!param: new_pan - The new default panning position.
	//!retval: True if successful, false otherwise.

	virtual bool is_playing(CL_SoundBuffer_Session **session=NULL, CL_SoundCard *card=NULL) const =0;
	//: Returns true if an instance of this soundbuffer is playing
	//!param: session - Pointer to session-pointer, set to point to the first session 
	//!param:           playing this soundbuffer. Return value is only valid if function returns true

	virtual void stop(CL_SoundCard *card=NULL)=0;
	//: Stops any sessions playing this soundbuffer

	virtual CL_SoundBuffer_Session play(bool looping=false, CL_SoundCard *card=NULL)=0;
	//: Plays the soundbuffer on the specified soundcard 
	//: and using the specified playback description.
	//!param: card - Soundcard to be used - NULL means use the current selected
	//!param:        sound card (CL_Sound::get_selected_card().
	//!retval: The playback session.

	virtual CL_SoundBuffer_Session prepare(bool looping=false, CL_SoundCard *card=NULL)=0;
	//: Prepares the soundbuffer for playback on the specified soundcard.
	//!param: card - Soundcard to be used or NULL to use
	//!param:        CL_Sound::get_selected_card().
	//!retval: The playback session.
};

#endif
