// 
// Copyright (c) 2006-2010, Benjamin Kaufmann
// 
// This file is part of Clasp. See http://www.cs.uni-potsdam.de/clasp/ 
// 
// Clasp is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// Clasp is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with Clasp; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//
#ifndef CLASP_SOLVE_ALGORITHMS_H_INCLUDED
#define CLASP_SOLVE_ALGORITHMS_H_INCLUDED

#ifdef _MSC_VER
#pragma warning (disable : 4200) // nonstandard extension used : zero-sized array
#pragma once
#endif

#include <clasp/solver_strategies.h>

/*!
 * \file 
 * Defines top-level functions for solving problems.
 */
namespace Clasp { 

//! Aggregates restart-parameters to configure restarts during search.
/*!
 * \see ScheduleStrategy
 */
struct RestartParams {
	enum Type { 
		default_restarts = 0, /**< Restart based on schedule. */
		local_restarts   = 1, /**< Restart if number of conflicts in *one* branch exceed threshold. */
		dynamic_restarts = 2  /**< Dynamic restarts. Ignore schedule. */
	};
	RestartParams() : sched(), cirBump(9973), cir(0), type(default_restarts), bounded(false), resetOnModel(false) {}
	void initDynamic(uint32 base, double xLbd, double xCfl) {
		sched.base = base;
		sched.grow = xLbd;
		sched.outer= static_cast<uint64>(xCfl * 1e8);
		type       = dynamic_restarts;
	}
	void disable() { sched = ScheduleStrategy::none();  }
	ScheduleStrategy sched; /**< Restart schedule to use. */
	uint32 cirBump;         /**< Bump factor for counter implication restarts. */
	uint8  cir;             /**< Apply counter implication bump every cir restarts (0: disable). */
	uint8  type;            /**< Type of restarts. */
	bool   bounded;         /**< Allow (bounded) restarts after first solution was found. */
	bool   resetOnModel;    /**< Repeat restart strategy after each solution. */
};


//! Aggregates parameters for the nogood deletion heuristic used during search.
struct ReduceParams {
	ReduceParams() : cflSched(ScheduleStrategy::none()), growSched(ScheduleStrategy::null()), baseFrac(3.0f), remFrac(0.75f)
		, dbGrow(1.1f), dbMaxGrow(3.0f)
		, base(5000), baseMin(10), baseMax(UINT32_MAX)
		, reduceOnRestart(false), estimate(false) {}
	
	void disable() {
		cflSched = ScheduleStrategy::none();
		growSched= ScheduleStrategy::none();
	}
	uint32 initDbSize() const;
	uint32 maxDbSize()  const;
	ScheduleStrategy cflSched; /**< Conflict-based deletion schedule.               */
	ScheduleStrategy growSched;/**< Growth-based deletion schedule.                 */
	ReduceStrategy   strategy; /**< Strategy to apply during nogood deletion.       */
	float            baseFrac; /**< Initial db size is base / baseFrac.             */
	float            remFrac;  /**< Fraction of nogoods to remove ([0.0,1.0]).      */
	float            dbGrow;   /**< Growth factor for db.                           */
	float            dbMaxGrow;/**< Stop growth once db size is > base*dbMaxGrow.   */
	uint32           base;     /**< Initial problem size estimate.                  */
	uint32           baseMin;  /**< Minimal initial db size.                        */
	uint32           baseMax;  /**< Maximal initial db size.                        */
	bool             reduceOnRestart; /**< Delete some nogoods on each restart.     */
	bool             estimate; /**< Use estimate of problem complexity in db init.  */
};

//! Type for holding global solve limits.
struct SolveLimits {
	explicit SolveLimits(uint64 conf = UINT64_MAX, uint64 r = UINT64_MAX) 
		: conflicts(conf)
		, restarts(r) {
	}
	bool   reached() const { return conflicts == 0 || restarts == 0; }
	uint64 conflicts; /*!< Number of conflicts. */
	uint64 restarts;  /*!< Number of restarts.  */
};

//! Type for holding pre-solve options.
struct InitParams {
	InitParams() : randRuns(0), randConf(0), initLook(0), lookType(0) {}
	uint16 randRuns; /*!< Number of initial randomized-runs. */
	uint16 randConf; /*!< Number of conflicts comprising one randomized-run. */
	uint16 initLook; /*!< Number of initial lookahead operations. */
	uint16 lookType; /*!< Type of initial lookahead operations. */
};
///////////////////////////////////////////////////////////////////////////////
// Parameter-object for the solve function
// 
///////////////////////////////////////////////////////////////////////////////
//! Parameter-Object for configuring search-parameters.
/*!
 * \ingroup solver
 */
struct SolveParams {
	//! Creates a default-initialized object.
	/*!
	 * The following parameters are used:
	 * restart      : quadratic: 100*1.5^k / no restarts after first solution
	 * shuffle      : disabled
	 * deletion     : initial size: vars()/3, grow factor: 1.1, max factor: 3.0, do not reduce on restart
	 * randomization: disabled
	 * randomProp   : 0.0 (disabled)
	 */
	SolveParams();
	
	RestartParams restart;
	ReduceParams  reduce;
	InitParams    init;
	mutable SolveLimits  limits;
	
	//! Sets the shuffle-parameters to use during search.
	/*!
	 * \param first   Shuffle program after first restarts.
	 * \param next    Re-Shuffle program every next restarts.
	 * \note          If first is equal to 0, shuffling is disabled.
	 */
	void setShuffleParams(uint32 first, uint32 next) {
		shuffleFirst_ = first;
		shuffleNext_  = next;
	}

	//! Sets the randomization-parameters to use during path initialization.
	/*!
	 * \param runs Number of initial randomized-runs.
	 * \param cfl  Number of conflicts comprising one randomized-run.
	 */
	bool setRandomizeParams(uint32 runs, uint32 cfls) {
		if (!runs || !cfls) { runs = cfls = 0; }
		init.randRuns = (uint16)std::min(runs, (1u<<16)-1);
		init.randConf = (uint16)std::min(cfls, (1u<<16)-1);
		return true;
	}

	//! Sets the probability with which choices are made randomly instead of with the installed heuristic.
	bool setRandomProbability(double p) {
		if (p >= 0.0 && p <= 1.0) {
			randFreq_ = p;
		}
		return randFreq_ == p;
	}
	// accessors
	uint32  randRuns()          const { return init.randRuns; }
	uint32  randConflicts()     const { return init.randConf; }
	double  randomProbability() const { return randFreq_;     }
	uint32  shuffleBase()       const { return shuffleFirst_; }
	uint32  shuffleNext()       const { return shuffleNext_;  }
private:
	double randFreq_;
	uint32 shuffleFirst_;
	uint32 shuffleNext_;
};


///////////////////////////////////////////////////////////////////////////////
// Basic solve functions
///////////////////////////////////////////////////////////////////////////////

//! Basic sequential search.
/*!
 * \ingroup solver
 * \relates Solver
 * \param ctx The context containing the problem.
 * \param p   The solve parameters to use.
 *
 * \return
 *  - true:  if the search stopped before the search-space was exceeded.
 *  - false: if the search-space was completely examined.
 * 
 */
bool solve(SharedContext& ctx, const SolveParams& p);

//! Basic sequential search under assumptions.
/*!
 * \ingroup solver
 * \relates Solver
 * The use of assumptions allows for incremental solving. Literals contained
 * in assumptions are assumed to be true during search but are undone before solve returns.
 *
 * \param ctx The context containing the problem.
 * \param p   The solve parameters to use.
 * \param assumptions The list of initial unit-assumptions.
 *
 * \return
 *  - true:  if the search stopped before the search-space was exceeded.
 *  - false: if the search-space was completely examined.
 * 
 */
bool solve(SharedContext& ctx, const SolveParams& p, const LitVec& assumptions);

///////////////////////////////////////////////////////////////////////////////
// General solve
///////////////////////////////////////////////////////////////////////////////

//! Interface for solve algorithms.
/*!
 * \ingroup solver
 * \relates Solver
 * SolveAlgorithm objects wrap an enumerator and
 * implement concrete solve algorithms.
 */
class SolveAlgorithm {
public:
	explicit SolveAlgorithm();
	virtual ~SolveAlgorithm();
	
	//! Force termination of current solve process.
	/*!
	 * Shall return true if termination is supported, otherwise false.
	 */
	virtual bool   terminate() = 0;

	//! Runs the solve algorithm.
	/*!
	 * \param ctx    A fully initialized context object containing the problem.
	 * \param p      The solve parameters for the master solver.
	 * \param assume A list of initial unit-assumptions.
	 *
	 * \return
	 *  - true:  if the search stopped before the search-space was exceeded.
	 *  - false: if the search-space was completely examined.
	 *
	 * \note 
	 * The use of assumptions allows for incremental solving. Literals contained
	 * in assumptions are assumed to be true during search but are undone before solve returns.
	 */
	bool solve(SharedContext& ctx, const SolveParams& p, LitVec assume);
protected:
	//! The default implementation simply forwards the call to the enumerator.
	virtual bool backtrackFromModel(Solver& s);
	//! The default implementation simply forwards the call to the enumerator.
	virtual void reportProgress(int type, Solver& s, uint64 maxCfl, uint32 maxLearnt);
	virtual bool doSolve(Solver& s, const SolveParams& p, const LitVec& assume) = 0;
	bool     initPath(Solver& s,  const LitVec& gp, InitParams& params);
	ValueRep solvePath(Solver& s, const SolveParams& p);
private:
	SolveAlgorithm(const SolveAlgorithm&);
	SolveAlgorithm& operator=(const SolveAlgorithm&);
};

//! A basic algorithm for single-threaded sequential solving.
class SimpleSolve : public SolveAlgorithm {
public:
	SimpleSolve() : SolveAlgorithm() {}
	bool   terminate();
private:
	bool   doSolve(Solver& s, const SolveParams& p, const LitVec& assume);
};

}
#endif
