// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000, 2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// mapwindow.cpp
//
// History:
//  04-12-2000 - created
//  04-03-2001 - added unit and cursor "animation" when moving
//  02-04-2001 - added MinXHex(), MinYHex(), MaxXHex(), MaxYHex()
//  22-04-2001 - added DrawFog()
////////////////////////////////////////////////////////////////////////

#include "mapwindow.h"
#include "globals.h"

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::MapWindow
// DESCRIPTION: Set the basic map window variables.
// PARAMETERS : x     - left edge of window
//              y     - top edge of window
//              w     - window width
//              h     - window height
//              flags - window flags (see window.h for details)
//              view  - pointer to the view the window will be put on
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

MapWindow::MapWindow( short x, short y, unsigned short w, unsigned short h,
                unsigned short flags, View *view ) :
           Window( x, y, w, h, flags, view ) {
  cursor.x = cursor.y = -1;		// cursor off
  cursor_image = IMG_CURSOR_IDLE;

  curx = cury = maxx = maxy = 0;

  unit = NULL;
  player = NULL;

  tiles = new Surface;
  fog = false;
  show = false;

  panel = new Panel( this, view );
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::~MapWindow
// DESCRIPTION: Destroy the map window.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

MapWindow::~MapWindow( void ) {
  delete tiles;
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::Init
// DESCRIPTION: Initialize the values required for map display and
//              scrolling. Call this function after having loaded a
//              map.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void MapWindow::Init( void ) {
  maxx = MAX( 0, m_w * (GFX_WIDTH - GFX_OVERLAP_X) + GFX_OVERLAP_X - w );
  maxy = MAX( 0, m_h * GFX_HEIGHT + GFX_OVERLAP_Y - h );
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::LoadMapGfx
// DESCRIPTION: Load the graphics set to use for the map from a file.
// PARAMETERS : file - SDL_RWops descriptor of a data file
// RETURNS    : 0 on success, -1 on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int MapWindow::LoadMapGfx( SDL_RWops *file ) {
  if ( tiles->LoadImageData( file ) ) return -1;
  tiles->DisplayFormat();

  sfont->DisplayFormat();
  lfont->DisplayFormat();
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::Reset
// DESCRIPTION: Set map offsets back to 0 and enable map display.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void MapWindow::Reset( void ) {
  curx = cury = 0;
  show = true;           // reenable map display
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::Draw
// DESCRIPTION: Draw a part of the map onto the window surface.
// PARAMETERS : x    - of the window (!) area to update
//              y    - of the window area to update
//              w    - width of the update area
//              h    - height of the update area
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void MapWindow::Draw( short x, short y, unsigned short w, unsigned short h ) {
  if ( show ) {
    Flood( Color(CF_COLOR_SHADOW) );
    DrawMap( curx + x, cury + y, w, h, this, x, y );
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::DrawMap
// DESCRIPTION: Draw a part of the map onto a surface.
// PARAMETERS : x    - leftmost pixel of the map (!) to paint
//              y    - topmost pixel to paint
//              w    - width of the map part to draw
//              h    - height of the map part
//              dest - destination surface
//              dx   - where to start drawing on the surface
//              dy   - where to start drawing vertically
// RETURNS    : -
//
// HISTORY
//   02-04-2001 - use MinXHex(), MinYHex(), MaxXHex(), and MaxYHex() to
//                determine the display boundaries
//   22-04-2001 - use DrawFog() for drawing fog
////////////////////////////////////////////////////////////////////////

void MapWindow::DrawMap( short x, short y, unsigned short w,
                unsigned short h, Surface *dest, short dx, short dy ) {
  int hx1 = MinXHex( x );
  int hy1 = MinYHex( y );
  int hx2 = MaxXHex( x, w );
  int hy2 = MaxYHex( y, h );

  Rect clip( dx, dy, w, h );
  int sx, sy, tx, ty, yoff;
  for ( tx = hx1; tx <= hx2; tx++ ) {
    sx = tx * (GFX_WIDTH - GFX_OVERLAP_X) - x;

    if ( tx & 1 ) yoff = -y;
    else yoff = GFX_OVERLAP_Y - y;

    for ( ty = hy1; ty <= hy2; ty++ ) {
      sy = ty * GFX_HEIGHT + yoff;

      // draw the terrain image
      DrawHex( HexImage( tx, ty ), dest, sx, sy, clip );

      // draw unit
      if ( Unit *u = GetUnit( tx, ty ) ) {
        DrawHex( u->Image(), dest, sx, sy, clip );
        if ( !u->IsReady() ) DrawHex( IMG_NOT_AVAILABLE, dest, sx, sy, clip );
      }

      // draw fog
      if ( fog && (m_path[XY2Index( tx, ty )] == -1) )
        DrawFog( dest, sx, sy, clip );

      // draw cursor
      if ( (tx == cursor.x) && (ty == cursor.y) )
        DrawHex( cursor_image, dest, sx, sy, clip );
    }
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::Pixel2Hex
// DESCRIPTION: Convert window coordinates to hex coordinates.
// PARAMETERS : px  - pixel x relative to window border
//              py  - window pixel y relative to window border
//              hex - buffer to hold the resulting hex
// RETURNS    : 0 on success, -1 on error (invalid pixels);
//              hex will contain -1, -1 then
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int MapWindow::Pixel2Hex( short px, short py, Point &hex ) const {
  short hx = px + curx, hy = py + cury;

  hex.x = (hx - GFX_OVERLAP_X/2) / (GFX_WIDTH - GFX_OVERLAP_X);
  hex.y = (hy - ((hex.x & 1) ? 0 : GFX_OVERLAP_Y)) / GFX_HEIGHT;

  if ( Map::Contains( hex.x, hex.y ) ) return 0;
  hex.x = hex.y = -1;
  return -1;
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::Hex2Pixel
// DESCRIPTION: Get the pixel coordinates (top left edge) of a hex
//              relative to the map display offsets.
// PARAMETERS : hx  - horizontal hex coordinate
//              hy  - vertical hex coordinate
//              pix - buffer to hold the resulting pixel values
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void MapWindow::Hex2Pixel( short hx, short hy, Point &pix ) const {
  pix.x = hx * (GFX_WIDTH - GFX_OVERLAP_X) - curx;
  pix.y = hy * GFX_HEIGHT + ((hx & 1) ? 0 : GFX_OVERLAP_Y) - cury;
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::HexOnScreen
// DESCRIPTION: Check whether a given hex is currently visible in the
//              map window area.
// PARAMETERS : hx - horizontal hex position
//              hy - vertical hex position
// RETURNS    : true if hex (or part of it) on screen, false otherwise
//
// HISTORY
////////////////////////////////////////////////////////////////////////

bool MapWindow::HexOnScreen( short hx, short hy ) const {
  Point p;
  Hex2Pixel( hx, hy, p );  // get absolute pixel values

  return( (p.x >= 0) && (p.x < w - GFX_WIDTH) &&
          (p.y >= 0) && (p.y < h - GFX_HEIGHT) );
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::UpdateHex
// DESCRIPTION: Redraw a single hex and update the display.
// PARAMETERS : hx - horizontal hex position
//              hy - vertical hex position
// RETURNS    : -
//
// HISTORY
//   22-04-2001 - use DrawFog() for drawing fog
////////////////////////////////////////////////////////////////////////

void MapWindow::UpdateHex( short hx, short hy ) {
  Point p;
  Hex2Pixel( hx, hy, p );  // get absolute position

  Rect clip( p.x, p.y, GFX_WIDTH, GFX_HEIGHT );   // create clip rect
  clip.Clip( Rect(0,0,w,h) );

  // draw the terrain image
  DrawHex( HexImage( hx, hy ), this, p.x, p.y, clip );

  // draw unit
  if ( Unit *u = GetUnit( hx, hy ) ) {
    DrawHex( u->Image(), this, p.x, p.y, clip );
    if ( !u->IsReady() ) DrawHex( IMG_NOT_AVAILABLE, this, p.x, p.y, clip );
  }

  // draw fog
  if ( fog && (m_path[XY2Index( hx, hy )] == -1) )
    DrawFog( this, p.x, p.y, clip );

  // draw cursor
  if ( (hx == cursor.x) && (hy == cursor.y) )
    DrawHex( cursor_image, this, p.x, p.y, clip );

  Show( clip );  // update display
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::DrawHex
// DESCRIPTION: Draw a hex image to a surface.
// PARAMETERS : n    - image number
//              dest - destination surface
//              px   - horizontal offset on surface
//              py   - vertical offset on surface
//              clip - clipping rectangle
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void MapWindow::DrawHex( unsigned short n, Surface *dest,
                         short px, short py, const Rect &clip ) {
  // set up destination
  Rect dstrect( px, py, GFX_WIDTH, GFX_HEIGHT );
  if ( dstrect.x + dstrect.w < clip.x ) return;
  if ( dstrect.y + dstrect.h < clip.y ) return;
  if ( dstrect.x >= clip.x + clip.w ) return;
  if ( dstrect.y >= clip.y + clip.h ) return;

  // set up source
  Rect srcrect;
  srcrect.x = (n % GFX_PER_LINE) * GFX_WIDTH;
  srcrect.y = (n / GFX_PER_LINE) * GFX_HEIGHT;
  srcrect.w = GFX_WIDTH;
  srcrect.h = GFX_HEIGHT;

  // clip and blit to surface
  dstrect.ClipBlit( srcrect, clip );
  tiles->LowerBlit( dest, srcrect, dstrect.x, dstrect.y );
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::DrawFog
// DESCRIPTION: Draw the fog hex to a surface.
// PARAMETERS : dest - destination surface
//              px   - horizontal offset on surface
//              py   - vertical offset on surface
//              clip - clipping rectangle
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void MapWindow::DrawFog( Surface *dest, short px, short py, const Rect &clip ) {
  tiles->SetAlpha( FOG_ALPHA, SDL_SRCALPHA );
  DrawHex( IMG_SHADOW, dest, px, py, clip );
  tiles->SetAlpha( 0, 0 );
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::MoveHex
// DESCRIPTION: Smoothly move a hex image (usually a unit or cursor)
//              from one hex to another (adjacent) one.
// PARAMETERS : hx1   - horizontal source hex position
//              hy1   - vertical source hex position
//              hx2   - horizontal destination hex position
//              hy2   - vertical destination hex position
//              speed - total time (ms) for the animation
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void MapWindow::MoveHex( unsigned short img, short hx1, short hy1,
                         short hx2, short hy2, unsigned short speed ) {
  // allocate a new surface where we can store the "background" graphics
  Point psrc, pdst;
  Rect bg;
  Hex2Pixel( hx1, hy1, psrc );
  Hex2Pixel( hx2, hy2, pdst );

  bg.x = MIN( psrc.x, pdst.x );
  bg.y = MIN( psrc.y, pdst.y );
  bg.w = GFX_WIDTH + ABS(psrc.x-pdst.x);
  bg.h = GFX_HEIGHT + ABS(psrc.y-pdst.y);
  bg.Clip( *this );

  Surface *bgs = new Surface;
  if ( !bgs->Create( bg.w, bg.h, DISPLAY_BPP, 0 ) ) {
    unsigned long oldticks, ticks = 0;

    Blit( bgs, bg, 0, 0 );

    oldticks = SDL_GetTicks();
    while ( ticks < speed ) {
      short cx, cy;

      cx = (short)((pdst.x - psrc.x) * ticks);           // IS THIS A BUG IN GCC???: when I do
      cy = (short)((pdst.y - psrc.y) * ticks);           // cx = psrc.x + (pdst.x - psrc.x) * ticks / speed;
                                                         // the result becomes completely bogus...
      DrawHex( img, this, psrc.x + cx / speed, psrc.y + cy / speed, bg );

      Show( bg );
      bgs->Blit( this, Rect( 0, 0, bg.w, bg.h ), bg.x, bg.y );
      ticks = SDL_GetTicks() - oldticks;
    }
    DrawHex( img, this, pdst.x, pdst.y, bg );
    Show( bg );
  }
  delete bgs;
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::CenterOnHex
// DESCRIPTION: Center the display on a given hex if possible.
// PARAMETERS : hx - horizontal hex position
//              hy - vertical hex position
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void MapWindow::CenterOnHex( short hx, short hy ) {
  Point p;
  short off;
  Hex2Pixel( hx, hy, p );

  off = p.x + (GFX_WIDTH - w) / 2 + curx;
  if ( off > maxx ) curx = maxx;
  else if ( off < 0 ) curx = 0;
  else curx = off;

  off = p.y + (GFX_HEIGHT - h) / 2 + cury;
  if ( off > maxy ) cury = maxy;
  else if ( off < 0 ) cury = 0;
  else cury = off;
  Draw();
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::BoxAvoidHexes
// DESCRIPTION: Try to place a box on the screen so that it does not
//              cover the two given hexes.
// PARAMETERS : box - the box; left and top edge will be modified
//              hx1 - first hex horizontal position
//              hy1 - first hex vertical position
//              hx2 - second hex horizontal position
//              hy2 - second hex vertical position
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void MapWindow::BoxAvoidHexes( Rect &box, short hx1, short hy1,
                               short hx2, short hy2 ) const {
  // define some preferred positions; we'll try to take one of these
  Point defs[4] = { { (w - box.w) / 2, (h - box.h) / 2 },
              { (w - box.w) / 2, 40 }, { 40, (h - box.h) / 2 },
              { w - box.w - 40, (h - box.h) / 2 } };
  Point p1, p2;
  Hex2Pixel( hx1, hy1, p1 );
  Hex2Pixel( hx2, hy2, p2 );

  for ( int i = 0; i < 4; i++ ) {
    if ( (((defs[i].x > p1.x + GFX_WIDTH) || (defs[i].x + box.w <= p1.x)) ||
         ((defs[i].y > p1.y + GFX_HEIGHT) || (defs[i].y + box.h <= p1.y))) &&
         (((defs[i].x > p2.x + GFX_WIDTH) || (defs[i].x + box.w <= p2.x)) ||
         ((defs[i].y > p2.y + GFX_HEIGHT) || (defs[i].y + box.h <= p2.y))) ) {
      box.x = defs[i].x;
      box.y = defs[i].y;
      return;
    }
  }
} 

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::MinXHex
// DESCRIPTION: Get the leftmost visible hex number.
// PARAMETERS : x - leftmost pixel. If a value of -1 is given, use the
//                  current viewport settings (curx) to determine the
//                  hex.
// RETURNS    : leftmost visible hex column
//
// HISTORY
////////////////////////////////////////////////////////////////////////

unsigned short MapWindow::MinXHex( short x ) const {
  if ( x == -1 ) x = curx;
  return( MAX( (x - GFX_OVERLAP_X) / (GFX_WIDTH - GFX_OVERLAP_X), 0 ) );
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::MinYHex
// DESCRIPTION: Get the topmost visible hex number.
// PARAMETERS : y - topmost pixel. If a value of -1 is given, use the
//                  current viewport settings (cury) to determine the
//                  hex.
// RETURNS    : topmost visible hex row
//
// HISTORY
////////////////////////////////////////////////////////////////////////

unsigned short MapWindow::MinYHex( short y ) const {
  if ( y == -1 ) y = cury;
  return( MAX( (y - GFX_OVERLAP_Y) / GFX_HEIGHT, 0 ) );
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::MaxXHex
// DESCRIPTION: Get the rightmost visible hex number.
// PARAMETERS : x - leftmost pixel. If a value of -1 is given, use the
//                  current viewport settings (curx/MapWindow::Width())
//                  to determine the hex.
//              w - display width
// RETURNS    : rightmost visible hex column
//
// HISTORY
////////////////////////////////////////////////////////////////////////

unsigned short MapWindow::MaxXHex( short x, unsigned short w ) const {
  if ( x == -1 ) {
    x = curx;
    w = Width();
  }
  return( MIN( (x + w) / (GFX_WIDTH - GFX_OVERLAP_X), MapWidth() - 1 ) );
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::MaxYHex
// DESCRIPTION: Get the lowest visible hex number.
// PARAMETERS : y - topmost pixel. If a value of -1 is given, use the
//                  current viewport settings (cury/MapWindow::Width())
//                  to determine the hex.
//              h - display height
// RETURNS    : lowest visible hex row
//
// HISTORY
////////////////////////////////////////////////////////////////////////

unsigned short MapWindow::MaxYHex( short y, unsigned short h ) const {
  if ( y == -1 ) {
    y = cury;
    h = Height();
  }
  return( MIN( (y + h) / GFX_HEIGHT, MapHeight() - 1 ) );
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::ShowOff
// DESCRIPTION: Hide the actual map and display just blackness.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void MapWindow::ShowOff( void ) {
  show = false;
  Flood( Color(CF_COLOR_SHADOW) );
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::CheckScroll
// DESCRIPTION: Look at the current cursor position and map offsets and
//              decide whether we need to scroll the display. If so, do
//              it.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void MapWindow::CheckScroll( void ) {
  if ( cursor.x == -1 ) return;

  Point p;
  Hex2Pixel( cursor.x, cursor.y, p );

  if ( (curx > 0) && (p.x <= GFX_WIDTH) ) p.x = -w / 2;
  else if ( (curx < maxx) && (w - p.x - GFX_WIDTH <= GFX_WIDTH) ) p.x = w / 2;
  else p.x = 0;

  if ( (cury > 0) && (p.y <= GFX_HEIGHT) ) p.y = -h / 2;
  else if ( (cury < maxy) && (h - p.y - GFX_HEIGHT <= GFX_HEIGHT) ) p.y = h / 2;
  else p.y = 0;

  if ( p.x || p.y ) Scroll( p.x, p.y );
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::Scroll
// DESCRIPTION: Scroll the currently visible map area and update the
//              display surface.
// PARAMETERS : px - pixels to scroll horizontally
//              py - pixels to scroll vertically
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void MapWindow::Scroll( short px, short py ) {
  if ( curx + px < 0 ) px = -curx;
  else if ( curx + px > maxx ) px = maxx - curx;
  curx += px;

  if ( cury + py < 0 ) py = -cury;
  else if ( cury + py > maxy ) py = maxy - cury;
  cury += py;

#ifdef CF_SDL_LOCAL_BLIT
  // right now SDL cannot blit parts of a surface to another
  // place on the same surface if both areas overlap

  // calculate dirty rectangles
  Rect d1( 0, 0, 0, 0 );  // dirty 1
  Rect d2( 0, 0, 0, 0 );  // dirty 2
  Rect c( 0, 0, w, h );   // copy, still valid

  if ( px > 0 ) {
    d1 = Rect( w - px, 0, px, h );
    c.x = px;
    c.w = w - px;
  } else if ( px < 0 ) {
    d1 = Rect( 0, 0, -px, h );
    c.w = w + px;
  }

  if ( py > 0 ) {
    d2 = Rect( 0, h - py, w, py );
    c.y = py;
    c.h = h - py;
  } else if ( py < 0 ) {
    d2 = Rect( 0, 0, w, -py );
    c.h = h + py;
  }

  // eliminate overlapping parts
  if ( (d1.w > 0) && (d2.w > 0) ) {
    if ( d1.x == d2.x ) d2.x += d1.w;
    d2.w = MAX( 0, d2.w - d1.w );
  }

  // copy valid part to current position
  if ( c.w && c.h )
    Blit( this, c, (c.x == 0) ? -px : 0, (c.y == 0) ? -py : 0 );

  // update dirty parts
  if ( d1.w && d1.h ) Draw( d1.x, d1.y, d1.w, d1.h );
  if ( d2.w && d2.h ) Draw( d2.x, d2.y, d2.w, d2.h );
#else
  Draw();
#endif

  // update display
  view->Refresh( *this );
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::UnsetCursor
// DESCRIPTION: Remove the cursor from the display.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void MapWindow::UnsetCursor( void ) {
  if ( (cursor.x < 0) || (cursor.y < 0) ) return;

  Point old = cursor;
  cursor.x = cursor.y = -1;

  // update old cursor position
  UpdateHex( old.x, old.y );

  if ( player->Mode() == MODE_IDLE ) {
    // if we had highlighted a unit's target we need to remove that mark
    Unit *u = GetUnit( old.x, old.y );
    const Point *target;
    if ( u && (u->Owner() == player) && (target = u->Target()) )
      UpdateHex( target->x, target->y );
  } else if ( player->Mode() == MODE_BUSY ) SetCursorImage( IMG_CURSOR_SELECT );
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::SetCursor
// DESCRIPTION: Set the cursor onto a specified hex. If cursor display
//              was off, calling this function will reenable it.
// PARAMETERS : hx  - destination hex horizontal coordinate
//              hy  - destination hex vertical coordinate
//              upd - update display at previous cursor location
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void MapWindow::SetCursor( short hx, short hy, bool upd ) {
  if ( upd ) UnsetCursor();
  cursor.x = hx;
  cursor.y = hy;

  Unit *u;
  if ( (u = GetUnit( hx, hy )) ) {
    if ( u->Owner() != player ) {
      if ( (player->Mode() == MODE_BUSY) &&
           unit->CanHit( u ) ) SetCursorImage( IMG_CURSOR_ATTACK );
    } else if ( player->Mode() == MODE_IDLE ) {
      // if it's a unit of the active player, highlight its target if it has one
      const Point *target = u->Target();
      if ( target ) {
        Point p;
        Hex2Pixel( target->x, target->y, p );
        DrawHex( IMG_CURSOR_HIGHLIGHT, this, p.x, p.y, *this );
        Show( Rect(p.x, p.y, GFX_WIDTH, GFX_HEIGHT) );
      }
    }
  }

  panel->Update( u );
  UpdateHex( hx, hy );
  CheckScroll();
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::MoveCursor
// DESCRIPTION: Move the cursor into the given direction if possible.
// PARAMETERS : dir - requested direction
// RETURNS    : 0 on success, -1 on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int MapWindow::MoveCursor( Direction dir ) {
  Point dest, curs;

  if ( Dir2XY( cursor.x, cursor.y, dir, dest ) ) return -1;

  curs = cursor;
  UnsetCursor();
  MoveHex( cursor_image, curs.x, curs.y, dest.x, dest.y, ANIM_SPEED_CURSOR );
  SetCursor( dest.x, dest.y, false );
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::CursorOff
// DESCRIPTION: Do not update the cursor display. This function does not
//              remove any cursor that may currently be displayed, but
//              it prevents any new cursors from being drawn until the
//              next call to SetCursor().
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
//   23-06-2001 - clear panel
////////////////////////////////////////////////////////////////////////

void MapWindow::CursorOff( void ) {
  cursor.x = cursor.y = -1;
  SetCursorImage( IMG_CURSOR_IDLE );
  panel->Update( NULL );
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::DrawUnitInfo
// DESCRIPTION: Display unit type information.
// PARAMETERS : type - unit type (may be NULL)
//              dest - destination surface
//              rect - rectangle in which to display
// RETURNS    : -
//
// HISTORY
//   22-04-2001 - use alpha for fog
//   14-08-2001 - don't display range if (min range == max range == 1)
//                or (strength == 0)
////////////////////////////////////////////////////////////////////////

void MapWindow::DrawUnitInfo( const UnitType *type, Surface *dest,
                              const Rect &rect ) {
  char buf[12];
  unsigned short tiles[] = { TILE_ROAD, TILE_PLAINS, TILE_FOREST,
       TILE_SWAMP, TILE_MOUNTAINS, TILE_WATER_SHALLOW, TILE_WATER_DEEP };

  dest->DrawBox( rect, BOX_CARVED );
  if ( type == NULL ) return;

  sfont->Write( type->ut_name, dest,
                rect.x + (rect.w - sfont->TextWidth(type->ut_name)) / 2,
                rect.y + 4, rect );

  // show terrain access
  Rect terborder( rect.x + 10, rect.y + rect.h - GFX_HEIGHT - 10,
                  rect.w - 20, GFX_HEIGHT + 10 );
  dest->FillRect( terborder.x, terborder.y, terborder.w, 1, Color(CF_COLOR_SHADOW) );
  dest->FillRect( terborder.x, terborder.y + 1, terborder.w, 1, Color(CF_COLOR_HIGHLIGHT) );

  short xpos = terborder.x + (terborder.w - GFX_WIDTH * 7) / 2,
        ypos = terborder.y + 5, spacing;
  for ( int i = 0; i < 7; i++ ) {
    DrawHex( tiles[i], dest, xpos, ypos, terborder );
    if ( !(type->ut_terrain & (1 << i)) ) DrawFog( dest, xpos, ypos, terborder );
    xpos += GFX_WIDTH;
  }

  // show unit combat strength
  spacing = (rect.w - 2 * ICON_WIDTH) / 3;
  xpos = rect.x + spacing;
  ypos = rect.y + sfont->Height() + 10;
  itoa( type->ut_moves / MCOST_MIN, buf );
  sfont->Write( buf, dest, xpos + ICON_WIDTH + 4,
                ypos + (ICON_HEIGHT - sfont->Height())/2 );
  Images[ICON_SPEED]->Draw( dest, xpos, ypos );

  xpos += ICON_WIDTH + spacing;
  itoa( type->ut_defence, buf );
  sfont->Write( buf, dest, xpos + ICON_WIDTH + 4,
                ypos + (ICON_HEIGHT - sfont->Height())/2 );
  Images[ICON_DEFENCE]->Draw( dest, xpos, ypos );

  spacing = (rect.w - 3 * ICON_WIDTH) / 4;
  xpos = rect.x + spacing;
  ypos += ICON_HEIGHT - 5;
  Images[ICON_GROUND]->Draw( dest, xpos, ypos );
  if ( (type->ut_pow_ground == 0) ||
       ((type->ut_min_range_ground == type->ut_max_range_ground)
         && (type->ut_max_range_ground <= 1)) ) itoa( type->ut_pow_ground, buf );
  else sprintf( buf, "%2d (%d-%d)", type->ut_pow_ground,
                type->ut_min_range_ground, type->ut_max_range_ground );
  sfont->Write( buf, dest, xpos + (ICON_WIDTH - sfont->TextWidth(buf))/2,
                           ypos + ICON_HEIGHT );

  xpos += ICON_WIDTH + spacing;
  Images[ICON_AIRCRAFT]->Draw( dest, xpos, ypos );
  if ( (type->ut_pow_air == 0) ||
       ((type->ut_min_range_air == type->ut_max_range_air)
         && (type->ut_max_range_air <= 1)) ) itoa( type->ut_pow_air, buf );
  else sprintf( buf, "%2d (%d-%d)", type->ut_pow_air, type->ut_min_range_air,
                               type->ut_max_range_air );
  sfont->Write( buf, dest, xpos + (ICON_WIDTH - sfont->TextWidth(buf))/2,
                           ypos + ICON_HEIGHT );

  xpos += ICON_WIDTH + spacing;
  Images[ICON_SHIP]->Draw( dest, xpos, ypos );
  if ( (type->ut_pow_ship == 0) ||
       ((type->ut_min_range_ship == type->ut_max_range_ship)
         && (type->ut_max_range_ship <= 1)) ) itoa( type->ut_pow_ship, buf );
  else sprintf( buf, "%2d (%d-%d)", type->ut_pow_ship, type->ut_min_range_ship,
                               type->ut_max_range_ship );
  sfont->Write( buf, dest, xpos + (ICON_WIDTH - sfont->TextWidth(buf))/2,
                           ypos + ICON_HEIGHT );
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::VideoModeChange
// DESCRIPTION: This method is called by the view whenever the video
//              resolution changes. The window can then adjust itself
//              to the new dimensions. Afterwards a View::Refresh() is
//              performed, i. e. the window just needs to reblit its
//              contents to its internal buffer, and NOT call a view
//              update itself.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void MapWindow::VideoModeChange( void ) {
  SetSize( view->Width(), view->Height() );
  Init();
  if ( curx > maxx ) curx = maxx;
  if ( cury > maxy ) cury = maxy;

  if ( HexOnScreen( cursor.x, cursor.y ) ) Draw();
  else CenterOnHex( cursor.x, cursor.y );

  if ( (panel->TopEdge() != 0) &&
       (panel->TopEdge() + panel->Height() < h - 1) )
    panel->y = h - panel->Height();
}


////////////////////////////////////////////////////////////////////////
// NAME       : Panel::Panel
// DESCRIPTION: Create the panel window.
// PARAMETERS : mw   - pointer to map window
//              view - pointer to view
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Panel::Panel( MapWindow *mw, View *view ) :
       Window( 0, mw->Height() - DEFAULT_PANEL_HEIGHT, 0, 0, 0, view ) {
  unit = NULL;
  mapwin = mw;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Panel::Draw
// DESCRIPTION: Draw the panel window.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Panel::Draw( void ) {
  if ( unit ) {
    char buf[4];
    sprintf( buf, "(%d)", unit->GroupSize() );

    Window::Draw();
    Images[ICON_XP_BASE + unit->XPLevel()]->Draw( this, 5, (h - XP_ICON_HEIGHT)/2 );
    sfont->Write( unit->Name(), this, 10 + XP_ICON_WIDTH, (h - sfont->Height())/2 );
    sfont->Write( buf, this, 15 + XP_ICON_WIDTH + sfont->TextWidth( unit->Name() ),
                  (h - sfont->Height())/2 );
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Panel::Update
// DESCRIPTION: Set the information displayed on the panel and update
//              the display.
// PARAMETERS : unit - unit to display information for (may be NULL)
// RETURNS    : -
//
// HISTORY
//   21-05-2001 - dynamically resize and reposition the panel
//   17-09-2001 - only allow top left and bottom left positions
////////////////////////////////////////////////////////////////////////

void Panel::Update( Unit *unit ) {
  Rect refresh = *this;

  if ( unit == NULL ) {
    if ( this->unit != NULL ) {
      w = h = 0;
      y = mapwin->Height() - DEFAULT_PANEL_HEIGHT;
      view->Refresh( refresh );
      this->unit = NULL;
    }
  } else {
    this->unit = unit;

    // resize and reposition if necessary
    w = sfont->TextWidth( unit->Name() ) + 25 + XP_ICON_WIDTH + sfont->Width() * 3;
    h = refresh.h = DEFAULT_PANEL_HEIGHT;

    Point pcursor;
    mapwin->Hex2Pixel( mapwin->Cursor()->x, mapwin->Cursor()->y, pcursor );
    if ( pcursor.x <= w + GFX_WIDTH ) {
      if ( y == 0 ) {
        if ( pcursor.y <= h + GFX_HEIGHT ) {
          y = mapwin->Height() - h;
        }
      } else if ( pcursor.y > y - GFX_HEIGHT ) {
        y = 0;
      }
    }
    SetSize( w, h );   // allocate surface
    Draw();

    // update the old window position and show the new
    if ( refresh.y == y ) {
      if ( w > refresh.w ) refresh.w = w;
    } else Show();
    view->Refresh( refresh );
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Panel::HandleEvent
// DESCRIPTION: The window must pass all incoming events to the map
//              window which does proper processing.
// PARAMETERS : event - event as received from the event handler
// RETURNS    : GUI status
//
// HISTORY
////////////////////////////////////////////////////////////////////////

GUI_Status Panel::HandleEvent( const SDL_Event &event ) {
  return mapwin->HandleEvent( event );
}

