// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000, 2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// view.cpp
//
// History:
//  02-12-2000 - created
//  25-03-2001 - added CloseAllWindows()
//  03-04-2001 - added Update()
//  22-04-2001 - removed SetPalette()
//  21-05-2001 - added ActiveWindow()
//  02-06-2001 - added ToggleFullScreen() and SetEventFilter()
//  22-06-2001 - added PeekEvent()
//  20-10-2001 - added SetVideoMode() - allows for video mode switching
////////////////////////////////////////////////////////////////////////

#include "view.h"

////////////////////////////////////////////////////////////////////////
// NAME       : View::View
// DESCRIPTION: Initialize the view structure. It will also create a
//              display surface.
// PARAMETERS : w     - display surface width
//              h     - display surface height
//              bpp   - display depth
//              flags - display surface flags (see SDL/SDL.h)
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

View::View( unsigned short w, unsigned short h, short bpp,
            unsigned long flags ) {
  sfont = lfont = NULL;
  s_surface = NULL;

  SetVideoMode( w, h, bpp, flags );
  if ( !s_surface ) return;

  x = y = 0;
  allow_updates = true;
  filter = NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : View::SetVideoMode
// DESCRIPTION: Set the view's video mode. If the view already has a
//              video surface (i.e. the resultion is changed) all
//              windows are adjusted to the new size.
// PARAMETERS : w     - resolution width
//              h     - resolution height
//              bpp   - color depth
//              flags - video flags
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void View::SetVideoMode( unsigned short w, unsigned short h, short bpp,
                         unsigned long flags ) {
  bool change = (s_surface != NULL);

  if ( change ) SDL_FreeSurface( s_surface );

  s_surface = SDL_SetVideoMode( w, h, bpp, flags );

  if ( s_surface ) {
    this->w = s_surface->w;
    this->h = s_surface->h;

    if ( change ) {
      // adjust window positions and sizes
      Window *win = static_cast<Window *>( windows.Tail() );

      while ( win ) {
        win->VideoModeChange();
        win = static_cast<Window *>( win->Prev() );
      }

      Refresh();
    }
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : View::Update
// DESCRIPTION: Update the display surface without reblitting the
//              particular window surfaces.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void View::Update( void ) {
  Update( *this );
}

////////////////////////////////////////////////////////////////////////
// NAME       : View::Update
// DESCRIPTION: Update part of the display surface without reblitting
//              the particular window surfaces.
// PARAMETERS : rect - area to be updated
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void View::Update( const Rect &rect ) {
  if ( allow_updates )
    SDL_UpdateRect( s_surface, rect.x, rect.y, rect.w, rect.h );
}

////////////////////////////////////////////////////////////////////////
// NAME       : View::Refresh
// DESCRIPTION: Update the entire display surface.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void View::Refresh( void ) {
  Window *win = static_cast<Window *>( windows.Tail() );

  FillRect( *this, Color(CF_COLOR_SHADOW) );
  while ( win ) {
    if ( !win->Closed() ) {
      win->Blit( this, Rect( 0, 0, win->Width(), win->Height() ),
                 win->LeftEdge(), win->TopEdge() );
    }
    win = static_cast<Window *>( win->Prev() );
  }

  // now update the display
  Update();
}

////////////////////////////////////////////////////////////////////////
// NAME       : View::Refresh
// DESCRIPTION: Update parts of the display surface only.
// PARAMETERS : refresh - area to refresh
// RETURNS    : -
//
// HISTORY
//  03-10-2001 - use LowerBlit() instead of Blit()
////////////////////////////////////////////////////////////////////////

void View::Refresh( const Rect &refresh ) {
  Window *window = static_cast<Window *>( windows.Tail() );

  FillRect( refresh, Color(CF_COLOR_SHADOW) );
  while ( window ) {
    if ( !window->Closed() ) {

      // clip the window to the refreshable area
      Rect win = *window;
      Rect src( 0, 0, window->Width(), window->Height() );
      win.ClipBlit( src, refresh );
      if ( win.Width() && win.Height() )
        window->LowerBlit( this, src, win.LeftEdge(), win.TopEdge() );
    }
    window = static_cast<Window *>( window->Prev() );
  }
  Update( refresh );
}

////////////////////////////////////////////////////////////////////////
// NAME       : View::AddWindow
// DESCRIPTION: Attach a window to the view. The window will then
//              receive event messages and can be shown. This
//              function does not automatically update the display.
//              Call Window::Refresh to do so manually.
// PARAMETERS : window - window to add
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void View::AddWindow( Window *window ) {
  windows.AddHead( window );
}

////////////////////////////////////////////////////////////////////////
// NAME       : View::CloseWindow
// DESCRIPTION: Close a window. If there are other windows left, update
//              the display.
//              This function only marks the window closed. The actual
//              cleaning up is done in the event handling function.
// PARAMETERS : window - window to close
// RETURNS    : active window after closing, NULL if no windows left
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Window *View::CloseWindow( Window *window ) {
  Window *focus;

  window->Remove();                     // remove from windows list

  if ( !windows.IsEmpty() ) {
    focus = static_cast<Window *>( windows.Head() );
    Refresh( *window );
  } else focus = NULL;

  window->Close();
  windows.AddTail( window );            // put it back to the end of the list

  return focus;
}

////////////////////////////////////////////////////////////////////////
// NAME       : View::SelectWindow
// DESCRIPTION: Redirect all incoming event messages to another window.
// PARAMETERS : window - window to receive event messages
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void View::SelectWindow( Window *window ) {
  if ( window != static_cast<Window *>( windows.Head() ) ) {
    window->Remove();
    windows.AddHead( window );
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Window::CloseAllWindows
// DESCRIPTION: Close all windows. This function only marks the window
//              to be closed. The actual cleaning is done in the event
//              handling function.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void View::CloseAllWindows( void ) {
  while ( !windows.IsEmpty() ) delete windows.RemHead();
  Refresh();
}

////////////////////////////////////////////////////////////////////////
// NAME       : View::HandleEvents
// DESCRIPTION: Distribute event messages to the windows and react
//              appropriately.
// PARAMETERS : -
// RETURNS    : GUI status
//
// HISTORY
//   25-03-2001 - issue GUI_RESTART instead of GUI_QUIT when all windows
//                have been closed (obsolete)
////////////////////////////////////////////////////////////////////////

GUI_Status View::HandleEvents( void ) {
  SDL_Event event;
  GUI_Status rc = FetchEvent( event );

  if ( rc != GUI_ERROR ) {
    Window *win = static_cast<Window *>( windows.Head() );
    if ( win ) {
      rc = win->HandleEvent( event );
      if ( rc == GUI_CLOSE ) win = CloseWindow( win );
    }

    // collect destroyed windows
    while ( (win = static_cast<Window *>(windows.Tail())) && win->Closed() ) {
      win->Remove();
      delete win;
    }
  }
  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : View::FetchEvent
// DESCRIPTION: Get the next event from the event queue.
// PARAMETERS : event - buffer to hold the event information
// RETURNS    : GUI status
//
// HISTORY
//   19-06-2001 - use event filter callback
////////////////////////////////////////////////////////////////////////

GUI_Status View::FetchEvent( SDL_Event &event ) {
  GUI_Status rc;

  do {
    if ( SDL_WaitEvent( &event ) ) {
      if ( filter ) rc = filter( event, static_cast<Window *>(windows.Head()) );
      else rc = GUI_OK;
    } else rc = GUI_ERROR;
  } while ( rc == GUI_SKIP );

  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : View::PeekEvent
// DESCRIPTION: Get the next event from the event queue if there are any
//              events pending. If there are no events waiting, return
//              immediately.
// PARAMETERS : event - buffer to hold the event information
// RETURNS    : GUI status; GUI_NONE is returned if no events waiting
//
// HISTORY
////////////////////////////////////////////////////////////////////////

GUI_Status View::PeekEvent( SDL_Event &event ) {
  GUI_Status rc;

  do {
    if ( SDL_PollEvent( &event ) ) {
      if ( filter ) rc = filter( event, static_cast<Window *>(windows.Head()) );
      else rc = GUI_OK;
    } else rc = GUI_NONE;
  } while ( rc == GUI_SKIP );

  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : View::ToggleFullScreen
// DESCRIPTION: Switch from windowed to fullscreen mode or vice versa.
// PARAMETERS : -
// RETURNS    : 0 on success, non-zero on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int View::ToggleFullScreen( void ) {
  return !SDL_WM_ToggleFullScreen( s_surface );
}

