//=============================================================================
//
// Copyright (C) 1999,2000 by Eric Sunshine <sunshine@sunshineco.com>
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//   1. Redistributions of source code must retain the above copyright
//      notice, this list of conditions and the following disclaimer.
//   2. Redistributions in binary form must reproduce the above copyright
//      notice, this list of conditions and the following disclaimer in the
//      documentation and/or other materials provided with the distribution.
//   3. The name of the author may not be used to endorse or promote products
//      derived from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
// NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
// TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//=============================================================================
//-----------------------------------------------------------------------------
// NeXTKeymap.m
//
//	For each keyboard event generated by the host platform, Crystal Space
//	expects two pieces of information: the raw key code, and the cooked
//	character code.  The raw key code is the canonic code generated for
//	that key press with no modifiers applied.  The cooked character code is
//	the actual character generated with all modifiers applied.  Here are
//	some examples:
//
//          Input    Raw  Cooked
//          -------  ---  ------
//          ctrl-C   c    ^C
//          d        d    d
//          shift-e  e    E
//          F        f    F
//          alt-m    m    "mu" *
//          shift-4  4    $
//
//	(*) For alt-m, the cooked character depends upon the user's current key
//	mapping.  It may actually be mapped to any character, but is often
//	mapped to Greek "mu", as it was in this example.
//
//	Unlike OpenStep's -[NSEvent charactersIgnoringModifiers], NextStep's
//	NXEvent does not offer any way of retrieving the raw (unmodified) key
//	code.  Because a cooked character may be generated via an arbitrary
//	combination of modifiers (or none at all), and because a cooked
//	character may even be mapped to more than one keystroke / modifier
//	combination, there is no way to retrieve the raw key code from the
//	cooked character.
//
//	In order to work around this limitation, it is necessary to access the
//	(undocumented) raw keymapping data resource currently in use by the
//	WindowServer and AppKit.  This resource defines the exact mapping
//	between hardware scan codes and the characters which those scan codes
//	generate in the presence of various modifier key combinations.  Among
//	other information, the raw character generated for any given scan code
//	when no modifiers are active can be gleaned from the keymapping
//	resources.
//
//	Using the keymapping resource as its source of information, this class
//	(NeXTKeymap) builds a translation table which maps hardware scan codes
//	to raw key codes (that is, the character mapped to a scan code when no
//	modifiers are active).  Clients of this class can index into the
//	translation table using the hardware scan code which comes from the
//	NXEvent record in order to retrieve the raw key code which Crystal
//	Space expects.  The cooked character code, on the other hand, still
//	comes straight from the NXEvent record.
//
//	The bulk of the code in this file was extracted from the diagnostic
//	tool `dumpkeymap' which was written by Eric Sunshine
//	<sunshine@sunshineco.com> and is copyright (C)1999,2000 by Eric
//	Sunshine.  The license and copyright notice for dumpkeymap are
//	reproduced above.
//
//	dumpkeymap is a diagnostic utility which interprets and extracts all
//	information contained in an Apple/NeXT .keymapping file and displays it
//	in a human-readable format.  Of greater importance, dumpkeymap contains
//	extensive documentation regarding the exact layout and structure of a
//	keymapping resource.  This is the only known documentation regarding
//	the layout of keymapping resources.
//
//	This file (NeXTKeymap.m) makes no attempt to describe the layout of a
//	keymapping resource nor is the actual extraction method explained.  For
//	an exhaustively detailed explanation of each of these topics, refer
//	instead to the original dumpkeymap package.
//
//	The original dumpkeymap package can be found in the directory
//	CS/libs/cssys/next/support/dumpkeymap.  The very latest version of
//	dumpkeymap can also be requested from Eric Sunshine by sending an
//	appropriate mail message to sunshine@sunshineco.com.  Finally,
//	dumpkeymap is incorporated into the XFree86 (www.xfree86.org)
//	distribution as of release 4.0.2 for the Darwin and MacOS/X ports.
//
//	A fully-detailed description of the internal layout of a keymapping
//	resource can be viewed online as part of the dumpkeymap documentation
//	at http://www.xfree86.org/current/dumpkeymap.1.html.
//
//-----------------------------------------------------------------------------
#include "NeXTKeymap.h"
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <drivers/event_status_driver.h>

typedef unsigned char byte;
typedef unsigned short word;
typedef unsigned int natural;
typedef unsigned long dword;
typedef dword number;

#define INVALID_VALUE ((word)~0)

#define ASCII_SET 0x00
#define BIND_FUNCTION 0xfe
#define BIND_SPECIAL 0xff

//-----------------------------------------------------------------------------
// Data Stream Object
//	Can be configured to treat embedded "numbers" as being composed of
//	either 1, 2, or 4 bytes, apiece.
//-----------------------------------------------------------------------------
typedef struct _DataStream
{
  byte const* data;
  byte const* data_end;
  natural number_size; // Size in bytes of a "number" in the stream.
} DataStream;

static DataStream* new_data_stream(byte const* data, int size)
{
  DataStream* s = (DataStream*)malloc(sizeof(DataStream));
  s->data = data;
  s->data_end = data + size;
  s->number_size = 1; // Default to byte-sized numbers.
  return s;
}

static void destroy_data_stream(DataStream* s)
{
  free(s);
}

#if defined(END_OF_STREAM_USED)
static int end_of_stream(DataStream* s)
{
  return (s->data >= s->data_end);
}
#endif

static void expect_nbytes(DataStream* s, int nbytes)
{
  if (s->data + nbytes > s->data_end)
  {
    fputs("Insufficient data in keymapping data stream.\n", stderr);
    exit(-1);
  }
}

static byte get_byte(DataStream* s)
{
  expect_nbytes(s, 1);
  return *s->data++;
}

static word get_word(DataStream* s)
{
  word hi, lo;
  expect_nbytes(s, 2);
  hi = *s->data++;
  lo = *s->data++;
  return ((hi << 8) | lo);
}

static dword get_dword(DataStream* s)
{
  dword b1, b2, b3, b4;
  expect_nbytes(s, 4);
  b4 = *s->data++;
  b3 = *s->data++;
  b2 = *s->data++;
  b1 = *s->data++;
  return ((b4 << 24) | (b3 << 16) | (b2 << 8) | b1);
}

static number get_number(DataStream* s)
{
  switch (s->number_size)
  {
    case 4:  return get_dword(s);
    case 2:  return get_word(s);
    default: return get_byte(s);
  }
}


//-----------------------------------------------------------------------------
// Compute number of bits set in value.
//-----------------------------------------------------------------------------
static number bits_set(number mask)
{
  number n = 0;
  for (; mask != 0; mask >>= 1)
    if ((mask & 0x01) != 0)
      n++;
  return n;
}


//-----------------------------------------------------------------------------
// Skip over the Modifier records.
//-----------------------------------------------------------------------------
static void skip_modifiers(DataStream* s)
{
  number nmod = get_number(s); // Modifier record count
  while (nmod-- > 0)
  {
    number nscan;
    get_number(s); // Modifier description code.
    nscan = get_number(s);
    while (nscan-- > 0)
      get_number(s); // One scan code mapped to this modifier.
  }
}


//-----------------------------------------------------------------------------
// Fill in the Binding record with the content of the first Character record
// for this scan code.  This is the Character record for the unmodified
// keystroke (i.e. no shift, control, alternate, etc.).  We are only
// interested in characters which are in the Ascii, Symbol, or Function
// character set.  Ignore the so-called Special character set since this is
// only a pseudo-designation which does not actually represent a real character
// set.  Skip over all remaining bindings for this scan code following the
// first.
//-----------------------------------------------------------------------------
static void unparse_char_codes(
  DataStream* s, number ncodes, NeXTKeymapBinding* binding)
{
  if (ncodes != 0)
  {
    number i;
    for (i = 0; i < ncodes; i++)
    {
      number const char_set = get_number(s);
      number const code = get_number(s);
      if (i == 0 && char_set != BIND_SPECIAL)
      {
	binding->code = code;
	binding->character_set = char_set;
      }
    }
  }
}


//-----------------------------------------------------------------------------
// Unparse the number-size flag.
//-----------------------------------------------------------------------------
static void unparse_numeric_size(DataStream* s)
{
  word const numbers_are_shorts = get_word(s);
  s->number_size = numbers_are_shorts ? 2 : 1;
}


//=============================================================================
// NeXTKeymap Implementation
//=============================================================================
@implementation NeXTKeymap

//-----------------------------------------------------------------------------
// allocateBindingRecords:
//-----------------------------------------------------------------------------
- (void)allocateBindingRecords:(number)n
{
  number const nbytes = n * sizeof(bindings[0]);
  bindings = (NeXTKeymapBinding*)malloc(nbytes);
  memset(bindings, INVALID_VALUE, nbytes);
  nbindings = n;
}


//-----------------------------------------------------------------------------
// Unparse a list of scan code bindings.
//-----------------------------------------------------------------------------
- (void)unparseCharacters:(DataStream*)s
{
  number const NOT_BOUND = 0xff;
  number const nscans = get_number(s);
  number scan;
  [self allocateBindingRecords:nscans];
  for (scan = 0; scan < nscans; scan++)
  {
    number const mask = get_number(s);
    if (mask != NOT_BOUND)
    {
      number const bits = bits_set(mask);
      number const codes = 1 << bits;
      unparse_char_codes(s, codes, bindings + scan);
    }
  }
}


//-----------------------------------------------------------------------------
// unparseKeymapData:
//-----------------------------------------------------------------------------
- (void)unparseKeymapData:(DataStream*)s
{
  unparse_numeric_size(s);
  skip_modifiers(s);
  [self unparseCharacters:s];
}


//-----------------------------------------------------------------------------
// unparseActiveKeymap
//-----------------------------------------------------------------------------
- (BOOL)unparseActiveKeymap
{
  BOOL rc = NO;
  NXEventHandle const h = NXOpenEventStatus();
  if (h == 0)
    fputs("Unable to open event status driver.\n", stderr);
  else
  {
    NXKeyMapping km;
    km.size = NXKeyMappingLength(h);
    if (km.size <= 0)
      fprintf(stderr, "Bad key mapping length (%d).\n", km.size);
    else
    {
      km.mapping = (char*)malloc(km.size);
      if (NXGetKeyMapping(h, &km) == 0)
	fputs("Unable to get current key mapping.\n", stderr);
      else
      {
	DataStream* stream = new_data_stream((byte const*)km.mapping, km.size);
	[self unparseKeymapData:stream];
	destroy_data_stream(stream);
	rc = YES;
      }
      free(km.mapping);
    }
    NXCloseEventStatus(h);
  }
  return rc;
}


//-----------------------------------------------------------------------------
// isScanCodeBound:
//-----------------------------------------------------------------------------
- (BOOL)isScanCodeBound:(word)code
{
  NeXTKeymapBinding const* const b = [self bindingForScanCode:code];
  return (b != 0 &&
    b->code != INVALID_VALUE && b->character_set != INVALID_VALUE);
}


//-----------------------------------------------------------------------------
// bindingForScanCode:
//-----------------------------------------------------------------------------
- (NeXTKeymapBinding const*)bindingForScanCode:(word)code
{
  return (code < nbindings ? bindings + code : 0);
}


//-----------------------------------------------------------------------------
// init
//-----------------------------------------------------------------------------
- (id)init
{
  [super init];
  bindings = 0;
  nbindings = 0;
  if (![self unparseActiveKeymap])
  {
    fputs("Unable to parse active NeXT keymapping.\n", stderr);
    exit(1);
  }
  return self;
}


//-----------------------------------------------------------------------------
// free
//-----------------------------------------------------------------------------
- (id)free
{
  if (bindings != 0)
    free(bindings);
  return [super free];
}

@end
