/* MP3.m - this file is part of Cynthiune
 *
 * Copyright (C) 2002, 2003, 2004 Wolfgang Sourdeau
 *
 * portions are Copyright (C) 2000-2003 Robert Leslie
 *                        (C) 2001-2002 Bertrand Petit
 *
 * Author: Wolfgang Sourdeau <wolfgang@contre.com>
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#import <Foundation/Foundation.h>

#import <sys/types.h>
#import <sys/stat.h>
#import <string.h>
#import <mad.h>
#import <id3tag.h>

#import <CynthiuneBundle.h>
#import <Format.h>

#import "xing.h"
#import "MP3.h"

#define LOCALIZED(X) _b ([MP3 class], X)

/* header testing */
static inline int
seekOffset (FILE *_f)
{
  char testchar;
  int counter;

  counter = -1;
  testchar = 0;

  while (!testchar)
    {
      fread (&testchar, 1, 1, _f);
      counter++;
    }
  fseek (_f, counter, SEEK_SET);

  return counter;
}

static inline BOOL
testRiffHeader (char *buffer, FILE *_f, int offset)
{
  char tag;
  BOOL result;

  if (strncmp (buffer, "WAVE", 4) == 0)
    {
      fseek (_f, 20 + offset, SEEK_SET);
      fread (&tag, 1, 1, _f);
      result = (tag == 80 || tag == 85);
    }
  else
    result = NO;

  return result;
}

static inline BOOL
testMP3Header (const char *buffer)
{
  u_int16_t *header;

  header = (u_int16_t *) buffer;

  return ((NSHostByteOrder() == NS_LittleEndian)
          ? ((*header & 0xfeff) == 0xfaff
             || (*header & 0xfeff) == 0xfcff
             || (*header & 0xf8ff) == 0xf0ff
             || (*header & 0xf8ff) == 0xe0ff)
          : ((*header & 0xfffe) == 0xfffa
             || (*header & 0xfffe) == 0xfffc
             || (*header & 0xfff8) == 0xfff0
             || (*header & 0xfff8) == 0xffe0));
}

/* data decoding */
static inline int
calcInputRemain (MadStream *stream, unsigned char *iBuffer)
{
  int inputRemain;

  if (stream->next_frame)
    {
      inputRemain = stream->bufend - stream->next_frame;
      memcpy (iBuffer, stream->next_frame, inputRemain);
    }
  else
    inputRemain = 0;

  return inputRemain;
}

static signed long
audioLinearDither (MadFixed sample, audioDither *dither)
{
  unsigned int scalebits;
  MadFixed output, mask, random;

  /* noise shape */
  sample += dither->error[0] - dither->error[1] + dither->error[2];

  dither->error[2] = dither->error[1];
  dither->error[1] = dither->error[0] / 2;

  /* bias */
  output = sample + (1L << (MAD_F_FRACBITS - 16));

  scalebits = MAD_F_FRACBITS - 15;
  mask = (1L << scalebits) - 1;

  /* dither */
  random = (dither->random * 0x0019660dL + 0x3c6ef35fL) & 0xffffffffL;;
  output += (random & mask) - (dither->random & mask);
  dither->random = random;

  /* clip */
  if (output > (MAD_F_ONE - 1))
    {
      output = MAD_F_ONE - 1;

      if (sample > MAD_F_ONE - 1)
        sample = MAD_F_ONE - 1;
    }
  else if (output < -MAD_F_ONE)
    {
      output = -MAD_F_ONE;

      if (sample < -MAD_F_ONE)
        sample = -MAD_F_ONE;
    }

  /* quantize */
  output &= ~mask;

  /* error feedback */
  dither->error[0] = sample - output;

  /* scale */
  return output >> scalebits;
}

static inline void
fillPCMBuffer (CMp3 *self, unsigned char *buffer, int start, int limit)
{
  int i;
  unsigned char *oBPtr;
  register signed int sample;

  oBPtr = buffer;

  i = start;
  while (i < limit)
    {
      sample = audioLinearDither (self->synth.pcm.samples[0][i],
                                  &(self->leftDither));
      *oBPtr++ = sample >> 0;
      *oBPtr++ = sample >> 8;
      if (self->channels == 2)
        {
          sample = audioLinearDither (self->synth.pcm.samples[1][i],
                                      &(self->rightDither));
          *oBPtr++ = sample >> 0;
          *oBPtr++ = sample >> 8;
        }
      i++;
    }
}

static int
translateBufferToPCM (CMp3 *self, unsigned char *buffer, int bufferSize)
{
  int start, limit, mult, delta;

  mult = 2 * self->channels;
  if (self->oRemain)
    {
      start = self->synth.pcm.length - self->oRemain;
      limit = self->synth.pcm.length;
    }
  else
    {
      start = 0;
      limit = (bufferSize / mult);
      if (self->synth.pcm.length < limit)
        limit = self->synth.pcm.length;
    }

  delta = (limit - start) * mult - bufferSize;
  if (delta > 0)
    limit -= delta / mult;

  fillPCMBuffer (self, buffer, start, limit);

  self->oRemain = self->synth.pcm.length - limit;

  return ((limit - start) * mult);
}

static inline InputBufferStatus
decodeInputBuffer (CMp3 *self, int iRBytes)
{
  InputBufferStatus bufferStatus;

  mad_stream_buffer (&(self->stream), self->iBuffer, iRBytes + self->iRemain);
  if (mad_frame_decode (&(self->frame), &(self->stream)))
    {
      if (MAD_RECOVERABLE (self->stream.error)
          || self->stream.error == MAD_ERROR_BUFLEN)
        bufferStatus = BufferHasRecoverableError;
      else
        {
          NSLog (@"%s: unrecoverable frame level error (%s)",
                 __FILE__, mad_stream_errorstr (&(self->stream)));
          bufferStatus = BufferHasUnrecoverableError;
        }
    }
  else
    bufferStatus = BufferHasNoError;

  return bufferStatus;
}

@implementation MP3 : NSObject

+ (NSArray *) bundleClasses
{
  return [NSArray arrayWithObject: [self class]];
}

- (id) init
{
  if ((self = [super init]))
    {
      mf = NULL;
      id3File = NULL;
      id3Tag = NULL;

      channels = 0;
      rate = 0;
      duration = 0;
      size = 0;
      iRemain = 0;
      oRemain = 0;

      opened = NO;
    }

  return self;
}

- (void) _readStreamMetaData
{
  int iRBytes;
  InputBufferStatus bufferStatus;
  BOOL done;

  mad_stream_init (&stream);
  mad_frame_init (&frame);
  mad_synth_init (&synth);
  done = NO;

  while (!done)
    {
      stream.error = 0;
      iRemain = calcInputRemain (&stream, iBuffer);
      iRBytes = fread (iBuffer + iRemain, sizeof (char),
                       IBUFFER_SIZE - iRemain, mf);

      if (iRBytes > 0 && !feof (mf))
        {
          bufferStatus = decodeInputBuffer ((CMp3 *) self, iRBytes);
          if (bufferStatus == BufferHasNoError)
            {
              channels = MAD_NCHANNELS(&frame.header);
              rate = frame.header.samplerate;
              if (xing_parse (&xing, stream.anc_ptr, stream.anc_bitlen) == 0)
                {
                  duration = (int) (xing.bytes * 8
                                    / (frame.header.bitrate * channels));
                }
              else
                {
                  mad_timer_multiply (&frame.header.duration,
                                      size / (stream.next_frame
                                              - stream.this_frame));
                  duration = mad_timer_count (frame.header.duration,
                                              MAD_UNITS_SECONDS);
                }
              done = YES;
            }
        }
      else
        done = YES;
    }

  mad_stream_finish (&stream);
  mad_frame_finish (&frame);
  mad_synth_finish (&synth);
  memset (iBuffer, 0, IBUFFER_SIZE);
}

- (void) _streamInit
{
  [self _readStreamMetaData];

  xing_init (&xing);
  mad_stream_init (&stream);
  mad_frame_init (&frame);
  mad_synth_init (&synth);
}

- (void) _id3Init: (NSString *) fileName
{
  id3File = id3_file_open ([fileName cString], ID3_FILE_MODE_READONLY);
  if (id3File)
    id3Tag = id3_file_tag (id3File);
}

- (void) _readSize: (const char *) filename;
{
  struct stat stat_val;

  stat (filename, &stat_val);
  size = stat_val.st_size;
}

- (BOOL) streamOpen: (NSString *) fileName
{
  const char *filename;

  mf = NULL;
  if (!opened)
    {
      filename = [fileName cString];
      mf = fopen (filename, "r");
      [self _readSize: filename];

      if (mf)
        {
          [self _streamInit];
          [self _id3Init: fileName];
          opened = YES;
        }
      else
        NSLog (@"%s: no handle...", __FILE__);
    }
  else
    NSLog (@"%s: attempted to open opened stream", __FILE__);

  return (mf != NULL);
}

+ (BOOL) canTestFileHeaders
{
  return YES;
}

+ (BOOL) streamTestOpen: (NSString *) fileName
{
  FILE *_f;
  char buffer[4];
  BOOL result;
  int offset;

  _f = fopen ([fileName cString], "r");

  if (_f)
    {
      offset = seekOffset (_f);

      fread (buffer, 1, 4, _f);
      if (!strncmp (buffer, "RIFF", 4))
        {
          fseek (_f, 8 + offset, SEEK_SET);
          fread (buffer, 1, 4, _f);
          result = testRiffHeader (buffer, _f, offset);
        }
      else
        result = (testMP3Header (buffer)
                  || !strncmp (buffer, "ID3", 3));

      fclose (_f);
    }
  else
    result = NO;

  return result;
}

// - (NSString *) errorText
// {
//   return @"[error unimplemented]";
// }

/* FIXME: we should put some error handling here... */
- (int) readNextChunk: (unsigned char *) buffer
	     withSize: (unsigned int) bufferSize
{
  int iRBytes, decodedBytes;
  InputBufferStatus bufferStatus;
  BOOL done;

  if (oRemain)
    decodedBytes = translateBufferToPCM ((CMp3 *) self, buffer, bufferSize);
  else
    {
      done = NO;
      decodedBytes = 0;

      while (!done)
        {
          stream.error = 0;
          iRemain = calcInputRemain (&stream, iBuffer);
          iRBytes = fread (iBuffer + iRemain, sizeof (char),
                           IBUFFER_SIZE - iRemain, mf);

          if (iRBytes > 0 && !feof (mf))
            {
              bufferStatus = decodeInputBuffer ((CMp3 *) self, iRBytes);
              if (bufferStatus == BufferHasNoError)
                {
                  mad_synth_frame (&(self->synth), &(self->frame));
                  decodedBytes = translateBufferToPCM ((CMp3 *) self,
                                                       buffer, bufferSize);
                  done = YES;
                }
              else if (bufferStatus == BufferHasUnrecoverableError)
                done = YES;
            }
          else
            done = YES;
        }
    }

  return decodedBytes;
}

- (unsigned int) readChannels
{
  return channels;
}

- (unsigned long) readRate
{
  return rate;
}

- (unsigned int) readDuration
{
  return duration;
}

- (NSString *) readComment: (char *) commentTag
{
  NSString *comment;
  Id3Frame *id3Frame;
  Id3Field *field;
  const Id3UCS4 *string;
  char *UTF8String;

  comment = @"";
  id3Frame = id3_tag_findframe (id3Tag, commentTag, 0);
  if (id3Frame)
    {
      field = id3_frame_field (id3Frame, 1);
      if (field
          && id3_field_type (field) == ID3_FIELD_TYPE_STRINGLIST)
        {
          string = id3_field_getstrings (field, 0);
          if (string)
            {
              if (commentTag == ID3_FRAME_GENRE)
                string = id3_genre_name (string);
              UTF8String = id3_ucs4_utf8duplicate (string);
              comment = [NSString stringWithUTF8String: UTF8String];
              free (UTF8String);
            }
        }
    }

  return comment;
}

- (NSString *) readTitle
{
  return [self readComment: ID3_FRAME_TITLE];
}

- (NSString *) readGenre
{
  return [self readComment: ID3_FRAME_GENRE];
}

- (NSString *) readArtist
{
  return [self readComment: ID3_FRAME_ARTIST];
}

- (NSString *) readAlbum
{
  return [self readComment: ID3_FRAME_ALBUM];
}

- (NSString *) readTrackNumber
{
  return [self readComment: ID3_FRAME_TRACK];
}

- (void) streamClose
{
  if (opened)
    {
      mad_synth_finish (&synth);
      mad_frame_finish (&frame);
      mad_stream_finish (&stream);

      if (id3Tag)
        {
          id3_tag_delete (id3Tag);
          id3Tag = NULL;
        }

      if (id3File)
        {
          id3_file_close (id3File);
          id3File = NULL;
        }

      if (mf)
        {
          fclose (mf);
          mf = NULL;
        }

      opened = NO;
    }
  else
    NSLog (@"Attempted to close closed stream");
}

// Player Protocol
+ (NSArray *) acceptedFileExtensions
{
  return [NSArray arrayWithObjects: @"mp2", @"mp3", @"mpa",
                  @"mpga", @"mpega", nil];
}

/* FIXME: this might not be true */
- (BOOL) isSeekable
{
  return YES;
}

- (void) seek: (unsigned int) aPos
{
  unsigned long filePos;

  if (size)
    {
      filePos = (size * aPos) / duration;
      fseek (mf, filePos, SEEK_SET);
    }
  else
    NSLog (@"size not computed?");
}

@end
