/* Playlist.h - this file is part of Cynthiune
 *
 * Copyright (C) 2003, 2004  Wolfgang Sourdeau
 *
 * Author: Wolfgang Sourdeau <Wolfgang@Contre.COM>
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#import <Foundation/Foundation.h>

#import <time.h>

#import "Playlist.h"
#import "PlaylistFile.h"
#import "Song.h"

#define DefaultPlaylistName @"Playlist"

NSString *PlaylistChangedNotification = @"PlaylistChangedNotification";

static NSNotificationCenter *nc = nil;

@implementation Playlist : NSObject

+ (void) initialize
{
  nc = [NSNotificationCenter defaultCenter];
}

- (id) init
{
  time_t now;
  time (&now);
  srand (now);

  if ((self = [super init]))
    {
      shuffle = NO;
      delegate = nil;
      currentSongNumber = -1;
      _regularList = [NSMutableArray new];
      _shuffleList = nil;
    }

  return self;
}

- (void) setDelegate: (id) anObject
{
  if (delegate)
    [nc removeObserver: delegate name: nil object: self];

  delegate = anObject;

  if ([delegate respondsToSelector: @selector (playlistChanged:)])
    [nc addObserver: delegate
	selector: @selector (playlistChanged:)
	name: PlaylistChangedNotification
	object: self];
}

- (id) delegate
{
  return delegate;
}

- (int) _currentToReal
{
  Song *song;
  int songNbr;

  if (shuffle && currentSongNumber > -1)
    {
      song = [_regularList objectAtIndex: currentSongNumber];
      songNbr = [_shuffleList indexOfObject: song];
    }
  else
    songNbr = currentSongNumber;

  return songNbr;
}

- (void) _realToCurrent: (int) songNbr;
{
  Song *song;

  if (shuffle && songNbr > -1)
    {
      song = [_shuffleList objectAtIndex: songNbr];
      currentSongNumber = [_regularList indexOfObject: song];
    }
  else
    currentSongNumber = songNbr;
}

- (void) _appendShuffled: (Song *) aSong
{
  unsigned int randPos;

  randPos = ((float) ([_shuffleList count] * rand()))
             / (RAND_MAX) + .5;
  [_shuffleList insertObject: aSong atIndex: randPos];
}

- (void) reShuffle
{
  unsigned int count, max, shuffleNumber;

  max = [_regularList count];
  if (_shuffleList)
    [_shuffleList release];
  _shuffleList = [NSMutableArray arrayWithCapacity: max];
  [_shuffleList retain];

  for (count = 0; count < max; count++)
    [self _appendShuffled: [_regularList objectAtIndex: count]];

  if (currentSongNumber > -1)
    {
      shuffleNumber = [self _currentToReal];
      for (count = 0; count < shuffleNumber; count++)
        {
          [_shuffleList addObject: [_shuffleList objectAtIndex: 0]];
          [_shuffleList removeObjectAtIndex: 0];
        }
    }
}

- (BOOL) shuffle
{
  return shuffle;
}

- (void) setShuffle: (BOOL) aBool
{
  shuffle = aBool;

  if (aBool)
    [self reShuffle];
  else
    {
      if (_shuffleList)
        {
          [_shuffleList release];
          _shuffleList = nil;
        }
    }
}

- (BOOL) repeat
{
  return repeat;
}

- (void) setRepeat: (BOOL) aBool
{
  repeat = aBool;
}

- (NSArray *) getSongs
{
  return _regularList;
}

- (NSArray *) getSongsAsFilenames
{
  NSString *filename;
  NSMutableArray *songList;
  unsigned int count, max;

  max = [_regularList count];
  songList = [NSMutableArray arrayWithCapacity: max];

  for (count = 0; count < max; count++)
    {
      filename = [[_regularList objectAtIndex: count] filename];
      [songList addObject: filename];
    }

  return songList;
}

- (void) appendSong: (Song *) newSong
{
  [_regularList addObject: newSong];
  if (shuffle)
    [self _appendShuffled: newSong];
}

- (void) insertSong: (Song *) newSong atIndex: (unsigned int) aPos
{
  [_regularList insertObject: newSong atIndex: aPos];
  if (aPos <= currentSongNumber)
    currentSongNumber++;
  if (shuffle)
    [self _appendShuffled : newSong];
}

- (void) firstSong
{
  if ([_regularList count])
    [self _realToCurrent: 0];
  else
    currentSongNumber = -1;
}

- (void) lastSong
{
  unsigned int count;

  count = [_regularList count];

  if (count > 0)
    [self _realToCurrent: count - 1];
  else
    currentSongNumber = -1;
}

- (void) prevSong
{
  unsigned int count, songNbr;

  if (currentSongNumber > -1)
    {
      count = [_regularList count];

      if (count > 0)
        {
          songNbr = [self _currentToReal];

          if (songNbr > 0)
            songNbr--;
          else if (repeat)
            songNbr = count - 1;

          [self _realToCurrent: songNbr];
        }
    }
}

- (void) nextSong
{
  int songNbr;
  unsigned int count;

  if (currentSongNumber > -1)
    {
      count = [_regularList count];

      if (count > 0)
        {
          songNbr = [self _currentToReal];

          if (songNbr < (count - 1))
            songNbr++;
          else if (repeat)
            songNbr = 0;

          [self _realToCurrent: songNbr];
        }
    }
  else if ([_regularList count])
    [self _realToCurrent: 0];
}

- (void) removeArrayOfSongs: (NSArray *) anArray
{
  unsigned int count, max;

  max = [anArray count];

  for (count = 0; count < max; count++)
    if ([_regularList indexOfObject: [anArray objectAtIndex: count]]
        < currentSongNumber)
      currentSongNumber--;

  [anArray makeObjectsPerformSelector: @selector (retain)];
  [_regularList removeObjectsInArray: anArray];
  if (shuffle)
    [_shuffleList removeObjectsInArray: anArray];
  [anArray makeObjectsPerformSelector: @selector (release)];

  if (currentSongNumber >= (int) [_regularList count])
    currentSongNumber = (unsigned int) [_regularList count] - 1;

  [nc postNotificationName: PlaylistChangedNotification object: self];
}

- (void) removeAll
{
  if ([_regularList count])
    {
      [_regularList removeAllObjects];
      if (shuffle)
        [_shuffleList removeAllObjects];
      currentSongNumber = -1;
      [nc postNotificationName: PlaylistChangedNotification object: self];
    }
}

- (void) removeBadEntries
{
  NSMutableArray *badEntries;
  Song *currentSong;
  unsigned int count, max;

  badEntries = [NSMutableArray array];
  max = [_regularList count];
  for (count = 0; count < max; count++)
    {
      currentSong = [_regularList objectAtIndex: count];
      if ([currentSong status] != SongOK)
        [badEntries addObject: currentSong];
    }

  [self removeArrayOfSongs: badEntries];
}

- (unsigned int) moveSongsWithRows: (NSArray *) aRowsList
                                to: (unsigned int) newPos
{
  unsigned int count, max, firstRow, nbr;
  NSMutableArray *songs;
  Song *currentSong;

  currentSong = [self songAtIndex: currentSongNumber];

  max = [aRowsList count];
  songs = [NSMutableArray arrayWithCapacity: max];

  firstRow = newPos;
  for (count = 0; count < max; count++)
    {
      nbr = [[aRowsList objectAtIndex: count] intValue];
      [songs addObject: [_regularList objectAtIndex: nbr]];

      if (nbr < firstRow)
        firstRow--;
    }

  [_regularList removeObjectsInArray: songs];
  for (count = 0; count < max; count++)
    [_regularList insertObject: [songs objectAtIndex: count]
                  atIndex: firstRow + count];

  currentSongNumber = [self songIndex: currentSong];

  return firstRow;
}

- (void) selectSongNumber: (int) anInt
{
  currentSongNumber = anInt;
}

- (int) currentSongNumber
{
  return currentSongNumber;
}

- (int) songIndex: (Song *) aSong
{
  return (([_regularList containsObject: aSong])
          ? [_regularList indexOfObject: aSong]
          : -1);
}

- (Song *) songAtIndex: (int) anIndex
{
  return ((anIndex > -1)
          ? [_regularList objectAtIndex: anIndex]
          : nil);
}

- (Song *) songAtNSIndex: (NSNumber *) anIndex;
{
  return ((anIndex)
          ? [self songAtIndex: [anIndex intValue]]
          : nil);
}

- (unsigned int) count
{
  return [_regularList count];
}

- (void) sortByPlaylistRepresentation: (BOOL) descending
{
  Song *currentSong;
  SEL compareMethod;

  currentSong = [self songAtIndex: currentSongNumber];

  compareMethod = ((descending)
                   ? @selector (reverseCompareByPlaylistRepresentation:)
                   : @selector (compareByPlaylistRepresentation:));
  [_regularList sortUsingSelector: compareMethod];

  currentSongNumber = [self songIndex: currentSong];
}

- (void) sortByDuration: (BOOL) descending
{
  Song *currentSong;
  SEL compareMethod;

  currentSong = [self songAtIndex: currentSongNumber];
  compareMethod = ((descending)
                   ? @selector (reverseCompareByDuration:)
                   : @selector (compareByDuration:));
  [_regularList sortUsingSelector: compareMethod];

  currentSongNumber = [self songIndex: currentSong];
}

- (void) saveToDefaults
{
  NSArray *dirs;
  NSString *libraryCynthiune, *playlist;
  NSFileManager *manager;

  manager = [NSFileManager defaultManager];
  dirs = NSSearchPathForDirectoriesInDomains (NSLibraryDirectory,
                                              NSUserDomainMask, YES);
  libraryCynthiune = [[dirs objectAtIndex: 0]
                       stringByAppendingPathComponent: @"Cynthiune"];
  if (![manager fileExistsAtPath: libraryCynthiune])
    [manager createDirectoryAtPath: libraryCynthiune attributes: nil];

  playlist =
    [libraryCynthiune stringByAppendingPathComponent: DefaultPlaylistName];
  [NSKeyedArchiver archiveRootObject: _regularList toFile: playlist];
}

- (void) _readFromAndMigrateDefaults: (NSArray *) songList
{
  int count, max;
  NSString *songName;
  Song *aSong;

  max = [songList count];
  for (count = 0; count < max; count++)
    {
      songName = [songList objectAtIndex: count];
      aSong = [Song songWithFilename: songName];
      [self appendSong: aSong];
    }

  [self saveToDefaults];
  [[NSUserDefaults standardUserDefaults]
    removeObjectForKey: @"DefaultPlaylist"];
}

- (void) loadFromDefaults
{
  NSArray *songList, *dirs, *archivedList;
  NSString *playlist;

  [self removeAll];
  songList = [[NSUserDefaults standardUserDefaults]
               stringArrayForKey: @"DefaultPlaylist"];

  if (songList)
    [self _readFromAndMigrateDefaults: songList];
  else
    {
      dirs = NSSearchPathForDirectoriesInDomains (NSLibraryDirectory,
                                                  NSUserDomainMask, YES);
      playlist = [[[dirs objectAtIndex: 0]
                    stringByAppendingPathComponent: @"Cynthiune"]
                   stringByAppendingPathComponent: DefaultPlaylistName];
      archivedList = [NSKeyedUnarchiver unarchiveObjectWithFile: playlist];
      if (archivedList)
        [_regularList setArray: archivedList];
    }
}

- (void) loadFromFile: (NSString *) aFilename
{
  PlaylistFile *playlistFile;

  playlistFile = [PlaylistFile new];
  [playlistFile setFilename: aFilename];
  [playlistFile setPlaylist: self];
  [playlistFile load];
  [playlistFile release];
}

- (void) saveToFile: (NSString *) aFilename
{
  PlaylistFile *playlistFile;

  playlistFile = [PlaylistFile new];
  [playlistFile setFilename: aFilename];
  [playlistFile setPlaylist: self];
  [playlistFile save];
  [playlistFile release];
}

- (void) dealloc
{
  [_regularList release];
  if (_shuffleList)
    [_shuffleList release];
  [super dealloc];
}

@end
