/*
 * Copyright (C) 2000-2001 Chris Ross and Evan Webb
 * Copyright (C) 1999-2000 Chris Ross
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies of the Software, its documentation and marketing & publicity
 * materials, and acknowledgment shall be given in the documentation, materials
 * and software packages that this Software was used.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include "../config.h"
#include "ferite.h"

/*!
 *  \fn FeriteEnvironment *ferite_init( int argc, char *argv[] )
 *  \brief First call to the ferite engine to initialise it
 *  \param argc Number of arguments
 *  \param argc Standard array of pointers
 */
FeriteEnvironment *ferite_init( int argc, char *argv[] )
{
   int i = 0;

   FE_ENTER_FUNCTION;

   if( genv == NULL )
   {

      __ferite_memory_init = __ferite_jedi_memory_init;
      __ferite_memory_deinit = __ferite_jedi_memory_deinit;
      __ferite_malloc = __ferite_jedi_malloc;
      __ferite_calloc = __ferite_jedi_calloc;
      __ferite_realloc = __ferite_jedi_realloc;
      __ferite_free = __ferite_jedi_free;

      if( argv != NULL )
      {
		 for( i = 0; i < argc; i++ )
		 {
			if( strcmp( argv[i], "--fe-use-classic" ) == 0 )
			{
			   __ferite_memory_init = __ferite_classic_memory_init;
			   __ferite_memory_deinit = __ferite_classic_memory_deinit;
			   __ferite_malloc = __ferite_classic_malloc;
			   __ferite_calloc = __ferite_classic_calloc;
			   __ferite_realloc = __ferite_classic_realloc;
			   __ferite_free = __ferite_classic_free;
			}
            if( strcmp( argv[i], "--fe-debug" ) == 0 )
			  show_debug = 1;
            if( strcmp( argv[i], "--fe-show-mem-use" ) == 0 )
			  hide_mem_use = 0;
            if( strcmp( argv[i], "--fe-module-path" ) == 0 )
			  printf( "module path: \"%s\"\n", MODULE_DIR );
		 }
      }

      __ferite_memory_init();

#ifdef DEBUG
	  ferite_warning( NULL, "Debug Build.\n" );
#endif
	  
      if( !hide_mem_use )
      {
#ifdef FERITE_MEM_DEBUG
		 ferite_warning( NULL, "Memory Debugging Enabled\n" );
#endif
      }
      genv = fmalloc( sizeof( FeriteEnvironment ) );   /* create the enviroment */

      /* new gc, yesh :) */
      genv->gc = NULL;
      __ferite_init_gc();

      genv->total_ops_run = 0;
      __ferite_init_module_list();
      __ferite_init_regex();

      ARGV = __ferite_create_uarray_variable( "argv", argc );
      if( argv != NULL )
      {
		 for( i = 0; i < argc; i++ )
		 {
			__ferite_uarray_add( NULL, VAUA(ARGV), fe_new_str(argv[i], argv[i]), NULL, -1 );
		 }
      }
   }
   else
     ferite_warning( NULL, "Ferite is already initialised. No need to call ferite_init() more than once" );

   FE_LEAVE_FUNCTION( genv );
}

/*!
 *  \fn int ferite_deinit()
 *  \brief Shuts the ferite engine down, clears the garbage collector and frees up unused memory
 */
int ferite_deinit()
{
   FE_ENTER_FUNCTION;
   if( genv != NULL )
   {
      ferite_delete_search_paths();
      __ferite_variable_destroy( NULL, ARGV );
    /*  __ferite_delete_scripts( genv->scripts );*/
      __ferite_deinit_gc(); /* delete all objects within the system before we delete the classes
							 * we need not worry about the env->objects as they aren't done from
							 * classes. */
      __ferite_deinit_module_list();
      ffree( genv );
      __ferite_memory_deinit();
   }
   FE_LEAVE_FUNCTION( 1 );
}

/*!
 *  \fn int ferite_get_parameters( FeriteVariable **list, int num_args, ... )
 *  \brief Get the ferite variables from a list and put them into pointers to real c variables
 *  \param list The list of variables passed to the function calling this method
 *  \param num_args The number of arguements you wish to extract from the list
 *  \param ... Pointers to the real c variables
 *
 * This method is very useful for transaltion from the internal variable format to a standard c variable.
 *
 * \warning Numbers can be eighter double or long so these need to be handled carefully
 */
int ferite_get_parameters( FeriteVariable **list, int num_args, ... )
{
   va_list  ap;
   void    *vptr;
   int      current_arg = 0;

   FE_ENTER_FUNCTION;

   if( num_args < 1 )
   {
	  FE_LEAVE_FUNCTION(1);
   }
   if( list != NULL )
   {
      va_start( ap, num_args );
      while( current_arg < num_args && list[current_arg] != NULL )
      {
		 if( list[current_arg] != NULL )
		 {
			switch( list[current_arg]->type )
			{
			 case F_VAR_LONG:
			   vptr = va_arg( ap, int * );
			   *((double *)vptr) = (double)(list[current_arg]->data.lval);
			   break;
			 case F_VAR_DOUBLE:
			   vptr = va_arg( ap, double * );
			   *((double *)vptr) = list[current_arg]->data.dval;
			   break;
			 case F_VAR_STR:
			   vptr = va_arg( ap, char * );
			   if( vptr == NULL )
				 break;
			   strcpy( (char *)vptr, list[current_arg]->data.sval );
			   break;
			 case F_VAR_OBJ:
			   vptr = va_arg( ap, void * );
			   *((void **)vptr) = list[current_arg]->data.pval;
			   break;
			 case F_VAR_UARRAY:
			   vptr = va_arg(ap, void ** );
			   *((void **)vptr) = list[current_arg]->data.pval;
			   break;
			 case F_VAR_VOID:
			   vptr = va_arg(ap, void ** );
			   break;
			}
		 }
		 else
		   break;
		 current_arg++;
      }
      va_end( ap );
      FE_LEAVE_FUNCTION( current_arg );
   }
   else
   {
      FE_LEAVE_FUNCTION( -1 );
   }
}

/*!
 *  \fn int ferite_get_parameter_count( FeriteVariable **list )
 *  \brief Returns the number of paramters within a list
 *  \param list The list of variables passed to the function calling this method
 *
 *  This methodis useful for obtaining the number of variables obtained. It is suggested that this method is used rather than
 */
int ferite_get_parameter_count( FeriteVariable **list )
{
   int argcount = 0;

   FE_ENTER_FUNCTION;
   if( list != NULL )
   {
      while( list[argcount++] != NULL );
      FE_LEAVE_FUNCTION( --argcount );
   }
   else
   {
      FE_LEAVE_FUNCTION( -1 );
   }
}

void *ferite_get_parameter( FeriteVariable **list, int num )
{
   FE_ENTER_FUNCTION;
   if( list != NULL )
   {
      if( num < ferite_get_parameter_count(list) )
      {
		 FE_LEAVE_FUNCTION( list[num] );
      }
   }
   FE_LEAVE_FUNCTION( NULL );
}
