/*
 * Copyright (C) 2000-2001 Chris Ross and Evan Webb
 * Copyright (C) 1999-2000 Chris Ross
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies of the Software, its documentation and marketing & publicity
 * materials, and acknowledgment shall be given in the documentation, materials
 * and software packages that this Software was used.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "ferite.h"
#include <setjmp.h>

typedef struct _ferite_compile_record
{
   FeriteFunction     *function;
   FeriteVariableHash *variables;
   FeriteClass        *cclass;
   FeriteScript       *script;
   FeriteNamespace    *ns;
}
FeriteCompileRecord;

/* these are used to keep track of verious bit's and pieces, and makes
 * it possible to compile ferite in one pass rather than several passes
 * like other langyages :) */
FeriteStack  *__ferite_fwd_look_stack;
FeriteStack  *__ferite_bck_look_stack;
FeriteStack  *__ferite_break_look_stack;
FeriteStack  *__ferite_include_list;
extern FeriteStack *__ferite_compiled_arrays_stack;

/* mwhahhaa, we can keep track of these things :) */
FeriteStack  *__ferite_compile_stack;
FeriteCompileRecord *__ferite_current_compile;

/* make life easier :) */
#define CURRENT_NAMESPACE __ferite_current_compile->ns
#define CURRENT_FUNCTION  __ferite_current_compile->function
#define CURRENT_CLASS     __ferite_current_compile->cclass
#define CURRENT_VARS      __ferite_current_compile->variables
#define CURRENT_SCRIPT    __ferite_current_compile->script

/* stolen from ferite_scanner.l -> so we can report errors on files */
extern char   *__ferite_scanner_file;
extern int     __ferite_scanner_lineno;
int            __ferite_compile_error = 0;
jmp_buf        __ferite_compiler_jmpback;

FeriteBkRequest *__ferite_create_request( FeriteOp *op, int type )
{
   FeriteBkRequest *ptr;

   FE_ENTER_FUNCTION;
   ptr = fmalloc( sizeof( FeriteBkRequest ) );
   ptr->reqop = op;
   ptr->type = type;
   FE_LEAVE_FUNCTION( ptr );
}

void __ferite_destroy_request( FeriteBkRequest *ptr )
{
   FE_ENTER_FUNCTION;
   ffree( ptr );
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_delete_request_stack( FeriteStack *stack )
{
   int i;
   FE_ENTER_FUNCTION;
   for( i = 0; i <= stack->stack_ptr; i++ )
   {
      if( stack->stack[i] != NULL ){
		 ffree( stack->stack[i] );
	  }
   }
   ffree( stack->stack );
   ffree( stack );
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_clean_compiler()
{
   FE_ENTER_FUNCTION;
   ferite_clean_parser();
   __ferite_delete_request_stack( __ferite_fwd_look_stack );
   __ferite_delete_request_stack( __ferite_bck_look_stack );
   __ferite_delete_request_stack( __ferite_break_look_stack );
   __ferite_delete_request_stack( __ferite_compile_stack );
   __ferite_delete_request_stack( __ferite_include_list );
   __ferite_delete_stack( __ferite_compiled_arrays_stack );
   ffree( __ferite_current_compile );
   __ferite_fwd_look_stack = NULL;
   __ferite_bck_look_stack = NULL;
   __ferite_break_look_stack = NULL;
   __ferite_compile_stack = NULL;
   __ferite_include_list = NULL;
   __ferite_compiled_arrays_stack = NULL;
   FE_LEAVE_FUNCTION( NOWT );
}

/* this is a runtime equivelent to:
 *
 * uses "foo";
 *
 * it also allows us to compile a script and the inclue arbitary modules
 * and scripts, say, for example, cookies.feh in mod_ferite :)
 *
 */
void __ferite_script_include( FeriteScript *script, char *filename )
{
   FeriteScript *ptr;
   char *str = NULL;
   int retval = 0;
   
   FE_ENTER_FUNCTION;
   
   /* load the script... */
   ptr = ferite_new_script();
   ferite_script_load( ptr, filename );
   if( ptr->scripttext != NULL ){
	  str = ptr->scripttext;
	  if( ptr->filename != NULL )
		ffree( ptr->filename );
	  __ferite_delete_namespace( script, ptr->mainns );
	  __ferite_delete_stack( ptr->exec_stack );
	  ffree( ptr );	  
   } else {
	  FE_LEAVE_FUNCTION(NOWT);
   }

   /* ...and compile that baby! */
   __ferite_current_compile = fmalloc( sizeof( FeriteCompileRecord ) );
   CURRENT_SCRIPT = script;
   CURRENT_CLASS = NULL;
   CURRENT_VARS = NULL;
   CURRENT_FUNCTION = NULL;
   CURRENT_NAMESPACE = script->mainns;

   __ferite_fwd_look_stack = __ferite_create_stack( FE_COMPILER_INTERNAL_STACK_SIZE );
   __ferite_bck_look_stack = __ferite_create_stack( FE_COMPILER_INTERNAL_STACK_SIZE );
   __ferite_break_look_stack = __ferite_create_stack( FE_COMPILER_INTERNAL_STACK_SIZE );
   __ferite_compile_stack = __ferite_create_stack( FE_COMPILER_INTERNAL_STACK_SIZE );
   __ferite_include_list = __ferite_create_stack( FE_COMPILER_INTERNAL_STACK_SIZE );

   FUD(("Setting up parser\n"));
   ferite_prepare_parser( str );
   FUD(("Running parser\n"));
   
   retval = setjmp( __ferite_compiler_jmpback );
   if( retval == 0 )
   {  /* we compiled correctly */
      ferite_parse();
      FUD(("Cleaning Up Parser\n"));
      __ferite_clean_compiler();
      FE_LEAVE_FUNCTION(NOWT);
   }
   else
   {
	  if( retval == 2 ){
		 __ferite_clean_compiler();
#ifdef DEBUG /* to make things work */
		 __ferite_call_level = __ferite_depth + 1; /* EVIL EVIL EVIL EVIL EVIL EVIL EVIL */
#endif
		 /* this done purely because we jump the stack and therefore require some form of retribution :)
		  * otherwise ferite chokes and go's "'ang about, you are at the wrong call depth. */
		 ffree( str );
		 FE_LEAVE_FUNCTION(NOWT);
	  } else {
		 /* uh, *cough* we didn't */
		 ferite_error( CURRENT_SCRIPT, "Error including script `%s'\n", filename );
		 __ferite_clean_compiler();
		 ffree( str );
		 FE_LEAVE_FUNCTION(NOWT);
	  }
   }
   FE_LEAVE_FUNCTION(NOWT);
}

FeriteScript *ferite_script_compile( char *filename )
{
   FeriteScript *ptr;
   FeriteScript *compiled_script;

   FE_ENTER_FUNCTION;
   ptr = ferite_new_script();
   ferite_script_load( ptr, filename );
   if( ptr->scripttext != NULL )
   {
      ferite_set_filename( filename );
      compiled_script = __ferite_compile_string( ptr->scripttext );
      if( compiled_script != NULL )
      {
		 __ferite_delete_namespace( compiled_script, ptr->mainns );
		 ptr->mainns = compiled_script->mainns;
		 __ferite_delete_stack( compiled_script->exec_stack );
		 ffree( compiled_script );
		 FE_LEAVE_FUNCTION(ptr);
      }
      else
      {
		 ferite_error( ptr, "Fatal error compiling script \"%s\"\n", filename );
		 ferite_script_delete( ptr );
		 FE_LEAVE_FUNCTION( NULL );
      }
   }
   else
   {
      __ferite_delete_stack( ptr->exec_stack );
      __ferite_delete_namespace( ptr, ptr->mainns );
      ffree( ptr );
      ferite_error( NULL, "Can't open script %s\n", filename );
      FE_LEAVE_FUNCTION( NULL );
   }
   FE_LEAVE_FUNCTION( NULL );
}

FeriteScript *__ferite_compile_string( char *str )
{
   FeriteScript *new_script = NULL;
   FeriteVariable *var = NULL;
   
   FE_ENTER_FUNCTION;
   __ferite_compile_error = 0;
   new_script = ferite_new_script();
   __ferite_register_ns_variable( new_script, new_script->mainns, fe_new_obj( "err" ) );
   __ferite_register_ns_variable( new_script, new_script->mainns, fe_new_obj( "null" ) );
   __ferite_init_error_system( new_script, new_script->mainns );

   var = __ferite_duplicate_variable( new_script, ARGV );
   __ferite_register_ns_variable( new_script, new_script->mainns, var );

   __ferite_current_compile = fmalloc( sizeof( FeriteCompileRecord ) );
   CURRENT_SCRIPT = new_script;
   CURRENT_CLASS = NULL;
   CURRENT_VARS = NULL;
   CURRENT_FUNCTION = NULL;
   CURRENT_NAMESPACE = new_script->mainns;

   __ferite_fwd_look_stack = __ferite_create_stack( FE_COMPILER_INTERNAL_STACK_SIZE );
   __ferite_bck_look_stack = __ferite_create_stack( FE_COMPILER_INTERNAL_STACK_SIZE );
   __ferite_break_look_stack = __ferite_create_stack( FE_COMPILER_INTERNAL_STACK_SIZE );
   __ferite_compile_stack = __ferite_create_stack( FE_COMPILER_INTERNAL_STACK_SIZE );
   __ferite_include_list = __ferite_create_stack( FE_COMPILER_INTERNAL_STACK_SIZE );
   __ferite_compiled_arrays_stack = __ferite_create_stack( FE_COMPILER_INTERNAL_STACK_SIZE );
   
   FUD(("Setting up parser\n"));
   ferite_prepare_parser( str );
   FUD(("Running parser\n"));
   if( setjmp( __ferite_compiler_jmpback ) == 0 )
   {  /* we compiled correctly */
      ferite_parse();
      FUD(("Cleaning Up Parser\n"));
      __ferite_clean_compiler();
      FE_LEAVE_FUNCTION( new_script );
   }
   else
   {  /* uh, *cough* we didn't */
	  if( __ferite_scanner_file == NULL || strcmp( __ferite_scanner_file, "-e" ) == 0 )
		ferite_error( CURRENT_SCRIPT, "Fatal error compiling script\n" );
      __ferite_clean_compiler();
      ferite_script_delete( new_script );
#ifdef DEBUG
		 __ferite_call_level = __ferite_depth + 1; /* EVIL EVIL EVIL EVIL EVIL EVIL EVIL */
#endif	  
      FE_LEAVE_FUNCTION( NULL );
   }
   FE_LEAVE_FUNCTION( NULL );
}

void __ferite_do_load_module( char *name )
{
   int i=0, incmod=1;

   FE_ENTER_FUNCTION;
   FUD(( "COMPILER: Request to load module %s\n", name ));
   for( i = 0; i < __ferite_include_list->stack_ptr; i++ )
   {
      if( __ferite_include_list->stack[i] != NULL && strcmp( name, __ferite_include_list->stack[i] ) == 0 )
	  {
		 incmod = 0;
		 break;
      }
   }
   if( incmod )
   {
      __ferite_stack_push( __ferite_include_list, fstrdup(name) );
      __ferite_load_module( CURRENT_SCRIPT, CURRENT_SCRIPT->mainns, name );
      if( __ferite_compile_error )
      {
		 ferite_error( CURRENT_SCRIPT, "Can't load module \"%s\" into the ferite engine, make sure it installed.\n" );
		 ffree( name );
		 longjmp( __ferite_compiler_jmpback, 1 );
      }
   }
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_load_script( char *name )
{
   FeriteScript *script;
   char *scripttext;
   jmp_buf old___ferite_compiler_jmpback;
   int i = 0, incmod = 1;

   FE_ENTER_FUNCTION;
   FUD(( "COMPILER: Request to load script %s\n", name ));

   for( i = 0; i < __ferite_include_list->stack_ptr; i++ )
   {
      if( __ferite_include_list->stack[i] != NULL && strcmp( name, __ferite_include_list->stack[i] ) == 0 )
	  {
		 incmod = 0;
		 break; /* optimise the loop */
      }
   }
   if( incmod )
   {
      __ferite_stack_push( __ferite_include_list, fstrdup(name) );

      /* save the older enviroment */
      memcpy( &old___ferite_compiler_jmpback, &__ferite_compiler_jmpback, sizeof(jmp_buf) );

      ferite_save_lexer();
      script = fmalloc( sizeof(FeriteScript) );
      ferite_script_load( script, name );
      ferite_set_filename( name );
      FUD(( "COMPILER: Setting up parser (__ferite_do_load_script)\n"));
      scripttext = script->scripttext;
      if( script->filename != NULL )
		ffree( script->filename );
      ffree( script );

      if( scripttext != NULL )
		ferite_prepare_parser( scripttext );
      else
      {
		 ferite_error( CURRENT_SCRIPT, "Can't include file \"%s\" (check that the paths are correct)\n", name );
		 ferite_restore_lexer();
		 __ferite_compile_error = 1;
		 ffree( name ); /* free up the reseource given to us - as the lexer/parser wont in this case */
		 memcpy( &__ferite_compiler_jmpback, &old___ferite_compiler_jmpback, sizeof(jmp_buf) );
		 longjmp( __ferite_compiler_jmpback, 1 );
      }

      if( setjmp( __ferite_compiler_jmpback ) == 0 )
      {
		 FUD(( "COMPILER: Running parser (__ferite_do_load_script)\n" ));
		 ferite_parse();
		 FUD(( "COMPILER: Cleaning Up Parser (__ferite_do_load_script)\n" ));
		 ferite_clean_parser();
		 ferite_restore_lexer();
		 ffree( scripttext );
		 /* we need to do this as we is not wantin to av a start function */
		 __ferite_delete_namespace_element_from_namespace( CURRENT_SCRIPT, CURRENT_NAMESPACE, "_start" );
		 /* restore the enviroment */
		 memcpy( &__ferite_compiler_jmpback, &old___ferite_compiler_jmpback, sizeof(jmp_buf) );
      }
      else
      {
		 ferite_error( CURRENT_SCRIPT, "Can't include file \"%s\" (Due to compliation errors)\n", name );
		 ferite_clean_parser();
		 ferite_restore_lexer();
		 __ferite_compile_error = 1;
		 ffree( name ); /* free up the reseource given to us - as the lexer/parser wont in this case */
		 ffree( scripttext );
		 memcpy( &__ferite_compiler_jmpback, &old___ferite_compiler_jmpback, sizeof(jmp_buf) );
		 longjmp( __ferite_compiler_jmpback, 1 );
      }
   }
   FE_LEAVE_FUNCTION(NOWT);
}

void __ferite_do_function_header( char *name, char *returntype, int is_static )
{
   FeriteFunction *new_function;
   FeriteCompileRecord *rec;
   int isclass = 0;

   FE_ENTER_FUNCTION;

   new_function = __ferite_create_internal_function( CURRENT_SCRIPT, name );
   new_function->is_static = is_static;

   if( CURRENT_CLASS == NULL )
   { /* add to a script */
      if( __ferite_namespace_element_exists( CURRENT_SCRIPT, CURRENT_NAMESPACE, name ) != NULL )
      {
		 if( strcmp( name, "_start" ) != 0 ){
			ferite_error( CURRENT_SCRIPT, "Compile Error: function %s already exists in script", name );
			__ferite_compile_error = 1;
			ffree( name );
			ffree( returntype );
			__ferite_delete_function_list( CURRENT_SCRIPT, new_function );
			longjmp( __ferite_compiler_jmpback, 1 );
		 } else { /* this is a special case */
			__ferite_delete_function_list( CURRENT_SCRIPT, new_function );
			longjmp( __ferite_compiler_jmpback, 2 );
		 }
      }
      __ferite_register_ns_function( CURRENT_SCRIPT, CURRENT_NAMESPACE, new_function );
   }
   else
   { /* add to a class */
      if( __ferite_hash_get( CURRENT_SCRIPT, CURRENT_CLASS->functions, name ) != NULL )
      {
		 ferite_error( CURRENT_SCRIPT, "Compile Error: function %s already exists in class %s", name, CURRENT_CLASS->name );
		 __ferite_compile_error = 1;
		 ffree( name );
		 ffree( returntype );
		 __ferite_delete_function_list( CURRENT_SCRIPT, new_function );
		 longjmp( __ferite_compiler_jmpback, 1 );
      }
      ferite_register_class_function( CURRENT_SCRIPT, CURRENT_CLASS, new_function );
      isclass = 1;
   }

#define CHKNSET( a, b ) if( strcmp( returntype, a ) == 0 ) new_function->return_type = b
   CHKNSET( "void", F_VAR_VOID );
   CHKNSET( "number", F_VAR_NUM );
   CHKNSET( "string", F_VAR_STR );
   CHKNSET( "object", F_VAR_OBJ );
#undef CHKNSET
   
   new_function->ccode->filename = fstrdup( (__ferite_scanner_file == NULL ? "" : __ferite_scanner_file) );
   
   rec = __ferite_current_compile;
   __ferite_stack_push( __ferite_compile_stack, __ferite_current_compile );
   __ferite_current_compile = fmalloc( sizeof( FeriteCompileRecord ) );
   CURRENT_SCRIPT = rec->script;
   CURRENT_FUNCTION = new_function;
   CURRENT_VARS = new_function->localvars;
   CURRENT_NAMESPACE = rec->ns;
   CURRENT_CLASS = rec->cclass;
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_function_start()
{
   FE_ENTER_FUNCTION;
   if( CURRENT_CLASS != NULL && CURRENT_FUNCTION != NULL )
   {
	  if( CURRENT_FUNCTION->is_static == 0 ){
		 __ferite_do_add_variable_to_paramlist( "super", "object" );
		 __ferite_do_add_variable_to_paramlist( "self", "object" );
	  }
   }
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_function_footer()
{
   FE_ENTER_FUNCTION;
   __ferite_do_exit();
   if( show_debug ){
	  FUD(("COMPILER: Dumping opcode list for %s\n", CURRENT_FUNCTION->name ));
	  __ferite_opcode_dump( CURRENT_FUNCTION->ccode );
   }
   ffree( __ferite_current_compile );
   __ferite_current_compile = __ferite_stack_pop( __ferite_compile_stack );
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_class_header( char *name, char *extends )
{
   FeriteClass *classtmpl;

   FE_ENTER_FUNCTION;
   classtmpl = ferite_register_inherited_class( CURRENT_SCRIPT, CURRENT_NAMESPACE, name, extends );
   __ferite_stack_push( __ferite_compile_stack, __ferite_current_compile );
   __ferite_current_compile = fmalloc( sizeof( FeriteCompileRecord ) );
   CURRENT_VARS = classtmpl->variables;
   CURRENT_CLASS = classtmpl;
   CURRENT_SCRIPT = NULL;
   CURRENT_FUNCTION = NULL;
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_class_footer()
{
   FE_ENTER_FUNCTION;
   ffree( __ferite_current_compile );
   __ferite_current_compile = __ferite_stack_pop( __ferite_compile_stack );
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_namespace_header( char *name )
{
   FeriteNamespace *ns;
   FE_ENTER_FUNCTION;
   if( __ferite_namespace_element_exists( CURRENT_SCRIPT, CURRENT_NAMESPACE, name ) == NULL )
   {
      FUD(( "registering namespace %s\n", name ));
      ns = __ferite_register_namespace( CURRENT_SCRIPT, name, CURRENT_NAMESPACE );
      __ferite_stack_push( __ferite_compile_stack, __ferite_current_compile );
      __ferite_current_compile = fmalloc( sizeof( FeriteCompileRecord ) );
      CURRENT_VARS = NULL;
      CURRENT_CLASS = NULL;
      CURRENT_SCRIPT = NULL;
      CURRENT_FUNCTION = NULL;
      CURRENT_NAMESPACE = ns;
   }
   else
   {
      ferite_error( CURRENT_SCRIPT, "A namespace element called '%s' already exists.\n", name );
      longjmp( __ferite_compiler_jmpback, 1 );
   }
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_namespace_extends( char *name )
{
   FeriteNamespace *ns;
   FeriteNamespaceBucket *nsb;
   
   FE_ENTER_FUNCTION;
   if( (nsb=__ferite_namespace_element_exists( CURRENT_SCRIPT, CURRENT_NAMESPACE, name )) != NULL )
   {
      FUD(( "extending namespace %s\n", name ));
      ns = nsb->data;
      __ferite_stack_push( __ferite_compile_stack, __ferite_current_compile );
      __ferite_current_compile = fmalloc( sizeof( FeriteCompileRecord ) );
      CURRENT_VARS = NULL;
      CURRENT_CLASS = NULL;
      CURRENT_SCRIPT = NULL;
      CURRENT_FUNCTION = NULL;
      CURRENT_NAMESPACE = ns;
   }
   else
   {
	  __ferite_do_namespace_header( name );
   }
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_namespace_footer()
{
   FE_ENTER_FUNCTION;
   ffree( __ferite_current_compile );
   __ferite_current_compile = __ferite_stack_pop( __ferite_compile_stack );
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_add_variable( char *name, char *type, int is_global, int is_final, int is_static )
{
   FeriteVariable *new_variable = NULL;
   FeriteVariableHash *current_hash;

   FE_ENTER_FUNCTION;

   if( is_global )
     current_hash = CURRENT_NAMESPACE->space;
   else
   {
      if( CURRENT_VARS != NULL )
		current_hash = CURRENT_VARS;
      else
		current_hash = CURRENT_NAMESPACE->space;
   }
   if( (new_variable = __ferite_get_variable_from_hash( CURRENT_SCRIPT, current_hash, name )) == NULL )
   {
      if( strcmp( type, "number" ) == 0 )
		new_variable = __ferite_create_number_long_variable( name, 0 );
      if( strcmp( type, "string" ) == 0 )
        new_variable = __ferite_create_string_variable( name, "" );
      if( strcmp( type, "object" ) == 0 )
		new_variable = __ferite_create_object_variable( name );
      if( strcmp( type, "array" ) == 0 )
		new_variable = __ferite_create_uarray_variable( name, 0);
      if( strcmp( type, "void" ) == 0 )
		new_variable = __ferite_create_void_variable( name );
      if( new_variable != NULL )
      {
		 if( is_final )
		 {
			FUD(( "Marking variable %s as final.\n", name ));
			new_variable->flags.constant = 1;
		 }
		 new_variable->flags.is_static = is_static;
		 if( is_global )
		   __ferite_register_ns_variable( CURRENT_SCRIPT, CURRENT_NAMESPACE, new_variable );
		 else
		 {
			if( CURRENT_VARS != NULL )
			  __ferite_add_variable_to_hash( CURRENT_SCRIPT, CURRENT_VARS, new_variable );
			else
			  __ferite_register_ns_variable( CURRENT_SCRIPT, CURRENT_NAMESPACE, new_variable );
		 }
		 FE_LEAVE_FUNCTION( NOWT );
      }
      ferite_error( CURRENT_SCRIPT, "Compile Error: Trying to declare variable of unknown type \"%s\"", type );
      ffree( name );
      ffree( type );
      __ferite_compile_error = 1;
      longjmp( __ferite_compiler_jmpback, 1 );
      FE_LEAVE_FUNCTION( NOWT );
   }
   else
   {
      ferite_error( CURRENT_SCRIPT, "Compile Error: Variable \"%s\" already exist's", name );
      ffree( name );
      ffree( type );
      __ferite_compile_error = 1;
      longjmp( __ferite_compiler_jmpback, 1 );
   }
}

void __ferite_do_add_variable_with_value( char *name, FeriteVariable *new_variable, int is_global )
{
   FeriteVariableHash *current_hash;

   FE_ENTER_FUNCTION;

   if( is_global )
     current_hash = CURRENT_NAMESPACE->space;
   else
   {
      if( CURRENT_VARS != NULL )
		current_hash = CURRENT_VARS;
      else
		current_hash = CURRENT_NAMESPACE->space;
   }
   if( __ferite_get_variable_from_hash( CURRENT_SCRIPT, current_hash, name ) == NULL )
   {
      if( new_variable != NULL )
      {
		 if( is_global )
		   __ferite_register_ns_variable( CURRENT_SCRIPT, CURRENT_NAMESPACE, new_variable );
		 else
		 {
			if( CURRENT_VARS != NULL )
			  __ferite_add_variable_to_hash( CURRENT_SCRIPT, CURRENT_VARS, new_variable );
			else
			  __ferite_register_ns_variable( CURRENT_SCRIPT, CURRENT_NAMESPACE, new_variable );
		 }
		 FE_LEAVE_FUNCTION( NOWT );
      }
   }
   else
   { 
      ferite_error( CURRENT_SCRIPT, "Compile Error: Variable \"%s\" already exist's", name );
      ffree( name );
	  if( new_variable != NULL )
		__ferite_variable_destroy( CURRENT_SCRIPT, new_variable );
      __ferite_compile_error = 1;
      longjmp( __ferite_compiler_jmpback, 1 );
   }
}

#ifdef DEBUG
# define REDUCE_DEPTH __ferite_call_level -= 2
#else
# define REDUCE_DEPTH
#endif

#define CHECK_FUNCTION_CODE( blah ) \
   if( CURRENT_FUNCTION == NULL ) { \
      ferite_error( CURRENT_SCRIPT, "Compile Error: on line %d, in %s\n", __ferite_scanner_lineno, __ferite_scanner_file ); \
      __ferite_compile_error = 1; \
      blah; \
	  REDUCE_DEPTH; \
      longjmp( __ferite_compiler_jmpback, 1 ); \
   }

void __ferite_do_add_variable_to_paramlist( char *name, char *type )
{
   FeriteVariable *new_variable;

   FE_ENTER_FUNCTION;
   CHECK_FUNCTION_CODE({
	  ffree(name); ffree(type);
   });
   __ferite_do_add_variable( name, type, 0, 0, 0 ); /* add the variable to the function */
   /* now lets add the paramter */
   if( strcmp( type, "number" ) == 0 )
     new_variable = __ferite_create_number_long_variable( name, 0 );
   if( strcmp( type, "string" ) == 0 )
     new_variable = __ferite_create_string_variable( name, "" );
   if( strcmp( type, "object" ) == 0 )
     new_variable = __ferite_create_object_variable( name );
   if( strcmp( type, "array" ) == 0 )
     new_variable = __ferite_create_uarray_variable( name, 0 );
   if( strcmp( type, "void" ) == 0 )
     new_variable = __ferite_create_void_variable( name );
   CURRENT_FUNCTION->signature[CURRENT_FUNCTION->arg_count] = fmalloc( sizeof( FeriteParameterRecord* ) );
   CURRENT_FUNCTION->signature[CURRENT_FUNCTION->arg_count]->variable = new_variable;
   CURRENT_FUNCTION->signature[CURRENT_FUNCTION->arg_count]->has_default_value = 0;
   CURRENT_FUNCTION->arg_count++;
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_exit()
{
   FeriteOp *op;

   FE_ENTER_FUNCTION;
   CHECK_FUNCTION_CODE( NOWT );
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_EXIT;
   op->line = __ferite_scanner_lineno;
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_push( FeriteVariable *var )
{
   FeriteOp *op;

   FE_ENTER_FUNCTION;
   CHECK_FUNCTION_CODE({
	  __ferite_variable_destroy( CURRENT_SCRIPT, var );
   });
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   OP_PUSH( op, var );
   op->line = __ferite_scanner_lineno;
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_pop()
{
   FeriteOp *op;

   FE_ENTER_FUNCTION;
   CHECK_FUNCTION_CODE( NOWT );
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_POP;
   op->line = __ferite_scanner_lineno;
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_variable_push( char *name )
{
   FeriteOp *op;

   FE_ENTER_FUNCTION;

   FUD(("DO_PUSHVAR: WANTING TO PUSH VARIABLE %s\n", name ));
   CHECK_FUNCTION_CODE({
	  ffree( name );
   });
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_PUSHVAR;
   op->opdata = fstrdup( name );
   op->line = __ferite_scanner_lineno;
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_binary_op( void *opptr )
{
   FeriteOp *op;

   FE_ENTER_FUNCTION;
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   OP_BINARY( op, opptr, NULL );
   op->line = __ferite_scanner_lineno;
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_unary_op( void *opptr )
{
   FeriteOp *op;

   FE_ENTER_FUNCTION;
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   OP_UNARY( op, opptr, NULL );
   op->line = __ferite_scanner_lineno;
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_function_call( char *name )
{
   FeriteOp *op;

   FE_ENTER_FUNCTION;
   CHECK_FUNCTION_CODE({
	  ffree( name );
   });
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_FUNCTION;
   op->opdata = fstrdup( name );
   op->line = __ferite_scanner_lineno;
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_object_function_call( char *name )
{
   FeriteOp *op;

   FE_ENTER_FUNCTION;
   CHECK_FUNCTION_CODE({
	  ffree( name );
   });
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_METHOD;
   op->opdata = fstrdup( name );
   op->line = __ferite_scanner_lineno;
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_if_statement()
{
   FeriteBkRequest *req;
   FeriteOp *op;

   FE_ENTER_FUNCTION;
   CHECK_FUNCTION_CODE( NOWT );
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_BNE;
   op->line = __ferite_scanner_lineno;
   req = __ferite_create_request( op, IF_JUMP_THEN );
   __ferite_stack_push( __ferite_fwd_look_stack, req );
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_after_then_statement()
{
   FeriteBkRequest *req;
   FeriteOp *op;
   int addr;

   FE_ENTER_FUNCTION;
   addr = __ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
   op = __ferite_get_next_op_address( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_NOP;
   req = (FeriteBkRequest *)__ferite_stack_pop( __ferite_fwd_look_stack );
   FUD(("Popped %d off request stack\n", req->type));
   req->reqop->addr = addr;
   MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
   __ferite_destroy_request( req );
   FE_LEAVE_FUNCTION( NOWT );
}

/* blum */
void __ferite_do_after_then_before_else_statement()
{
   FeriteBkRequest *req;
   FeriteOp *jump_else_op, *do_else_op;
   int do_else_op_offset;

   FE_ENTER_FUNCTION;
   /* we jump over the else block */
   jump_else_op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   jump_else_op->OP_TYPE = F_OP_JMP;
   jump_else_op->line = __ferite_scanner_lineno;

   /* we jump to this op when we do else {} */
   do_else_op_offset = __ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
   do_else_op = __ferite_get_next_op_address( CURRENT_FUNCTION->ccode ); /* so we know where to jump, but dont put an NOP there */
   do_else_op->OP_TYPE = F_OP_NOP;

   req = (FeriteBkRequest *)__ferite_stack_pop( __ferite_fwd_look_stack );
   FUD(("Popped %d off request stack\n", req->type));
   req->reqop->addr = do_else_op_offset;
   MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
   __ferite_destroy_request( req );

   req = __ferite_create_request( jump_else_op, IF_JUMP_ELSE );
   __ferite_stack_push( __ferite_fwd_look_stack, req );
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_after_else_statement()
{
   FeriteBkRequest *req;
   FeriteOp *op;
   int next_op;

   FE_ENTER_FUNCTION;
   next_op = __ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
   op = __ferite_get_next_op_address( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_NOP;
   req = (FeriteBkRequest *)__ferite_stack_pop( __ferite_fwd_look_stack );
   FUD(("Popped %d off request stack\n", req->type));
   req->reqop->addr = next_op;
   MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
   __ferite_destroy_request( req );
   FE_LEAVE_FUNCTION( NOWT );
}

/* while stuff
 *
 * a:  expr
 *     BNE   b
 *     block
 *     JMP   a
 * b:
 */

void __ferite_do_while_begin()
{
   FeriteBkRequest *req;
   FeriteOp *op;
   int next_op;

   FE_ENTER_FUNCTION;
   next_op = __ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
   op = __ferite_get_next_op_address( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_NOP;
   req = __ferite_create_request( op, WHILE_JUMP_TO );
   req->addr = next_op;
   __ferite_stack_push( __ferite_bck_look_stack, req );
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_while_after_expr()
{
   FeriteBkRequest *req;
   FeriteOp *op;

   FE_ENTER_FUNCTION;
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_BNE;
   op->line = __ferite_scanner_lineno;
   req = __ferite_create_request( op, WHILE_JUMP_BLOCK );
   __ferite_stack_push( __ferite_fwd_look_stack, req );
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_while_end()
{
   FeriteBkRequest *req;
   FeriteOp *op;
   int next_addr;

   FE_ENTER_FUNCTION;
   /* hook up jump back */
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_JMP;
   op->line = __ferite_scanner_lineno;

   next_addr = __ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
   __ferite_process_breaks( WHILE_JUMP_TO, next_addr );

   req = (FeriteBkRequest *)__ferite_stack_pop( __ferite_bck_look_stack );
   FUD(("Popped %d off request stack\n", req->type));
/*   fprintf( stderr, "while end type %d\n", req->type );*/
   op->addr = req->addr;
   MARK_VARIABLE_AS_COMPILED( PTR2VAR(op->opdata) );
   __ferite_destroy_request( req );

   /* hook up jump over */
   op = __ferite_get_next_op_address( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_NOP;
   req = (FeriteBkRequest *)__ferite_stack_pop( __ferite_fwd_look_stack );
   FUD(("Popped %d off request stack\n", req->type));
   req->reqop->addr = next_addr;
   MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
   __ferite_destroy_request( req );
   FE_LEAVE_FUNCTION( NOWT );
}

/************ FOR LOOP STUFF */

/* for( init , */

void __ferite_do_for_loop_start()
{
   FeriteBkRequest *req;
   FeriteOp *op;
   int next_op;

   FE_ENTER_FUNCTION;
   next_op = __ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
   op = __ferite_get_next_op_address( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_NOP;
   req = __ferite_create_request( op, FOR_TEST_START );
   req->addr = next_op;
   __ferite_stack_push( __ferite_bck_look_stack, req );
   FE_LEAVE_FUNCTION( NOWT );
}

/* test , */

void __ferite_do_for_loop_itterate()
{

   FeriteBkRequest *req;
   FeriteOp *op;
   int next_op;

   FE_ENTER_FUNCTION;
   /* jump out of the for loop */
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_BNE;
   op->line = __ferite_scanner_lineno;
   req = __ferite_create_request( op, FOR_JUMP_BLOCK );
   __ferite_stack_push( __ferite_fwd_look_stack, req );

   /* jump over the increment block (3rd expr) of the while loop */
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_JMP;
   op->line = __ferite_scanner_lineno;
   req = __ferite_create_request( op, FOR_JUMP_INCR );
   __ferite_stack_push( __ferite_fwd_look_stack, req );

   /* push on the addrs of the increment block start */
   next_op = __ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
   op = __ferite_get_next_op_address( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_NOP;
   req = __ferite_create_request( op, FOR_INCR_START );
   req->addr = next_op;
   __ferite_stack_push( __ferite_bck_look_stack, req );
   FE_LEAVE_FUNCTION( NOWT );
}

/*    INCR ) */

void __ferite_do_for_loop_block()
{
   FeriteBkRequest *req, *req_incr_start, *req_test_start;
   FeriteOp *op;
   int next_op;

   FE_ENTER_FUNCTION;
   /* so we dont lose this request */
   req_incr_start = __ferite_stack_pop( __ferite_bck_look_stack );
   FUD(("Popped %d off stack\n", req_incr_start->type ));

   /* setupa jmp op to go back to the test */
   req_test_start = __ferite_stack_pop( __ferite_bck_look_stack );
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_JMP;
   op->addr = req_test_start->addr;
   op->line = __ferite_scanner_lineno;
   MARK_VARIABLE_AS_COMPILED( PTR2VAR(op->opdata) );
   FUD(("Popped %d off stack\n", req_test_start->type ));
   __ferite_destroy_request( req_test_start );

   /* push first pop back onto the stack */
   __ferite_stack_push( __ferite_bck_look_stack, req_incr_start );
   FUD(("Popped %d off stack\n", req_incr_start->type ));

   /* now lets sort out the jump increment block code */
   next_op = __ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
   op = __ferite_get_next_op_address( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_NOP;
   req = __ferite_stack_pop( __ferite_fwd_look_stack );
   FUD(("Popped %d off stack\n", req->type ));
   req->reqop->addr = next_op;
   MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
   __ferite_destroy_request( req );

   FE_LEAVE_FUNCTION( NOWT );
}

/* block */

void __ferite_do_for_loop_end()
{
   FeriteBkRequest *req;
   FeriteOp *op;
   int next_op;

   FE_ENTER_FUNCTION;
   /* jump back to incr block */
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_JMP;
   op->line = __ferite_scanner_lineno;
   next_op = __ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );

   /* process dem breaks's */
   __ferite_process_breaks( FOR_INCR_START, next_op );

   req = __ferite_stack_pop( __ferite_bck_look_stack );
   FUD(("Popped %d off stack\n", req->type ));
   op->addr = req->addr;
   MARK_VARIABLE_AS_COMPILED( PTR2VAR(op->opdata) );
   __ferite_destroy_request( req );

   /* jump over block */
/*   next_op = __ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );*/
   op = __ferite_get_next_op_address( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_NOP;
   req = __ferite_stack_pop( __ferite_fwd_look_stack );
   FUD(("Popped %d off stack\n", req->type ));
   req->reqop->addr = next_op;
   MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
   __ferite_destroy_request( req );
   FE_LEAVE_FUNCTION( NOWT );
}

/************** do stuff */

void __ferite_do_do_start()
{
   FeriteBkRequest *req;
   FeriteOp *op;
   int next_op;

   FE_ENTER_FUNCTION;
   next_op = __ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
   op = __ferite_get_next_op_address( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_NOP;
   req = __ferite_create_request( op, DO_START );
   req->addr = next_op;
   __ferite_stack_push( __ferite_bck_look_stack, req );
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_do_end()
{
   FeriteBkRequest *req;
   FeriteOp *op;
   int addr;

   FE_ENTER_FUNCTION;
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   op->line = __ferite_scanner_lineno;
   op->OP_TYPE = F_OP_BIE;

   /* process dem breaks's */
   addr = __ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
   __ferite_process_breaks( DO_START, addr );

   req = __ferite_stack_pop( __ferite_bck_look_stack );
   FUD(("Popped %d off stack\n", req->type ));
   op->addr = req->addr;
   MARK_VARIABLE_AS_COMPILED( PTR2VAR(op->opdata) );
   __ferite_destroy_request( req );
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_new_object()
{
   FeriteOp *op;

   FE_ENTER_FUNCTION;
   CHECK_FUNCTION_CODE(NOWT);
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_NEWOBJ;
   op->line = __ferite_scanner_lineno;
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_regex( char *pattern )
{
   FeriteOp *op;

   FE_ENTER_FUNCTION;
   CHECK_FUNCTION_CODE(
					 {
						ffree( pattern );
					 }
					   );
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_RGX;
   op->opdata = __ferite_generate_regex( pattern );
   op->line = __ferite_scanner_lineno;
   FE_LEAVE_FUNCTION( NOWT );
}

/****************************************************************************************/
/********* ERROR STUFF ******************************************************************/
void __ferite_do_iferr_block()
{
   FeriteBkRequest *req;
   FeriteOp *op;

   FE_ENTER_FUNCTION;
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_ERR;
   op->line = __ferite_scanner_lineno;
   req = __ferite_create_request( op, ERR_START );
   __ferite_stack_push( __ferite_fwd_look_stack, req );
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_before_fix_block()
{
   FeriteBkRequest *req, *req2;
   FeriteOp *op, *op_two;
   int addr;

   FE_ENTER_FUNCTION;

   op_two = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   op_two->OP_TYPE = F_OP_JMP;
   op_two->line = __ferite_scanner_lineno;
   req2 = __ferite_create_request( op_two, FIX_BLOCK_JMP ); /* jump instruction to jump fix block */

   /* this sets the code to jump to the fix block if there is an error
    * goes back and sets the address to jump to */
   addr = __ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_ERR;
   op->addr = -1;
   op->line = __ferite_scanner_lineno;
   MARK_VARIABLE_AS_COMPILED( PTR2VAR(op->opdata) );

   req = (FeriteBkRequest *)__ferite_stack_pop( __ferite_fwd_look_stack );
   FUD(("Popped %d off request stack\n", req->type));
   req->reqop->addr = addr;
   MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
   __ferite_destroy_request( req );

   __ferite_stack_push( __ferite_fwd_look_stack, req2 ); /* jump the fix block */

   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_after_fix_block()
{
   FeriteBkRequest *req;
   FeriteOp *op;
   int addr;

   /* we dont have an else block */
   FE_ENTER_FUNCTION;
   /* the address to jump over the fix block */
   addr = __ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
   op = __ferite_get_next_op_address( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_NOP;
   req = (FeriteBkRequest *)__ferite_stack_pop( __ferite_fwd_look_stack );
   req->reqop->addr = addr;
   MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
   __ferite_destroy_request( req );
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_after_fix_before_else_block()
{
   /* we have an else block */
   FeriteBkRequest *req, *req2;
   FeriteOp *op;
   int addr;

   FE_ENTER_FUNCTION;
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_JMP;
   op->line = __ferite_scanner_lineno;
   req2 = __ferite_create_request( op, JMP_ERR_ELSE ); /* jump instruction to jump else block */

   /* this sets the code to jump to the else block if there sn't an error */
   addr = __ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_ERR;
   op->addr = -1;
   op->line = __ferite_scanner_lineno;
   MARK_VARIABLE_AS_COMPILED( PTR2VAR(op->opdata) );

   req = (FeriteBkRequest *)__ferite_stack_pop( __ferite_fwd_look_stack );
   FUD(("Popped %d off request stack\n", req->type));
   req->reqop->addr = addr;
   MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
   __ferite_destroy_request( req );

   __ferite_stack_push( __ferite_fwd_look_stack, req2 ); /* jump the fix block */

   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_after_fix_else_statement()
{
   FeriteBkRequest *req;
   FeriteOp *op;
   int addr;

   /* we need this to jump over the else block */
   FE_ENTER_FUNCTION;
   /* the address to jump over the else block */
   addr = __ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
   op = __ferite_get_next_op_address( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_NOP;
   req = (FeriteBkRequest *)__ferite_stack_pop( __ferite_fwd_look_stack );
   req->reqop->addr = addr;
   MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
   __ferite_destroy_request( req );
   FE_LEAVE_FUNCTION( NOWT );
}

/* the dreaded switch statment */

void __ferite_do_pre_switch()
{
   FeriteBkRequest *req;
   FeriteOp *jmp_op, *pop_op;
   int addr;

   FE_ENTER_FUNCTION;
   
   /* JMP */
   jmp_op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   jmp_op->OP_TYPE = F_OP_JMP;
   jmp_op->line = __ferite_scanner_lineno;

   /* POP */
   pop_op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   pop_op->OP_TYPE = F_OP_POP;
   pop_op->line = __ferite_scanner_lineno;
   
   addr = __ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
   jmp_op->addr = addr;
   MARK_VARIABLE_AS_COMPILED( PTR2VAR(jmp_op->opdata) );

   /* this is where the continue jumps to in a switch statement */
   req = __ferite_create_request( jmp_op, SWITCH_CONTINUE_JUMP_TO );
   req->addr = addr;
   __ferite_stack_push( __ferite_bck_look_stack, req );
   
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_case_block_start()
{
   FeriteOp *op = NULL;
   FeriteBkRequest *req = NULL;
   int addr = 0;
   
   FE_ENTER_FUNCTION;
   
   /* case expr */
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   OP_BINARY( op, FERITE_OP_CALL(case), NULL );
   op->line = __ferite_scanner_lineno;
   
   /* BNE */
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_BNE;
   op->line = __ferite_scanner_lineno;
   
   /* here we hook up the addresses such that we come here if we hit this case block */
   addr = __ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
   req = __ferite_stack_pop( __ferite_fwd_look_stack );
   if( req == NULL || req->type != SWITCH_NEXT_CASE_BLOCK )
   {
/*	  fprintf( stderr, "foobar %d:%d\n", __LINE__, (req == NULL ? -1 : req->type) );*/
	  if( req != NULL ){
		 __ferite_stack_push( __ferite_fwd_look_stack, req );
/*		 fprintf( stderr, "pushed back onto stack\n" );*/
	  }
   } else {
	  req->reqop->addr = addr;
	  MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
	  __ferite_destroy_request( req );
   }
   
   /* this sets up the branch to the next case block if the case statement doesn't match */
   req = __ferite_create_request( op, SWITCH_NEXT_CASE );
   __ferite_stack_push( __ferite_fwd_look_stack, req );
   
   FE_LEAVE_FUNCTION( NOWT );   
}

void __ferite_do_default_block_start()
{
   FeriteBkRequest *req = NULL;
   int addr = 0;
   
   FE_ENTER_FUNCTION;
         
   /* here we hook up the addresses such that we come here if we hit this case block */
   addr = __ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
   req = __ferite_stack_pop( __ferite_fwd_look_stack );
   if( req == NULL || req->type != SWITCH_NEXT_CASE_BLOCK )
   {
/*	  fprintf( stderr, "foobar %d:%d\n", __LINE__, (req == NULL ? -1 : req->type) );*/
	  if( req != NULL ){
		 __ferite_stack_push( __ferite_fwd_look_stack, req );
/*		 fprintf( stderr, "pushed back onto stack\n" );*/
	  }
   } else {
	  req->reqop->addr = addr;
	  MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
	  __ferite_destroy_request( req );
   }
   
   /* this sets up the branch to the next case block if the case statement doesn't match */
/*   req = __ferite_create_request( op, SWITCH_NEXT_CASE_BLOCK );
   __ferite_stack_push( __ferite_fwd_look_stack, req );*/
   
   FE_LEAVE_FUNCTION( NOWT );   
}

void __ferite_do_case_block_end()
{
   FeriteOp *op = NULL;
   FeriteBkRequest *req = NULL;
   int addr= 0;
   
   FE_ENTER_FUNCTION;
   
   /* jump over the next case block */
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_JMP;
   op->line = __ferite_scanner_lineno;

   /* link up the jump to the next case statement */
   addr = __ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
   req = __ferite_stack_pop( __ferite_fwd_look_stack );
   if( req->type != SWITCH_NEXT_CASE )
   {
/*	  fprintf( stderr, "foobar %d:%d\n", __LINE__, req->type );*/
	  __ferite_stack_push( __ferite_fwd_look_stack, req );
   } else {
	  req->reqop->addr = addr;
	  MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
	  __ferite_destroy_request( req );
   }
   
   /* this is for the afore mentioned jump */
   req = __ferite_create_request( op, SWITCH_NEXT_CASE_BLOCK );
   __ferite_stack_push( __ferite_fwd_look_stack, req );

   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_do_post_switch()
{
   FeriteOp *op;
   FeriteBkRequest *req = NULL;
   int addr;
   
   FE_ENTER_FUNCTION;
   
   /* here we hook up the addresses such that we come here if we hit this case block */
   addr = __ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
   req = __ferite_stack_pop( __ferite_fwd_look_stack );
   if( req->type != SWITCH_NEXT_CASE_BLOCK )
   {
/*	  fprintf( stderr, "foobar %d:%d\n", __LINE__, req->type );*/
	  __ferite_stack_push( __ferite_fwd_look_stack, req );
   } else {
	  req->reqop->addr = addr;
	  MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
	  __ferite_destroy_request( req );
   }
   
   /* hook up breaks to jump to the pop */
   addr = __ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
   __ferite_process_breaks( SWITCH_CONTINUE_JUMP_TO, addr );   

   req = __ferite_stack_pop( __ferite_bck_look_stack );
   __ferite_destroy_request( req );
							 
   /* POP */
   op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
   op->OP_TYPE = F_OP_POP;
   
   FE_LEAVE_FUNCTION( NOWT );   
}

/* loop control */
void __ferite_do_break()
{
   FeriteOp *op = NULL;
   FeriteBkRequest *req, *breq;
   int i;

   FE_ENTER_FUNCTION;
   for( i = __ferite_bck_look_stack->stack_ptr; i > 0; i-- )
   {
      req = __ferite_bck_look_stack->stack[i];
      if( req->type == FOR_INCR_START || req->type == WHILE_JUMP_TO || req->type == DO_START || req->type == SWITCH_CONTINUE_JUMP_TO )
      {
		 op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
		 op->OP_TYPE = F_OP_JMP;
		 op->line = __ferite_scanner_lineno;
		 breq = __ferite_create_request( op, req->type );
		 __ferite_stack_push( __ferite_break_look_stack, breq );
		 break;
      }
   }
   if( op == NULL )
   {
      ferite_warning( CURRENT_SCRIPT, "Trying to use break in non-looping block. (ignoring)\n" );
   }
   FE_LEAVE_FUNCTION(NOWT);
}

/* for this we simply go up the __ferite_bck_look_stack and get the address of the first
 * structure :) */
void __ferite_do_continue()
{
   FeriteOp *op = NULL;
   FeriteBkRequest *req;
   int i;

   FE_ENTER_FUNCTION;
   for( i = __ferite_bck_look_stack->stack_ptr; i > 0; i-- )
   {
      req = __ferite_bck_look_stack->stack[i];
      if( req->type == FOR_INCR_START || req->type == WHILE_JUMP_TO || req->type == DO_START || req->type == SWITCH_CONTINUE_JUMP_TO )
      {
		 op = __ferite_get_next_op( CURRENT_FUNCTION->ccode );
		 op->OP_TYPE = F_OP_JMP;
		 op->addr = req->addr;
		 op->line = __ferite_scanner_lineno;
		 MARK_VARIABLE_AS_COMPILED( PTR2VAR(op->opdata) );
		 break;
      }
   }
   if( op == NULL )
   {
      ferite_warning( CURRENT_SCRIPT, "Trying to use continue in non-looping block. (ignoring)\n" );
   }
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_process_breaks( int starttag, int address )
{
   FeriteBkRequest *req;

   FE_ENTER_FUNCTION;
   req = __ferite_stack_top( __ferite_break_look_stack );
   while( req != NULL && req->type == starttag )
   {
      __ferite_stack_pop( __ferite_break_look_stack );

      req->reqop->addr = address;
      MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata ) );

      __ferite_destroy_request( req );
      req = __ferite_stack_top( __ferite_break_look_stack );
   }
   FE_LEAVE_FUNCTION( NOWT );
}
