/* Utility functions used  */ 
/* COPYRIGHT (C) 2000 THE VICTORIA UNIVERSITY OF MANCHESTER and John Levon
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details. 
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA. 
 */
/*
 * $Log: util.c,v $
 * Revision 1.3  2000/12/17 00:57:43  moz
 * examples, filled open splines, highlight_objects
 *
 * Revision 1.2  2000/12/06 20:56:07  moz
 * GPL stuff.
 *
 * Revision 1.1.1.1  2000/08/21 01:05:31  moz
 *
 *
 * Revision 1.1.1.1  2000/07/19 22:45:31  moz
 * CVS Import
 *
 * Revision 1.36  2000/05/04 03:38:35  moz
 * Got font faking working fairly sensibly.
 *
 * Revision 1.35  2000/05/04 02:27:01  moz
 * Fix font scaling to some degree.
 *
 * Revision 1.34  2000/03/13 02:46:53  moz
 * redraw_object fixes for lw.
 *
 * Revision 1.33  2000/03/09 23:43:52  moz
 * No end styles for closed splines.
 *
 * Revision 1.32  2000/03/09 23:29:40  moz
 * Don't allow joinstyle for splines/arcs.
 *
 * Revision 1.31  2000/03/06 02:28:49  moz
 * Compile fixes.
 *
 * Revision 1.30  2000/03/06 01:06:18  moz
 * Show arrow button in spline icon.
 *
 * Revision 1.29  2000/03/04 20:49:24  moz
 * Fonts should be scaled by screen_ppi.
 *
 * Revision 1.28  2000/02/25 00:16:22  moz
 * Set node fillcolour and fillstyle.
 *
 * Revision 1.27  2000/02/04 01:55:26  moz
 * switch_icons(): don't rejig all icons if resulting from
 * object selection.
 *
 * Revision 1.26  2000/02/01 16:52:20  moz
 * switch_icons() fixes.
 *
 * Revision 1.25  2000/01/28 17:32:12  moz
 * Inline cosround and sinround.
 *
 * Revision 1.24  2000/01/25 01:37:36  moz
 * Don't set a border.
 *
 * Revision 1.23  2000/01/21 11:50:55  moz
 * Removed dead comment.
 *
 * Revision 1.22  1999/11/15 23:01:21  moz
 * Back out window groups change.
 *
 * Revision 1.21  1999/11/15 02:00:55  moz
 * Rounded sin and cos for rotation.
 * Name change.
 * Set window group hint.
 *
 * Revision 1.20  1999/07/04 22:11:34  moz
 * Place view icons correctly for more than one view.
 *
 * Revision 1.19  1999/06/30 15:32:59  moz
 * Updated the view icon sizes in switch_icons().
 *
 * Revision 1.18  1999/05/22 02:49:08  moz
 * Fixes for redrawing compounds.
 *
 * Revision 1.17  1999/05/19 17:06:43  moz
 * 1.0 Checkin.
 *
 * Revision 1.16  1999/05/04 15:23:51  moz
 * Moved snap() from obsolete grid.c
 *
 * Revision 1.15  1999/05/04 14:40:58  moz
 * Allow arrow dialog for arc ellipses.
 *
 * Revision 1.14  1999/05/01 23:21:08  moz
 * Changes to dynamic view_icon_window.
 *
 * Revision 1.13  1999/04/29 01:35:10  moz
 * Make sure we don't try to switch icons to a user-defined colour.
 *
 * Revision 1.12  1999/04/29 00:26:17  moz
 * Removed node box icon.
 *
 * Revision 1.11  1999/04/29 00:12:55  moz
 * Disabled NO_INTERSECT in ellipse_box as it's broken.
 *
 * Revision 1.10  1999/04/28 03:29:39  moz
 * coredump fix in switch_icons.
 *
 * Revision 1.9  1999/04/27 23:55:25  moz
 * switch_icons() extended to set view icons from selected object.
 *
 * Revision 1.8  1999/04/27 18:48:19  moz
 * Enable colour change for TEXT objects.
 *
 * Revision 1.7  1999/04/27 16:00:51  moz
 * Flip angles in x and y axis.
 *
 * Revision 1.6  1999/04/27 04:24:30  moz
 * draw_view parameters change.
 *
 * Revision 1.5  1999/04/27 04:12:44  moz
 * Remove extraneous variable.
 *
 * Revision 1.4  1999/04/23 00:40:13  moz
 * redraw_view_window change.
 *
 * Revision 1.3  1999/04/04 01:47:03  moz
 * Merge multiple store redraw requests before sending.
 *
 * Revision 1.2  1999/04/03 23:13:51  moz
 * Don't show Set Spline icon when in Pointer mode (editing).
 *
 * Revision 1.1  1999/03/30 00:07:04  moz
 * Initial revision
 *
 */    

#include <stdlib.h> 
#include <pwd.h>
#include <sys/types.h>
#include <unistd.h> 
#include <string.h> 
#include <math.h> 
 
#include "include/figurine.h" 
#include "include/extern.h" 

/* for use of store_redraw_object */  
static Boolean storedflag = FALSE; 
static long slw = 0;
static long sjs = JoinRound;
static long sfarrow_w = 0;
static long sfarrow_h = 0;
static long sbarrow_w = 0;
static long sbarrow_h = 0;
static VRegion sbbox; 
static Boolean scompound=FALSE; 

void 
free_list(void *v)
{
	freelist = add_to_list(freelist,0,0,v);
}

/* set X clipping region  */  
inline void
set_clip(GC agc, int x, int y, int w, int h)
{
	static XRectangle rectangles[1];

	rectangles[0].x = 0;
	rectangles[0].y = 0;
	rectangles[0].width = w;
	rectangles[0].height = h;
	
	XSetClipRectangles(display,agc, x, y, rectangles, 1, Unsorted);
}

void 
set_clip_v(GC agc, View *v, VRegion bbox)
{
	XRectangle rectangles[1];

	rectangles[0].x = 0;
	rectangles[0].y = 0;
	rectangles[0].width = XD2P(bbox.x2,v)-XD2P(bbox.x1,v)+1;
	rectangles[0].height = YD2P(bbox.y2,v)-YD2P(bbox.y1,v)+1;
	XSetClipRectangles(display,agc, XD2P(bbox.x1,v), YD2P(bbox.y1,v), rectangles, 1, Unsorted);
} 

/* determine user's $HOME  */  
void 
home_directory(char *result)
{
	uid_t uid;
	struct passwd *password = NULL;
	char *tmp;

	if ((tmp=getenv("HOME"))!=NULL)
		{
		strcpy(result, tmp);
		return; 
		}
	else
		{
		if ((tmp=getenv("USER"))!=NULL)
			{
			password = getpwnam(tmp);
			}
		else
			{
			uid = getuid();
			password = getpwuid(uid);
			};
		};

	if (password)
		{
		strcpy(result,password->pw_dir);
		return;
		};
	
	*result = '\0';
}
	
inline Boolean 
is_in_box(int px, int py, int x, int y, int w, int h)
{
	return (px>=x && px <=x+w && py>=y && py<=y+h);
} 

inline Boolean
is_in_bbox(int px, int py, int x1, int y1, int x2, int y2)
{
	return (px>=x1 && px<=x2 && py>=y1 && py<=y2);
}
 
inline Boolean
is_in_object(long px, long py, Object *ob)
{
	return (px>=ob->bbox.x1 && px<=ob->bbox.x2 && py>=ob->bbox.y1 && py<=ob->bbox.y2);
}
 
Boolean 
is_in_object_p(View *v, long x, long y, Object *ob)
{
	return is_in_object(XP2D(x,v), YP2D(y,v), ob);
}

/* usage is_contained(outer_box, inner_box)  */  
inline Boolean
is_contained(int x1, int y1, int x2, int y2,int x3, int y3, int x4, int y4)
{
	return (is_in_bbox(x3,y3,x1,y1,x2,y2) && is_in_bbox(x4,y4,x1,y1,x2,y2));
} 

/* usage bbox = merge_boxes(box1, box2)  */ 
/* box1 box2 must both be normalised  */  
VRegion 
merge_boxes(VRegion b1, VRegion b2)
{
	VRegion r;
	
	/* this works because b1,b2 are normalised, right ? */  
	r.x1 = min(b1.x1, b2.x1);
	r.y1 = min(b1.y1, b2.y1);
	r.x2 = max(b1.x2, b2.x2);
	r.y2 = max(b1.y2, b2.y2);

	return r;
}

inline Boolean
is_contained_v(VRegion a, VRegion b)
{
	return (is_in_bbox(b.x1,b.y1,a.x1,a.y1,a.x2,a.y2) && is_in_bbox(b.x2,b.y2,a.x1,a.y1,a.x2,a.y2));
}

/* determine whether two boxes intersect  */  
inline Boolean
intersects(int x1, int y1, int x2, int y2,int x3, int y3, int x4, int y4)
{
	/* first test if a corner is in the other rectangle */  
	if (is_in_bbox(x1,y1,x3,y3,x4,y4))
		return TRUE;
	
	if (is_in_bbox(x2,y2,x3,y3,x4,y4))
		return TRUE;

	if (is_in_bbox(x1,y2,x3,y3,x4,y4))
		return TRUE;
	
	if (is_in_bbox(x2,y1,x3,y3,x4,y4))
		return TRUE;

	/* if contained, TRUE  */ 
	 
	if (is_contained(x1,y1,x2,y2,x3,y3,x4,y4))
		return TRUE;
	
	if (is_contained(x3,y3,x4,y4,x1,y1,x2,y2))
		return TRUE;
	 
	/* better double check for other rectangle */   
	if (is_in_bbox(x3,y3,x1,y1,x2,y2))
		return TRUE;
	
	if (is_in_bbox(x4,y4,x1,y1,x2,y2))
		return TRUE;

	if (is_in_bbox(x3,y4,x1,y1,x2,y2))
		return TRUE;
	
	if (is_in_bbox(x4,y3,x1,y1,x2,y2))
		return TRUE;

	/* if contained, TRUE  */ 
	 
	if (((x3>=x1 && x3<=x2) || (x4>=x1 && x4<=x2)) && y3<=y1 && y4>=y2)
		return TRUE;
	
	if (((y3>=y1 && y3<=y2) || (y4>=y1 && y4<=y2)) && x3<=x1 && x4>=x2)
		return TRUE;
	
	return FALSE;
}

Boolean 
intersects_v(VRegion v1, VRegion v2)
{
	return intersects(v1.x1,v1.y1,v1.x2,v1.y2, v2.x1,v2.y1,v2.x2,v2.y2);
}

/* normalised rectangle !!!  */ 
/* forces a point to be in the box x1,y1-x2,y2  */  
void 
force_in_box(int *xp, int *yp, int x1, int y1, int x2, int y2)
{
	if (*xp < x1 || *xp > x2)
		*xp = (x1+x2)/2;

	if (*yp < y1 || *yp > y2)
		*yp = (y1+y2)/2;

}

void 
force_in_box_l(long *xp, long *yp, long x1, long y1, long x2, long y2)
{
	*xp = max(x1,*xp);
	*xp = min(x2,*xp);
	*yp = max(y1,*yp);
	*yp = min(y2,*yp);
	
}

/* simple equation of line */  
signed long 
line_x_at_y(long x1, long y1, long x2, long y2, long y)
{
	return (signed long)( ((((double)(x2-x1))/(y2-y1))*(y-y1))+x1);
}

/* simple equation of line */  
signed long 
line_y_at_x(long x1, long y1, long x2, long y2, long x)
{
	return (signed long)( ((((double)(y2-y1))/(x2-x1))*(x-x1))+y1);
}
 
/* rounded sin and cos useful for us*/  
inline double
sinround(double angle)
{
	double a=sin(angle);

	if (between(a,-0.03,0.03))
		return 0.0;
	
	if (between(a,0.97,1.0))
		return 1.0;

	if (between(a,-1.0,-0.97))
		return -1.0;
	
	return a;	
}
 
inline double
cosround(double angle)
{
	double a=cos(angle);

	if (between(a,-0.03,0.03))
		return 0.0;
	
	if (between(a,0.97,1.0))
		return 1.0;

	if (between(a,-1.0,-0.97))
		return -1.0;
	
	return a;
}
 
/* flip an angle in the x axis */   
double
flip_angle_x(double angle)
{
	double x,y;

	x = - cosround(angle); 
	y = sinround (angle);

	angle = atan2(y,x);
	 
	while (angle>2*PI)
		angle -= 2*PI;
	
	while (angle<0)
		angle += 2*PI;
	
	return angle;
}

/* flip an angle in the y axis */   
double
flip_angle_y(double angle)
{
	double x,y;

	x = cosround(angle); 
	y = - sinround(angle);

	angle = atan2(y,x);
	 
	while (angle>2*PI)
		angle -= 2*PI;
	
	while (angle<0)
		angle += 2*PI;
	
	return angle;
}

/* normalise a rectangle described by two points  */  
void 
normalise_rectangle(long *x1, long *y1, long *x2, long *y2)
{
	long a;

	if (*x1<=*x2 && *y1<=*y2) /* line from tl to br  */ 
		return;
	
	if (*x1>=*x2 && *y1>=*y2) /* line from br to tl  */ 
		{
		a = *x2;
		*x2 = *x1;
		*x1 = a;
		a = *y2;
		*y2 = *y1;
		*y1 = a;
		return;
		};
	
	if (*x1<=*x2 && *y1>=*y2) /* line from bl to tr  */ 
		{
		a = *y1; 
		*y1 = *y2;
		*y2 = a;
		return;
		};

	if (*x1>=*x2 && *y1<=*y2) /* line from tr to bl */  
		{
		a = *x1;
		*x1 = *x2;
		*x2 = a;
		return;
		};
	
}
 
/* send a redraw for a section */  
void 
send_redraw(WindowStruct *window, int x, int y, int w, int h)
{
	XEvent ev;

	set_clip(ugc, x, y, w, h); 
	set_clip(handlegc, x, y, w, h); 
	set_clip(blackxorgc, x, y, w, h); 
	set_clip(dashgc, x, y, w, h); 
	set_clip(fillgc, x, y, w, h); 
	 
	ev.type = Expose; 
	ev.xexpose.window = window->win;
	ev.xexpose.x = x;
	ev.xexpose.y = y;
	ev.xexpose.width = w;
	ev.xexpose.height = h;
	ev.xexpose.count = 0;

	XSendEvent(display, window->win, False, 0, &ev);
}

void 
send_redraw_v(View *view, VRegion bb)
{
	int x1,y1,x2,y2;
	List l;

	l = view->doc->views;

	while (l!=NULL)
		{
			
		x1 = XD2P(bb.x1,VIEW(l));
		y1 = YD2P(bb.y1,VIEW(l));
		x2 = XD2P(bb.x2,VIEW(l));
		y2 = YD2P(bb.y2,VIEW(l));
		
		if (intersects(x1,y1,x2,y2,0,0,(int)VIEW(l)->draw_window->w,(int)VIEW(l)->draw_window->h))
			{ 
			x1 = max(0,x1);
			y1 = max(0,y1);
			x2 = min(x2-x1,(int)VIEW(l)->draw_window->w);
			y2 = min(y2-y1,(int)VIEW(l)->draw_window->h);
			
			send_redraw(VIEW(l)->draw_window, x1, y1, x2+1, y2+1);  
			}; 
		l = l->next; 
		};
}
 
/* store shape of first object  */  
void
store_redraw_object(Object *ob)
{
	/* if we already have stored, but not sent, merge the requests */  
	if (storedflag) 
		sbbox = merge_boxes(sbbox,ob->bbox); 
	else 
		sbbox = ob->bbox; 
 
	if (ob->type!=COMPOUND)
		{ 
		slw = ob->lw;
		sjs = ob->js;
			 
		if (ob->farrow!=NULL)
			{
			sfarrow_w = ob->farrow->w;
			sfarrow_h = ob->farrow->h;
			}
		else
			sfarrow_w = 0; /* checked for  */ 

		if (ob->barrow!=NULL)
			{
			sbarrow_w = ob->barrow->w;
			sbarrow_h = ob->barrow->h;
			}
		else
			sbarrow_w = 0; /* checked for  */ 
		}
	else
		scompound=TRUE; 
 
	storedflag = TRUE;
}
  
/* merge ob with value stored above  */  
/* and send redraw  */  
void 
send_stored_redraw_object(View *view, Object *ob)
{
	VRegion bb;
	VRegion sbb; 
	int x1,y1,x2,y2;
	long v =0; 
	List l;
	long lw; 

	storedflag = FALSE; 
	l = view->doc->views;
	 
	while (l!=NULL)
		{
		v = 0; 
		if (ob->type == POLYLINE || ob->type==POLYGON)
			{ 
			lw = I2D((((double)ob->lw)/80.0),VIEW(l));
			/* if join miter, the point can be miles away from bb  */  
			if (ob->js==JoinMiter)
				lw = I2D((((double)ob->lw)/80.0),VIEW(l))*6;
				 
			if (ob->farrow!=NULL)
				v = I2D((((double)max(ob->farrow->w,ob->farrow->h))/80.0),VIEW(l));

			if (ob->barrow!=NULL)
				v = max(v, I2D((((double)max(ob->barrow->w,ob->barrow->h))/80.0),VIEW(l)));
			
			lw += v;
			}
		else
			lw = 0; 
		 
		bb.x1 = ob->bbox.x1-lw;
		bb.y1 = ob->bbox.y1-lw;
		bb.x2 = ob->bbox.x2+lw;
		bb.y2 = ob->bbox.y2+lw;

		/* account for handles in editing mode  */  
		bb.x1 = bb.x1-P2D(HANDLE_PIXEL_SIZE,VIEW(l));
		bb.y1 = bb.y1-P2D(HANDLE_PIXEL_SIZE,VIEW(l));
		bb.x2 = bb.x2+P2D(HANDLE_PIXEL_SIZE,VIEW(l));
		bb.y2 = bb.y2+P2D(HANDLE_PIXEL_SIZE,VIEW(l));
			 
		v = 0;
		if (!scompound)
			{ 
			lw = I2D((((double)slw)/80.0),VIEW(l));
			/* if join miter, the point can be miles away from bb  */  
			if (sjs==JoinMiter)
				lw = I2D((((double)slw)/80.0),VIEW(l))*6;
				 
			if (sfarrow_w!=0)
				v = I2D((((double)max(sfarrow_w,sfarrow_h))/80.0),VIEW(l));
			if (sbarrow_w!=0)
				v = I2D((((double)max(sbarrow_w,sbarrow_h))/80.0),VIEW(l));

			lw += v;
			}
		else
			lw = I2D(1,VIEW(l)); 
		 
		sbb.x1 = sbbox.x1-lw;
		sbb.y1 = sbbox.y1-lw;
		sbb.x2 = sbbox.x2+lw;
		sbb.y2 = sbbox.y2+lw;

		/* account for handles in editing mode  */  
		sbb.x1 = sbb.x1-P2D(HANDLE_PIXEL_SIZE,VIEW(l));
		sbb.y1 = sbb.y1-P2D(HANDLE_PIXEL_SIZE,VIEW(l));
		sbb.x2 = sbb.x2+P2D(HANDLE_PIXEL_SIZE,VIEW(l));
		sbb.y2 = sbb.y2+P2D(HANDLE_PIXEL_SIZE,VIEW(l));
	
		bb = merge_boxes(bb,sbb);

		x1 = XD2P(bb.x1,VIEW(l));
		y1 = YD2P(bb.y1,VIEW(l));
		x2 = XD2P(bb.x2,VIEW(l));
		y2 = YD2P(bb.y2,VIEW(l));

		if (intersects(x1,y1,x2,y2,0,0,(int)VIEW(l)->draw_window->w,(int)VIEW(l)->draw_window->h))
			{
			x1 = max(0,x1);
			y1 = max(0,y1);
			x2 = min(x2-x1,(int)VIEW(l)->draw_window->w);
			y2 = min(y2-y1,(int)VIEW(l)->draw_window->h);

			send_redraw(VIEW(l)->draw_window, x1, y1, x2+1, y2+1);               
			}; 
		   
		l = l->next;
		};
	scompound=FALSE; 
}

/* redraw the whole view window  */  
void
redraw_view_window(View *view)
{
	XEvent xev;

	/* remove queued redraw events on this window  */  
	while (XCheckTypedWindowEvent(display, view->draw_window->win, Expose, &xev))
		; 

	draw_view(view,0,0,(long)view->draw_window->w,(long)view->draw_window->h);
}

/* redraw an object, taking account
	of object properties */  
void 
send_redraw_object(View *view, Object *ob)
{
	VRegion bb;
	int x1,y1,x2,y2;
	long v =0; 
	List l;
	long lw=0; 

	l = view->doc->views;
	 
	while (l!=NULL)
		{
		if (ob->type == POLYLINE || ob->type==POLYGON)
			{ 
			lw = I2D((((double)ob->lw)/80.0),VIEW(l));
			/* if join miter, the point can be miles away from bb  */  
			if (ob->js==JoinMiter)
				lw = I2D((((double)ob->lw)/80.0),VIEW(l))*6;
				 
			if (ob->farrow!=NULL)
				v = I2D((((double)max(ob->farrow->w,ob->farrow->h))/80.0),VIEW(l));

			if (ob->barrow!=NULL)
				v = max(v, I2D((((double)max(ob->barrow->w,ob->barrow->h))/80.0),VIEW(l)));
			
			lw += v;
			} 
		else
			lw = 0; 

		bb.x1 = ob->bbox.x1-lw;
		bb.y1 = ob->bbox.y1-lw;
		bb.x2 = ob->bbox.x2+lw;
		bb.y2 = ob->bbox.y2+lw;
 
		/* account for handles in editing mode  */  
		bb.x1 = bb.x1-P2D(HANDLE_PIXEL_SIZE,VIEW(l));
		bb.y1 = bb.y1-P2D(HANDLE_PIXEL_SIZE,VIEW(l));
		bb.x2 = bb.x2+P2D(HANDLE_PIXEL_SIZE,VIEW(l));
		bb.y2 = bb.y2+P2D(HANDLE_PIXEL_SIZE,VIEW(l));
			 
		x1 = XD2P(bb.x1,VIEW(l));
		y1 = YD2P(bb.y1,VIEW(l));
		x2 = XD2P(bb.x2,VIEW(l));
		y2 = YD2P(bb.y2,VIEW(l));

		if (intersects(x1,y1,x2,y2,0,0,(int)VIEW(l)->draw_window->w,(int)VIEW(l)->draw_window->h))
			{ 
			x1 = max(0,x1);
			y1 = max(0,y1);
			x2 = min(x2-x1,(int)VIEW(l)->draw_window->w);
			y2 = min(y2-y1,(int)VIEW(l)->draw_window->h);

			send_redraw(VIEW(l)->draw_window, x1, y1, x2+1, y2+1);               
			}; 
		l = l->next;
		};
}
		
/* configure a window  */ 
void 
send_configure(WindowStruct *window, int x, int y, int w, int h)
{
	XEvent ev;

	ev.type = ConfigureNotify;
	ev.xconfigure.window = window->win;
	ev.xconfigure.x = x;
	ev.xconfigure.y = y;
	ev.xconfigure.width = w;
	ev.xconfigure.height = h;
	ev.xconfigure.border_width = 0;
	ev.xconfigure.above = None;
	ev.xconfigure.override_redirect = FALSE;

	XSendEvent(display, window->win, False, 0, &ev);
} 

/* set a window cursor type  */  
void 
set_window_cursor(Window win, int c)
{
	Cursor tc;
	XSetWindowAttributes wattr;

	tc = XCreateFontCursor(display,(uint)c);
	wattr.cursor = tc;

	XChangeWindowAttributes(display,win, CWCursor, &wattr);
}

/* create a normal window  */  
Boolean 
create_window(WindowStruct *window, Window parent, List *list, int type, void *data)
{
	List templ; 
	XGCValues gcvalues; 
	
	XGetGCValues(display, window->gc, (GCForeground | GCBackground), &gcvalues);
	  
	window->win = XCreateSimpleWindow(display, parent,
								window->x,window->y,window->w,window->h,
								0, gcvalues.foreground,
								gcvalues.background);
	 
	set_window_cursor(window->win,window->cursor_num);


	if ((window->wm_hints = XAllocWMHints())!=NULL)
		{
		window->wm_hints->initial_state = NormalState;
		window->wm_hints->input = True;
		window->wm_hints->flags = StateHint | InputHint;             
		XSetWMHints(display,window->win, window->wm_hints); 
		};
	if ((window->size_hints = XAllocSizeHints())!=NULL)
		{
		window->size_hints->flags = PPosition | PSize | PMinSize;
		window->size_hints->min_width = 50;
		window->size_hints->min_height = 50;
		XSetWMNormalHints(display,window->win,window->size_hints); 
		}; 

	XSelectInput(display, window->win, KeyPressMask | ButtonPressMask 
					 | ButtonReleaseMask | EnterWindowMask | LeaveWindowMask
					 | PointerMotionMask | ButtonMotionMask
					 | StructureNotifyMask
					 | ExposureMask | StructureNotifyMask | FocusChangeMask);

	if (parent==RootWindow(display,screen))
		XSetWMProtocols(display,window->win, &windelete, 1);

	XMoveResizeWindow(display, window->win, window->x, window->y, window->w, window->h); 
	 
	templ = add_to_list(*list,(ulong)window->win,(ulong)type,data);  
	 
	if (templ==NULL)
		return FALSE;
	
	*list = templ;
	  
	return TRUE;  
}

/* set the window title  */  
void 
set_window_name(WindowStruct *window,char *name)
{
	XTextProperty textprop;

	XStringListToTextProperty(&name,1,&textprop);

	XSetWMName(display,window->win, &textprop);             
} 

/* constrain a line to 90 degrees  */  
inline void
constrain_line(long x1, long y1, long *x2, long *y2)
{
	((abs(*x2-x1))>(abs(*y2-y1))) ? (*y2 = y1) : (*x2 = x1); 
} 

/* constrain proportionally  */ 
inline void
constrain_resize(double *xr, double *yr)
{
	(*xr>*yr) ? (*yr= *xr) : (*xr = *yr); 
} 

/* constrain proportionally  */ 
inline void
constrain_ellipse(long *xr, long *yr)
{
	(*xr>*yr) ? (*yr= *xr) : (*xr = *yr); 
} 

/* constrain proportionally  */ 
inline void
constrain_rectangle(long x1, long y1, long *x2, long *y2)
{
	((abs(*x2-x1))>(abs(*y2-y1))) ? (*y2 = (*x2-x1+y1)) : (*x2 = (*y2-y1+x1)); 
}

void 
deselect_everything()
{
	List l;
	Boolean redraw=FALSE; 
	Object *ob=NULL; 

	l = state.views;

	while (l!=NULL)
		{
		if (VIEW(l)->selected_object!=NULL)
			{
			redraw = TRUE;
			ob = VIEW(l)->selected_object->ob;
			VIEW(l)->selected_object = NULL; 
			};

		if (VIEW(l)->highlighted_object!=NULL)
			{
			redraw = TRUE;
			ob = VIEW(l)->highlighted_object->ob;
			VIEW(l)->highlighted_object = NULL; 
			};

		if (VIEW(l)->edited_object!=NULL)
			{
			redraw = TRUE;
			ob = VIEW(l)->edited_object->ob;
			};

		if (redraw)
			send_redraw_object(VIEW(l), ob);
	
		VIEW(l)->edited_object = NULL;
		 
		redraw = FALSE;

		l = l->next;
		};

}

/* is a point x,y in an ellipse  */  
Boolean 
point_in_ellipse(long cx, long cy, long a, long b, long x, long y)
{
	double xda = ((double)(x-cx))/((double)a); 
	double ydb = ((double)(y-cy))/((double)b); 
	
	return ((xda*xda)+(ydb*ydb) < 1.0);
}

/* does a box and an ellipse intersect  */  
int 
ellipse_box(long cx, long cy, long a, long b, long x1, long y1, long x2, long y2)
{
   /* returns CONTAINS if ellipse contains box */
   /* returns CONTAINED if box contains ellipse or intersects */
	
   if (point_in_ellipse(cx,cy,a,b,x1,y1) && point_in_ellipse(cx,cy,a,b,x2,y1) &&
		 point_in_ellipse(cx,cy,a,b,x1,y2) && point_in_ellipse(cx,cy,a,b,x2,y2))
		return CONTAINS;

	return CONTAINED;
}

/* get a font of specified size and type  */  
/* nb point size is *scaled* point size  */  
VFont *
get_font(int fontnumber, int pointsize)
{
	char name[100];
	char a[10];
	int realsize;
	VFont *vf; 
	List l;
	char name2[100];
	char **availfonts;
	int bfontsize=0;
	int numf=0;
	
	if (pointsize < 20) 
		return NULL; 
	/* i.e. greek the text, no font  */  

	if (pointsize >3000)
		return NULL;

	l = afonts;

	while (l)
		{
		if (fontnumber==FONT(l)->num && FONT(l)->size==pointsize)
			break;
		l = l->next;
		};
	
	if (l)
		return FONT(l);
	 
	/* this code is set up to get scaled bitmap fonts as per Nye A.3.2 */
	/* they look terrible but that's X for you */ 
	 
	/* -adobe-courier-bold-o-normal-*-10-100-75-75-m-60-iso8859-1 */
	/* -foundry-family-weight-slant-setwidth-"unknown"-pixels-10thpoints-hdpi-vdpi-spacing-avewidth-charset */
	vf = (VFont *)malloc(sizeof(VFont));
	vf->num = fontnumber;
	vf->size = pointsize;
	strcpy(name,fontnames[fontnumber][0]);
	realsize = ((int)((pointsize/(POINTS_IN_INCH))*screen_ppi)); 
	sprintf(a,"*-*-%d",realsize);
	strcat(name,a); 
	/* REQUIRES the iso8859-1 to load */ 
	strcat(name,"-75-75-*-*-iso8859-1");
	 
	/* assignment used as truth value :) */ 
	if ((vf->x=XLoadQueryFont(display,name)))
		{ 
		afonts = add_to_list(afonts,0,0,(void *)vf);
		return vf;
		}; 
		 
	/* if we get here, font server won't scale bitmaps or font really isn't available */ 
	/* there are a surprising number of font servers out there not doing this :( */ 
	
	/* search for the nearest */
			
	strcpy(name2,fontnames[fontnumber][0]);
	strcat(name2,"*-*-*-*-*-*-*-iso8859-1");
	/* not ordered by size */ 
	availfonts = XListFonts(display, name2, 200, &numf);
	if (availfonts)
		{
		while (numf)
			{
			char *f = *(availfonts+numf-1);
			int asize;
			char *g;
			 
			 
			while (*f!='\0' && (*f!='-' || *(f+1)!='-'))
				f++;

			if (*f=='\0')
				continue;

			/* -- */
			f+=2;
			/* skip pixel size */
			while(*f!='\0' && *f!='-')
				f++;

			if (*f=='\0')
				continue;

			/* - */ 
			g = ++f;

			while (*g!='\0' && *g!='-')
				g++;

			*g='\0';
			/* pointsize */ 
			sscanf(f,"%d",&asize);
			/* pick biggest smaller than realsize */ 
			if (asize<=realsize && asize>bfontsize)
				bfontsize=asize;
			 
			numf--;
			};

		if (bfontsize)
			{
			char name3[10]; 
			/* XListFonts returns stuff like "-adobe-times-medium-r-normal--12-120" */
			strcpy(name2,fontnames[fontnumber][0]);
			sprintf(name3,"*-*-%d",bfontsize);
			strcat(name2,name3);
			strcat(name2,"-*-*-*-*-iso8859-1");
			if ((vf->x=XLoadQueryFont(display,name2))!=NULL)
				{
				/* if the font is way out, warn user */ 
				if (abs(bfontsize-realsize)>100) 
					fprintf(stderr,"figurine: warning: using font %s\nfigurine:instead of %s\n",name2,name);

				XFreeFontNames(availfonts);
				afonts = add_to_list(afonts,0,0,(void *)vf);
				return vf;
				};
			};
		/* failure */ 
		XFreeFontNames(availfonts);
		};         			       
	 
	fprintf(stderr, "figurine: Could not load font %s\n", name);
	free(vf);
	 
	/* careful not to infinitely recurse */ 
	if (fontnumber!=0 && fontnumber!=35) 
		return get_font(0, pointsize); /* first try times replica  */ 
	 
	return NULL; /* ok, have to greek  */ 
}

/* switch the icons on every view window to reflect current
	drawing mode / object */  
	
#define off(a) stk_hide_icon((a),v->view_icon_window->win)
#define onx(a) do { stk_show_icon((a),v->view_icon_window->win); \
		stk_set_x((a),v->view_icon_window->win, x); \
		x += VIEW_ICON_WIDTH; } while (0)
	

void 
switch_icons(View *us)
{
	List l;
	View *v;
	int a=state.current_icon; 
	IconType temp; 
	int x;

	l = state.views;

	while (l!=NULL)
		{
		v = VIEW(l);
		x=4*VIEW_ICON_WIDTH; 
		 
		if (v!=us && us->selected_object && state.current_icon==POINTERICON)  
			{
			l=l->next;
			continue;
			};

		off(LINESTYLEICON); 
		off(COLOURICON); 
		off(FILLSTYLEICON); 
		off(FILLCOLOURICON); 
		off(LINEWIDTHICON); 
		off(ENDSTYLEICON);
		off(JOINSTYLEICON); 
		off(ARROWICON);
		off(SETPOLYGONICON); 
		off(FONTICON);
		off(JUSTIFICATIONICON); 
		off(SETSPLINEICON);
		off(ROUNDBOXICON); 
		off(SETARCELLIPSEICON); 
		
		if (state.current_icon!=POINTERICON || !v->selected_object)
			a = state.current_icon;
		
		if (state.current_icon==POINTERICON && v->selected_object!=NULL)
			{
			switch (v->selected_object->ob->type)
				{ 
				case ARCELLIPSE: a = ARCELLIPSEICON; break;
				case ELLIPSE: a = ELLIPSEICON; break;
				case POLYLINE: a = POLYLINEICON; break;
				case POLYGON: a = POLYGONICON; break;
				case SPLINE: 
				case ARC: 
					a = SPLINEICON;
					break;
				 
				case TEXT:
					if (v->selected_object->ob->ob.text.node)
						{
						onx(LINESTYLEICON); 
						onx(COLOURICON); 
						onx(FILLSTYLEICON); 
						onx(FILLCOLOURICON); 
						onx(LINEWIDTHICON); 
						} 
					else
						{ 
						a = 0; 
						onx(COLOURICON); 
						}; 
					break;


				case ROUNDBOX: 
					onx(LINESTYLEICON);
					onx(COLOURICON);
					onx(FILLSTYLEICON);
					onx(FILLCOLOURICON);
					onx(LINEWIDTHICON);
					onx(ROUNDBOXICON);
					break; 
		
				default: a = state.current_icon; break;
				};
			}; 
	 
		switch(a)
			{
			case 0: break; 
			case COMPOUNDICON: break;

			case POLYLINEICON:
				onx(LINESTYLEICON); onx(COLOURICON);
				onx(LINEWIDTHICON); onx(ENDSTYLEICON);
				onx(JOINSTYLEICON); onx(ARROWICON);
				break;

			case ARCELLIPSEICON:
				onx(LINESTYLEICON); onx(COLOURICON);
				onx(FILLSTYLEICON); onx(FILLCOLOURICON);
				onx(LINEWIDTHICON); onx(ENDSTYLEICON);
				onx(SETARCELLIPSEICON); onx(ARROWICON);
				break;

			case ELLIPSEICON:
			case BOXELLIPSEICON:
				onx(LINESTYLEICON); onx(COLOURICON);
				onx(FILLSTYLEICON); onx(FILLCOLOURICON);
				onx(LINEWIDTHICON);
				break;

			case RECTANGLEICON:
				onx(LINESTYLEICON); onx(COLOURICON);
				onx(FILLSTYLEICON); onx(FILLCOLOURICON);
				onx(LINEWIDTHICON); onx(JOINSTYLEICON);
				onx(ROUNDBOXICON);
				break; 
			
			case POLYGONICON: 
				onx(LINESTYLEICON); onx(COLOURICON);
				onx(FILLSTYLEICON); onx(FILLCOLOURICON);
				onx(LINEWIDTHICON); onx(JOINSTYLEICON);
				if (state.current_icon!=POINTERICON) 
					onx(SETPOLYGONICON);
				break; 
				 
			case SPLINEICON:
				/* yuk */  
				onx(LINESTYLEICON); onx(COLOURICON);
				if (state.current_icon!=POINTERICON || v->selected_object)
					{ 
					onx(FILLSTYLEICON); 
					onx(FILLCOLOURICON);
					}; 
				onx(LINEWIDTHICON); 
				if (state.current_icon!=POINTERICON || v->selected_object)
					{ 
					onx(ENDSTYLEICON);
					onx(ARROWICON);
					}; 
				if (state.current_icon!=POINTERICON) 
					onx(SETSPLINEICON);
				break;

			case TEXTICON:
				onx(COLOURICON); onx(FONTICON);
				onx(JUSTIFICATIONICON);
				break; 

			case NODEICON:
				onx(LINESTYLEICON); onx(COLOURICON);
				onx(FILLSTYLEICON); onx(FILLCOLOURICON);
				onx(LINEWIDTHICON); onx(FONTICON);
				onx(JUSTIFICATIONICON);
				break; 
			
			case ARCICON:
				onx(LINESTYLEICON); onx(COLOURICON);
				onx(LINEWIDTHICON); onx(ENDSTYLEICON);
				onx(ARROWICON);
				onx(FONTICON); onx(JUSTIFICATIONICON);
				break; 
			}; 
			 
		if (v->selected_object!=NULL)
			{
			Object *ob = v->selected_object->ob;
			
			/* linestyle  */  
			if (ob->type!=TEXT && ob->type!=COMPOUND)
				{
				switch((signed)ob->ls)
					{
					case SOLID: temp=SOLIDICON; break; 
					case DASHED: temp=DASHEDICON; break; 
					case DOTTED: temp=DOTTEDICON; break; 
					case DASH_DOTTED: temp=DASHDOTTEDICON; break; 
					case DASH_DOUBLE_DOTTED: temp=DASHDOUBLEDOTTEDICON; break; 
					case DASH_TRIPLE_DOTTED: temp=DASHTRIPLEDOTTEDICON; break; 
					case -1: temp=SOLIDICON; break; 
					default: temp=0; break; 
					};
				stk_swap_pixmap(LINESTYLEICON, v->view_icon_window->win,(signed)temp,linestyle_menu);
				}; 

			/* endstyle  */  
			if (ob->type==POLYLINE || ((ob->type==SPLINE || ob->type==ARC) && !ob->ob.spline.closed))
			 	{
				switch(ob->es)
					{
					case CapRound: temp=ROUNDENDICON; break;
					case CapButt: temp=FLATENDICON; break;
					case CapProjecting: temp=PROJECTINGENDICON; break;
					default: temp=0; break;
					};
				stk_swap_pixmap(ENDSTYLEICON, v->view_icon_window->win,(signed)temp,endstyle_menu);
				};

			/* joinstyle  */  
			if (ob->type==POLYGON || ob->type==POLYLINE)
			 	{
				switch(ob->js)
					{
					case JoinRound: temp=ROUNDJOINICON; break;
					case JoinMiter: temp=SHARPJOINICON; break;
					case JoinBevel: temp=BEVELJOINICON; break;
					default: temp=0; break; 
					};
				stk_swap_pixmap(JOINSTYLEICON, v->view_icon_window->win,(signed)temp,joinstyle_menu);
				};

			/* colour  */  
			if (ob->type==ARCELLIPSE || ob->type==POLYGON ||
				ob->type==ROUNDBOX || ob->type==ELLIPSE || 
				ob->type==POLYLINE || ob->type==TEXT ||
				ob->type==SPLINE || ob->type==ARC)
				{
				if (ob->colour<332) 
					{ 
					stk_swap_pixmap(COLOURICON, v->view_icon_window->win,(signed)ob->colour,colour_menu); 
					v->colour = ob->colour;
					}; 
				};
			
			/* fill colour  */  
			if (ob->type==ARCELLIPSE || ob->type==POLYGON ||
				 ob->type==ROUNDBOX || ob->type==ELLIPSE || 
				 ob->type==SPLINE || ob->type==ARC)
				{
				if (ob->fillcolour<432) 
					{ 
					stk_swap_pixmap(FILLCOLOURICON, v->view_icon_window->win,(signed)ob->fillcolour,fillcolour_menu); 
					v->fillcolour = ob->fillcolour;
					}; 
				};
			
			/* fill style  */  
			if (ob->type==ARCELLIPSE || ob->type==POLYGON ||
				 ob->type==ROUNDBOX || ob->type==ELLIPSE || 
				 ob->type==SPLINE || ob->type==ARC)
				{
				if (ob->fs==-1)
					stk_swap_pixmap(FILLSTYLEICON, v->view_icon_window->win, 243, fillstyle_menu);
				else 
					stk_swap_pixmap(FILLSTYLEICON, v->view_icon_window->win, STARTOFFILLSTYLES+ob->fs, fillstyle_menu);
				v->fillstyle = ob->fs;
				};
				
			/* node special cases */  
			if (ob->type==TEXT && ob->ob.text.node)
				{
				if (ob->fillcolour<432)
					{
					stk_swap_pixmap(FILLCOLOURICON, v->view_icon_window->win,
							(signed)ob->ob.text.ellipse->fillcolour, fillcolour_menu);
					v->fillcolour = ob->ob.text.ellipse->fillcolour;
					};
				
				if (ob->ob.text.ellipse->fs==-1)
					stk_swap_pixmap(FILLSTYLEICON, v->view_icon_window->win, 243, fillstyle_menu);
				else
					stk_swap_pixmap(FILLSTYLEICON, v->view_icon_window->win, 
							STARTOFFILLSTYLES+ob->ob.text.ellipse->fs, fillstyle_menu);
				v->fillstyle = ob->ob.text.ellipse->fs;
				};

			/* arcellipse type  */  
			if (ob->type==ARCELLIPSE)
				{
				if (ob->ob.arcellipse.open) 
					stk_swap_pixmap(SETARCELLIPSEICON, v->view_icon_window->win, OPENARCELLIPSEICON, setarcellipse_menu);
				else
					stk_swap_pixmap(SETARCELLIPSEICON, v->view_icon_window->win, CLOSEDARCELLIPSEICON, setarcellipse_menu);
				v->openarc = ob->ob.arcellipse.open; 
				};
			};
		l = l->next; 
		};

}

/* calc a mid point bent by angle  */  
void 
bent_midpoint(long x1,long y1,long x2,long y2,double angle, short *ox, short *oy)
{
	long x3,y3;
	double sina,cosa; 
	
	sina = sin(angle);
	cosa = cos(angle);

	x3 = (x2-x1)/2;
	y3 = (y2-y1)/2;

	*ox = x1 + cosa*x3 - sina*y3;
	*oy = y1 + sina*x3 + cosa*y3;
}

/* snap a value by grid size  */  
long 
snap(long size, long val)
{
	if ((val % size) > (size/2))
		return ((val/size)+1)*size;
	else
		return (val/size)*size;
}
