#!/usr/bin/env python
#############################################################################
# Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1997, 1998
# All Rights Reserved.
#
# The software contained on this media is the property of the DSTC Pty
# Ltd.  Use of this software is strictly in accordance with the
# license agreement in the accompanying LICENSE.HTML file.  If your
# distribution of this software does not contain a LICENSE.HTML file
# then you have no rights to use this software in any manner and
# should contact DSTC at the address below to determine an appropriate
# licensing arrangement.
# 
#      DSTC Pty Ltd
#      Level 7, GP South
#      Staff House Road
#      University of Queensland
#      St Lucia, 4072
#      Australia
#      Tel: +61 7 3365 4310
#      Fax: +61 7 3365 4311
#      Email: enquiries@dstc.edu.au
# 
# This software is being provided "AS IS" without warranty of any
# kind.  In no event shall DSTC Pty Ltd be liable for damage of any
# kind arising out of or in connection with the use or performance of
# this software.
#
# Project:      Distributed Environment
# File:         $Source: /cvsroot/fnorb/fnorb/script/fnmetrics.py,v $
# Version:      @(#)$RCSfile: fnmetrics.py,v $ $Revision: 1.1 $
#
#############################################################################
""" IDL metrics generator. """


# Standard/built-in modules.
import commands, os, string, sys

# Fnorb modules.
from Fnorb.orb    import CORBA
from Fnorb.parser import IDLParser
from Fnorb.utils  import MetricsGenerator

# Interface Repository modules.
from Fnorb.cos.interface_repository import IntRepImpl


# Commands to run the C/C++ pre-processor on Unixen and Windows 95/NT.
if sys.platform == 'win32':
    CPP_CMD = 'cl /E %s %s'

# Unixen.
else:
    # Look for the forces of goodness and light first ;^)
    (result, output) = commands.getstatusoutput('which gcc')
    if result == 0:
	CPP_CMD = 'gcc -x c -E %s %s'

    # And then the other stuff!
    else:
	(result, output) = commands.getstatusoutput('which cpp')
	if result == 0:
	    CPP_CMD = 'cpp %s %s'

	else:
	    raise "No C/C++ pre-processor on your $PATH!"


def main(argv):
    """ Do it! """

    # Any argument not starting with '-' is deemed to be an IDL file!
    cpp_flags = []
    idl_files = []

    for arg in argv[1:]:
	if arg[0] == '-':
	    cpp_flags.append(arg)

	else:
	    idl_files.append(arg)

    # If no files were specified on the command line, then parse from stdin!
    if len(idl_files) == 0:
	result = main_interactive()

    else:
	result = main_batch(cpp_flags, idl_files)

    return result


def main_interactive():
    """ Parse IDL from stdin! """

    # Create the parser.
    parser = IDLParser.IDLParser()

    # Create an instance of the 'Repository' implementation class.
    ifr = IntRepImpl.RepositoryImpl()

    # Do the parsing!
    print 'Enter IDL...\n'
    (result, contents) = parser.parse(ifr, 'stdin', sys.stdin)

    # If the parsing succeeded.
    if result == 0:
	if len(contents) > 0:
	    # Create a new metrics instance for the global scope.
	    metrics = MetricsGenerator.ModuleMetrics('Global Scope')

	    # Create the Metrics Generator do the work!
	    mg = MetricsGenerator.MetricsGenerator(metrics)

	    for ir_object in contents:
		mg.generate(None, sys.stdout, None, ir_object, 0)

	    # Write the results on stdout.
	    mg.write_results(sys.stdout)
		
	    # Write the summary on stdout.
	    mg.write_summary(sys.stdout)

    return result


def main_batch(cpp_flags, idl_files):
    """ Parse IDL from files! """

    # Create the parser.
    parser = IDLParser.IDLParser()

    # Parse each file.
    for idl_file in idl_files:
	# Format the command to run the C/C++ pre-processor.
	cmd = CPP_CMD % (string.join(cpp_flags), idl_file)

	# Run the pre-processor and use its output as the lexer's input stream.
	yyin = os.popen(cmd, 'r')

	# Create an instance of the 'Repository' implementation class.
	ifr = IntRepImpl.RepositoryImpl()

	# If there are definitions at the global scope that are NOT IDL module
	# definitions then we will create Python stub and skeleton packages
	# with the same base name as the IDL file (ie. if the IDL file 
	# 'foo.idl' contains definitions at the global scope then we create
	# Python packages 'foo' and 'foo_skel').
	(path, filename) = os.path.split(idl_file)
	(base, ext) = os.path.splitext(filename)

	# Do the parsing!
	(result, contents) = parser.parse(ifr, filename, yyin)

	# If the parsing succeeded.
	if result == 0:
	    if len(contents) > 0:
		# Create a new metrics instance for the global scope.
		metrics = MetricsGenerator.ModuleMetrics('Global Scope')

		# Create the Metrics Generator do the work!
		mg = MetricsGenerator.MetricsGenerator(metrics)

		for ir_object in contents:
		    mg.generate(None, sys.stdout, None, ir_object, 0)

		# Write the results on stdout.
		mg.write_results(sys.stdout)
		
		# Write the summary on stdout.
		mg.write_summary(sys.stdout)

	# If a parsing error occured then bail out.
	else:
	    break

	# Close the pipe.
	yyin.close()

    return result

#############################################################################

if __name__ == '__main__':
    # Do it!
    sys.exit(main(sys.argv))

#############################################################################
