/*
  Copyright 2005, 2006, 2007 David Cad, Damien Stehl.

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; see the file COPYING.  If not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

  This program implements ideas from the paper "Floating-point LLL Revisited", 
  by Phong Nguyen and Damien Stehl, in the Proceedings of Eurocrypt'2005, 
  Springer-Verlag; and was partly inspired by Shoup's NTL library: 
  http://www.shoup.net/ntl/

*/



/* prec=53, eta=0.501, dim < dim_double_max [ (delta / 100.0) + 25 ] */
static const double dim_double_max[75]=
  {0,26,29.6,28.1,31.1,32.6,34.6,34,37.7,38.8,39.6,41.8,40.9,43.6,44.2,47,46.8,50.6,49.1,51.5,52.5,54.8,54.6,57.4,57.6,59.9,61.8,62.3,64.5,67.1,68.8,68.3,69.9,73.1,74,76.1,76.8,80.9,81.8,83,85.3,87.9,89,90.1,89,94.6,94.8,98.7,99,101.6,104.9,106.8,108.2,107.4,110,112.7,114.6,118.1,119.7,121.8,122.9,126.6,128.6,129,133.6,126.9,135.9,139.5,135.2,137.2,139.3,142.8,142.4,142.5,145.4};

static const double eta_dep[10]=
  {1.,//0.5
   1.,//0.55
   1.0521,//0.6
   1.1254,//0.65
   1.2535,//0.7
   1.3957,//0.75
   1.6231,//0.8
   1.8189,//0.85
   2.1025,//0.9
   2.5117};//0.95



inline void present(const char* s, int p)
{
#ifdef VERBOSE
  cerr<<"\n====== LLL method : "<<s;
  if (p)
    cerr<<" (precision = "<<p<<")";
  cerr<<" ======\n\n";
#endif
}
inline void echecpresent(int kappa)
{
#ifdef VERBOSE
  cerr<<"\n====== LLL method end : ";
  if (kappa==0)
    cerr<<"success";
  else if (kappa>0)
    cerr<<"Size-reduction failed. (kappa="<<kappa<<")";
  else
    cerr<<"Too many iterations.";

  cerr<<" ======\n\n";
#endif
}
inline int min(int x,int y)
{
  return x<y?x:y;
}

wrapper::wrapper(ZZ_mat<mpz_t>*B,int precision,double eta_,double delta_)
{
  B1=B;
  eta=eta_;
  delta=delta_;
  int x=B->getMaxExp();
  max_exp=x;
  dim=B->GetNumCols();
  
  double rho = (1.0 + eta) * (1.0+eta) / (delta - eta*eta);
  unsigned int d= B1->GetNumCols();
  unsigned int goodprec = (unsigned int) (7.0 + 0.2*d + d* log(rho)/ log(2.0) 
			     +  2.0 * log ( (double) d )  
			     - log( (eta-0.5)*(1.0-delta) ) / log(2.));
  prec=(precision==0 ? goodprec : precision);
}

bool wrapper::little(int kappa,int precision)
{
  /*one may add here dimension arguments with respect to eta and delta */
  int dm=(int)(delta*100.-25.);
  if (dm<0) dm=0;
  if (dm>74) dm=74;

  int em=(int) ((eta-0.5)*20);
  if (em<0) em=0;
  if (em>9) em=9;
  
  double p=precision<53?53.:((double)precision);
  p/=53.0;
  
  p*=eta_dep[em] ;/*eta dependance*/
  p*=dim_double_max[dm];
#if 0
#ifdef VERBOSE
  cerr << kappa << " compared to " << p << "\n";
#endif
#endif
  return kappa<p;
}

int wrapper::heuristicLoop(int precision)
{
  int kappa;
  if (precision<=PREC_DOUBLE)
    {
      if (max_exp>MAX_EXP_DOUBLE)
	{
	  present("heuristic<mpz_t, dpe_t>",0);
	  kappa=heuristic<mpz_t,dpe_t>(B1,0,eta,delta).LLL();
	  echecpresent(kappa);
	}
      else
	{
	  present("heuristic<mpz_t, double>",0);
	  kappa=heuristic<mpz_t,double>(B1,0,eta,delta).LLL();
	  echecpresent(kappa);
	}
    }
  else
    {
      present("heuristic<mpz_t, mpfr_t>",precision);
      kappa=heuristic<mpz_t,mpfr_t>(B1,precision,eta,delta).LLL();
      echecpresent(kappa);
    }
  if (little(kappa,precision) && kappa!=0)
    return provedLoop(precision);
  else if (kappa!=0 && precision<prec)
    return heuristicLoop(min(precision*2,prec));
  // this point should never be reached
  return -1;
}
int wrapper::provedLoop(int precision)
{
  int kappa;
  if (precision<=PREC_DOUBLE)
    {
      if (max_exp>MAX_EXP_DOUBLE)
	{
	  present("proved<mpz_t, dpe_t>",0);
	  kappa=proved<mpz_t,dpe_t>(B1,0,eta,delta).LLL();
	  echecpresent(kappa);
	}
      else
	{
	  present("proved<mpz_t, double>",0);
	  kappa=proved<mpz_t,double>(B1,0,eta,delta).LLL();
	  echecpresent(kappa);
	}
    }
  else
    {
      present("proved<mpz_t, mpfr_t>",precision);
      kappa=proved<mpz_t,mpfr_t>(B1,precision,eta,delta).LLL();
      echecpresent(kappa);
    }
  
  if (little(kappa,precision) && kappa!=0)
    return kappa;
  else if (kappa!=0 && precision*2<prec)
    return provedLoop(precision*2);
  // this point should never be reached
  return -1;
}

int wrapper::LastLLL()
{
  int kappa;
  if (prec<=PREC_DOUBLE)
    {
      present("proved<mpz_t, dpe_t>(prec<=PREC_DOUBLE)",prec);
      kappa=proved<mpz_t,dpe_t>(B1,prec,eta,delta).LLL();
      echecpresent(kappa);
    }
  else
    {
      present("proved<mpz_t, mpfr_t>",prec);
      kappa=proved<mpz_t,mpfr_t>(B1,prec,eta,delta).LLL();
      echecpresent(kappa);
    }
  return kappa;
}

int wrapper::LLL()
{
  int kappa;
  if (max_exp>MAX_EXP_DOUBLE)
    {
      present("fast<mpz_t, double>",0);
      kappa=fast<mpz_t,double>(B1,0,eta,delta).LLL();
      echecpresent(kappa);
      if (little(kappa,53)&& kappa!=0)
	kappa=heuristicLoop(PREC_DOUBLE);
      else if (kappa!=0)
	kappa=heuristicLoop(min(PREC_DOUBLE*2,prec));
    }
  else
    {
      present("heuristic<mpz_t, double>",0);
      kappa=heuristic<mpz_t,double>(B1,0,eta,delta).LLL();
      echecpresent(kappa);
      if (little(kappa,53) && kappa!=0)
	kappa=provedLoop(PREC_DOUBLE);
      else if (kappa!=0)
	kappa=heuristicLoop(min(PREC_DOUBLE*2,prec));
    }

  return LastLLL();
}
