//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: MeshOfTetrahedra.cpp,v 1.4 2004/04/15 15:21:08 delpinux Exp $

#include <set>
#include <stack>

#include <Types.hpp>
#include <TinyVector.hpp>
#include <TinyMatrix.hpp>

#include <ConnectivityBuilder.hpp>
#include <ConformTransformation.hpp>

#include <MeshOfTetrahedra.hpp>
#include <P1FiniteElement.hpp>

void MeshOfTetrahedra::buildLocalizationTools()
{
  if (this->numberOfVertices()==0) {
    fferr(0) << "Oops: mesh contains no vertices\n";
    std::exit(1);
  }

  __connectivity = new Connectivity<Tetrahedron>(this->numberOfCells());
  ConnectivityBuilder<MeshOfTetrahedra> (*this, *__connectivity, __surfaceMesh);
  if (__surfaceMesh != 0) {
    (*__surfaceMesh).setBackgroundMesh(this);
  }

  // getting bounding box size
  __a = (this->vertex(0));
  __b = __a;
  for (size_t i=1; i<this->numberOfVertices(); ++i) {
    Vertex& x = this->vertex(i);
    for (size_t k=0; k<3; ++k) {
      __a[k] = (__a[k]<x[k])?__a[k]:x[k];
      __b[k] = (__b[k]>x[k])?__b[k]:x[k];
    }
  }
  ffout(3) << "- Bounding box is " << __a << ',' << __b << '\n';

  ffout(3) << "- Building the octree\n";
  __octree = new Octree<size_t>(__a,__b);

  for (size_t i = 0; i < this->numberOfCells(); ++i) {
    ConformTransformationP1Tetrahedron T(this->cell(i));

    TinyVector<3,real_t> X;
    T.value(P1FiniteElement::massCenter(), X);

    (*__octree).add(i,X);
  }
}

MeshOfTetrahedra::const_iterator
MeshOfTetrahedra::find(const double& x,
		       const double& y,
		       const double& z) const
{
  TinyVector<3,real_t> X(x,y,z);
  Octree<size_t>::iterator i = (*__octree).fuzzySearch(X);

  size_t guessedCellNumber = (*i).value();
  const_iterator t0(*this, guessedCellNumber);

  bool found=false;

  std::set<MeshOfTetrahedra::const_iterator> visited;
  std::set<MeshOfTetrahedra::const_iterator> toVisitSet;
  std::stack<MeshOfTetrahedra::const_iterator> toVisitStack;

  toVisitSet.insert(t0);
  toVisitStack.push(t0);
  const_iterator c (*this);

  do {
    // treating next cell
    c = toVisitStack.top();

    toVisitSet.erase(c);
    toVisitStack.pop();
    visited.insert(c);

    const Tetrahedron& t = *c;
    size_t cellNumber = this->cellNumber(t);

    TinyVector<4, real_t> lambda;

    t.getBarycentricCoordinates(X, lambda);

    found = true;
    for (size_t i=0; i<4; ++i) {
#warning Should use a relevent epsilon here ...
      if (lambda[i] < -1E-6) {
	found = false;
	const Tetrahedron* newT = (*__connectivity)(cellNumber)[i];
	if (newT != 0) {
	  const_iterator t1(*this, this->cellNumber(*newT));
  
	  if (visited.find(t1) == visited.end()) {
	    if (toVisitSet.find(t1) == toVisitSet.end()) {
	      toVisitSet.insert(t1);
	      toVisitStack.push(t1);
	    }
	  }
	}
      }
    }
  } while (not(found) and not(toVisitStack.empty()));

  if (not(found)) {
    c = const_iterator(*this,const_iterator::End);
  }
  return  c;
}

