#include "inotifytools/inotifytools.h"
#include "inotifytools/inotify.h"

#include <stdio.h>
#include <string.h>

#define test(A) _test(A, #A, __LINE__, __FILE__)
#define test_s(B,A) _test(A, B, __LINE__, __FILE__)
#define test_err(A) _test_err(A, #A, __LINE__, __FILE__)
#define test_err_s(B,A) _test_err(A, B, __LINE__, __FILE__)
#define test_ex(A,B) _test_ex(A, B, #A, __LINE__, __FILE__)
#define test_ex_s(C,A,B) _test_ex(A, B, C, __LINE__, __FILE__)


static int tests_failed;
static int tests_succeeded;

int _test( int cond, char const * teststr, int line, char const * file ) {
	if ( cond ) {
		++tests_succeeded;
		fprintf(stderr, "%s:%d Test succeeded\n", file, line);
		return cond;
	}

	++tests_failed;
	fprintf(stderr, "%s:%d Test failed: %s\n", file, line, teststr);
	return cond;
}

int _test_err( int cond, char const * teststr, int line, char const * file ) {
	if ( cond ) {
		++tests_succeeded;
		fprintf(stderr, "%s:%d Test succeeded\n", file, line);
		return cond;
	}

	++tests_failed;
	fprintf(stderr, "%s:%d Test failed: %s\n", file, line, teststr);
	fprintf(stderr, "%s:%d inotifytools_error() returns %s\n", file, line,
	        strerror( inotifytools_error() ) );
	return cond;
}

int _test_ex( int value, int expval, char const * teststr, int line,
              char const * file ) {
	if ( value == expval ) {
		++tests_succeeded;
		fprintf(stderr, "%s:%d Test succeeded\n", file, line);
		return 1;
	}

	++tests_failed;
	fprintf(stderr, "%s:%d Test failed: %s\n", file, line, teststr);
	fprintf(stderr,"%s:%d Expected value %d, got %d\n",file,line,expval, value);
	return 0;
}

int main() {
	tests_failed = 0;
	tests_succeeded = 0;

	if ( !test_s( "inotifytools_event_to_str",
	  !strcmp( inotifytools_event_to_str( IN_OPEN | IN_MODIFY | IN_ACCESS ),
	           "OPEN,MODIFY,ACCESS" ) ||
	  !strcmp( inotifytools_event_to_str( IN_OPEN | IN_MODIFY | IN_ACCESS ),
	           "OPEN,ACCESS,MODIFY" ) ||
	  !strcmp( inotifytools_event_to_str( IN_OPEN | IN_MODIFY | IN_ACCESS ),
	           "ACCESS,OPEN,MODIFY" ) ||
	  !strcmp( inotifytools_event_to_str( IN_OPEN | IN_MODIFY | IN_ACCESS ),
	           "MODIFY,OPEN,ACCESS" ) ||
	  !strcmp( inotifytools_event_to_str( IN_OPEN | IN_MODIFY | IN_ACCESS ),
	           "ACCESS,MODIFY,OPEN" ) ||
	  !strcmp( inotifytools_event_to_str( IN_OPEN | IN_MODIFY | IN_ACCESS ),
	           "MODIFY,ACCESS,OPEN" )
	) ) {
		fprintf(stderr, "got %s, expected OPEN,MODIFY,ACCESS or similar\n",
		inotifytools_event_to_str( IN_OPEN | IN_MODIFY | IN_ACCESS ) );
	}

	if ( !test_s( "inotifytools_event_to_str_sep",
	  !strcmp( inotifytools_event_to_str_sep(IN_OPEN|IN_MODIFY|IN_ACCESS,':'),
	           "OPEN:MODIFY:ACCESS" ) ||
	  !strcmp( inotifytools_event_to_str_sep(IN_OPEN|IN_MODIFY|IN_ACCESS,':'),
	           "OPEN:ACCESS:MODIFY" ) ||
	  !strcmp( inotifytools_event_to_str_sep(IN_OPEN|IN_MODIFY|IN_ACCESS,':'),
	           "ACCESS:OPEN:MODIFY" ) ||
	  !strcmp( inotifytools_event_to_str_sep(IN_OPEN|IN_MODIFY|IN_ACCESS,':'),
	           "MODIFY:OPEN:ACCESS" ) ||
	  !strcmp( inotifytools_event_to_str_sep(IN_OPEN|IN_MODIFY|IN_ACCESS,':'),
	           "ACCESS:MODIFY:OPEN" ) ||
	  !strcmp( inotifytools_event_to_str_sep(IN_OPEN|IN_MODIFY|IN_ACCESS,':'),
	           "MODIFY:ACCESS:OPEN" )
	) ) {
		fprintf(stderr, "got %s, expected OPEN:MODIFY:ACCESS or similar\n",
		inotifytools_event_to_str_sep( IN_OPEN | IN_MODIFY | IN_ACCESS, ':' ));
	}





	/* ====================================================================== */
	test_ex( inotifytools_str_to_event("open,modify,access"),
	         IN_OPEN | IN_MODIFY | IN_ACCESS );
	test_ex( inotifytools_str_to_event(",open,modify,access"), 0 );
	test_ex( inotifytools_str_to_event("open,modify,access,"), 0 );
	test_ex( inotifytools_str_to_event("open,modify,,access,close"), 0 );
	test_ex( inotifytools_str_to_event("open,mod,access,close"), -1 );
	test_ex( inotifytools_str_to_event("mod"), -1 );
	test_ex( inotifytools_str_to_event(","), 0 );
	test_ex( inotifytools_str_to_event(",,"), 0 );
	test_ex( inotifytools_str_to_event("open"), IN_OPEN );
	test_ex( inotifytools_str_to_event("close"), IN_CLOSE );
	test_ex( inotifytools_str_to_event(",close"), 0 );
	test_ex( inotifytools_str_to_event(",,close"), 0 );
	test_ex( inotifytools_str_to_event("close,"), 0 );
	test_ex( inotifytools_str_to_event("close,,"), 0 );







	/* ====================================================================== */
	test_ex( inotifytools_str_to_event_sep("open:modify:access", ':'),
	         IN_OPEN | IN_MODIFY | IN_ACCESS );
	test_ex( inotifytools_str_to_event_sep(":open:modify:access", ':'), 0 );
	test_ex( inotifytools_str_to_event_sep("open:modify:access:", ':'), 0 );
	test_ex( inotifytools_str_to_event_sep("open:modify::access:close", ':'),0);
	test_ex( inotifytools_str_to_event_sep("open:mod:access:close", ':'), -1 );
	test_ex( inotifytools_str_to_event_sep("mod", ':'), -1 );
	test_ex( inotifytools_str_to_event_sep(":", ':'), 0 );
	test_ex( inotifytools_str_to_event_sep("::", ':'), 0 );
	test_ex( inotifytools_str_to_event_sep("open", ':'), IN_OPEN );
	test_ex( inotifytools_str_to_event_sep("close", ':'), IN_CLOSE );
	test_ex( inotifytools_str_to_event_sep(":close", ':'), 0 );
	test_ex( inotifytools_str_to_event_sep("::close", ':'), 0 );
	test_ex( inotifytools_str_to_event_sep("close:", ':'), 0 );
	test_ex( inotifytools_str_to_event_sep("close::", ':'), 0 );
	test_ex( inotifytools_str_to_event_sep("open:modify:access", ','), -1 );
	test_ex( inotifytools_str_to_event_sep("open:modify:access", 'o'), -1 );



	test_err( inotifytools_initialize() );
	test_err( inotifytools_watch_file( "/", IN_CLOSE ) );
	test_ex( inotifytools_wd_from_filename( "/" ), 1 );
	test_ex( inotifytools_wd_from_filename( "foobar" ), -1 );
	test( !strcmp( inotifytools_filename_from_wd(1), "/" ) );
	test_err( inotifytools_remove_watch_by_filename( "/" ) );
	test_ex( inotifytools_wd_from_filename( "/" ), -1 );
	test_ex( inotifytools_filename_from_wd( 1 ), 0 );
	test_err( inotifytools_watch_file( "/", IN_CLOSE ) );
	test_ex( inotifytools_wd_from_filename( "/" ), 2 );
	test_ex( inotifytools_wd_from_filename( "foobar" ), -1 );
	test( !strcmp( inotifytools_filename_from_wd(2), "/" ) );
	test_err( inotifytools_remove_watch_by_wd( 2 ) );
	test_ex( inotifytools_wd_from_filename( "/" ), -1 );
	test_ex( inotifytools_filename_from_wd( 2 ), 0 );



	printf("Out of %d tests, %d succeeded and %d failed.\n",
	       tests_failed + tests_succeeded, tests_succeeded, tests_failed );

	if ( tests_failed ) return -tests_failed;
}
