/*********************************************************************
 * Filename:      intuitively.c
 * Version:       $Id: intuitively.c,v 1.29 2002/02/09 11:35:36 tfheen Exp $
 * Description:   
 * Author:        Tollef Fog Heen <tollef@add.no>
 * Created at:    Wed Oct 11 10:38:46 2000
 * Modified at:   Sat Feb  9 12:23:44 2002
 * Modified by:   Tollef Fog Heen <tollef@add.no>
 ********************************************************************/

#define _GNU_SOURCE

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include "intuitively.h"
#include <libnet.h>
#include <pcap.h>
#include "utility.h"
#include <net/route.h>
#include <sys/param.h>
#include <getopt.h>
#include <string.h>

int main (int argc, char** argv) {
  int i;
  struct entry *tmp;
  pcap_t* pd;

  e = malloc(sizeof(struct entry));
  bzero(&switches,sizeof(struct switches));
  switches.run_scripts = 1;

  parse_commandline(argc,argv,&switches);
  read_conffile(&switches);

  /* Print out the configuration - if verbose > 1 */

  tmp = e;
  if (switches.verbose > 1) {
    for (;tmp != NULL;tmp = tmp->next) {
      dump_entry_as_text(tmp);
    }
  }

  if (!switches.device) {
    switches.device = NULL;
  }

  init_libnet(&switches.device);

  if (switches.verbose >= 1) {
    fprintf(stderr, "Selected device: %s\n",switches.device);
  }
  pd = init_pcap(switches.device);

  if (switches.locationname) {
    tmp = e;
    while (tmp != NULL) {
      if ((strcasecmp(switches.locationname,tmp->description) == 0) ||
	  (strcasecmp(switches.locationname,tmp->name) == 0)) {
	setup_location(tmp);
      }
      tmp = tmp->next;
    } 
    /* We should never come here.. must be an invalid location */
    fprintf(stderr,"Invalid location specified.\n");
    exit(1);
  }

  for (i = 0; i < 6; i++) {
    struct host *answer;
    send_arps(switches.device,e);
    answer = get_answers(pd,e);

    if (answer != NULL) {
      struct entry *tmp=e; 

      while (tmp != NULL) {
	char *a = NULL, *b;
	int q;
	b = strdup(net_ntoa(answer->ip));
	if (tmp->dontsetip == 0) {
	  a = strdup(net_ntoa(tmp->myip));
	  if ((strncmp(a,b,MAX(strlen(a),strlen(b))) == 0) && (switches.verbose >= 1)) {
	    /* oops, ip already taken. bad, bad, bad */
	    fprintf(stderr,"Blacklisting unavailable IP %s\n",
		    net_ntoa(answer->ip));
	    tmp->pingips = 0;		/* blacklist */
	  }
	}
	for (q=0; q < tmp->pingips; q++) {
	  if (!q && a != NULL && tmp->dontsetip == 0) 
	    free(a);
	  a = strdup(net_ntoa(tmp->pingip[q].ip));
	  if (strncmp(a,b,MAX(strlen(a),strlen(b))) == 0) {
	    if (tmp->pingip[q].mac != NULL && answer->mac != NULL) {
	      if (strcasecmp(tmp->pingip[q].mac,answer->mac) == 0) {
		setup_location(tmp);
	      }
	    } else 
	      setup_location(tmp);
	  }
	}
	free(a);
	free(b);
	a = b = NULL;
	tmp = tmp->next;
      }
    }
    sleep(1);
  }
  /* search for default location */
  tmp = e;
  while (tmp != NULL) {
    if (tmp->defaultlocation == 1)
      setup_location(tmp);
    tmp = tmp->next;
  }
  
  if (switches.verbose >= 0) {
    fprintf(stderr, "Sorry, no answers received and no default location defined.\n");
  }
  
  return -1;
}

int parse_commandline(int argc,char **argv, struct switches* s) {
  int c;

  while (1) {
    int option_index = 0;
    static struct option long_options[] = {
      {"config", 1, 0, 'c'},
      {"force", 0, 0, 'f'},
      {"help", 0, 0, 'h'},
      {"interface", 1, 0, 'i'},
      {"location", 1, 0, 'l'},
      {"quiet", 0, 0, 'q'},
      {"usage", 0, 0, 'h'},
      {"verbose", 0, 0, 'v'},
      {"version", 0, 0, 'V'},
      {"no-script", 0, 0, 'S'},
      {"locationfile", 0, 0, 'r'},
      {"writeconfig", 0, 0, 'w'},
      {"dry-run", 0, 0, 'N'},
      {0, 0, 0, 0}};
    
    c = getopt_long (argc, argv, "c:fhi:l:qr:Vv?wSN",
		     long_options, &option_index);
    if (c == -1)
            break;

    switch (c) {
    case 'c':
      s->conffile = strdup(optarg);
      break;

    case 'f':
      s->force++;
      break;

    case 'S':
      s->run_scripts = 0;
      break;
      
    case '?':
    case 'h':
      fprintf(stderr,"intuitively:\n"
	  "Usage: intuitively [OPTION]\n"
	  "-c, --config            use alternate configuration file\n"
	  "-f, --force             force overwriting of non-symlinks when linking\n"
	  "-h, --help              this help screen\n"
	  "-i, --interface=IFACE   select interface\n"
	  "-l, --location=LOC      force location\n"
	  "-q, --quiet             be more quiet\n"
	  "-S, --no-scripts        don't run the script assosciated with a entry\n"
	  "-V, --version           display version and quit\n"
	  "-v, --verbose           be more verbose\n"
	  "-r, --locationfile=FILE write location into FILE\n"
	  "-w, --writeconfig       try to guess current configuration and print to stderr\n"
	      );

      exit(0);
      break;

    case 'i':
      s->device = strdup(optarg);
      break;
      
    case 'l':
      s->locationname = strdup(optarg);
      break;

    case 'N':
      s->dry_run = 1;
      break;

    case 'q':
      s->verbose--;
      break;

    case 'r':
      s->locationfile = strdup(optarg);
      break;

    case 'V':
      fprintf(stderr,"intuitively version " VERSION "\n");
      exit(0);
      break;

    case 'v':
      s->verbose++;
      break;

    case 'w':
      dump_entry_as_text(get_current_entry(s->device));
      exit(0);
      break;

    default:
      fprintf (stderr, "?? getopt returned character code 0%o ??\n", c);
    }
  }
  
  if (optind < argc) {
    fprintf (stderr, "Illegal arguments: ");
    while (optind < argc) {
      fprintf (stderr, "%s ", argv[optind++]);
    }
    fprintf(stderr, "\n");
    exit(1);
  }
  return 0;
};

/*
 * Function get_answers (pd, e)
 *
 *    Checks if there are any answers to our ARP Requests.  If so, it
 *    returns the a pointer to an in_addr struct.  The structure must
 *    be free'd by the client.
 *
 */
struct host* get_answers(pcap_t *pd, struct entry *e) {
  struct host *answer = malloc(sizeof(struct host));
  struct entry *tmp = e;
  struct sockaddr_storage* result = malloc(sizeof(struct sockaddr_storage));
  int i,f;

  result->ss_family = AF_INET;
  f = tmp->pingips;
  while (tmp->next != NULL) {
    tmp = tmp->next;
    f += tmp->pingips;
  }

  for (; f != 0; f--) {
    memset(answer,'\0',sizeof(struct host));
    i = pcap_dispatch(pd, 0, get_answer, (u_char *) answer);
    if (answer->ip.ss_family == AF_INET) {
      return answer;
    }
  }
  return NULL;
}

/*
 * Function get_answer (answer, pc, packet)
 *
 *    Answer is really a struct host.
 *
 */
void get_answer(u_char* answer, const struct pcap_pkthdr* pc, const u_char* packet) {
  struct libnet_ethernet_hdr *p;
  struct libnet_arp_hdr *a;
  struct host *ans = malloc(sizeof(struct host));

  p = (struct libnet_ethernet_hdr *)(packet);

  if (ntohs(p->ether_type) == ETHERTYPE_ARP) {
    a = (struct libnet_arp_hdr *)(packet + ETH_H);
    if (ntohs(a->ar_op) == ARPOP_REPLY) {
      struct sockaddr_storage target;

      if (ntohs(a->ar_hrd) == ARPHRD_ETHER) {
	ans->mac = ether_to_ascii(a->ar_sha);
      } else {
	ans->mac = NULL;
      }
      
      (((struct sockaddr_in *)&ans->ip)->sin_addr).s_addr = *((int*)&a->ar_spa);
      (((struct sockaddr_in *)&target)->sin_addr).s_addr = *((int*)&a->ar_tpa);

      ans->ip.ss_family = AF_INET;
      target.ss_family = AF_INET;

      if (switches.verbose >= 1)
	fprintf(stderr, "Got ARP reply from %s to %s (%s)\n", net_ntoa(ans->ip), 
	       net_ntoa(target), ans->mac);

      memcpy(answer,ans,sizeof(struct host));
      /*      strcpy(answer,net_ntoa(source));*/
    }
  }
}


/*
 * Function init_pcap (device)
 *
 *    Initializes libpcap.
 *
 */
pcap_t *init_pcap(unsigned char* device) {
  pcap_t* pd;
  char errbuf[PCAP_ERRBUF_SIZE];


  pd = pcap_open_live(device,LIBNET_ARP_H+LIBNET_ETH_H,0,20,errbuf);
  if (pd == NULL) {
    fprintf(stderr,"intuitively: pcap_open_live failed: %s\n", errbuf);
    exit(1);
  }
  return pd;
}

/*
 * Function init_libnet (device)
 *
 *    Initalizes libnet.
 *
 */
int init_libnet(char** device) {
  struct sockaddr_in sin;
  unsigned char errbuf[PCAP_ERRBUF_SIZE];
  char *dev;

  if (switches.verbose >= 3) {
    fprintf(stderr,"init_libnet(%s)\n",*device);
  }

  if (libnet_select_device(&sin,device,errbuf) == -1) {
    /* Couldn't automatically select device - this usually means that
       eth0 isn't up yet - bring it up and try again.  This is kind of
       a hack - but it shouldn't pose any problems to anybody.  The
       only problem which might arise is that, if the interface was
       configured already, that configuration is lost.
    */

    struct entry link_local;

    int i = switches.verbose;

    memset(&link_local,'\0',sizeof(struct entry));
    link_local.myip.ss_family = AF_INET;
    net_aton("169.254.5.174",&link_local.myip); /* Random value ;) */
    link_local.netmask.ss_family = AF_INET;
    net_aton("255.255.0.0",&link_local.netmask);
    link_local.gw.ss_family = AF_INET;
    net_aton("0.0.0.0",&link_local.gw);
    
    link_local.broadcast.ss_family = AF_INET;
    net_aton("169.254.255.255",&link_local.broadcast);

    switches.verbose = -100;
    if (*device) {
      dev = strdup(*device);
    } else {
      dev = strdup("eth0");
    }
    configure_iface(&link_local,dev);
    free(dev);
    switches.verbose = i;
    if (libnet_select_device(&sin,device,errbuf) == -1) {
      
      fprintf(stderr,"intuitively: libnet_select_device failed: %s\n", errbuf);
      exit(-1);
    }
  }
  return 0;
}

void read_conffile(struct switches* s) {
  FILE *conffile;
  struct entry *e_begin = e;
  if (s->conffile == NULL) {
    conffile = fopen(INTUITIVELY_CONFFILE,"r");
  } else {
    conffile = fopen(s->conffile,"r");
  }
  if (!conffile) {
    perror("cannot open conffile");
    exit(1);
  }
  yyin = conffile;
  yyparse();
  e = e_begin;
}

/*
 * Function send_arps (device, e_list)
 *
 *    First, sends out arps to myself, it then sends out all the arp
 *    requests, using sendarp(...).
 *
 */
void send_arps(unsigned char* device, struct entry* e_list) {
  struct entry *tmp=e_list;
  struct sockaddr_storage *null_addr = malloc(sizeof(struct sockaddr_storage));

  while (tmp) {

    if (tmp->myip.ss_family == AF_INET) {
      
      null_addr->ss_family = AF_INET;
      net_aton("0.0.0.0",null_addr);
      sendarp(null_addr,&tmp->myip,device);
    } else {
      if (switches.verbose >= 0) {
	fprintf(stderr,"Unsupported network protocol");
      }
    } 
    tmp=tmp->next;
    
  }

  tmp=e_list;
  while (tmp) {
    int i;
    for (i=0; i<tmp->pingips; i++)
      sendarp(null_addr,&tmp->pingip[i].ip,device);
    tmp=tmp->next;
  }
}

/*
 * Function sendarp (myip, hisip, device)
 *
 *    Handles the actual sending of the ARP requests.
 *
 */
void sendarp(struct sockaddr_storage *myip, struct sockaddr_storage *hisip, unsigned char* device) {
  static int initialized=0;
  static struct ether_addr *my_ether;
  static struct libnet_link_int *l;
  static u_char enet_bcast[6] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};
  static u_char enet_nulls[6] = {0,0,0,0,0,0};
  static char *buf;
  char errbuf[PCAP_ERRBUF_SIZE];

  if (!initialized) {
    buf = malloc(LIBNET_ARP_H + LIBNET_ETH_H);
    l = libnet_open_link_interface(device, errbuf);
    if (!l) {
      fprintf(stderr,"sendarp::libnetopen_link_interface failed: %s\n", errbuf);
    }
    my_ether = libnet_get_hwaddr(l,device,errbuf);
    if (!my_ether) {
      fprintf(stderr,"libnet_get_hwaddr failed: %s\n", errbuf);
    }
    initialized = 1;
  }

  switch(myip->ss_family) {
  case AF_INET:

  /*
    *  Ethernet header
    */
  libnet_build_ethernet(
	  enet_bcast,             /* broadcast ethernet address */
	  my_ether->ether_addr_octet,    /* source ethernet address */
	  ETHERTYPE_ARP,          /* this frame holds an ARP packet */
	  NULL,                   /* payload */
	  0,                      /* payload size */
	  buf);                   /* packet header memory */

  /*
    *  ARP header
    */
  libnet_build_arp(
	  ARPHRD_ETHER,           /* hardware address type */
	  ETHERTYPE_IP,           /* protocol address type */
	  ETHER_ADDR_LEN,         /* hardware address length */
	  4,                      /* protocol address length */
	  ARPOP_REQUEST,          /* packet type - ARP request */
	  my_ether->ether_addr_octet,    /* source (local) ethernet address */
	  (u_char *)&((struct sockaddr_in *)myip)->sin_addr,        
	                          /* source (local) IP address */
	  enet_nulls,             /* target's ethernet address (broadcast) */
	  (u_char *)&((struct sockaddr_in *)hisip)->sin_addr,       /* target's IP address */
	  NULL,                   /* payload */
	  0,                      /* payload size */
	  buf + ETH_H);           /* packet header memory */

  if (libnet_write_link_layer(l, device, buf, LIBNET_ARP_H + LIBNET_ETH_H) == -1) {
      fprintf(stderr,"libnet_write_link_layer failed\n");
  }
  break;
  default: 
    fprintf(stderr,"sendarp::Unknown address type\n");
  }
}
