/*                                                                     
 *                                                                     
 * Detailed logging daemon for ipac-ng, now works with QUEUE target in 
 * 	iptables
 * Copyright (C) 2001 Al Zaharov
 *                                                                     
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or   
 * (at your option) any later version.                                 
 *                                                                     
 * This program is distributed in the hope that it will be useful,     
 * but WITHOUT ANY WARRANTY; without even the implied warranty of      
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the       
 * GNU General Public License for more details.                        
 *                                                                     
 * You should have received a copy of the GNU General Public License   
 * along with this program; if not, write to the Free Software         
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.           
 *                                                                     
 * The author can be reached via email: kaiser13@mail2000.ru, or by    
 * fido: Al_Zaharov@p88.f58.n5005.z2.fidonet.org                       
 *
 */ 
 
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <signal.h>


#include <netinet/in.h>
#include <netinet/ip.h>
#include <linux/netfilter.h>
#include <linux/netfilter_ipv4.h>
#include <netinet/udp.h>
#include <netinet/tcp.h>

#ifndef IFNAMSIZ
#define IFNAMSIZ        16
#endif

#ifndef PROC_SYS_MODPROBE
#define PROC_SYS_MODPROBE "/proc/sys/kernel/modprobe"
#endif

#include <linux/netfilter_ipv4/ip_tables.h>

#include "libipq.h"

#include <libpq-fe.h>

#include "config.h"
#include "ipac.h"

#define BUFSIZE 2048
#define IP_OFFSET       0x1FFF          /* "Fragment Offset" part       */

// number of packets to collect before flush
#define PACKETS_TO_COLLECT 200
// timeout in microseconds waiting for next packet
#define IPQ_TIMEOUT 30000

static char *pghost = NULL;   
static char *pgport = NULL;   
static char *pgoptions = NULL;
static char *pgtty = NULL;    
static char *dbname = "ipac"; 

static PGconn *conn;          
static PGresult *res;         

static int pkt_num = 0;

void usage(void)
{
	printf("ipac-ng daemon v0.17\n");
	printf("usage:\n");
	printf ("\t-d\tstart daemon\n"
		"\t-q\tstop daemon\n"
		"\t-f\tflush queue\n");
	exit(0);
}

void stopit (int rc)
{
        exit(rc);
}

void sigerr(int sig)                                                            
{                                                                               
//        char *sigs[]={"","HUP","INT","QUIT","ILL","TRAP","IOT","BUS","FPE",     
//                          "KILL","USR1","SEGV","USR2","PIPE","ALRM","TERM"};
	signal(sig, SIG_DFL);
        switch(sig) {
    		case SIGSEGV:   
	        case SIGFPE:    
		case SIGBUS:    
		        abort();
		default:        
		        exit(1);
	}               
}

void sigchild(int sig)            
{                                 
        int rc, wr;               
        signal(SIGCHLD, sigchild);
        wr=wait(&rc);
        if(wr<0);// log("wait() returned %d (%d)", wr, rc);
//        rc=WEXITSTATUS(rc)&S_MASK;                      
//        if(rc==S_OK || rc==S_REDIAL) {                  
//                do_rescan=1;                            
//        }                                               	
}

void 
sighup(int sig)           
{                              
	pkt_num = PACKETS_TO_COLLECT;
//        signal(SIGHUP, sighup);
}

int
open_db(void)
{
	conn=PQsetdb(pghost, pgport, pgoptions, pgtty, dbname);
	if(PQstatus(conn) == CONNECTION_BAD) {
	        fprintf(stderr, "Connection to database '%s' failed.\n", dbname);
		fprintf(stderr, "%s", PQerrorMessage(conn));
		PQfinish(conn);
                return(1);
        }
	if (PQsetnonblocking(conn, 1)) {
		fprintf(stderr, "Can't set nonblocking mode for database "
			    "connection\n");
		return(1);
	}
        return(0);
}

int
dump_iphdr(unsigned char *packet, char *buf)
{
        struct iphdr *iph = (struct iphdr *)packet;
        void *protoh = (u_int32_t *)iph + iph->ihl;
	sprintf(buf+strlen(buf), "insert into raw_logs (source, destination, "
		    "len, proto, sport, dport, time) values ('%u.%u.%u.%u', "
		    "'%u.%u.%u.%u', '%u', ", 
            	    (ntohl(iph->saddr) >> 24) & 0xFF,
		    (ntohl(iph->saddr) >> 16) & 0xFF,
		    (ntohl(iph->saddr) >> 8) & 0xFF,
		    (ntohl(iph->saddr)) & 0xFF,
		    (ntohl(iph->daddr) >> 24) & 0xFF,
		    (ntohl(iph->daddr) >> 16) & 0xFF,
		    (ntohl(iph->daddr) >> 8) & 0xFF,
		    (ntohl(iph->daddr)) & 0xFF,
		    ntohs(iph->tot_len));
	switch (iph->protocol) {
		case IPPROTO_TCP: {
                        struct tcphdr *tcph = protoh;
			
                        sprintf(buf+strlen(buf), "'TCP', ");
                        if (ntohs(iph->frag_off) & IP_OFFSET) {
				sprintf(buf+strlen(buf), "'0', '0', ");
	                        break;
			}
			sprintf(buf+strlen(buf), "'%u', '%u', ",
                                   ntohs(tcph->source), ntohs(tcph->dest));
			break;
		}
		case IPPROTO_UDP: {
                        struct udphdr *udph = protoh;
			
                        sprintf(buf+strlen(buf), "'UDP', ");
                        if (ntohs(iph->frag_off) & IP_OFFSET) {
				sprintf(buf+strlen(buf), "'0', '0', ");
	                        break;
			}
			sprintf(buf+strlen(buf), "'%u', '%u', ",
                                   ntohs(udph->source), ntohs(udph->dest));
			break;
		}
		case IPPROTO_ICMP: {
			sprintf(buf+strlen(buf), "'ICMP', '0', '0', ");
			break;
		}
		default: 
			sprintf(buf+strlen(buf), "'%u', '0', '0', ", iph->protocol);
	}
	sprintf(buf+strlen(buf), "'%lu');", time(NULL));
	return 0;
}

static char
*get_modprobe(void)
{
        int procfile;
        char *ret;

        procfile = open(PROC_SYS_MODPROBE, O_RDONLY);
        if (procfile < 0)
                return NULL;

        ret = malloc(1024);
        if (ret) {
                switch (read(procfile, ret, 1024)) {
	                case -1: goto fail;
	                case 1024: goto fail; /* Partial read.  Weird */
                }
                if (ret[strlen(ret)-1]=='\n')
                        ret[strlen(ret)-1]=0;
                close(procfile);
                return ret;
        }
 fail:
        free(ret);
        close(procfile);
        return NULL;
}

static int
ipacd_insmod(const char *modname, const char *modprobe)
{
        char *buf = NULL;
        char *argv[3];

	//       If they don't explicitly set it, read out of kernel
        if (!modprobe) {
                buf = get_modprobe();
                if (!buf)
                        return -1;
                modprobe = buf;
        }

        switch (fork()) {
	        case 0:
	                argv[0] = (char *)modprobe;
	                argv[1] = (char *)modname;
	                argv[2] = NULL;
	                execv(argv[0], argv);

	//               not usually reached
	                exit(0);
	        case -1:
	                return -1;
			
	        default: // parent
	                wait(NULL);
        }

        free(buf);
        return 0;
}

void
daemon_mode(void)
{
	unsigned char *buf;
	int rc;
	struct ipq_handle *h;
        int status;
	unsigned char inbuf[BUFSIZE];
        int verdict = NF_ACCEPT;
        const char *modprobe = NULL;

        if (getppid()!=1) {
                signal(SIGTTOU, SIG_IGN);
                signal(SIGTTIN, SIG_IGN);
                signal(SIGTSTP, SIG_IGN);
                if((rc=fork())>0)
                        exit(0);
                if(rc<0) {
                        fprintf(stderr, "can't spawn daemon!\n");
                        exit(1);
                }
                setsid();
        }
        signal(SIGINT, sigerr);
        signal(SIGTERM, sigerr);
        signal(SIGFPE, sigerr);
        signal(SIGCHLD, sigchild);
        signal(SIGHUP, sighup);
        if (lock(PIDFILE))
                stopit(1);
	if (open_db())
		stopit(1);
        h = ipq_create_handle(0, PF_INET);
        if (!h) {
		ipacd_insmod("ip_queue", modprobe);
		h = ipq_create_handle(0, PF_INET);
		if (!h) {
			fprintf(stderr, "Can't create ip_queue handle\n"
				    "maybe \"modprobe ip_queue\" will help\n");
			stopit(1);
		}
	}
	/* copy only first 24 bytes - it will be enought */
        status = ipq_set_mode(h, IPQ_COPY_PACKET, 24);
        if (status < 0) {
		fprintf(stderr, "Cant setup ip_queue in proper manner\n"
				    "maybe \"modprobe ip_queue\" will help\n");
		stopit(1);
	}
	buf=xcalloc(300, PACKETS_TO_COLLECT);
	while (1) {
		sprintf(buf, "begin; ");
		/* collect 200 packets */
		for (pkt_num=0; pkt_num<PACKETS_TO_COLLECT; pkt_num++) {
			status = ipq_read(h, inbuf, BUFSIZE, IPQ_TIMEOUT);
			if (status < 0) {
				fprintf(stderr, "Error reading queue\n");
				stopit(1);
			}
			if (status==0)	// got timeout from ipq_read
				continue;
			switch (ipq_message_type(inbuf)) {
	                        case NLMSG_ERROR:
                            		fprintf(stderr, "Received error message"
						    " %d\n", ipq_get_msgerr(inbuf));
					fprintf(stderr, "%s\n", 
								ipq_errstr());
    		                        break;
	                        case IPQM_PACKET: {
	                                ipq_packet_msg_t *m = ipq_get_packet(inbuf);
	                                unsigned char *packet = NULL;

					if (!m)
						break;
					status = ipq_set_verdict(h, 
					    m->packet_id, verdict, 0, NULL);
					if (m->data_len>=sizeof(struct iphdr)) {
						packet=(unsigned char *) m
								+ sizeof(*m);
    						dump_iphdr(packet, buf);
					}
					if (status<0) {
						fprintf(stderr, "Can't send verdict\n");
						stopit(0);
					}
					break;
				}
				default:
					fprintf(stderr, "Got unknown message from queue\n");
					break;
			}
			if (!PQisBusy(conn)) {
				res=PQgetResult(conn);
				PQclear(res);
			}
			
		}
		sprintf(buf+strlen(buf), "commit;");
		while ((res=PQgetResult(conn))) PQclear(res);
		if (!PQsendQuery(conn, buf)) {
			fprintf(stderr, "Some error with PQsendQuery occured\n");
			stopit(1);
		}
		buf[0]='\0';
	}
}

void killdaemon(int sig)                                                      
{                                                                             
        FILE *f=fopen(PIDFILE, "rt");
        pid_t pid;                                                            

        if(!f) {
                fprintf(stderr, "can't open pid file - no daemon killed\n");
                return;
        }
        fscanf(f, "%d", &pid);fclose(f);
        if(kill(pid, sig))
                fprintf(stderr, "can't send signal\n");
//        else
//                fprintf(stderr, "daemon succesfully stopped\n");
	if (sig != SIGHUP)
		unlock(PIDFILE);
}

int main(int argc, char *argv[], char *envp[])
{
	int c, daemon = -1;
	
        while((c=getopt(argc, argv, "dqf"))!=EOF) {
                switch(c) {	
			case 'd':
				daemon=1;
				break;
			case 'q':
				daemon=2;
				break;
			case 'f':
				daemon=3;
		}
	}
	if (daemon<0) usage();
	switch(daemon) {
		case 3:
			killdaemon(SIGHUP);
			exit(0);
		case 2:
			killdaemon(SIGTERM);
			exit(0);
		case 1:
			daemon_mode();
			exit(0);
	}
	return(0);
}
