// --------------------------------------------------------------------
// Ipelet to insert an image
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipelib.h"

#include <qfiledialog.h>
#include <qimage.h>
#include <qmessagebox.h>
#include <qfile.h>
#include <qclipboard.h>
#include <qapplication.h>

// --------------------------------------------------------------------

/*
 JPG reading code
 Copyright (c) 1996-2002 Han The Thanh, <thanh@pdftex.org>

 This code is part of pdfTeX.

 pdfTeX is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.
*/

#define JPG_GRAY  1     /* Gray color space, use /DeviceGray  */
#define JPG_RGB   3     /* RGB color space, use /DeviceRGB    */
#define JPG_CMYK  4     /* CMYK color space, use /DeviceCMYK  */

enum JPEG_MARKER {      /* JPEG marker codes                    */
  M_SOF0  = 0xc0,       /* baseline DCT                         */
  M_SOF1  = 0xc1,       /* extended sequential DCT              */
  M_SOF2  = 0xc2,       /* progressive DCT                      */
  M_SOF3  = 0xc3,       /* lossless (sequential)                */

  M_SOF5  = 0xc5,       /* differential sequential DCT          */
  M_SOF6  = 0xc6,       /* differential progressive DCT         */
  M_SOF7  = 0xc7,       /* differential lossless                */

  M_JPG   = 0xc8,       /* JPEG extensions                      */
  M_SOF9  = 0xc9,       /* extended sequential DCT              */
  M_SOF10 = 0xca,       /* progressive DCT                      */
  M_SOF11 = 0xcb,       /* lossless (sequential)                */

  M_SOF13 = 0xcd,       /* differential sequential DCT          */
  M_SOF14 = 0xce,       /* differential progressive DCT         */
  M_SOF15 = 0xcf,       /* differential lossless                */

  M_DHT   = 0xc4,       /* define Huffman tables                */

  M_DAC   = 0xcc,       /* define arithmetic conditioning table */

  M_RST0  = 0xd0,       /* restart                              */
  M_RST1  = 0xd1,       /* restart                              */
  M_RST2  = 0xd2,       /* restart                              */
  M_RST3  = 0xd3,       /* restart                              */
  M_RST4  = 0xd4,       /* restart                              */
  M_RST5  = 0xd5,       /* restart                              */
  M_RST6  = 0xd6,       /* restart                              */
  M_RST7  = 0xd7,       /* restart                              */

  M_SOI   = 0xd8,       /* start of image                       */
  M_EOI   = 0xd9,       /* end of image                         */
  M_SOS   = 0xda,       /* start of scan                        */
  M_DQT   = 0xdb,       /* define quantization tables           */
  M_DNL   = 0xdc,       /* define number of lines               */
  M_DRI   = 0xdd,       /* define restart interval              */
  M_DHP   = 0xde,       /* define hierarchical progression      */
  M_EXP   = 0xdf,       /* expand reference image(s)            */

  M_APP0  = 0xe0,       /* application marker, used for JFIF    */
  M_APP1  = 0xe1,       /* application marker                   */
  M_APP2  = 0xe2,       /* application marker                   */
  M_APP3  = 0xe3,       /* application marker                   */
  M_APP4  = 0xe4,       /* application marker                   */
  M_APP5  = 0xe5,       /* application marker                   */
  M_APP6  = 0xe6,       /* application marker                   */
  M_APP7  = 0xe7,       /* application marker                   */
  M_APP8  = 0xe8,       /* application marker                   */
  M_APP9  = 0xe9,       /* application marker                   */
  M_APP10 = 0xea,       /* application marker                   */
  M_APP11 = 0xeb,       /* application marker                   */
  M_APP12 = 0xec,       /* application marker                   */
  M_APP13 = 0xed,       /* application marker                   */
  M_APP14 = 0xee,       /* application marker, used by Adobe    */
  M_APP15 = 0xef,       /* application marker                   */

  M_JPG0  = 0xf0,       /* reserved for JPEG extensions         */
  M_JPG13 = 0xfd,       /* reserved for JPEG extensions         */
  M_COM   = 0xfe,       /* comment                              */

  M_TEM   = 0x01,       /* temporary use                        */

  M_ERROR = 0x100       /* dummy marker, internal use only      */
};

inline int read2bytes(QFile &f)
{
  int c = f.getch();
  return (c << 8) + f.getch();
}

// --------------------------------------------------------------------

class ImageIpelet : public Ipelet {
public:
  virtual int IpelibVersion() const { return IPELIB_VERSION; }
  virtual int NumFunctions() const { return 3; }
  virtual const char *SubLabel(int function) const;
  virtual const char *Label() const { return "Insert image"; }
  virtual void Run(int, IpePage *page, IpeletHelper *helper);
private:
  void InsertBitmap(IpePage *page, IpeletHelper *helper, QString name);
  void InsertJpeg(IpePage *page, IpeletHelper *helper, QString name);
  void Fail(QString msg);
  bool ReadJpegInfo(QFile &file);
  IpeRect ComputeRect(IpeletHelper *helper);
private:
  int iWidth;
  int iHeight;
  IpeBitmap::TColorSpace iColorSpace;
  int iBitsPerComponent;
  IpeVector iDotsPerInch;
};

// --------------------------------------------------------------------

const char * const sublabel[] = {
  "Insert JPEG image",
  "Insert bitmap image",
  "Insert bitmap from clipboard"
};

const char *ImageIpelet::SubLabel(int function) const
{
  return sublabel[function];
}

void ImageIpelet::Run(int fn, IpePage *page, IpeletHelper *helper)
{
  QString name;
  if (fn != 2) {
    name = QFileDialog::getOpenFileName();
    if (name.isNull())
      return;
  }
  switch (fn) {
  case 0: // JPEG
    InsertJpeg(page, helper, name);
    break;
  case 1: // bitmap
    InsertBitmap(page, helper, name);
    break;
  case 2: // bitmap from clipboard
#ifdef WIN32
    Fail("This function is not supported on Windows");
#else
    InsertBitmap(page, helper, QString::null);
#endif
    break;
  default:
    break;
  }
}

void ImageIpelet::Fail(QString msg)
{
  QMessageBox::information(0, "Insert image ipelet",
			   "<qt>" + msg + "</qt>", "Dismiss");
}

// --------------------------------------------------------------------

IpeRect ImageIpelet::ComputeRect(IpeletHelper *helper)
{
  IpeRect media = helper->Document()->Properties().iMedia;

  double dx = (iWidth * 72.0) / iDotsPerInch.iX;
  double dy = (iHeight * 72.0) / iDotsPerInch.iY;

  double xfactor = 1.0;
  if (dx > media.Width())
    xfactor = media.Width() / dx;
  double yfactor = 1.0;
  if (dy > media.Height())
    yfactor = media.Height() / dy;
  double factor = (xfactor < yfactor) ? xfactor : yfactor;
  IpeRect rect(IpeVector::Zero, factor * IpeVector(dx, dy));
  IpeVector v = 0.5 * IpeVector((media.Min().iX + media.Max().iX) -
				(rect.Min().iX + rect.Max().iX),
				(media.Min().iY + media.Max().iY) -
				(rect.Min().iY + rect.Max().iY));
  return IpeRect(rect.Min() + v, rect.Max() + v);
}

// --------------------------------------------------------------------

// 72 points per inch
const double InchPerMeter = (1000.0 / 25.4);

void ImageIpelet::InsertBitmap(IpePage *page, IpeletHelper *helper,
			       QString name)
{
  qDebug("InsertBitmap");
  QImage im;
  if (name.isNull()) {
    QClipboard *cb = QApplication::clipboard();
    qDebug("about to retrieve image");
    im = cb->image();
    qDebug("image retrieved %d", im.width());
    if (im.isNull()) {
      Fail("The clipboard contains no image, or perhaps\n"
	   "an image in a format not supported by Qt.");
      return;
    }
  } else {
    if (!im.load(name)) {
      Fail("The image could not be loaded.\n"
	   "Perhaps the format is not supported by Qt.");
      return;
    }
  }
  QImage im1 = im.convertDepth(32);
  iWidth = im1.width();
  iHeight = im1.height();
  iDotsPerInch = IpeVector(72, 72);

  if (im1.dotsPerMeterX())
    iDotsPerInch.iX = double(im1.dotsPerMeterX()) / InchPerMeter;
  if (im1.dotsPerMeterY())
    iDotsPerInch.iY = double(im1.dotsPerMeterY()) / InchPerMeter;

  bool isGray = im1.allGray();
  iColorSpace = isGray ? IpeBitmap::EDeviceGray : IpeBitmap::EDeviceRGB;
  int datalen = iWidth * iHeight * (isGray ? 1 : 3);
  IpeBuffer data(datalen);
  char *d = data.data();
  for (int y = 0; y < iHeight; ++y) {
    uint *p = (uint *) im1.scanLine(y);
    for (int x = 0; x < iWidth; ++x) {
      if (isGray) {
	*d++ = qRed(*p++);
      } else {
	*d++ = qRed(*p);
	*d++ = qGreen(*p);
	*d++ = qBlue(*p++);
      }
    }
  }

  IpeBitmap bitmap(iWidth, iHeight, iColorSpace, 8, data,
		   IpeBitmap::EDirect, true);
  IpeImage *obj = new IpeImage(ComputeRect(helper), bitmap);
  page->push_back(IpePgObject(IpePgObject::ESecondary,
			      helper->CurrentLayer(), obj));
}

// --------------------------------------------------------------------

bool ImageIpelet::ReadJpegInfo(QFile &file)
{
  static char jpg_id[] = "JFIF";
  int units;
  int xres;
  int yres;
  int fpos;
  iDotsPerInch = IpeVector(72.0, 72.0);

  file.at(0);
  if (read2bytes(file) != 0xFFD8) {
    Fail("The file does not appear to be a JPEG image");
    return false;
  }
  if (read2bytes(file) == 0xFFE0) { /* JFIF APP0 */
    read2bytes(file);
    for (int i = 0; i < 5; i++) {
      if (file.getch() != jpg_id[i]) {
	Fail("Reading JPEG image failed");
	return false;
      }
    }
    read2bytes(file);
    units = file.getch();
    xres = read2bytes(file);
    yres = read2bytes(file);
    switch (units) {
    case 1: /* pixels per inch */
      iDotsPerInch = IpeVector(xres, yres);
      break;
    case 2: /* pixels per cm */
      iDotsPerInch = IpeVector(xres * 2.54, yres * 2.54);
      break;
    default:
      break;
    }
  }

  file.at(0);
  while(1) {
    if (file.atEnd() || file.getch() != 0xFF) {
      Fail("Reading JPEG image failed");
      return false;
    }
    switch (file.getch()) {
    case M_SOF5:
    case M_SOF6:
    case M_SOF7:
    case M_SOF9:
    case M_SOF10:
    case M_SOF11:
    case M_SOF13:
    case M_SOF14:
    case M_SOF15:
      Fail("Unsupported type of JPEG compression");
      return false;
    case M_SOF2:
      //if (cfgpar(cfgpdf12compliantcode) > 0)
      //pdftex_fail("cannot use progressive DCT with PDF-1.2");
    case M_SOF0:
    case M_SOF1:
    case M_SOF3:
      read2bytes(file);    /* read segment length  */
      iBitsPerComponent = file.getch();
      iHeight = read2bytes(file);
      iWidth = read2bytes(file);
      switch (file.getch()) {
      case JPG_GRAY:
	iColorSpace = IpeBitmap::EDeviceGray;
        break;
      case JPG_RGB:
	iColorSpace = IpeBitmap::EDeviceRGB;
        break;
      case JPG_CMYK:
	iColorSpace = IpeBitmap::EDeviceCMYK;
        // pdf_puts("/DeviceCMYK\n/Decode [1 0 1 0 1 0 1 0]\n");
	break;
      default:
	Fail("Unsupported color space in JPEG image");
	return false;
      }
      file.at(0);
      return true;
    case M_SOI:        /* ignore markers without parameters */
    case M_EOI:
    case M_TEM:
    case M_RST0:
    case M_RST1:
    case M_RST2:
    case M_RST3:
    case M_RST4:
    case M_RST5:
    case M_RST6:
    case M_RST7:
      break;
    default:           /* skip variable length markers */
      fpos = file.at();
      file.at(fpos + read2bytes(file));
      break;
    }
  }
}

void ImageIpelet::InsertJpeg(IpePage *page, IpeletHelper *helper,
			     QString name)
{
  QFile file(name);
  if (!file.open(IO_ReadOnly)) {
    Fail(QString("Could not open file '%1'").arg(name));
    return;
  }
  if (!ReadJpegInfo(file))
    return;

  QByteArray a = file.readAll();
  file.close();

  IpeBitmap bitmap(iWidth, iHeight, iColorSpace,
		   iBitsPerComponent, IpeBuffer(a.data(), a.size()),
		   IpeBitmap::EDCTDecode);
  IpeImage *obj = new IpeImage(ComputeRect(helper), bitmap);
  page->push_back(IpePgObject(IpePgObject::ESecondary,
			      helper->CurrentLayer(), obj));
}

// --------------------------------------------------------------------

IPELET_DECLARE Ipelet *NewIpelet()
{
  return new ImageIpelet;
}

// --------------------------------------------------------------------
